'use client';

import React, { useState, useEffect, useCallback, useMemo } from 'react';
import {
  useReactTable,
  getCoreRowModel,
  getSortedRowModel,
  getFilteredRowModel,
  getPaginationRowModel,
  flexRender,
  createColumnHelper,
  type SortingState,
  type ColumnFiltersState,
  type VisibilityState,
  type RowSelectionState,
} from '@tanstack/react-table';
import { useAuth } from '@/hooks/useAuth';
import { getAuthHeaders } from '@/utils/api';
import { DocumentPreviewModal } from '@/components/DocumentPreviewModal';
import {
  Files,
  Upload,
  Download,
  Trash2,
  Eye,
  Search,
  Filter,
  RefreshCw,
  FileText,
  Archive,
  Tag,
  Calendar,
  User,
  CheckCircle,
  Clock,
  AlertCircle,
  XCircle,
  X,
  FolderOpen,
  Folder,
  Plus,
  ChevronRight,
  ChevronDown,
  Users,
  UserCheck,
  ChevronLeft
} from 'lucide-react';
import { Button } from "@/components/ui/button";

interface ChatDocument {
  id: number;
  filename: string;
  filepath: string;
  filesize: number;
  mimetype: string;
  user_id: string;
  upload_status: 'uploaded' | 'processing' | 'completed' | 'failed';
  embedding_status: 'pending' | 'processing' | 'completed' | 'failed';
  project_name?: string;
  tags?: string[];
  metadata?: Record<string, any>;
  error_message?: string;
  processed_at?: string;
  created_at: string;
  updated_at: string;
}

interface DocumentFilters {
  search: string;
  embeddingStatus: string;
  projectName: string;
  fileType: string;
  userId?: string;
  folderId?: string;
}

interface Project {
  id: number;
  name: string;
  description?: string;
  user_id: string;
  document_count: number;
  created_at: string;
}

interface FolderStructure {
  userId: string;
  username: string;
  documentCount: number;
  projects: Project[];
}

interface TreeNode {
  id: string;
  type: 'user' | 'project' | 'all';
  name: string;
  userId?: string;
  projectId?: string | number;
  documentCount: number;
  children?: TreeNode[];
  expanded?: boolean;
}

interface FileManagementModalProps {
  isOpen: boolean;
  onClose: () => void;
  userId?: string;
  userRole?: string;
}

export function FileManagementModal({ isOpen, onClose, userId, userRole }: FileManagementModalProps) {
  const { user } = useAuth();
  const [documents, setDocuments] = useState<ChatDocument[]>([]);
  const [loading, setLoading] = useState(true);
  // TanStack Table 상태들
  const [sorting, setSorting] = useState<SortingState>([]);
  const [columnFilters, setColumnFilters] = useState<ColumnFiltersState>([]);
  const [columnVisibility, setColumnVisibility] = useState<VisibilityState>({});
  const [rowSelection, setRowSelection] = useState<RowSelectionState>({});
  const [globalFilter, setGlobalFilter] = useState('');
  
  // 기존 상태들 (호환성 유지)
  const [selectedDocuments, setSelectedDocuments] = useState<Set<number>>(new Set());
  const [filters, setFilters] = useState<DocumentFilters>({
    search: '',
    embeddingStatus: '',
    projectName: '',
    fileType: ''
  });
  const [showFilters, setShowFilters] = useState(false);
  const [showUploadModal, setShowUploadModal] = useState(false);
  const [uploadProgress, setUploadProgress] = useState(0);
  const [isUploading, setIsUploading] = useState(false);

  // 문서 미리보기 상태
  const [showDocPreview, setShowDocPreview] = useState(false);
  const [previewFileName, setPreviewFileName] = useState<string>('');
  const [treeData, setTreeData] = useState<TreeNode[]>([]);
  const [selectedNode, setSelectedNode] = useState<TreeNode | null>(null);
  const [projects, setProjects] = useState<Project[]>([]);
  const [showCreateProjectModal, setShowCreateProjectModal] = useState(false);
  const [expandedNodes, setExpandedNodes] = useState<Set<string>>(new Set(['all']));
  const [activeTab, setActiveTab] = useState<'documents' | 'projects'>('documents');

  // 페이지네이션 상태
  const [currentPage, setCurrentPage] = useState(1);
  const [itemsPerPage, setItemsPerPage] = useState(20);
  const [totalItems, setTotalItems] = useState(0);

  // 정렬 상태
  const [sortBy, setSortBy] = useState<keyof ChatDocument>('created_at');
  const [sortOrder, setSortOrder] = useState<'asc' | 'desc'>('desc');

  // 알림 모달 상태
  const [showAlert, setShowAlert] = useState(false);
  const [alertMessage, setAlertMessage] = useState('');
  const [alertType, setAlertType] = useState<'success' | 'error' | 'warning' | 'info'>('info');

  // 확인 모달 상태
  const [showConfirm, setShowConfirm] = useState(false);
  const [confirmMessage, setConfirmMessage] = useState('');
  const [confirmCallback, setConfirmCallback] = useState<(() => void) | null>(null);

  // 알림 표시 함수
  const showAlertModal = (message: string, type: 'success' | 'error' | 'warning' | 'info' = 'info') => {
    setAlertMessage(message);
    setAlertType(type);
    setShowAlert(true);
  };

  // 확인 표시 함수
  const showConfirmModal = (message: string, callback: () => void) => {
    setConfirmMessage(message);
    setConfirmCallback(() => callback);
    setShowConfirm(true);
  };

  // 트리 구조 데이터 불러오기
  const fetchTreeData = useCallback(async () => {
    try {
      const headers = await getAuthHeaders();
      const response = await fetch('/api/documents/tree', {
        headers,
      });

      if (response.ok) {
        const data = await response.json();
        if (data.success) {
          setTreeData(data.data || []);
        }
      }
    } catch (error) {
      console.error('트리 데이터 불러오기 실패:', error);
    }
  }, []);

  // 프로젝트 목록 불러오기
  const fetchProjects = useCallback(async () => {
    try {
      const headers = await getAuthHeaders();
      const response = await fetch('/api/projects', {
        headers,
      });

      if (response.ok) {
        const data = await response.json();
        if (data.success) {
          setProjects(data.data || []);
        }
      }
    } catch (error) {
      console.error('프로젝트 목록 불러오기 실패:', error);
    }
  }, []);

  // 문서 목록 불러오기 (필터 적용)
  const fetchDocuments = useCallback(async () => {
    try {
      setLoading(true);
      const headers = await getAuthHeaders();

      // 선택된 노드에 따른 쿼리 파라미터 구성
      const params = new URLSearchParams();
      if (selectedNode) {
        if (selectedNode.type === 'user' && selectedNode.userId) {
          params.append('userId', selectedNode.userId);
        } else if (selectedNode.type === 'project' && selectedNode.projectId) {
          params.append('projectId', selectedNode.projectId.toString());
        }
      }

      const response = await fetch(`/api/documents?${params.toString()}`, {
        headers,
      });

      if (response.ok) {
        const data = await response.json();
        if (data.success) {
          setDocuments(data.data || []);
        }
      }
    } catch (error) {
      console.error('문서 목록 불러오기 실패:', error);
    } finally {
      setLoading(false);
    }
  }, [selectedNode]);

  useEffect(() => {
    if (isOpen && user) {
      fetchTreeData();
      fetchProjects();
      fetchDocuments();
    }
  }, [isOpen, user, fetchTreeData, fetchProjects, fetchDocuments]);

  // 선택된 노드 변경 시 문서 목록 새로고침
  useEffect(() => {
    if (user && selectedNode && isOpen) {
      fetchDocuments();
    }
  }, [user, selectedNode, fetchDocuments, isOpen]);

  // 상태 아이콘 및 색상 가져오기
  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'completed':
        return <CheckCircle className="w-4 h-4 text-green-500" />;
      case 'pending':
        return <Clock className="w-4 h-4 text-gray-500" />;
      default:
        return <AlertCircle className="w-4 h-4 text-gray-400" />;
    }
  };

  // 파일 크기 포맷팅
  const formatFileSize = (bytes: number) => {
    if (bytes === 0) return '0B';
    const k = 1024;
    const sizes = ['B', 'K', 'M', 'G'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    const size = bytes / Math.pow(k, i);
    return (size < 10 ? size.toFixed(1) : Math.round(size)) + sizes[i];
  };

  // 날짜 포맷팅
  const formatDate = (dateString: string) => {
    const date = new Date(dateString);
    const month = String(date.getMonth() + 1).padStart(2, '0');
    const day = String(date.getDate()).padStart(2, '0');
    const hours = String(date.getHours()).padStart(2, '0');
    const minutes = String(date.getMinutes()).padStart(2, '0');
    return `${month}/${day} ${hours}:${minutes}`;
  };

  // TanStack Table 컬럼 정의
  const columnHelper = createColumnHelper<ChatDocument>();
  
  const columns = useMemo(() => [
    // 선택 체크박스
    columnHelper.display({
      id: 'select',
      header: ({ table }) => (
        <input
          type="checkbox"
          className="rounded"
          checked={table.getIsAllRowsSelected()}
          onChange={table.getToggleAllRowsSelectedHandler()}
        />
      ),
      cell: ({ row }) => (
        <input
          type="checkbox"
          className="rounded"
          checked={row.getIsSelected()}
          onChange={row.getToggleSelectedHandler()}
        />
      ),
      size: 40,
      enableSorting: false,
      enableResizing: false,
    }),
    
    // 파일명
    columnHelper.accessor('filename', {
      header: '파일명',
      cell: (info) => (
        <span
          className="font-medium text-sm truncate block"
          title={info.getValue()}
        >
          {info.getValue()}
        </span>
      ),
      size: 250,
      minSize: 150,
      maxSize: 400,
    }),
    
    // 파일 크기
    columnHelper.accessor('filesize', {
      header: '크기',
      cell: (info) => (
        <span className="text-sm text-gray-600 dark:text-gray-400">
          {formatFileSize(info.getValue())}
        </span>
      ),
      size: 80,
      minSize: 60,
      maxSize: 100,
    }),
    
    // 임베딩 상태
    columnHelper.accessor('embedding_status', {
      header: '상태',
      cell: (info) => (
        <div
          className="flex justify-center"
          title={`임베딩 상태: ${info.getValue()}`}
        >
          {getStatusIcon(info.getValue())}
        </div>
      ),
      size: 60,
      enableResizing: false,
    }),
    
    // 프로젝트
    columnHelper.accessor('project_name', {
      header: '프로젝트',
      cell: (info) => {
        const value = info.getValue();
        return value ? (
          <span className="px-2 py-1 bg-blue-100 dark:bg-blue-900 text-blue-800 dark:text-blue-200 rounded-full text-sm">
            {value}
          </span>
        ) : null;
      },
      size: 120,
      minSize: 100,
      maxSize: 200,
    }),
    
    // 생성일
    columnHelper.accessor('created_at', {
      header: '생성일',
      cell: (info) => (
        <span className="text-sm text-gray-600 dark:text-gray-400">
          {formatDate(info.getValue())}
        </span>
      ),
      size: 120,
      minSize: 100,
      maxSize: 150,
    }),
    
    // 작업
    columnHelper.display({
      id: 'actions',
      header: '작업',
      cell: ({ row }) => {
        const doc = row.original;
        return (
          <div className="flex items-center justify-center gap-1">
            <button
              className="p-0.5 hover:bg-gray-100 dark:hover:bg-gray-600 rounded"
              title="문서 보기"
              onClick={() => {
                setPreviewFileName(doc.filename);
                setShowDocPreview(true);
              }}
            >
              <Eye className="w-4 h-4" />
            </button>
            <button
              className="p-0.5 hover:bg-gray-100 dark:hover:bg-gray-600 rounded"
              onClick={() => handleDeleteDocument(doc.id, doc.filename)}
              title="삭제"
            >
              <Trash2 className="w-4 h-4 text-red-500" />
            </button>
          </div>
        );
      },
      size: 100,
      enableSorting: false,
      enableResizing: false,
    }),
  ], []);

  // TanStack Table 인스턴스 생성
  const table = useReactTable({
    data: documents,
    columns,
    getCoreRowModel: getCoreRowModel(),
    getSortedRowModel: getSortedRowModel(),
    getFilteredRowModel: getFilteredRowModel(),
    getPaginationRowModel: getPaginationRowModel(),
    onSortingChange: setSorting,
    onColumnFiltersChange: setColumnFilters,
    onColumnVisibilityChange: setColumnVisibility,
    onRowSelectionChange: setRowSelection,
    onGlobalFilterChange: setGlobalFilter,
    enableRowSelection: true,
    enableColumnResizing: true,
    columnResizeMode: 'onChange',
    initialState: {
      pagination: {
        pageSize: 10,
      },
    },
    state: {
      sorting,
      columnFilters,
      columnVisibility,
      rowSelection,
      globalFilter,
    },
  });

  // 선택된 문서들 동기화
  useEffect(() => {
    const selectedRows = table.getSelectedRowModel().rows;
    const newSelectedDocuments = new Set(selectedRows.map(row => row.original.id));
    setSelectedDocuments(newSelectedDocuments);
  }, [rowSelection, table]);

  // 파일 업로드 처리
  const handleFileUpload = async (event: React.FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    const formData = new FormData(event.currentTarget);
    const file = formData.get('file') as File;

    if (!file) {
      showAlertModal('파일을 선택해주세요.', 'warning');
      return;
    }

    try {
      setIsUploading(true);
      setUploadProgress(0);

      const headers = await getAuthHeaders();
      const uploadFormData = new FormData();
      uploadFormData.append('file', file);
      uploadFormData.append('project_name', formData.get('project_name') as string || '');
      uploadFormData.append('tags', formData.get('tags') as string || '');

      const response = await fetch('/api/documents', {
        method: 'POST',
        headers: {
          ...headers,
        },
        body: uploadFormData,
      });

      if (response.ok) {
        const data = await response.json();
        if (data.success) {
          setShowUploadModal(false);
          fetchDocuments();
          showAlertModal('파일이 성공적으로 업로드되었습니다.', 'success');
        } else {
          showAlertModal(data.error?.message || '업로드에 실패했습니다.', 'error');
        }
      } else {
        showAlertModal('업로드에 실패했습니다.', 'error');
      }
    } catch (error) {
      console.error('업로드 오류:', error);
      showAlertModal('업로드 중 오류가 발생했습니다.', 'error');
    } finally {
      setIsUploading(false);
      setUploadProgress(0);
    }
  };

  // 문서 삭제 처리
  const handleDeleteDocument = async (id: number, filename: string) => {
    showConfirmModal(`"${filename}" 문서를 정말 삭제하시겠습니까?`, async () => {
      await performDeleteDocument(id, filename);
    });
  };

  const performDeleteDocument = async (id: number, filename: string) => {
    try {
      const headers = await getAuthHeaders();
      const response = await fetch(`/api/documents/${id}`, {
        method: 'DELETE',
        headers,
      });

      if (response.ok) {
        const data = await response.json();
        if (data.success) {
          fetchDocuments();
          showAlertModal('파일이 성공적으로 삭제되었습니다.', 'success');
        } else {
          showAlertModal(data.error?.message || '삭제에 실패했습니다.', 'error');
        }
      } else {
        showAlertModal('삭제에 실패했습니다.', 'error');
      }
    } catch (error) {
      console.error('삭제 오류:', error);
      showAlertModal('삭제 중 오류가 발생했습니다.', 'error');
    }
  };


  // 대량 삭제 처리
  const handleBulkDelete = async () => {
    if (selectedDocuments.size === 0) {
      showAlertModal('삭제할 파일을 선택해주세요.', 'warning');
      return;
    }

    showConfirmModal(`선택한 ${selectedDocuments.size}개의 파일을 정말 삭제하시겠습니까?`, async () => {
      await performBulkDelete();
    });
  };

  const performBulkDelete = async () => {
    try {
      const headers = await getAuthHeaders();
      const deletePromises = Array.from(selectedDocuments).map(id =>
        fetch(`/api/documents/${id}`, {
          method: 'DELETE',
          headers,
        })
      );

      await Promise.all(deletePromises);
      setSelectedDocuments(new Set());
      fetchDocuments();
      showAlertModal('선택한 파일들이 성공적으로 삭제되었습니다.', 'success');
    } catch (error) {
      console.error('대량 삭제 오류:', error);
      showAlertModal('삭제 중 오류가 발생했습니다.', 'error');
    }
  };

  // 트리 노드 토글
  const toggleNode = (nodeId: string) => {
    const newExpanded = new Set(expandedNodes);
    if (newExpanded.has(nodeId)) {
      newExpanded.delete(nodeId);
    } else {
      newExpanded.add(nodeId);
    }
    setExpandedNodes(newExpanded);
  };

  // 노드 선택
  const selectNode = (node: TreeNode) => {
    setSelectedNode(node);
  };

  // 프로젝트 생성
  const handleCreateProject = async (event: React.FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    const formData = new FormData(event.currentTarget);
    const name = formData.get('name') as string;
    const description = formData.get('description') as string;

    try {
      const headers = await getAuthHeaders();
      const response = await fetch('/api/projects', {
        method: 'POST',
        headers: {
          ...headers,
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          name,
          description,
        }),
      });

      if (response.ok) {
        const data = await response.json();
        if (data.success) {
          setShowCreateProjectModal(false);
          fetchProjects();
          fetchTreeData();
          showAlertModal('프로젝트가 성공적으로 생성되었습니다.', 'success');
        } else {
          showAlertModal(data.error?.message || '프로젝트 생성에 실패했습니다.', 'error');
        }
      }
    } catch (error) {
      console.error('프로젝트 생성 오류:', error);
      showAlertModal('프로젝트 생성 중 오류가 발생했습니다.', 'error');
    }
  };

  // 프로젝트 삭제 (project_name 기반)
  const handleDeleteProject = async (projectName: string) => {
    showConfirmModal(`"${projectName}" 프로젝트를 정말 삭제하시겠습니까?\n프로젝트 내 모든 파일도 함께 삭제됩니다.`, async () => {
      await performDeleteProject(projectName);
    });
  };

  const performDeleteProject = async (projectName: string) => {
    try {
      const headers = await getAuthHeaders();
      const response = await fetch(`/api/projects/${encodeURIComponent(projectName)}`, {
        method: 'DELETE',
        headers,
      });

      if (response.ok) {
        const data = await response.json();
        if (data.success) {
          fetchProjects();
          fetchTreeData();
          fetchDocuments();
          if (selectedNode && selectedNode.type === 'project' && selectedNode.projectId === projectName) {
            setSelectedNode(null);
          }
          showAlertModal(`프로젝트가 성공적으로 삭제되었습니다. (${data.data.deletedDocuments}개 파일 삭제)`, 'success');
        } else {
          showAlertModal(data.error?.message || '프로젝트 삭제에 실패했습니다.', 'error');
        }
      }
    } catch (error) {
      console.error('프로젝트 삭제 오류:', error);
      showAlertModal('프로젝트 삭제 중 오류가 발생했습니다.', 'error');
    }
  };

  // 트리 노드 렌더링
  const renderTreeNode = (node: TreeNode, level: number = 0) => {
    const isExpanded = expandedNodes.has(node.id);
    const isSelected = selectedNode?.id === node.id;
    const hasChildren = node.children && node.children.length > 0;

    return (
      <div key={node.id} className="select-none">
        <div
          className={`flex items-center gap-2 px-3 py-2 cursor-pointer hover:bg-gray-100 dark:hover:bg-gray-700 ${
            isSelected ? 'bg-blue-100 dark:bg-blue-900/50 text-blue-700 dark:text-blue-300' : ''
          }`}
          style={{ paddingLeft: `${12 + level * 16}px` }}
          onClick={() => selectNode(node)}
        >
          {hasChildren && (
            <button
              onClick={(e) => {
                e.stopPropagation();
                toggleNode(node.id);
              }}
              className="p-1 hover:bg-gray-200 dark:hover:bg-gray-600 rounded"
            >
              {isExpanded ? (
                <ChevronDown className="w-4 h-4" />
              ) : (
                <ChevronRight className="w-4 h-4" />
              )}
            </button>
          )}
          {!hasChildren && <div className="w-6" />}

          {node.type === 'all' && <Files className="w-4 h-4 text-blue-500" />}
          {node.type === 'user' && <User className="w-4 h-4 text-green-500" />}
          {node.type === 'project' && <FolderOpen className="w-4 h-4 text-orange-500" />}

          <span className="flex-1 text-sm font-medium">{node.name}</span>
          <span className="text-sm px-2 py-1 rounded-full" style={{
            backgroundColor: 'var(--neutral-700)',
            color: 'var(--neutral-300)'
          }}>
            {node.documentCount}
          </span>

          {node.type === 'project' && user?.role === 'admin' && (
            <button
              onClick={(e) => {
                e.stopPropagation();
                if (node.projectId) {
                  handleDeleteProject(node.projectId as string);
                }
              }}
              className="p-1 hover:bg-red-100 dark:hover:bg-red-900/20 rounded text-red-500"
              title="프로젝트 삭제"
            >
              <Trash2 className="w-3 h-3" />
            </button>
          )}
        </div>

        {hasChildren && isExpanded && (
          <div>
            {node.children?.map(child => renderTreeNode(child, level + 1))}
          </div>
        )}
      </div>
    );
  };

  // 필터링된 문서 목록
  const filteredDocuments = documents.filter(doc => {
    if (filters.search && !doc.filename.toLowerCase().includes(filters.search.toLowerCase())) {
      return false;
    }
    if (filters.embeddingStatus && doc.embedding_status !== filters.embeddingStatus) {
      return false;
    }
    if (filters.projectName && doc.project_name !== filters.projectName) {
      return false;
    }
    if (filters.fileType && !doc.mimetype.includes(filters.fileType)) {
      return false;
    }
    return true;
  });

  // 정렬된 문서 목록
  const sortedDocuments = filteredDocuments.sort((a, b) => {
    const aValue = a[sortBy];
    const bValue = b[sortBy];

    if (typeof aValue === 'string' && typeof bValue === 'string') {
      return sortOrder === 'asc'
        ? aValue.localeCompare(bValue)
        : bValue.localeCompare(aValue);
    }

    if (typeof aValue === 'number' && typeof bValue === 'number') {
      return sortOrder === 'asc' ? aValue - bValue : bValue - aValue;
    }

    return 0;
  });

  // 페이지네이션 적용
  const totalPages = Math.ceil(sortedDocuments.length / itemsPerPage);
  const startIndex = (currentPage - 1) * itemsPerPage;
  const endIndex = startIndex + itemsPerPage;
  const paginatedDocuments = sortedDocuments.slice(startIndex, endIndex);

  // 필터 변경 시 첫 페이지로 이동
  useEffect(() => {
    setCurrentPage(1);
  }, [filters, sortBy, sortOrder, selectedNode]);

  // 프로젝트 이름 목록
  const projectNames = Array.from(new Set(
    documents.map(doc => doc.project_name).filter(Boolean)
  ));

  // 파일 타입 목록
  const fileTypes = Array.from(new Set(
    documents.map(doc => doc.mimetype.split('/')[0]).filter(Boolean)
  ));

  if (!isOpen) return null;

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
      <div className="bg-white dark:bg-gray-800 rounded-lg w-[95vw] h-[85vh] max-w-6xl max-h-[700px] flex flex-col shadow-2xl">
        {/* 모달 헤더 */}
        <div className="flex items-center justify-between p-4 border-b border-gray-200 dark:border-gray-700">
          <h2 className="text-xl font-semibold">파일 관리</h2>
          <button
            onClick={onClose}
            className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300"
          >
            <X className="w-6 h-6" />
          </button>
        </div>

        {/* 모달 내용 */}
        <div className="flex-1 flex overflow-hidden">
          {/* 왼쪽 문서 탐색기 */}
          <div className="w-48 border-r border-gray-200 dark:border-gray-700 overflow-y-auto" style={{ backgroundColor: 'var(--card-bg)' }}>
            <div className="p-4 border-b border-gray-200 dark:border-gray-700">
              <div className="flex items-center justify-between">
                <h3 className="text-lg font-semibold flex items-center gap-2">
                  <Folder className="w-5 h-5" />
                  파일 탐색기
                </h3>
              </div>
            </div>

            <div className="p-2">
              {treeData.map(node => renderTreeNode(node))}
            </div>
          </div>

          {/* 오른쪽 파일 목록 */}
          <div className="flex-1 flex flex-col overflow-hidden" style={{ backgroundColor: 'var(--card-bg)' }}>
            <div className="p-4 border-b border-gray-200 dark:border-gray-700">
              <div className="flex items-center justify-between mb-4">
                <h3 className="text-lg font-semibold">
                  파일 목록
                  {selectedNode && (
                    <span className="text-sm text-gray-600 dark:text-gray-400 font-normal ml-2">
                      {selectedNode.type === 'all' && '(전체)'}
                      {selectedNode.type === 'user' && `(${selectedNode.name})`}
                      {selectedNode.type === 'project' && `(${selectedNode.name})`}
                    </span>
                  )}
                </h3>
                <div className="flex items-center gap-2">
                  <Button
                    onClick={() => setShowFilters(!showFilters)}
                    variant="outline"
                    size="sm"
                  >
                    <Filter className="w-4 h-4 mr-1" />
                    필터
                  </Button>
                  <Button
                    onClick={() => setShowUploadModal(true)}
                    className="bg-blue-600 hover:bg-blue-700 text-white"
                    size="sm"
                  >
                    <Upload className="w-4 h-4 mr-1" />
                    업로드
                  </Button>
                </div>
              </div>

              {/* 문서 통계 */}
              <div className="grid grid-cols-4 gap-3">
                <div className="p-3 rounded-lg border" style={{ backgroundColor: 'var(--card-bg)' }}>
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-sm text-gray-600 dark:text-gray-400">전체 파일</p>
                      <p className="text-xl font-bold">{documents.length}</p>
                    </div>
                    <FileText className="w-6 h-6 text-blue-500" />
                  </div>
                </div>
                <div className="p-3 rounded-lg border" style={{ backgroundColor: 'var(--card-bg)' }}>
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-sm text-gray-600 dark:text-gray-400">임베딩 완료</p>
                      <p className="text-xl font-bold text-green-600">
                        {documents.filter(d => d.embedding_status === 'completed').length}
                      </p>
                    </div>
                    <CheckCircle className="w-6 h-6 text-green-500" />
                  </div>
                </div>
                <div className="p-3 rounded-lg border" style={{ backgroundColor: 'var(--card-bg)' }}>
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-sm text-gray-600 dark:text-gray-400">대기 중</p>
                      <p className="text-xl font-bold text-gray-600">
                        {documents.filter(d => d.embedding_status === 'pending').length}
                      </p>
                    </div>
                    <Clock className="w-6 h-6 text-gray-500" />
                  </div>
                </div>
              </div>
            </div>

            {/* 필터 패널 */}
            {showFilters && (
              <div className="p-4 border-b border-gray-200 dark:border-gray-700" style={{ backgroundColor: 'var(--card-bg)' }}>
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                  <div>
                    <label className="block text-sm font-medium mb-2">검색</label>
                    <div className="relative">
                      <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 text-gray-400" />
                      <input
                        type="text"
                        placeholder="파일명 검색..."
                        className="w-full pl-10 pr-3 py-2 border rounded-md dark:bg-gray-700 dark:border-gray-600 text-sm"
                        value={filters.search}
                        onChange={(e) => setFilters({...filters, search: e.target.value})}
                      />
                    </div>
                  </div>
                  <div>
                    <label className="block text-sm font-medium mb-2">임베딩 상태</label>
                    <select
                      className="w-full p-2 border rounded-md dark:bg-gray-700 dark:border-gray-600 text-sm"
                      value={filters.embeddingStatus}
                      onChange={(e) => setFilters({...filters, embeddingStatus: e.target.value})}
                    >
                      <option value="">전체</option>
                      <option value="pending">대기중</option>
                      <option value="completed">완료</option>
                    </select>
                  </div>
                  <div>
                    <label className="block text-sm font-medium mb-2">프로젝트</label>
                    <select
                      className="w-full p-2 border rounded-md dark:bg-gray-700 dark:border-gray-600 text-sm"
                      value={filters.projectName}
                      onChange={(e) => setFilters({...filters, projectName: e.target.value})}
                    >
                      <option value="">전체</option>
                      {projectNames.map(name => (
                        <option key={name} value={name}>{name}</option>
                      ))}
                    </select>
                  </div>
                  <div>
                    <label className="block text-sm font-medium mb-2">파일 타입</label>
                    <select
                      className="w-full p-2 border rounded-md dark:bg-gray-700 dark:border-gray-600 text-sm"
                      value={filters.fileType}
                      onChange={(e) => setFilters({...filters, fileType: e.target.value})}
                    >
                      <option value="">전체</option>
                      {fileTypes.map(type => (
                        <option key={type} value={type}>{type}</option>
                      ))}
                    </select>
                  </div>
                </div>
              </div>
            )}

            {/* 문서 테이블 */}
            <div className="flex-1 overflow-auto">
              <div className="rounded-lg border overflow-hidden" style={{ backgroundColor: 'var(--card-bg)' }}>
                {loading ? (
                  <div className="p-8 text-center">
                    <RefreshCw className="w-8 h-8 animate-spin mx-auto mb-4 text-gray-400" />
                    <p className="text-gray-600 dark:text-gray-400">파일 목록을 불러오고 있습니다...</p>
                  </div>
                ) : paginatedDocuments.length === 0 ? (
                  <div className="p-8 text-center">
                    <Files className="w-12 h-12 mx-auto mb-4 text-gray-400" />
                    <p className="text-gray-600 dark:text-gray-400">파일이 없습니다.</p>
                  </div>
                ) : (
                  <div className="overflow-x-auto px-1 py-1">
                    <div
                      className="w-full border border-gray-200 dark:border-gray-700 rounded-lg"
                      style={{ minWidth: table.getCenterTotalSize(), width: '100%' }}
                    >
                      {/* 테이블 헤더 */}
                      <div style={{ backgroundColor: 'var(--sidebar-bg)' }}>
                        {table.getHeaderGroups().map(headerGroup => (
                          <div key={headerGroup.id} className="flex">
                            {headerGroup.headers.map(header => (
                              <div
                                key={header.id}
                                className="p-2 text-left font-medium border-r border-gray-200 dark:border-gray-700 last:border-r-0 relative group text-sm"
                                style={{ width: header.getSize() }}
                              >
                                <div 
                                  className={`flex items-center gap-2 ${
                                    header.column.getCanSort() ? 'cursor-pointer select-none' : ''
                                  }`}
                                  onClick={header.column.getToggleSortingHandler()}
                                >
                                  {flexRender(header.column.columnDef.header, header.getContext())}
                                  {header.column.getCanSort() && (
                                    <span className="text-gray-400">
                                      {{
                                        asc: '↑',
                                        desc: '↓',
                                      }[header.column.getIsSorted() as string] ?? '↕'}
                                    </span>
                                  )}
                                </div>
                                
                                {/* 컬럼 리사이저 */}
                                {header.column.getCanResize() && (
                                  <div
                                    className="absolute right-0 top-0 h-full w-1 bg-transparent hover:bg-blue-500 cursor-col-resize opacity-0 group-hover:opacity-100 transition-opacity"
                                    onMouseDown={header.getResizeHandler()}
                                    onTouchStart={header.getResizeHandler()}
                                  />
                                )}
                              </div>
                            ))}
                          </div>
                        ))}
                      </div>
                      
                      {/* 테이블 바디 */}
                      <div>
                        {table.getRowModel().rows.map(row => (
                          <div 
                            key={row.id} 
                            className="flex border-t border-gray-200 dark:border-gray-700 hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors"
                          >
                            {row.getVisibleCells().map(cell => (
                              <div
                                key={cell.id}
                                className="p-2 border-r border-gray-200 dark:border-gray-700 last:border-r-0 flex items-center"
                                style={{ width: cell.column.getSize() }}
                              >
                                {flexRender(cell.column.columnDef.cell, cell.getContext())}
                              </div>
                            ))}
                          </div>
                        ))}
                      </div>
                    </div>
                  </div>
                )}

                {/* TanStack Table 페이지네이션 */}
                {table.getPageCount() > 1 && (
                  <div className="px-4 py-3 border-t border-gray-200 dark:border-gray-700 flex items-center justify-between">
                    <div className="flex items-center gap-4">
                      <div className="flex items-center gap-2">
                        <span className="text-sm text-gray-600 dark:text-gray-400">페이지당 항목:</span>
                        <select
                          value={table.getState().pagination.pageSize}
                          onChange={(e) => {
                            table.setPageSize(Number(e.target.value));
                          }}
                          className="px-2 py-1 border rounded text-sm dark:bg-gray-700 dark:border-gray-600"
                        >
                          {[10, 20, 30, 40, 50].map(pageSize => (
                            <option key={pageSize} value={pageSize}>
                              {pageSize}
                            </option>
                          ))}
                        </select>
                      </div>
                      <span className="text-sm text-gray-600 dark:text-gray-400">
                        {table.getState().pagination.pageIndex * table.getState().pagination.pageSize + 1}-
                        {Math.min((table.getState().pagination.pageIndex + 1) * table.getState().pagination.pageSize, table.getFilteredRowModel().rows.length)} / 
                        {table.getFilteredRowModel().rows.length}개 항목
                      </span>
                    </div>

                    <div className="flex items-center gap-2">
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => table.previousPage()}
                        disabled={!table.getCanPreviousPage()}
                      >
                        <ChevronLeft className="w-4 h-4" />
                        이전
                      </Button>

                      <div className="flex items-center gap-1">
                        {Array.from({ length: Math.min(5, table.getPageCount()) }, (_, i) => {
                          const currentPage = table.getState().pagination.pageIndex;
                          const totalPages = table.getPageCount();
                          let pageNum;
                          
                          if (totalPages <= 5) {
                            pageNum = i;
                          } else if (currentPage <= 2) {
                            pageNum = i;
                          } else if (currentPage >= totalPages - 3) {
                            pageNum = totalPages - 5 + i;
                          } else {
                            pageNum = currentPage - 2 + i;
                          }

                          return (
                            <Button
                              key={pageNum}
                              variant={currentPage === pageNum ? "default" : "outline"}
                              size="sm"
                              onClick={() => table.setPageIndex(pageNum)}
                              className="w-8 h-8 p-0"
                            >
                              {pageNum + 1}
                            </Button>
                          );
                        })}
                      </div>

                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => table.nextPage()}
                        disabled={!table.getCanNextPage()}
                      >
                        다음
                        <ChevronRight className="w-4 h-4" />
                      </Button>
                    </div>
                  </div>
                )}

                {/* Manual pagination controls */}
                {totalPages > 1 && (
                  <div className="px-4 py-3 border-t border-gray-200 dark:border-gray-700 flex items-center justify-between">
                    <div className="flex items-center gap-4">
                      <div className="flex items-center gap-2">
                        <span className="text-sm text-gray-600 dark:text-gray-400">페이지당 항목:</span>
                        <select
                          value={itemsPerPage}
                          onChange={(e) => {
                            setItemsPerPage(Number(e.target.value));
                            setCurrentPage(1);
                          }}
                          className="px-2 py-1 border rounded text-sm dark:bg-gray-700 dark:border-gray-600"
                        >
                          <option value={10}>10</option>
                          <option value={20}>20</option>
                          <option value={50}>50</option>
                          <option value={100}>100</option>
                        </select>
                      </div>
                      <span className="text-sm text-gray-600 dark:text-gray-400">
                        {startIndex + 1}-{Math.min(endIndex, sortedDocuments.length)} / {sortedDocuments.length}개 항목
                      </span>
                    </div>

                    <div className="flex items-center gap-2">
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => setCurrentPage(Math.max(1, currentPage - 1))}
                        disabled={currentPage === 1}
                      >
                        <ChevronLeft className="w-4 h-4" />
                        이전
                      </Button>

                      <div className="flex items-center gap-1">
                        {Array.from({ length: Math.min(5, totalPages) }, (_, i) => {
                          let pageNum;
                          if (totalPages <= 5) {
                            pageNum = i + 1;
                          } else if (currentPage <= 3) {
                            pageNum = i + 1;
                          } else if (currentPage >= totalPages - 2) {
                            pageNum = totalPages - 4 + i;
                          } else {
                            pageNum = currentPage - 2 + i;
                          }

                          return (
                            <Button
                              key={pageNum}
                              variant={currentPage === pageNum ? "default" : "outline"}
                              size="sm"
                              onClick={() => setCurrentPage(pageNum)}
                              className="w-8 h-8 p-0"
                            >
                              {pageNum}
                            </Button>
                          );
                        })}
                        {totalPages > 5 && currentPage < totalPages - 2 && (
                          <>
                            <span className="text-gray-400">...</span>
                            <Button
                              variant="outline"
                              size="sm"
                              onClick={() => setCurrentPage(totalPages)}
                              className="w-8 h-8 p-0"
                            >
                              {totalPages}
                            </Button>
                          </>
                        )}
                      </div>

                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => setCurrentPage(Math.min(totalPages, currentPage + 1))}
                        disabled={currentPage === totalPages}
                      >
                        다음
                        <ChevronRight className="w-4 h-4" />
                      </Button>
                    </div>
                  </div>
                )}
              </div>
            </div>
          </div>
        </div>

        {/* 파일 업로드 모달 */}
        {showUploadModal && (
          <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
            <div className="bg-white dark:bg-gray-800 p-6 rounded-lg max-w-md w-full mx-4">
              <div className="flex items-center justify-between mb-4">
                <h3 className="text-lg font-semibold">파일 업로드</h3>
                <button
                  onClick={() => setShowUploadModal(false)}
                  className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300"
                >
                  <X className="w-5 h-5" />
                </button>
              </div>

              <form onSubmit={handleFileUpload} className="space-y-4">
                <div>
                  <label className="block text-sm font-medium mb-2">파일 선택</label>
                  <input
                    type="file"
                    name="file"
                    required
                    className="w-full p-2 border rounded-md dark:bg-gray-700 dark:border-gray-600"
                    accept=".pdf,.doc,.docx,.txt,.md"
                  />
                </div>

                <div>
                  <label className="block text-sm font-medium mb-2">프로젝트명 (선택사항)</label>
                  <input
                    type="text"
                    name="project_name"
                    placeholder="프로젝트명을 입력하세요"
                    className="w-full p-2 border rounded-md dark:bg-gray-700 dark:border-gray-600"
                  />
                </div>

                <div>
                  <label className="block text-sm font-medium mb-2">태그 (선택사항)</label>
                  <input
                    type="text"
                    name="tags"
                    placeholder="태그를 쉼표로 구분하여 입력하세요"
                    className="w-full p-2 border rounded-md dark:bg-gray-700 dark:border-gray-600"
                  />
                  <p className="text-sm text-gray-500 mt-1">예: 문서, 분석, 보고서</p>
                </div>

                {isUploading && (
                  <div className="w-full bg-gray-200 rounded-full h-2">
                    <div
                      className="bg-blue-600 h-2 rounded-full transition-all duration-300"
                      style={{ width: `${uploadProgress}%` }}
                    ></div>
                  </div>
                )}

                <div className="flex gap-2 pt-4">
                  <Button
                    type="button"
                    variant="outline"
                    onClick={() => setShowUploadModal(false)}
                    disabled={isUploading}
                    className="flex-1"
                  >
                    취소
                  </Button>
                  <Button
                    type="submit"
                    disabled={isUploading}
                    className="flex-1 bg-blue-600 hover:bg-blue-700 text-white"
                  >
                    {isUploading ? '업로드 중...' : '업로드'}
                  </Button>
                </div>
              </form>
            </div>
          </div>
        )}


        {/* 알림 모달 */}
        {showAlert && (
          <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
            <div className="bg-white dark:bg-gray-800 p-6 rounded-lg max-w-md w-full mx-4">
              <div className="flex items-center justify-between mb-4">
                <div className="flex items-center gap-3">
                  {alertType === 'success' && <CheckCircle className="w-6 h-6 text-green-500" />}
                  {alertType === 'error' && <XCircle className="w-6 h-6 text-red-500" />}
                  {alertType === 'warning' && <AlertCircle className="w-6 h-6 text-yellow-500" />}
                  {alertType === 'info' && <AlertCircle className="w-6 h-6 text-blue-500" />}
                  <h3 className="text-lg font-semibold">
                    {alertType === 'success' && '성공'}
                    {alertType === 'error' && '오류'}
                    {alertType === 'warning' && '경고'}
                    {alertType === 'info' && '알림'}
                  </h3>
                </div>
                <button
                  onClick={() => setShowAlert(false)}
                  className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300"
                >
                  <X className="w-5 h-5" />
                </button>
              </div>

              <p className="text-gray-600 dark:text-gray-300 mb-6">
                {alertMessage}
              </p>

              <div className="flex justify-end">
                <Button
                  onClick={() => setShowAlert(false)}
                  className="bg-blue-600 hover:bg-blue-700 text-white"
                >
                  확인
                </Button>
              </div>
            </div>
          </div>
        )}

        {/* 확인 모달 */}
        {showConfirm && (
          <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
            <div className="bg-white dark:bg-gray-800 p-6 rounded-lg max-w-md w-full mx-4">
              <div className="flex items-center justify-between mb-4">
                <div className="flex items-center gap-3">
                  <AlertCircle className="w-6 h-6 text-orange-500" />
                  <h3 className="text-lg font-semibold">확인</h3>
                </div>
                <button
                  onClick={() => {
                    setShowConfirm(false);
                    setConfirmCallback(null);
                  }}
                  className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300"
                >
                  <X className="w-5 h-5" />
                </button>
              </div>

              <p className="text-gray-600 dark:text-gray-300 mb-6 whitespace-pre-line">
                {confirmMessage}
              </p>

              <div className="flex gap-2">
                <Button
                  variant="outline"
                  onClick={() => {
                    setShowConfirm(false);
                    setConfirmCallback(null);
                  }}
                  className="flex-1"
                >
                  취소
                </Button>
                <Button
                  onClick={() => {
                    if (confirmCallback) {
                      confirmCallback();
                    }
                    setShowConfirm(false);
                    setConfirmCallback(null);
                  }}
                  className="flex-1 bg-red-600 hover:bg-red-700 text-white"
                >
                  확인
                </Button>
              </div>
            </div>
          </div>
        )}

        {/* 문서 미리보기 모달 */}
        {showDocPreview && (
          <DocumentPreviewModal
            isOpen={showDocPreview}
            onClose={() => setShowDocPreview(false)}
            fileName={previewFileName}
            userId={user?.username || 'admin'}
          />
        )}
      </div>
    </div>
  );
}