'use client';

import { useState, useEffect, useRef } from 'react';
import { useRouter, usePathname } from 'next/navigation';
import Link from 'next/link';
import { useLanguage } from '@/contexts/LanguageContext';
import { externalApiClient } from '@/lib/apiClient';
import {
  MessageSquare,
  Code,
  Bot,
  User,
  Rocket,
  FileText,
  HelpCircle,
  FolderOpen,
  Brain,
  Cpu,
  TrendingUp,
  Users,
  Settings,
  BarChart3,
  Sun,
  Moon,
  Menu,
  ChevronDown,
  Building2,
  Activity,
  Files,
  FormInput
} from 'lucide-react';

interface User {
  id: string;
  username: string;
  email: string;
  role: string;
  profileImageUrl?: string;
}

// 환경 변수 기반 메뉴 제어 함수
const getMenuVisibility = () => {
  return {
    chat: process.env.NEXT_PUBLIC_ENABLE_CHAT === 'true',
    code: process.env.NEXT_PUBLIC_ENABLE_CODE === 'true',
    agent: process.env.NEXT_PUBLIC_ENABLE_AGENT === 'true',
    assistant: process.env.NEXT_PUBLIC_ENABLE_ASSISTANT === 'true',
    document: process.env.NEXT_PUBLIC_ENABLE_DOCUMENT === 'true',
    flowai: process.env.NEXT_PUBLIC_ENABLE_FLOWAI === 'true',
    business: process.env.NEXT_PUBLIC_ENABLE_BUSINESS === 'true',
    formbuilder: process.env.NEXT_PUBLIC_ENABLE_FORMBUILDER === 'true',
    support: process.env.NEXT_PUBLIC_ENABLE_SUPPORT === 'true',
    ragAdmin: process.env.NEXT_PUBLIC_ENABLE_RAG_ADMIN === 'true',
    modelAdmin: process.env.NEXT_PUBLIC_ENABLE_MODEL_ADMIN === 'true',
    evaluation: process.env.NEXT_PUBLIC_ENABLE_EVALUATION === 'true',
    userAdmin: process.env.NEXT_PUBLIC_ENABLE_USER_ADMIN === 'true',
    settings: process.env.NEXT_PUBLIC_ENABLE_SETTINGS === 'true',
    monitoring: process.env.NEXT_PUBLIC_ENABLE_MONITORING === 'true',
  };
};

// 서비스 등급 정보 가져오기
const getServiceTier = () => {
  return process.env.NEXT_PUBLIC_SERVICE_TIER || 'basic';
};

export default function Header() {
  const [user, setUser] = useState<User | null>(null);
  const [isDropdownOpen, setIsDropdownOpen] = useState(false);
  const [isMobileMenuOpen, setIsMobileMenuOpen] = useState(false);
  const router = useRouter();
  const pathname = usePathname();
  const [isDarkMode, setIsDarkMode] = useState(false);
  const [isClient, setIsClient] = useState(false);
  const dropdownRef = useRef<HTMLDivElement>(null);
  const dropdownButtonRef = useRef<HTMLButtonElement>(null);
  const { t } = useLanguage();
  const [customLogo, setCustomLogo] = useState<string | null>(null);
  const [serverStatus, setServerStatus] = useState<'online' | 'offline' | 'checking'>('checking');
  const [apiServerUrl, setApiServerUrl] = useState<string>('');

  // 메뉴 가시성 설정
  const menuVisibility = getMenuVisibility();
  const serviceTier = getServiceTier();

  // 현재 섹션 확인
  const section = pathname.split('/')[1] || 'home';

  // 일반 사용자용 메뉴 (환경 변수 기반 필터링)
  const userNavigationItems = [
    { name: t('header.chat', '대화하기'), path: '/chat', icon: MessageSquare, section: 'chat', enabled: menuVisibility.chat },
    { name: t('header.code', '코드'), path: '/code', icon: Code, section: 'code', enabled: menuVisibility.code },
    { name: t('header.agent', '자동화'), path: '/agent', icon: Bot, section: 'agent', enabled: menuVisibility.agent },
    { name: t('header.assistant', 'Assistant'), path: '/assistant', icon: User, section: 'assistant', enabled: menuVisibility.assistant },
    { name: t('header.document', '문서'), path: '/document', icon: FileText, section: 'document', enabled: menuVisibility.document },
    { name: t('header.formbuilder', '폼빌더'), path: '/formbuilder', icon: FormInput, section: 'formbuilder', enabled: menuVisibility.formbuilder },
    { name: t('header.business', '사업분석'), path: '/business', icon: Building2, section: 'business', enabled: menuVisibility.business },
    { name: t('header.support', '고객센터'), path: '/support', icon: HelpCircle, section: 'support', enabled: menuVisibility.support }
  ].filter(item => item.enabled);

  // FlowAI 메뉴 (admin 전용)
  const flowAINavigationItems = [
    { name: t('header.flowai', 'FlowAI'), path: '/flowai', icon: Rocket, section: 'flowai', enabled: menuVisibility.flowai }
  ].filter(item => item.enabled);

  // 관리자용 추가 메뉴 (환경 변수 기반 필터링)
  const adminNavigationItems = [
    { name: t('header.rag', 'RAG'), path: '/admin/rag', icon: FolderOpen, section: 'rag', enabled: menuVisibility.ragAdmin },
    { name: t('header.health', '헬스'), path: '/admin/health-batch', icon: Activity, section: 'health-batch', enabled: true },
    { name: t('header.esg', 'ESG'), path: '/admin/esg-analysis', icon: Files, section: 'esg-analysis', enabled: true },
    { name: t('header.models', '파인튜닝'), path: '/admin/models', icon: Brain, section: 'models', enabled: menuVisibility.modelAdmin },
    { name: t('header.evaluation', '평가'), path: '/admin/evaluation', icon: TrendingUp, section: 'evaluation', enabled: menuVisibility.evaluation },
    { name: t('header.users', '사용자'), path: '/admin/users', icon: Users, section: 'users', enabled: menuVisibility.userAdmin },
    { name: t('header.settings', '설정'), path: '/settings', icon: Settings, section: 'settings', enabled: menuVisibility.settings },
    { name: t('header.monitoring', '모니터링'), path: '/monitoring', icon: BarChart3, section: 'monitoring', enabled: menuVisibility.monitoring }
  ].filter(item => item.enabled);

  // 매니저용 메뉴 (환경 변수 기반 필터링)
  const managerNavigationItems = [
    { name: t('header.rag', 'RAG'), path: '/admin/rag', icon: FolderOpen, section: 'rag', enabled: menuVisibility.ragAdmin }
  ].filter(item => item.enabled);

  // 사용자 역할에 따른 메뉴 결정
  const navigationItems = user?.role === 'admin' 
    ? [...userNavigationItems, ...flowAINavigationItems, ...adminNavigationItems]
    : user?.role === 'manager'
    ? [...userNavigationItems, ...managerNavigationItems] 
    : userNavigationItems;

  // 외부 클릭 감지를 위한 이벤트 핸들러
  useEffect(() => {
    const handleClickOutside = (event: MouseEvent) => {
      if (
        dropdownRef.current && 
        !dropdownRef.current.contains(event.target as Node) &&
        dropdownButtonRef.current && 
        !dropdownButtonRef.current.contains(event.target as Node)
      ) {
        setIsDropdownOpen(false);
      }
    };

    // 전역 클릭 이벤트 리스너 등록
    document.addEventListener('mousedown', handleClickOutside);
    
    return () => {
      // 컴포넌트 언마운트 시 리스너 제거
      document.removeEventListener('mousedown', handleClickOutside);
    };
  }, []);

  // 프로필 이미지 변경 이벤트 리스너
  useEffect(() => {
    if (!user?.id) return;
    // 프로필 이미지 변경 이벤트 핸들러
    const refreshProfileImage = async () => {
      const url = await fetchProfileImage(user.id);
      setUser((prev) => prev ? { ...prev, profileImageUrl: url } : prev);
    };
    window.addEventListener('profileImageUpdated', refreshProfileImage);
    return () => {
      window.removeEventListener('profileImageUpdated', refreshProfileImage);
    };
  }, [user?.id]);

  // 프로필 이미지 불러오기
  const fetchProfileImage = async (userId: string): Promise<string> => {
    try {
      const response = await fetch(`/api/profile/image?userId=${userId}`, {
        method: 'GET',
        credentials: 'include'
      });

      if (response.ok) {
        const data = await response.json();
        if (data.success && data.data.imageUrl) {
          return data.data.imageUrl;
        }
      }
    } catch (error) {
      console.error('Profile image loading failed:', error);
    }
    return '/images/user.png'; // 기본 이미지
  };

  // 커스텀 로고 불러오기
  const fetchCustomLogo = async (): Promise<string | null> => {
    try {
      const response = await fetch('/api/config/logo', {
        method: 'GET',
        credentials: 'include',
      });
      
      if (response.ok) {
        const data = await response.json();
        if (data.success && data.data?.logoUrl) {
          return data.data.logoUrl;
        }
      }
    } catch (error) {
      console.error('Custom logo loading failed:', error);
    }
    return null;
  };

  // API 서버 상태 확인
  const checkServerStatus = async (): Promise<'online' | 'offline'> => {
    try {
      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), 5000);
      
      const response = await fetch('/api/server-status', {
        method: 'GET',
        signal: controller.signal,
      });
      
      clearTimeout(timeoutId);
      
      if (response.ok) {
        const data = await response.json();
        return data.status || 'offline';
      } else {
        return 'offline';
      }
    } catch (error) {
      // 연결 거부나 타임아웃은 정상적인 오프라인 상태이므로 에러 로그를 출력하지 않음
      if (error instanceof Error) {
        // AbortError는 타임아웃에 의한 것이므로 조용히 처리
        if (error.name === 'AbortError') {
          console.debug('Server status check timeout - server is offline.');
        } else if (error.message.includes('Failed to fetch') || error.message.includes('CONNECTION_REFUSED')) {
          console.debug('Server connection failed - server is offline.');
        } else {
          console.warn('Unexpected error during server status check:', error.message);
        }
      }
      return 'offline';
    }
  };
  
  // 페이지 변경 감지 및 드롭다운 닫기
  useEffect(() => {
    setIsDropdownOpen(false);
    setIsMobileMenuOpen(false);
  }, [pathname]);

  // 초기화 (클라이언트에서만 실행)
  useEffect(() => {
    setIsClient(true);
    
    // 테마 설정 초기화
    const savedTheme = localStorage.getItem('theme');
    const prefersDark = window.matchMedia('(prefers-color-scheme: dark)').matches;
    
    if (savedTheme === 'dark' || (!savedTheme && prefersDark)) {
      document.documentElement.classList.add('dark');
      setIsDarkMode(true);
    } else {
      document.documentElement.classList.remove('dark');
      setIsDarkMode(false);
    }
    
    // 커스텀 로고 초기 로드
    const loadCustomLogo = async () => {
      const logoUrl = await fetchCustomLogo();
      setCustomLogo(logoUrl);
    };
    loadCustomLogo();



    // 간단한 로컬 스토리지 체크
    const checkAuth = async () => {
      if (typeof window === 'undefined') return;
      
      const token = localStorage.getItem('userToken');
      const username = localStorage.getItem('username');
      const userId = localStorage.getItem('userId');
      
      // anonymous 사용자 정보가 있으면 제거
      if (username === 'anonymous' || userId === 'anonymous') {
        localStorage.removeItem('userToken');
        localStorage.removeItem('username');
        localStorage.removeItem('userId');
        localStorage.removeItem('userEmail');
        localStorage.removeItem('userRole');
        localStorage.removeItem('apiKey');
        setUser(null);
        console.log('Header: Anonymous user info removed');
        return;
      }
      
      if (token && username && userId && username !== 'anonymous') {
        const result = await externalApiClient.get('/auth/me');
        if (result.success && result.data) {
          // 프로필 이미지 불러오기
          const profileImageUrl = await fetchProfileImage(userId);
          
                  setUser({
          id: userId,
          username,
          email: localStorage.getItem('userEmail') || '',
          role: localStorage.getItem('userRole') || 'user',
          profileImageUrl
        });
        } else {
          setUser(null);
        }
      } else {
        setUser(null);
      }
    };
    
    // 초기 체크
    checkAuth();
    
    // 이벤트 리스너
    window.addEventListener('storage', checkAuth);
    
    // 커스텀 이벤트
    const handleAuthEvent = () => {
      checkAuth();
    };
    
    const handleLogoUpdate = (event: CustomEvent) => {
      setCustomLogo(event.detail.logoUrl);
    };
    
    window.addEventListener('login', handleAuthEvent);
    window.addEventListener('logout', handleAuthEvent);
    window.addEventListener('logoUpdated', handleLogoUpdate as EventListener);
    
    return () => {
      window.removeEventListener('storage', checkAuth);
      window.removeEventListener('login', handleAuthEvent);
      window.removeEventListener('logout', handleAuthEvent);
      window.removeEventListener('logoUpdated', handleLogoUpdate as EventListener);
    };
  }, []);

  // API 서버 상태 확인 (별도 useEffect)
  useEffect(() => {
    if (!isClient) return;

    const serverUrl = process.env.NEXT_PUBLIC_API_SERVER_URL;
    if (!serverUrl) return;

    setApiServerUrl(serverUrl);

    let interval: NodeJS.Timeout | null = null;

    // 서버 상태 확인 함수
    const checkStatus = async () => {
      const status = await checkServerStatus();
      setServerStatus(status);
      
      // 서버가 오프라인인 경우에만 더 자주 체크
      if (status === 'offline') {
        if (interval) clearInterval(interval);
        interval = setInterval(checkStatus, 30000); // 30초마다 재시도
      } else if (status === 'online') {
        if (interval) clearInterval(interval);
        interval = setInterval(checkStatus, 300000); // 5분마다 체크
      }
    };

    // 초기 상태 확인
    checkStatus();

    return () => {
      if (interval) clearInterval(interval);
    };
  }, [isClient]);

  // 로그아웃 처리
  const handleLogout = async () => {
    const result = await externalApiClient.post('/auth/logout');
    if (result.success) {
      localStorage.removeItem('userToken');
      localStorage.removeItem('userRole');
      localStorage.removeItem('username');
      localStorage.removeItem('userId');
      localStorage.removeItem('userEmail');
      localStorage.removeItem('apiKey');
      
      sessionStorage.removeItem('isLoggedIn');
      
      setUser(null);
      window.dispatchEvent(new Event('logout'));
      
      // 홈으로 이동
      window.location.href = '/';
    }
  };

  // 테마 전환 함수
  const toggleTheme = () => {
    if (isDarkMode) {
      document.documentElement.classList.remove('dark');
      localStorage.setItem('theme', 'light');
      setIsDarkMode(false);
      
      // 스타일 즉시 적용을 위한 추가 코드
      document.body.style.backgroundColor = 'var(--body-bg)';
      document.body.style.color = 'var(--text-primary)';
      
      // 테마 변경 이벤트 발생
      const event = new CustomEvent('themeChange', { detail: { theme: 'light' } });
      window.dispatchEvent(event);
    } else {
      document.documentElement.classList.add('dark');
      localStorage.setItem('theme', 'dark');
      setIsDarkMode(true);
      
      // 스타일 즉시 적용을 위한 추가 코드
      document.body.style.backgroundColor = 'var(--body-bg)';
      document.body.style.color = 'var(--text-primary)';
      
      // 테마 변경 이벤트 발생
      const event = new CustomEvent('themeChange', { detail: { theme: 'dark' } });
      window.dispatchEvent(event);
    }
  };

      return (
    <>
    <style dangerouslySetInnerHTML={{
      __html: `
        /* Chromium-compatible header layout */
        .header-container {
          display: grid !important;
          grid-template-columns: 1fr auto !important;
          grid-gap: 1rem !important;
          align-items: center !important;
          width: 100% !important;
        }
        .header-left {
          display: flex !important;
          align-items: center !important;
          min-width: 0 !important;
          overflow: hidden !important;
        }
        .header-right {
          display: flex !important;
          align-items: center !important;
          white-space: nowrap !important;
          flex-shrink: 0 !important;
        }

        /* Fallback for older browsers */
        @supports not (display: grid) {
          .header-container {
            display: flex !important;
            justify-content: space-between !important;
          }
          .header-left {
            flex: 1 !important;
            max-width: calc(100% - 200px) !important;
          }
          .header-right {
            width: 200px !important;
            flex-shrink: 0 !important;
          }
        }
      `
    }} />
    <header className="border-b border-gray-200 dark:border-gray-700 w-full" style={{ backgroundColor: 'var(--header-bg)' }}>
      <div className="header-container p-4 pb-4">
        {/* 왼쪽: 로고 + 네비 */}
        <div className="header-left gap-2 md:gap-8">
          <Link href="/" className="flex items-center space-x-3 rtl:space-x-reverse">
            {customLogo ? (
              <div className="flex items-center justify-center">
                <img
                  src={customLogo}
                  alt="Custom Logo"
                  className="object-contain transition-all duration-200 hover:scale-105 w-24 h-auto md:w-48"
                  onError={(e) => {
                    console.error('Custom logo loading failed:', e);
                    // 오류 시 기본 로고로 폴백
                    setCustomLogo(null);
                  }}
                />
              </div>
            ) : (
              <div className="flex items-center justify-center">
                <img
                  src="/images/airun_logo.png"
                  alt="AI Workspace Logo"
                  className="object-contain transition-all duration-200 hover:scale-105 w-24 h-auto md:w-48"
                />
              </div>
            )}         
          </Link>
          
          {/* 데스크톱 네비게이션 */}
          <nav className="hidden lg:flex items-center space-x-0 pl-4 overflow-hidden flex-1 min-w-0 max-w-none">
            {/* 일반 사용자 메뉴 */}
            {isClient && userNavigationItems.map((item) => {
              const isActive = section === item.section || (item.path === '/' && section === 'home');
              const IconComponent = item.icon;
              return (
                <Link
                  key={item.path}
                  href={item.path}
                  className={`flex flex-col items-center justify-center px-1 py-1 rounded-xl transition-all duration-200 min-w-[70px] ${
                    isActive
                      ? 'bg-blue-100 dark:bg-blue-900/50 text-blue-700 dark:text-blue-300 shadow-md'
                      : 'text-gray-600 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 hover:text-gray-900 dark:hover:text-white hover:shadow-md'
                  }`}
                >
                  <IconComponent className="w-7 h-7 mb-2" />
                  <span className="text-base font-medium text-center leading-tight">{item.name}</span>
                </Link>
              );
            })}

            {/* FlowAI 메뉴 (admin 전용) */}
            {isClient && user?.role === 'admin' && flowAINavigationItems.length > 0 && (
              <>
                {flowAINavigationItems.map((item) => {
                  const isActive = section === item.section;
                  const IconComponent = item.icon;
                  return (
                    <Link
                      key={item.path}
                      href={item.path}
                      className={`flex flex-col items-center justify-center px-1 py-1 rounded-xl transition-all duration-200 min-w-[70px] ${
                        isActive
                          ? 'bg-purple-100 dark:bg-purple-900/50 text-purple-700 dark:text-purple-300 shadow-md'
                          : 'text-gray-600 dark:text-gray-300 hover:bg-purple-50 dark:hover:bg-purple-900/20 hover:text-purple-700 dark:hover:text-purple-300 hover:shadow-md'
                      }`}
                      title={t('header.adminOnly', 'Admin Only')}
                    >
                      <IconComponent className="w-7 h-7 mb-2" />
                      <span className="text-base font-medium text-center leading-tight">{item.name}</span>
                    </Link>
                  );
                })}
              </>
            )}
            
            {/* 관리자 메뉴 (구분선과 함께) */}
            {isClient && user?.role === 'admin' && adminNavigationItems.length > 0 && (
              <>
                <div className="h-12 w-px bg-gray-300 dark:bg-gray-600 mx-2"></div>
                {adminNavigationItems.map((item) => {
                  const isActive = section === item.section;
                  const IconComponent = item.icon;
                  return (
                    <Link
                      key={item.path}
                      href={item.path}
                      className={`flex flex-col items-center justify-center px-2 py-2 rounded-xl transition-all duration-200 min-w-[70px] ${
                        isActive
                          ? 'bg-red-100 dark:bg-red-900/50 text-red-700 dark:text-red-300 shadow-md'
                          : 'text-gray-600 dark:text-gray-300 hover:bg-red-50 dark:hover:bg-red-900/20 hover:text-red-700 dark:hover:text-red-300 hover:shadow-md'
                      }`}
                      title={t('header.adminOnly', 'Admin Only')}
                    >
                      <IconComponent className="w-8 h-8 mb-2" />
                      <span className="text-base font-medium text-center leading-tight">{item.name}</span>
                    </Link>
                  );
                })}
              </>
            )}
            
            {/* 매니저 메뉴 (구분선과 함께) */}
            {isClient && user?.role === 'manager' && managerNavigationItems.length > 0 && (
              <>
                <div className="h-12 w-px bg-gray-300 dark:bg-gray-600 mx-2"></div>
                {managerNavigationItems.map((item) => {
                  const isActive = section === item.section;
                  const IconComponent = item.icon;
                  return (
                    <Link
                      key={item.path}
                      href={item.path}
                      className={`flex flex-col items-center justify-center px-2 py-2 rounded-xl transition-all duration-200 min-w-[70px] ${
                        isActive
                          ? 'bg-orange-100 dark:bg-orange-900/50 text-orange-700 dark:text-orange-300 shadow-md'
                          : 'text-gray-600 dark:text-gray-300 hover:bg-orange-50 dark:hover:bg-orange-900/20 hover:text-orange-700 dark:hover:text-orange-300 hover:shadow-md'
                      }`}
                      title={t('header.managerOnly', 'Manager Only')}
                    >
                      <IconComponent className="w-8 h-8 mb-2" />
                      <span className="text-base font-medium text-center leading-tight">{item.name}</span>
                    </Link>
                  );
                })}
              </>
            )}
          </nav>

          {/* 중간 크기 화면용 축약된 네비게이션 */}
          <nav className="hidden md:flex lg:hidden items-center space-x-1 pl-4 overflow-hidden flex-1 min-w-0">
            {/* 핵심 메뉴만 표시 */}
            {isClient && userNavigationItems.slice(0, 3).map((item) => {
              const isActive = section === item.section || (item.path === '/' && section === 'home');
              const IconComponent = item.icon;
              return (
                <Link
                  key={item.path}
                  href={item.path}
                  className={`flex items-center justify-center p-2 rounded-lg transition-all duration-200 min-w-[40px] ${
                    isActive
                      ? 'bg-blue-100 dark:bg-blue-900/50 text-blue-700 dark:text-blue-300'
                      : 'text-gray-600 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 hover:text-gray-900 dark:hover:text-white'
                  }`}
                  title={item.name}
                >
                  <IconComponent className="w-5 h-5" />
                </Link>
              );
            })}
          </nav>
        </div>
        {/* 오른쪽: 유저/로그인/테마 */}
        <div className="header-right gap-3">
          {/* 모바일 메뉴 버튼 */}
          <button
            onClick={() => setIsMobileMenuOpen(!isMobileMenuOpen)}
            className="md:hidden p-2 text-gray-500 hover:bg-gray-100 dark:text-gray-400 dark:hover:bg-gray-700 rounded-md"
          >
            <Menu className="w-6 h-6" />
          </button>

          {/* 테마 전환 버튼 */}
          <button
            onClick={toggleTheme}
            className="rounded-full p-2 text-gray-500 hover:bg-gray-100 dark:text-gray-400 dark:hover:bg-gray-700 transition-colors duration-300"
            aria-label={t('header.toggleTheme', '테마 전환')}
          >
            {isDarkMode ? (
              <Sun className="h-5 w-5" />
            ) : (
              <Moon className="h-5 w-5" />
            )}
          </button>
          
          {user ? (
            <div className="relative">
              <button 
                onClick={() => setIsDropdownOpen(!isDropdownOpen)}
                className="flex items-center text-sm font-medium text-gray-900 rounded-full hover:text-blue-600 dark:hover:text-blue-500 md:mr-0 focus:ring-4 focus:ring-gray-100 dark:focus:ring-gray-700 dark:text-white"
                type="button"
                style={{ color: isDarkMode ? 'var(--text-primary)' : 'var(--text-primary)' }}
                ref={dropdownButtonRef}
              >
                <span className="sr-only">{t('header.openUserMenu', '사용자 메뉴 열기')}</span>
                {user.profileImageUrl && user.profileImageUrl !== '/images/user.png' ? (
                  <img
                    src={user.profileImageUrl}
                    alt="Profile Image"
                    className="w-8 h-8 rounded-full object-cover mr-2 border border-gray-200 dark:border-gray-600"
                  />
                ) : (
                  <div className="w-8 h-8 rounded-full bg-blue-600 text-white flex items-center justify-center mr-2">
                    {user.username.charAt(0).toUpperCase()}
                  </div>
                )}
                <span className="hidden md:block" style={{ color: isDarkMode ? 'var(--text-primary)' : 'var(--text-primary)' }}>
                  {user.username}
                </span>
                <svg className="w-4 h-4 ml-2" aria-hidden="true" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                  <ChevronDown className="w-4 h-4 ml-2" />
                </svg>
              </button>
              {isDropdownOpen && (
                <div ref={dropdownRef} className="absolute right-0 mt-2 w-40 py-1 bg-white dark:bg-gray-700 rounded-md shadow-xl z-[9999] border border-gray-200 dark:border-gray-600">
                  <Link 
                    href="/profile" 
                    className="flex items-center px-3 py-2 text-xs text-gray-700 hover:bg-gray-100 dark:text-gray-300 dark:hover:bg-gray-700 transition-colors"
                    onClick={() => setIsDropdownOpen(false)}
                  >
                    <span className="mr-2 text-sm">👤</span>
                    {t('header.myProfile', '내 프로필')}
                  </Link>
                  <button 
                    onClick={() => {
                      setIsDropdownOpen(false);
                      handleLogout();
                    }}
                    className="w-full flex items-center text-left px-3 py-2 text-xs text-red-600 hover:bg-gray-100 dark:text-red-400 dark:hover:bg-gray-700 transition-colors"
                  >
                    <span className="mr-2 text-sm">🚪</span>
                    {t('header.logout', '로그아웃')}
                  </button>
                </div>
              )}
            </div>
          ) : (
            <Link 
              href="/login"
              className="text-sm font-medium text-blue-600 dark:text-blue-500 hover:underline"
              style={{ color: '#3b82f6 !important' }}
            >
              {t('header.login', '로그인')}
            </Link>
          )}
        </div>
      </div>

      {/* 서버 상태 표시 - 가운데 정렬 */}
      {isClient && apiServerUrl && (
        <div className="hidden md:block px-4 pb-2 border-t border-gray-100 dark:border-gray-800">
          <div className="flex flex-col md:flex-row md:items-center md:justify-end gap-2 md:gap-4">
            {/* 푸터 링크들 - 모바일에서는 가운데 정렬, 데스크톱에서는 오른쪽 정렬 */}
            <div className="flex items-center justify-center md:justify-end gap-3 md:mr-4 text-xs text-gray-500 dark:text-gray-400 order-2 md:order-1">
              <Link
                href="/terms"
                className="hover:text-gray-700 dark:hover:text-gray-300 transition-colors whitespace-nowrap"
              >
                {t('footer.terms', '이용약관')}
              </Link>
              <span>•</span>
              <Link
                href="/privacy"
                className="hover:text-gray-700 dark:hover:text-gray-300 transition-colors whitespace-nowrap"
              >
                {t('footer.privacy', '개인정보처리방침')}
              </Link>
            </div>
            
            {/* API 상태 표시 - 모바일에서는 가운데 정렬, 데스크톱에서는 오른쪽 정렬 */}
            <div className="flex items-center justify-center md:justify-end gap-2 order-1 md:order-2">
              <div className={`w-2 h-2 rounded-full ${
                serverStatus === 'online' ? 'bg-green-500' : 
                serverStatus === 'offline' ? 'bg-red-500' : 'bg-yellow-500'
              }`}></div>
              <span className="text-xs text-gray-600 dark:text-gray-400">
                {t('status.api', 'API')} :
                <a
                  href={`${apiServerUrl}/docs`}
                  target="_blank"
                  rel="noopener noreferrer"
                  className="text-blue-600 dark:text-blue-400 hover:underline ml-1"
                >
                  {serverStatus === 'online' ? t('status.online', '온라인') : serverStatus === 'offline' ? t('status.offline', '오프라인') : t('status.checking', '확인 중')}
                </a>
              </span>
            </div>
          </div>
        </div>
      )}

      {/* 모바일 메뉴 - 간소화된 형태 */}
      {isClient && isMobileMenuOpen && (
        <div className="md:hidden border-t border-gray-200 dark:border-gray-700 shadow-lg" style={{ backgroundColor: 'var(--header-bg)' }}>
          <div className="px-4 py-6">
            {/* 간소화된 메뉴 - 대화하기, 고객센터, 로그인만 */}
            <div className="grid grid-cols-3 gap-3">
              {/* 대화하기 */}
              <Link
                href="/chat"
                className="flex flex-col items-center justify-center p-3 rounded-xl transition-all duration-200 min-h-[80px] bg-gray-50 dark:bg-gray-800 text-gray-600 dark:text-gray-300 hover:bg-blue-50 dark:hover:bg-blue-900/20 hover:text-blue-700 dark:hover:text-blue-300 hover:shadow-md"
                onClick={() => setIsMobileMenuOpen(false)}
              >
                <MessageSquare className="w-8 h-8 mb-2" />
                <span className="text-sm font-medium text-center leading-tight">{t('header.chat', '대화하기')}</span>
              </Link>
              
              {/* 고객센터 */}
              <Link
                href="/support"
                className="flex flex-col items-center justify-center p-3 rounded-xl transition-all duration-200 min-h-[80px] bg-gray-50 dark:bg-gray-800 text-gray-600 dark:text-gray-300 hover:bg-green-50 dark:hover:bg-green-900/20 hover:text-green-700 dark:hover:text-green-300 hover:shadow-md"
                onClick={() => setIsMobileMenuOpen(false)}
              >
                <HelpCircle className="w-8 h-8 mb-2" />
                <span className="text-sm font-medium text-center leading-tight">{t('header.support', '고객센터')}</span>
              </Link>
              
              {/* 로그인/사용자 */}
              {user ? (
                <Link
                  href="/profile"
                  className="flex flex-col items-center justify-center p-3 rounded-xl transition-all duration-200 min-h-[80px] bg-gray-50 dark:bg-gray-800 text-gray-600 dark:text-gray-300 hover:bg-purple-50 dark:hover:bg-purple-900/20 hover:text-purple-700 dark:hover:text-purple-300 hover:shadow-md"
                  onClick={() => setIsMobileMenuOpen(false)}
                >
                  {user.profileImageUrl && user.profileImageUrl !== '/images/user.png' ? (
                    <img
                      src={user.profileImageUrl}
                      alt="Profile Image"
                      className="w-8 h-8 rounded-full object-cover mb-2 border border-gray-200 dark:border-gray-600"
                    />
                  ) : (
                    <div className="w-8 h-8 rounded-full bg-blue-600 text-white flex items-center justify-center mb-2 text-sm">
                      {user.username.charAt(0).toUpperCase()}
                    </div>
                  )}
                  <span className="text-sm font-medium text-center leading-tight">{user.username}</span>
                </Link>
              ) : (
                <Link
                  href="/login"
                  className="flex flex-col items-center justify-center p-3 rounded-xl transition-all duration-200 min-h-[80px] bg-blue-50 dark:bg-blue-900/20 text-blue-600 dark:text-blue-400 hover:bg-blue-100 dark:hover:bg-blue-900/30 hover:text-blue-700 dark:hover:text-blue-300 hover:shadow-md"
                  onClick={() => setIsMobileMenuOpen(false)}
                >
                  <User className="w-8 h-8 mb-2" />
                  <span className="text-sm font-medium text-center leading-tight">{t('header.login', '로그인')}</span>
                </Link>
              )}
            </div>

          </div>
        </div>
      )}
    </header>
    </>
  );
} 