'use client';

import { useState, useEffect } from 'react';
import { Download, Calendar, User, Paperclip, File, ChevronLeft } from 'lucide-react';
import Modal from './Modal';

interface Attachment {
  id: number;
  filename: string;
  original_filename: string;
  file_path: string;
  file_size: number;
  mime_type: string;
  created_at: string;
}

interface FormResponse {
  id: number;
  response_data: Record<string, any>;
  submitted_by: string;
  submitted_ip: string;
  created_at: string;
  attachments?: Attachment[];
}

interface FormField {
  id: string;
  label: string;
  type: string;
}

interface FormInfo {
  title: string;
  description: string;
  form_schema: FormField[];
  responses: FormResponse[];
  response_count: number;
}

interface ResponsesViewModalProps {
  isOpen: boolean;
  onClose: () => void;
  viewLink: string;
}

export default function ResponsesViewModal({ isOpen, onClose, viewLink }: ResponsesViewModalProps) {
  const [formInfo, setFormInfo] = useState<FormInfo | null>(null);
  const [isLoading, setIsLoading] = useState(true);
  const [selectedResponse, setSelectedResponse] = useState<FormResponse | null>(null);
  const [currentPage, setCurrentPage] = useState(1);
  const itemsPerPage = 10;

  useEffect(() => {
    if (isOpen && viewLink) {
      loadResponses();
    }
  }, [isOpen, viewLink]);

  const loadResponses = async () => {
    setIsLoading(true);
    try {
      const response = await fetch(`/api/formbuilder/responses/${viewLink}`);
      if (response.ok) {
        const data = await response.json();
        setFormInfo(data.data);
      } else {
        alert('응답을 찾을 수 없습니다.');
        onClose();
      }
    } catch (error) {
      console.error('Failed to load responses:', error);
      alert('응답을 불러오는 중 오류가 발생했습니다.');
      onClose();
    } finally {
      setIsLoading(false);
    }
  };

  const exportToCSV = () => {
    if (!formInfo || formInfo.responses.length === 0) return;

    // CSV 헤더 생성
    const headers = ['제출자', '제출 시간', ...formInfo.form_schema.map(f => f.label)];

    // CSV 데이터 생성
    const rows = formInfo.responses.map(response => {
      const row = [
        response.submitted_by,
        new Date(response.created_at).toLocaleString('ko-KR'),
        ...formInfo.form_schema.map(field => {
          const value = response.response_data[field.id];
          if (typeof value === 'boolean') return value ? '예' : '아니오';
          if (Array.isArray(value)) return value.join(', ');
          return value || '';
        })
      ];
      return row.map(cell => `"${String(cell).replace(/"/g, '""')}"`).join(',');
    });

    // CSV 문자열 생성
    const csvContent = [headers.join(','), ...rows].join('\n');

    // BOM 추가 (엑셀에서 한글 정상 표시)
    const BOM = '\uFEFF';
    const blob = new Blob([BOM + csvContent], { type: 'text/csv;charset=utf-8;' });
    const url = URL.createObjectURL(blob);

    // 다운로드
    const link = document.createElement('a');
    link.setAttribute('href', url);
    link.setAttribute('download', `${formInfo.title}_responses_${Date.now()}.csv`);
    link.click();
    URL.revokeObjectURL(url);
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleString('ko-KR', {
      year: 'numeric',
      month: '2-digit',
      day: '2-digit',
      hour: '2-digit',
      minute: '2-digit',
    });
  };

  const formatValue = (value: any) => {
    if (typeof value === 'boolean') return value ? '예' : '아니오';
    if (Array.isArray(value)) return value.join(', ');
    if (value === null || value === undefined) return '-';
    return String(value);
  };

  const handleClose = () => {
    setFormInfo(null);
    setSelectedResponse(null);
    setCurrentPage(1);
    onClose();
  };

  const handleBackToList = () => {
    setSelectedResponse(null);
  };

  // 페이지네이션 계산
  const getPaginatedResponses = () => {
    if (!formInfo) return [];
    const startIndex = (currentPage - 1) * itemsPerPage;
    const endIndex = startIndex + itemsPerPage;
    return formInfo.responses.slice(startIndex, endIndex);
  };

  const totalPages = formInfo ? Math.ceil(formInfo.responses.length / itemsPerPage) : 0;

  const goToPage = (page: number) => {
    setCurrentPage(page);
  };

  return (
    <Modal isOpen={isOpen} onClose={handleClose} size="xl">
      {isLoading ? (
        <div className="flex items-center justify-center py-20">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600"></div>
        </div>
      ) : formInfo ? (
        <>
          {/* 헤더 */}
          <div className="mb-6">
            <div className="flex flex-col md:flex-row md:items-start md:justify-between">
              <div className="flex-1">
                <h1 className="text-3xl font-bold mb-2 text-gray-900 dark:text-white">{formInfo.title}</h1>
                {formInfo.description && (
                  <p className="text-gray-600 dark:text-gray-400">{formInfo.description}</p>
                )}
              </div>
              <div className="mt-4 md:mt-0 flex items-center gap-4">
                <div className="text-center">
                  <div className="text-3xl font-bold text-blue-600">{formInfo.response_count}</div>
                  <div className="text-sm text-gray-600 dark:text-gray-400">응답 수</div>
                </div>
                <button
                  onClick={exportToCSV}
                  disabled={formInfo.responses.length === 0}
                  className="px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors disabled:opacity-50 disabled:cursor-not-allowed flex items-center gap-2"
                >
                  <Download className="w-5 h-5" />
                  CSV 다운로드
                </button>
              </div>
            </div>
          </div>

          <div className="border-t border-gray-200 dark:border-gray-700 mb-6"></div>

          {/* 응답 목록 또는 상세 보기 */}
          {selectedResponse ? (
            // 응답 상세 보기
            <div>
              <button
                onClick={handleBackToList}
                className="mb-4 flex items-center gap-2 text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-gray-200 transition-colors"
              >
                <ChevronLeft className="w-5 h-5" />
                <span>목록으로 돌아가기</span>
              </button>

              <div className="bg-gray-50 dark:bg-gray-900/30 rounded-lg p-6 border border-gray-200 dark:border-gray-700">
                <div className="flex items-center justify-between mb-4 pb-4 border-b border-gray-200 dark:border-gray-700">
                  <div className="flex items-center gap-4">
                    <div className="flex items-center gap-2 text-gray-600 dark:text-gray-400">
                      <User className="w-5 h-5" />
                      <span className="font-medium">{selectedResponse.submitted_by}</span>
                    </div>
                    <div className="flex items-center gap-2 text-gray-600 dark:text-gray-400">
                      <Calendar className="w-5 h-5" />
                      <span>{formatDate(selectedResponse.created_at)}</span>
                    </div>
                  </div>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  {formInfo.form_schema.map((field) => (
                    <div key={field.id}>
                      <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                        {field.label}
                      </label>
                      <div className="px-4 py-3 bg-white dark:bg-gray-800 rounded border border-gray-200 dark:border-gray-700 text-gray-900 dark:text-gray-100">
                        {formatValue(selectedResponse.response_data[field.id])}
                      </div>
                    </div>
                  ))}
                </div>

                {/* 첨부파일 섹션 */}
                {selectedResponse.attachments && selectedResponse.attachments.length > 0 && (
                  <div className="mt-6 pt-6 border-t border-gray-200 dark:border-gray-700">
                    <div className="flex items-center gap-2 mb-4">
                      <Paperclip className="w-5 h-5 text-gray-600 dark:text-gray-400" />
                      <span className="font-medium text-gray-700 dark:text-gray-300">
                        첨부파일 ({selectedResponse.attachments.length})
                      </span>
                    </div>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                      {selectedResponse.attachments.map((attachment) => (
                        <a
                          key={attachment.id}
                          href={attachment.file_path}
                          download={attachment.original_filename}
                          target="_blank"
                          rel="noopener noreferrer"
                          className="flex items-center gap-3 px-4 py-3 bg-white dark:bg-gray-800 rounded border border-gray-200 dark:border-gray-700 hover:border-blue-400 dark:hover:border-blue-600 transition-colors group"
                        >
                          <File className="w-5 h-5 text-gray-500 group-hover:text-blue-600 flex-shrink-0" />
                          <div className="flex-1 min-w-0">
                            <p className="text-sm font-medium text-gray-700 dark:text-gray-300 truncate group-hover:text-blue-600">
                              {attachment.original_filename}
                            </p>
                            <p className="text-xs text-gray-500 dark:text-gray-400">
                              {(attachment.file_size / 1024).toFixed(2)} KB
                            </p>
                          </div>
                          <Download className="w-5 h-5 text-gray-400 group-hover:text-blue-600 flex-shrink-0" />
                        </a>
                      ))}
                    </div>
                  </div>
                )}
              </div>
            </div>
          ) : (
            // 응답 목록
            <div>
              <div className="max-h-[500px] overflow-y-auto mb-4">
                {formInfo.responses.length === 0 ? (
                  <div className="py-12 text-center">
                    <p className="text-gray-500 dark:text-gray-400 text-lg">아직 응답이 없습니다</p>
                  </div>
                ) : (
                  <div className="space-y-3">
                    {getPaginatedResponses().map((response, index) => {
                      const globalIndex = (currentPage - 1) * itemsPerPage + index;
                      return (
                        <button
                          key={response.id}
                          onClick={() => setSelectedResponse(response)}
                          className="w-full bg-gray-50 dark:bg-gray-900/30 rounded-lg p-4 border border-gray-200 dark:border-gray-700 hover:bg-gray-100 dark:hover:bg-gray-800 hover:border-blue-400 dark:hover:border-blue-600 transition-all text-left"
                        >
                          <div className="flex items-center justify-between">
                            <div className="flex items-center gap-4">
                              <span className="text-lg font-bold text-gray-400">#{formInfo.responses.length - globalIndex}</span>
                              <div className="flex items-center gap-2 text-gray-600 dark:text-gray-400">
                                <User className="w-4 h-4" />
                                <span className="text-sm">{response.submitted_by}</span>
                              </div>
                              <div className="flex items-center gap-2 text-gray-600 dark:text-gray-400">
                                <Calendar className="w-4 h-4" />
                                <span className="text-sm">{formatDate(response.created_at)}</span>
                              </div>
                              {response.attachments && response.attachments.length > 0 && (
                                <div className="flex items-center gap-1 text-gray-600 dark:text-gray-400">
                                  <Paperclip className="w-4 h-4" />
                                  <span className="text-sm">{response.attachments.length}</span>
                                </div>
                              )}
                            </div>
                            <ChevronLeft className="w-5 h-5 text-gray-400 rotate-180" />
                          </div>
                        </button>
                      );
                    })}
                  </div>
                )}
              </div>

              {/* 페이지네이션 */}
              {totalPages > 1 && (
                <div className="flex items-center justify-center gap-2 mt-6 pt-4 border-t border-gray-200 dark:border-gray-700">
                  <button
                    onClick={() => goToPage(currentPage - 1)}
                    disabled={currentPage === 1}
                    className="px-3 py-2 rounded border border-gray-300 dark:border-gray-600 hover:bg-gray-100 dark:hover:bg-gray-800 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
                  >
                    이전
                  </button>

                  <div className="flex items-center gap-1">
                    {Array.from({ length: totalPages }, (_, i) => i + 1).map((page) => {
                      // 페이지가 많을 경우 일부만 표시
                      if (
                        page === 1 ||
                        page === totalPages ||
                        (page >= currentPage - 2 && page <= currentPage + 2)
                      ) {
                        return (
                          <button
                            key={page}
                            onClick={() => goToPage(page)}
                            className={`px-3 py-2 rounded border transition-colors ${
                              currentPage === page
                                ? 'bg-blue-600 text-white border-blue-600'
                                : 'border-gray-300 dark:border-gray-600 hover:bg-gray-100 dark:hover:bg-gray-800'
                            }`}
                          >
                            {page}
                          </button>
                        );
                      } else if (page === currentPage - 3 || page === currentPage + 3) {
                        return <span key={page} className="px-2 text-gray-500">...</span>;
                      }
                      return null;
                    })}
                  </div>

                  <button
                    onClick={() => goToPage(currentPage + 1)}
                    disabled={currentPage === totalPages}
                    className="px-3 py-2 rounded border border-gray-300 dark:border-gray-600 hover:bg-gray-100 dark:hover:bg-gray-800 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
                  >
                    다음
                  </button>
                </div>
              )}
            </div>
          )}
        </>
      ) : (
        <div className="py-12 text-center">
          <h2 className="text-2xl font-bold mb-2 text-gray-900 dark:text-white">응답을 찾을 수 없습니다</h2>
          <p className="text-gray-600 dark:text-gray-400">유효하지 않은 링크입니다.</p>
        </div>
      )}
    </Modal>
  );
}
