'use client';

import { FC, useState, KeyboardEvent, useRef, ChangeEvent, useEffect } from 'react';
import Image from 'next/image';
import { uploadFile } from '@/utils/api';

interface ChatInputProps {
  onSendMessage: (content: string, options?: any) => Promise<void>;
  isLoading: boolean;
  useRag: boolean;
  setUseRag: React.Dispatch<React.SetStateAction<boolean>>;
  useWebSearch: boolean;
  setUseWebSearch: React.Dispatch<React.SetStateAction<boolean>>;
  isStreaming?: boolean;
}

const ChatInput: FC<ChatInputProps> = ({
  onSendMessage,
  isLoading,
  useRag,
  setUseRag,
  useWebSearch,
  setUseWebSearch,
  isStreaming = false
}) => {
  const [input, setInput] = useState('');
  const [uploadedFiles, setUploadedFiles] = useState<Array<{id: string, name: string, type: string, base64?: string}>>([]);
  const textareaRef = useRef<HTMLTextAreaElement>(null);
  const fileInputRef = useRef<HTMLInputElement>(null);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    const trimmedInput = input.trim();
    if (!trimmedInput || isLoading || isStreaming) return;

    try {
      await onSendMessage(trimmedInput, {
        useRag,
        useWebSearch,
        image: uploadedFiles.find(f => f.type === 'image')?.base64
      });
      setInput('');
      // 메시지 전송 후 업로드된 파일 초기화
      setUploadedFiles([]);
    } catch (error) {
      console.error('메시지 전송 오류:', error);
    }
  };

  const handleKeyDown = (e: React.KeyboardEvent<HTMLTextAreaElement>) => {
    if (e.key === 'Enter' && !e.shiftKey) {
      e.preventDefault();
      handleSubmit(e);
    }
  };

  // 입력창 자동 높이 조절
  const handleInput = () => {
    const textarea = textareaRef.current;
    if (textarea) {
      textarea.style.height = 'auto';
      textarea.style.height = `${Math.min(textarea.scrollHeight, 200)}px`;
    }
  };

  // 컴포넌트 마운트시 높이 초기화
  useEffect(() => {
    if (textareaRef.current) {
      handleInput();
    }
  }, []);

  // 파일 업로드 처리
  const handleFileUpload = async (e: ChangeEvent<HTMLInputElement>) => {
    const files = e.target.files;
    if (!files || files.length === 0) return;
    
    const file = files[0];
    
    // 파일 타입 확인
    const acceptedTypes = [
      'text/plain', 'text/markdown', 'application/pdf',
      'application/vnd.hancom.hwp', 'application/vnd.hancom.hwpx',
      'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
      'application/vnd.ms-powerpoint', 'application/vnd.openxmlformats-officedocument.presentationml.presentation',
      'application/vnd.ms-excel', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
      'image/jpeg', 'image/png', 'image/gif'
    ];
    
    const fileExtension = file.name.split('.').pop()?.toLowerCase();
    const acceptedExtensions = ['txt', 'md', 'pdf', 'hwp', 'hwpx', 'doc', 'docx', 'ppt', 'pptx', 'xls', 'xlsx', 'jpg', 'jpeg', 'png', 'gif'];
    
    if (!acceptedTypes.includes(file.type) && !acceptedExtensions.includes(fileExtension || '')) {
      alert(`지원하지 않는 파일 형식입니다: ${file.type || fileExtension}`);
      return;
    }
    
    try {
      // 이미지인 경우 Base64로 변환하여 바로 저장
      const isImage = file.type.startsWith('image/');
      
      if (isImage) {
        const reader = new FileReader();
        reader.onloadend = () => {
          const base64String = reader.result as string;
          
          // 이미지 파일 정보 저장
          setUploadedFiles(prev => [...prev, {
            id: `image-${Date.now()}`,
            name: file.name,
            type: 'image',
            base64: base64String
          }]);
        };
        reader.readAsDataURL(file);
        
        return; // 이미지는 API 업로드 처리하지 않고 여기서 종료
      }
      
      // 파일 업로드 API 호출 (이미지가 아닌 문서 파일)
      const data = await uploadFile('/rag/upload', file);
      
      if (!data.success) {
        throw new Error(data.error?.message || '파일 업로드에 실패했습니다.');
      }
      
      // 성공적으로 업로드된 파일 정보를 상태에 추가
      const fileId = data.data?.id || `file-${Date.now()}`;
      const fileType = file.type.split('/')[1] || 'document';
      
      setUploadedFiles(prev => [...prev, {
        id: fileId,
        name: file.name,
        type: fileType
      }]);
      
      // RAG 토글 활성화 (문서 파일인 경우)
      setUseRag(true);
      
    } catch (err) {
      console.error('파일 업로드 오류:', err);
      alert(err instanceof Error ? err.message : '파일 업로드 중 오류가 발생했습니다.');
    } finally {
      // 파일 입력 초기화
      if (fileInputRef.current) {
        fileInputRef.current.value = '';
      }
    }
  };
  
  // 파일 제거
  const removeFile = (id: string) => {
    setUploadedFiles(prev => prev.filter(file => file.id !== id));
  };

  return (
    <div className="chat-input relative w-full">
      <form onSubmit={handleSubmit} className="flex flex-col w-full">
        <div className="flex flex-col w-full rounded-lg border border-gray-300 dark:border-gray-600 bg-white dark:bg-gray-700 overflow-hidden">
          <div className="relative">
            {/* 업로드된 파일 목록 */}
            {uploadedFiles.length > 0 && (
              <div className="px-3 py-2 border-b border-gray-200 dark:border-gray-600 flex flex-wrap gap-2">
                {uploadedFiles.map(file => (
                  <div key={file.id} className="relative group flex items-center px-2 py-1 bg-gray-100 dark:bg-gray-700 rounded-md border border-gray-200 dark:border-gray-600">
                    {file.type === 'image' && file.base64 ? (
                      <div className="mr-2 h-5 w-5 relative overflow-hidden rounded">
                        <Image 
                          src={file.base64} 
                          alt={file.name}
                          width={20}
                          height={20}
                          className="object-cover"
                        />
                      </div>
                    ) : (
                      <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4 mr-1 text-blue-500" viewBox="0 0 20 20" fill="currentColor">
                        <path fillRule="evenodd" d="M4 4a2 2 0 012-2h4.586A2 2 0 0112 2.586L15.414 6A2 2 0 0116 7.414V16a2 2 0 01-2 2H6a2 2 0 01-2-2V4z" clipRule="evenodd" />
                      </svg>
                    )}
                    <span className="text-xs truncate max-w-[100px]">{file.name}</span>
                    <button
                      type="button"
                      onClick={() => removeFile(file.id)}
                      className="ml-1 p-0.5 text-gray-500 hover:text-red-500 rounded-full"
                    >
                      <svg xmlns="http://www.w3.org/2000/svg" className="h-3 w-3" viewBox="0 0 20 20" fill="currentColor">
                        <path fillRule="evenodd" d="M4.293 4.293a1 1 0 011.414 0L10 8.586l4.293-4.293a1 1 0 111.414 1.414L11.414 10l4.293 4.293a1 1 0 01-1.414 1.414L10 11.414l-4.293 4.293a1 1 0 01-1.414-1.414L8.586 10 4.293 5.707a1 1 0 010-1.414z" clipRule="evenodd" />
                      </svg>
                    </button>
                  </div>
                ))}
              </div>
            )}

            <div className="flex items-end p-2 relative">
              <div className="absolute left-2 top-2 flex items-center">
                <button
                  type="button"
                  onClick={() => fileInputRef.current?.click()}
                  className="p-2 text-gray-500 hover:text-blue-500 transition-colors"
                  disabled={isLoading || isStreaming}
                  title="파일 첨부"
                >
                  <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5" viewBox="0 0 20 20" fill="currentColor">
                    <path fillRule="evenodd" d="M8 4a3 3 0 00-3 3v4a5 5 0 0010 0V7a1 1 0 112 0v4a7 7 0 11-14 0V7a5 5 0 0110 0v4a3 3 0 11-6 0V7a1 1 0 012 0v4a1 1 0 102 0V7a3 3 0 00-3-3z" clipRule="evenodd" />
                  </svg>
                </button>
                <input
                  ref={fileInputRef}
                  type="file"
                  className="hidden"
                  onChange={handleFileUpload}
                  disabled={isLoading || isStreaming}
                />
              </div>

              <div className="absolute right-2 top-2 flex items-center">
                {/* RAG 토글 버튼 */}
                <button
                  type="button"
                  onClick={() => setUseRag(!useRag)}
                  className={`p-1 rounded-full transition-colors mr-1 ${
                    useRag ? 'bg-green-100 text-green-600 dark:bg-green-900 dark:text-green-400' : 'text-gray-400 hover:text-green-500'
                  }`}
                  disabled={isLoading || isStreaming}
                  title="RAG 활성화"
                >
                  <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4" viewBox="0 0 20 20" fill="currentColor">
                    <path d="M9 4.804A7.968 7.968 0 005.5 4c-1.255 0-2.443.29-3.5.804v10A7.969 7.969 0 015.5 14c1.669 0 3.218.51 4.5 1.385A7.962 7.962 0 0114.5 14c1.255 0 2.443.29 3.5.804v-10A7.968 7.968 0 0014.5 4c-1.255 0-2.443.29-3.5.804V12a1 1 0 11-2 0V4.804z" />
                  </svg>
                </button>

                {/* 웹 검색 토글 버튼 */}
                <button
                  type="button"
                  onClick={() => setUseWebSearch(!useWebSearch)}
                  className={`p-1 rounded-full transition-colors mr-1 ${
                    useWebSearch ? 'bg-purple-100 text-purple-600 dark:bg-purple-900 dark:text-purple-400' : 'text-gray-400 hover:text-purple-500'
                  }`}
                  disabled={isLoading || isStreaming}
                  title="웹 검색 활성화"
                >
                  <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4" viewBox="0 0 20 20" fill="currentColor">
                    <path fillRule="evenodd" d="M8 4a4 4 0 100 8 4 4 0 000-8zM2 8a6 6 0 1110.89 3.476l4.817 4.817a1 1 0 01-1.414 1.414l-4.816-4.816A6 6 0 012 8z" clipRule="evenodd" />
                  </svg>
                </button>
              </div>

              <textarea
                ref={textareaRef}
                placeholder="무엇이든 물어보세요..."
                className="flex-1 p-2 pl-10 pr-10 rounded-md outline-none resize-none min-h-[40px] max-h-[200px] bg-transparent mt-8"
                value={input}
                onChange={(e) => setInput(e.target.value)}
                onKeyDown={handleKeyDown}
                onInput={handleInput}
                disabled={isLoading || isStreaming}
                rows={1}
              />

              <button
                type="submit"
                className={`absolute right-2 bottom-2 p-1.5 rounded-full ${
                  !input.trim() || isLoading || isStreaming
                    ? 'bg-gray-300 dark:bg-gray-600 text-gray-500 dark:text-gray-400 cursor-not-allowed'
                    : 'bg-blue-500 hover:bg-blue-600 text-white'
                }`}
                disabled={!input.trim() || isLoading || isStreaming}
              >
                {isLoading || isStreaming ? (
                  <svg className="animate-spin h-4 w-4" viewBox="0 0 24 24">
                    <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4" fill="none"></circle>
                    <path
                      className="opacity-75"
                      fill="currentColor"
                      d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"
                    ></path>
                  </svg>
                ) : (
                  <svg
                    xmlns="http://www.w3.org/2000/svg"
                    className="h-4 w-4"
                    viewBox="0 0 20 20"
                    fill="currentColor"
                  >
                    <path
                      d="M10.894 2.553a1 1 0 00-1.788 0l-7 14a1 1 0 001.169 1.409l5-1.429A1 1 0 009 15.571V11a1 1 0 112 0v4.571a1 1 0 00.725.962l5 1.428a1 1 0 001.17-1.408l-7-14z"
                    />
                  </svg>
                )}
              </button>
            </div>
          </div>
        </div>
      </form>
    </div>
  );
};

export default ChatInput; 