'use client';

import { useState, useRef, useEffect } from 'react';
import type { NodeData, ExecutionResult } from '@/types/flowai';
import ConfigPanel from './ConfigPanel';
import ExecutionPanel from './ExecutionPanel';

interface SlidingPanelProps {
  isOpen: boolean;
  type: 'config' | 'execution' | null;
  selectedNode?: NodeData | null;
  executionResults?: ExecutionResult[];
  isExecuting?: boolean;
  onNodeUpdate?: (nodeId: string, updates: Partial<NodeData>) => void;
  onNodeDelete?: (nodeId: string) => void;
  onClose: () => void;
}

export default function SlidingPanel({
  isOpen,
  type,
  selectedNode,
  executionResults = [],
  isExecuting = false,
  onNodeUpdate,
  onNodeDelete,
  onClose,
}: SlidingPanelProps) {
  const [isDragging, setIsDragging] = useState(false);
  // Process nodes need more width for test UI
  const defaultWidth = selectedNode?.type === 'process' ? 800 : 400;
  const [panelWidth, setPanelWidth] = useState(defaultWidth);
  const [startX, setStartX] = useState(0);
  const [startWidth, setStartWidth] = useState(0);
  const panelRef = useRef<HTMLDivElement>(null);
  const resizerRef = useRef<HTMLDivElement>(null);

  const minWidth = 300;
  const maxWidth = 600;

  useEffect(() => {
    const handleMouseMove = (e: MouseEvent) => {
      if (!isDragging) return;
      
      const deltaX = startX - e.clientX; // 왼쪽으로 드래그하면 양수
      const newWidth = Math.min(maxWidth, Math.max(minWidth, startWidth + deltaX));
      setPanelWidth(newWidth);
    };

    const handleMouseUp = () => {
      setIsDragging(false);
      document.body.style.cursor = '';
      document.body.style.userSelect = '';
    };

    if (isDragging) {
      document.addEventListener('mousemove', handleMouseMove);
      document.addEventListener('mouseup', handleMouseUp);
      document.body.style.cursor = 'ew-resize';
      document.body.style.userSelect = 'none';
    }

    return () => {
      document.removeEventListener('mousemove', handleMouseMove);
      document.removeEventListener('mouseup', handleMouseUp);
    };
  }, [isDragging, startX, startWidth]);

  const handleResizerMouseDown = (e: React.MouseEvent) => {
    e.preventDefault();
    setIsDragging(true);
    setStartX(e.clientX);
    setStartWidth(panelWidth);
  };

  const handleKeyDown = (e: React.KeyboardEvent) => {
    if (e.key === 'Escape') {
      onClose();
    }
  };

  if (!isOpen || !type) {
    return null;
  }

  return (
    <>
      {/* 배경 오버레이 */}
      <div
        className="fixed inset-0 bg-black bg-opacity-20 z-40 transition-opacity"
        onClick={onClose}
      />

      {/* 슬라이딩 패널 */}
      <div
        ref={panelRef}
        className={`fixed top-16 right-0 bottom-8 bg-white dark:bg-gray-800 border-l border-gray-200 dark:border-gray-700 shadow-xl z-50 transform transition-transform duration-300 ease-in-out flex flex-col ${
          isOpen ? 'translate-x-0' : 'translate-x-full'
        }`}
        style={{ width: panelWidth }}
        onKeyDown={handleKeyDown}
        tabIndex={-1}
      >
        {/* 리사이저 */}
        <div
          ref={resizerRef}
          className="absolute left-0 top-0 bottom-0 w-1 cursor-ew-resize bg-gray-200 dark:bg-gray-600 hover:bg-blue-500 transition-colors group"
          onMouseDown={handleResizerMouseDown}
        >
          <div className="absolute left-0 top-1/2 transform -translate-y-1/2 -translate-x-1 w-3 h-8 bg-gray-300 dark:bg-gray-600 group-hover:bg-blue-500 rounded-l opacity-0 group-hover:opacity-100 transition-opacity">
            <div className="flex items-center justify-center h-full">
              <svg className="w-3 h-3 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8 9l4-4 4 4m0 6l-4 4-4-4" />
              </svg>
            </div>
          </div>
        </div>

        {/* 패널 헤더 */}
        <div className="flex-shrink-0 flex items-center justify-between p-4 border-b border-gray-200 dark:border-gray-700">
          <h2 className="text-lg font-semibold text-gray-800 dark:text-gray-200">
            {type === 'config' ? '노드 설정' : '실행 결과'}
          </h2>
          <button
            onClick={onClose}
            className="p-2 text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200 hover:bg-gray-50 dark:hover:bg-gray-700 rounded-xl border border-transparent hover:border-gray-200 dark:hover:border-gray-600 hover:scale-[1.02] transition-all duration-150"
          >
            <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
            </svg>
          </button>
        </div>

        {/* 패널 컨텐츠 */}
        <div className="flex-1 flex flex-col min-h-0">
          {type === 'config' && selectedNode && onNodeUpdate ? (
            <ConfigPanel
              node={selectedNode}
              onNodeUpdate={onNodeUpdate}
              onNodeDelete={onNodeDelete}
              onClose={onClose}
            />
          ) : type === 'execution' ? (
            <ExecutionPanel
              results={executionResults}
              isExecuting={isExecuting}
              onClose={onClose}
            />
          ) : (
            <div className="flex items-center justify-center h-full text-gray-500 dark:text-gray-400">
              <div className="text-center">
                <div className="text-4xl mb-3">⚙️</div>
                <p className="text-sm text-gray-600 dark:text-gray-400">노드를 선택하여</p>
                <p className="text-sm text-gray-600 dark:text-gray-400">설정을 편집하세요</p>
              </div>
            </div>
          )}
        </div>


      </div>
    </>
  );
}