import { useState } from 'react';
import type { NodeConfigProps } from './types';

export default function InputNodeConfig({ node, onNodeUpdate }: NodeConfigProps) {
  const [localConfig, setLocalConfig] = useState(node.data.config || { inputType: 'text' });
  const [localData, setLocalData] = useState(node.data);

  const updateConfig = (key: string, value: any) => {
    const newConfig = { ...localConfig, [key]: value };
    setLocalConfig(newConfig);
    onNodeUpdate(node.id, {
      data: {
        ...localData,
        config: newConfig,
      },
    });
  };

  const updateData = (key: string, value: any) => {
    const newData = { ...localData, [key]: value };
    setLocalData(newData);
    onNodeUpdate(node.id, {
      data: newData,
    });
  };

  const renderInputComponent = () => {
    const inputType = localConfig.inputType || 'text';
    
    switch (inputType) {
      case 'text':
        return (
          <div>
            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
              텍스트 입력
            </label>
            <textarea
              value={localData.value || ''}
              onChange={(e) => updateData('value', e.target.value)}
              placeholder="텍스트를 입력하세요"
              rows={6}
              className="w-full px-3 py-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150 resize-none"
            />
          </div>
        );
        
      case 'number':
        return (
          <div>
            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
              숫자 입력
            </label>
            <input
              type="number"
              value={localData.value || ''}
              onChange={(e) => updateData('value', e.target.value)}
              placeholder="숫자를 입력하세요"
              min={localConfig.min}
              max={localConfig.max}
              step={localConfig.step || 1}
              className="w-full px-3 py-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150"
            />
          </div>
        );
        
      case 'file':
        return (
          <div className="space-y-3">
            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
              파일 입력
            </label>
            <div className="space-y-2">
              <div>
                <input
                  type="file"
                  accept={getFileAccept()}
                  onChange={handleFileChange}
                  className="w-full text-sm text-gray-500 dark:text-gray-400 file:mr-4 file:py-2 file:px-4 file:rounded-xl file:border-0 file:text-sm file:font-medium file:bg-blue-50 file:text-blue-700 hover:file:bg-blue-100 dark:file:bg-blue-900/20 dark:file:text-blue-400"
                />
              </div>
              <div className="text-xs text-gray-500 dark:text-gray-400">
                또는 URL 입력:
              </div>
              <input
                type="url"
                value={localData.url || ''}
                onChange={(e) => updateData('url', e.target.value)}
                placeholder="https://example.com/file.pdf"
                className="w-full px-3 py-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150"
              />
            </div>
          </div>
        );
        
      case 'image':
        return (
          <div className="space-y-3">
            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
              이미지 입력
            </label>
            <div className="space-y-2">
              <div>
                <input
                  type="file"
                  accept="image/*"
                  onChange={handleFileChange}
                  className="w-full text-sm text-gray-500 dark:text-gray-400 file:mr-4 file:py-2 file:px-4 file:rounded-xl file:border-0 file:text-sm file:font-medium file:bg-green-50 file:text-green-700 hover:file:bg-green-100 dark:file:bg-green-900/20 dark:file:text-green-400"
                />
              </div>
              <div className="text-xs text-gray-500 dark:text-gray-400">
                또는 이미지 URL 입력:
              </div>
              <input
                type="url"
                value={localData.url || ''}
                onChange={(e) => updateData('url', e.target.value)}
                placeholder="https://example.com/image.jpg"
                className="w-full px-3 py-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150"
              />
              {(localData.value || localData.url) && (
                <div className="mt-2">
                  <img 
                    src={localData.url || (localData.value ? URL.createObjectURL(localData.value) : '')} 
                    alt="Preview" 
                    className="max-w-full h-32 object-contain rounded-lg border border-gray-200 dark:border-gray-700"
                    onError={(e) => {
                      const target = e.target as HTMLImageElement;
                      target.style.display = 'none';
                    }}
                  />
                </div>
              )}
            </div>
          </div>
        );
        
      case 'json':
        return (
          <div>
            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
              JSON 입력
            </label>
            <textarea
              value={localData.value || ''}
              onChange={handleJsonChange}
              placeholder='{"key": "value"}'
              rows={8}
              className={`w-full px-3 py-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150 resize-none font-mono text-sm ${
                localData.jsonError ? 'border-red-500 focus:border-red-500 focus:ring-red-500' : ''
              }`}
            />
            {localData.jsonError && (
              <div className="text-red-500 text-xs mt-1">
                {localData.jsonError}
              </div>
            )}
          </div>
        );
        
      default:
        return null;
    }
  };

  const getFileAccept = () => {
    const fileType = localConfig.fileType || 'any';
    switch (fileType) {
      case 'text': return '.txt,.md,.csv';
      case 'image': return 'image/*';
      case 'pdf': return '.pdf';
      case 'doc': return '.doc,.docx,.odt';
      default: return '*';
    }
  };

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (file) {
      updateData('value', file);
      updateData('fileName', file.name);
      updateData('fileSize', file.size);
      updateData('fileType', file.type);
      // URL이 있다면 제거
      updateData('url', '');
    }
  };

  const handleJsonChange = (e: React.ChangeEvent<HTMLTextAreaElement>) => {
    const value = e.target.value;
    updateData('value', value);
    
    if (value.trim()) {
      try {
        JSON.parse(value);
        updateData('jsonError', '');
      } catch (error) {
        updateData('jsonError', 'JSON 형식이 올바르지 않습니다.');
      }
    } else {
      updateData('jsonError', '');
    }
  };

  return (
    <div className="space-y-4">
      <div>
        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
          입력 타입
        </label>
        <select
          value={localConfig.inputType || 'text'}
          onChange={(e) => {
            updateConfig('inputType', e.target.value);
            // 타입 변경 시 기존 값 초기화
            updateData('value', '');
            updateData('url', '');
            updateData('jsonError', '');
          }}
          className="w-full px-3 py-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150"
        >
          <option value="text">텍스트</option>
          <option value="number">숫자</option>
          <option value="file">파일</option>
          <option value="image">이미지</option>
          <option value="json">JSON</option>
        </select>
      </div>

      {/* 입력 타입별 컴포넌트 렌더링 */}
      {renderInputComponent()}

      {/* 숫자 타입 추가 설정 */}
      {localConfig.inputType === 'number' && (
        <div className="grid grid-cols-3 gap-3">
          <div>
            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
              최솟값
            </label>
            <input
              type="number"
              value={localConfig.min || ''}
              onChange={(e) => updateConfig('min', e.target.value)}
              className="w-full px-2 py-1 text-sm bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded text-gray-900 dark:text-gray-100"
            />
          </div>
          <div>
            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
              최댓값
            </label>
            <input
              type="number"
              value={localConfig.max || ''}
              onChange={(e) => updateConfig('max', e.target.value)}
              className="w-full px-2 py-1 text-sm bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded text-gray-900 dark:text-gray-100"
            />
          </div>
          <div>
            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
              단위
            </label>
            <input
              type="number"
              value={localConfig.step || 1}
              onChange={(e) => updateConfig('step', e.target.value)}
              min="0.01"
              step="0.01"
              className="w-full px-2 py-1 text-sm bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded text-gray-900 dark:text-gray-100"
            />
          </div>
        </div>
      )}

      {/* 파일 타입 추가 설정 */}
      {localConfig.inputType === 'file' && (
        <div>
          <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
            허용할 파일 형식
          </label>
          <select
            value={localConfig.fileType || 'any'}
            onChange={(e) => updateConfig('fileType', e.target.value)}
            className="w-full px-3 py-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150"
          >
            <option value="any">모든 파일</option>
            <option value="text">텍스트 파일 (.txt, .md, .csv)</option>
            <option value="image">이미지 파일</option>
            <option value="pdf">PDF 파일</option>
            <option value="doc">문서 파일 (.doc, .docx)</option>
          </select>
        </div>
      )}


    </div>
  );
}