'use client';

import React, { useState, useEffect, useCallback, useRef } from 'react';
import { XMarkIcon } from '@heroicons/react/24/outline';

interface PdfPreprocessModalProps {
  isOpen: boolean;
  onClose: () => void;
  onProcess: (options: PdfProcessOptions) => void;
}

interface PdfProcessOptions {
  file: File;
  autoRotate: boolean;
  detectTwoUp: boolean;
  deskew: boolean;
  cropMargins: number;
  maxPages: number;
  aiRecommendations?: {
    rotation: number;
    isTwoUp: boolean;
    suggestedCropMargin: number;
  };
}

interface PdfPreview {
  pageCount: number;
  pages: {
    pageNumber: number;
    thumbnail: string;
    width: number;
    height: number;
    rotation?: number;
    isTwoUp?: boolean;
  }[];
}

export default function PdfPreprocessModal({ isOpen, onClose, onProcess }: PdfPreprocessModalProps) {
  const [processingStatus, setProcessingStatus] = useState<{
    isProcessing: boolean;
    step: string;
    progress: number;
  }>({
    isProcessing: false,
    step: '',
    progress: 0
  });
  const [file, setFile] = useState<File | null>(null);
  const [preview, setPreview] = useState<PdfPreview | null>(null);
  const [isProcessing, setIsProcessing] = useState(false);
  const [currentPage, setCurrentPage] = useState(1);

  // 전처리 옵션
  const [autoRotate, setAutoRotate] = useState(true);
  const [detectTwoUp, setDetectTwoUp] = useState(true);
  const [deskew, setDeskew] = useState(false);
  const [cropMargins, setCropMargins] = useState(4);
  const [maxPages, setMaxPages] = useState(2000);
  const [aiMode, setAiMode] = useState(false);
  const [aiRecommendations, setAiRecommendations] = useState<any>(null);

  const canvasRef = useRef<HTMLCanvasElement>(null);

  // 모달 상태 초기화 함수
  const resetModalState = useCallback(() => {
    setFile(null);
    setPreview(null);
    setIsProcessing(false);
    setCurrentPage(1);
    setAutoRotate(true);
    setDetectTwoUp(true);
    setDeskew(false);
    setCropMargins(4);
    setMaxPages(2000);
    setAiMode(false);
    setAiRecommendations(null);
    setProcessingStatus({
      isProcessing: false,
      step: '',
      progress: 0
    });
  }, []);

  // 모달이 닫힐 때 상태 초기화
  useEffect(() => {
    if (!isOpen) {
      resetModalState();
    }
  }, [isOpen, resetModalState]);

  // 파일 업로드 처리
  const handleFileUpload = useCallback(
    async (e: React.ChangeEvent<HTMLInputElement>) => {
      const uploadedFile = e.target.files?.[0];
      if (!uploadedFile) return;

      setFile(uploadedFile);
      setIsProcessing(true);

      try {
        // PDF 미리보기 생성 요청
        const formData = new FormData();
        formData.append('file', uploadedFile);
        formData.append('maxPages', '10'); // 미리보기용 최대 10페이지

        const response = await fetch('/api/preprocess/pdf/preview', {
          method: 'POST',
          body: formData
        });

        if (response.ok) {
          const result = await response.json();
          console.log('PDF 미리보기 응답:', result);

          if (result.success && result.data) {
            const previewData = result.data;
            // pages 배열 확인 및 초기화
            if (!previewData.pages) {
              previewData.pages = [];
            }
            setPreview(previewData);

            // AI 모드일 경우 자동 분석 요청
            if (aiMode) {
              await requestAiAnalysis(uploadedFile, previewData);
            }
          } else {
            console.error('PDF 미리보기 생성 실패:', result);
            alert('PDF 미리보기를 생성할 수 없습니다: ' + (result.error || '알 수 없는 오류'));
          }
        } else {
          const errorText = await response.text();
          console.error('PDF 미리보기 요청 실패:', response.status, errorText);
          alert('PDF 파일 처리 중 서버 오류가 발생했습니다.');
        }
      } catch (error) {
        console.error('PDF 미리보기 생성 실패:', error);
        alert('PDF 파일을 읽는 중 오류가 발생했습니다.');
      } finally {
        setIsProcessing(false);
      }
    },
    [aiMode]
  );

  // AI 분석 요청
  const requestAiAnalysis = async (file: File, preview: PdfPreview) => {
    try {
      const formData = new FormData();
      formData.append('file', file);
      formData.append('preview', JSON.stringify(preview));

      const response = await fetch('/api/preprocess/pdf/analyze', {
        method: 'POST',
        body: formData
      });

      if (response.ok) {
        const analysis = await response.json();
        setAiRecommendations(analysis);

        // AI 추천사항 자동 적용
        if (analysis.autoRotate !== undefined) {
          setAutoRotate(analysis.autoRotate);
        }
        if (analysis.detectTwoUp !== undefined) {
          setDetectTwoUp(analysis.detectTwoUp);
        }
        if (analysis.suggestedCropMargin !== undefined) {
          setCropMargins(analysis.suggestedCropMargin);
        }
      }
    } catch (error) {
      console.error('AI 분석 요청 실패:', error);
    }
  };

  // 페이지 렌더링
  const renderPage = useCallback(() => {
    if (!preview || !preview.pages || !canvasRef.current) return;

    const page = preview.pages.find((p) => p.pageNumber === currentPage);
    if (!page) return;

    const canvas = canvasRef.current;
    const ctx = canvas.getContext('2d');
    if (!ctx) return;

    const img = new Image();
    img.onload = () => {
      // 캔버스 크기 설정
      const maxWidth = 600;
      const scale = maxWidth / img.width;
      canvas.width = img.width * scale;
      canvas.height = img.height * scale;

      // 이미지 그리기
      ctx.clearRect(0, 0, canvas.width, canvas.height);
      ctx.drawImage(img, 0, 0, canvas.width, canvas.height);

      // 2-up 감지 시각화
      if (detectTwoUp && page.isTwoUp) {
        ctx.strokeStyle = 'red';
        ctx.lineWidth = 2;
        ctx.setLineDash([5, 5]);
        ctx.beginPath();
        ctx.moveTo(canvas.width / 2, 0);
        ctx.lineTo(canvas.width / 2, canvas.height);
        ctx.stroke();

        // 라벨 추가
        ctx.fillStyle = 'red';
        ctx.font = '16px Arial';
        ctx.fillText('2-UP 감지됨', 10, 30);
      }

      // 회전 표시
      if (autoRotate && page.rotation) {
        ctx.fillStyle = 'blue';
        ctx.font = '16px Arial';
        ctx.fillText(`회전: ${page.rotation}°`, 10, 60);
      }

      // 여백 크롭 영역 표시
      if (cropMargins > 0) {
        const margin = (cropMargins / 100) * Math.min(canvas.width, canvas.height);
        ctx.strokeStyle = 'green';
        ctx.lineWidth = 1;
        ctx.setLineDash([3, 3]);
        ctx.strokeRect(margin, margin, canvas.width - 2 * margin, canvas.height - 2 * margin);
      }
    };
    img.src = page.thumbnail;
  }, [preview, currentPage, detectTwoUp, autoRotate, cropMargins]);

  useEffect(() => {
    renderPage();
  }, [renderPage]);

  // 전처리 실행
  const handleProcess = async () => {
    if (!file) return;

    const options: PdfProcessOptions = {
      file,
      autoRotate,
      detectTwoUp,
      deskew,
      cropMargins,
      maxPages,
      aiRecommendations
    };

    // 처리 중 상태로 설정
    setProcessingStatus({
      isProcessing: true,
      step: 'PDF 분석 중...',
      progress: 10
    });

    try {
      // 진행 상태를 시뮬레이션 (실제로는 서버에서 진행률을 받아와야 함)
      setTimeout(() => setProcessingStatus((prev) => ({ ...prev, step: '페이지 추출 중...', progress: 30 })), 1000);
      setTimeout(() => setProcessingStatus((prev) => ({ ...prev, step: '이미지 보정 중...', progress: 60 })), 2000);
      setTimeout(() => setProcessingStatus((prev) => ({ ...prev, step: '최종 처리 중...', progress: 90 })), 3000);

      // 전처리 실행
      await onProcess(options);

      // 완료 후 상태 초기화 및 모달 닫기
      setTimeout(() => {
        setProcessingStatus({ isProcessing: false, step: '', progress: 0 });
        onClose();
      }, 4000);
    } catch (error) {
      console.error('전처리 실패:', error);
      setProcessingStatus({ isProcessing: false, step: '', progress: 0 });
      alert('전처리 중 오류가 발생했습니다.');
    }
  };

  if (!isOpen) return null;

  // 전처리 중일 때 표시할 오버레이
  if (processingStatus.isProcessing) {
    return (
      <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
        <div className="rounded-lg shadow-xl w-full max-w-md p-8" style={{ backgroundColor: 'var(--sidebar-bg)', borderColor: 'var(--card-border)' }}>
          <div className="text-center">
            {/* 로딩 스피너 */}
            <div className="mb-6 flex justify-center">
              <div className="relative">
                <div className="w-16 h-16 border-4 rounded-full animate-pulse" style={{ borderColor: 'var(--primary-200)' }}></div>
                <div className="absolute top-0 left-0 w-16 h-16 border-4 rounded-full border-t-transparent animate-spin" style={{ borderColor: 'var(--primary-600)' }}></div>
              </div>
            </div>

            {/* 상태 메시지 */}
            <h3 className="text-xl font-semibold mb-2" style={{ color: 'var(--text-primary)' }}>
              PDF 파일 전처리 중
            </h3>
            <p className="text-sm mb-6" style={{ color: 'var(--text-secondary)' }}>
              {processingStatus.step || '처리 중입니다...'}
            </p>

            {/* 프로그레스 바 */}
            <div className="w-full rounded-full h-2 mb-4" style={{ backgroundColor: 'var(--neutral-200)' }}>
              <div
                className="h-2 rounded-full transition-all duration-300"
                style={{
                  width: `${processingStatus.progress}%`,
                  backgroundColor: 'var(--primary-600)'
                }}
              ></div>
            </div>
            <p className="text-sm" style={{ color: 'var(--text-muted)' }}>
              {processingStatus.progress}% 완료
            </p>

            {/* 안내 메시지 */}
            <div className="mt-6 p-3 rounded-lg border" style={{ backgroundColor: 'var(--info-bg)', borderColor: 'var(--info-border)' }}>
              <p className="text-xs" style={{ color: 'var(--info)' }}>
                💡 파일 크기와 페이지 수에 따라 처리 시간이 다를 수 있습니다.
              </p>
            </div>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
      <div className="rounded-lg shadow-xl w-full max-w-6xl max-h-[90vh] overflow-hidden" style={{ backgroundColor: 'var(--sidebar-bg)', borderColor: 'var(--card-border)' }}>
        {/* 헤더 */}
        <div
          className="flex items-center justify-between p-6 border-b"
          style={{
            backgroundColor: 'var(--sidebar-bg)',
            borderColor: 'var(--card-border)'
          }}
        >
          <h2 className="text-2xl font-bold" style={{ color: 'var(--text-primary)' }}>
            PDF 전처리
          </h2>
          <button
            onClick={onClose}
            className="p-2 rounded-lg transition-colors"
            style={{
              backgroundColor: 'transparent',
              color: 'var(--text-secondary)'
            }}
            onMouseEnter={(e) => {
              e.currentTarget.style.backgroundColor = 'var(--card-hover)';
              e.currentTarget.style.color = 'var(--text-primary)';
            }}
            onMouseLeave={(e) => {
              e.currentTarget.style.backgroundColor = 'transparent';
              e.currentTarget.style.color = 'var(--text-secondary)';
            }}
          >
            <XMarkIcon className="w-6 h-6" />
          </button>
        </div>

        {/* 본문 */}
        <div className="p-6 overflow-y-auto max-h-[calc(90vh-180px)]" style={{ backgroundColor: 'var(--sidebar-bg)' }}>
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
            {/* 왼쪽: 설정 */}
            <div>
              {/* 파일 업로드 */}
              <div className="card p-4 mb-6" style={{ backgroundColor: 'var(--card-bg)' }}>
                <label className="block text-base font-medium mb-2" style={{ color: 'var(--text-primary)' }}>
                  PDF 파일 선택
                </label>
                <input type="file" accept=".pdf" onChange={handleFileUpload} className="block w-full text-base file:mr-4 file:py-2 file:px-4 file:rounded-full file:border-0 file:text-base file:font-semibold file:bg-blue-50 file:text-blue-700 hover:file:bg-blue-100" style={{ color: 'var(--text-secondary)' }} />
              </div>

              {/* AI 모드 */}
              <div className="card p-4 mb-6" style={{ backgroundColor: 'var(--card-bg)' }}>
                <label className="flex items-center">
                  <input type="checkbox" checked={aiMode} onChange={(e) => setAiMode(e.target.checked)} className="mr-2" />
                  <span className="text-base font-medium" style={{ color: 'var(--text-primary)' }}>
                    AI 자동 분석 사용
                  </span>
                </label>
              </div>

              {/* AI 분석 결과 */}
              {aiMode && aiRecommendations && (
                <div className="card p-4 mb-6" style={{ backgroundColor: 'var(--primary-100)' }}>
                  <h3 className="font-semibold mb-2" style={{ color: 'var(--primary-900)' }}>
                    AI 분석 결과
                  </h3>
                  <ul className="text-base space-y-1" style={{ color: 'var(--primary-700)' }}>
                    {aiRecommendations.isTwoUp && <li>• 2-up 레이아웃 감지됨 (페이지 분할 권장)</li>}
                    {aiRecommendations.rotation && <li>• {aiRecommendations.rotation}° 회전 필요</li>}
                    {aiRecommendations.hasSkew && <li>• 기울기 보정 필요</li>}
                    {aiRecommendations.suggestedCropMargin && <li>• 권장 여백 크롭: {aiRecommendations.suggestedCropMargin}%</li>}
                  </ul>
                </div>
              )}

              {/* 전처리 옵션 */}
              <div className="card p-4 space-y-4" style={{ backgroundColor: 'var(--card-bg)' }}>
                <h3 className="text-base font-medium" style={{ color: 'var(--text-primary)' }}>
                  전처리 옵션
                </h3>

                <label className="flex items-center">
                  <input type="checkbox" checked={autoRotate} onChange={(e) => setAutoRotate(e.target.checked)} className="mr-2" />
                  <span className="text-base" style={{ color: 'var(--text-primary)' }}>
                    자동 회전 감지
                  </span>
                </label>

                <label className="flex items-center">
                  <input type="checkbox" checked={detectTwoUp} onChange={(e) => setDetectTwoUp(e.target.checked)} className="mr-2" />
                  <span className="text-base" style={{ color: 'var(--text-primary)' }}>
                    2-up 레이아웃 감지 및 분할
                  </span>
                </label>

                <label className="flex items-center">
                  <input type="checkbox" checked={deskew} onChange={(e) => setDeskew(e.target.checked)} className="mr-2" />
                  <span className="text-base" style={{ color: 'var(--text-primary)' }}>
                    기울기 보정 (데스큐)
                  </span>
                </label>

                <div>
                  <label className="block text-base mb-1" style={{ color: 'var(--text-primary)' }}>
                    여백 크롭 (%)
                  </label>
                  <input type="range" min="0" max="20" value={cropMargins} onChange={(e) => setCropMargins(Number(e.target.value))} className="w-full" />
                  <span className="text-xs" style={{ color: 'var(--text-muted)' }}>
                    {cropMargins}%
                  </span>
                </div>

                <div>
                  <label className="block text-base mb-1" style={{ color: 'var(--text-primary)' }}>
                    최대 페이지 수
                  </label>
                  <input
                    type="number"
                    min="1"
                    max="10000"
                    value={maxPages}
                    onChange={(e) => setMaxPages(Number(e.target.value))}
                    className="w-full px-3 py-2 border rounded-lg input"
                    style={{
                      borderColor: 'var(--card-border)',
                      backgroundColor: 'var(--card-bg)',
                      color: 'var(--text-primary)'
                    }}
                  />
                </div>
              </div>

              {/* 파일 정보 */}
              {preview && (
                <div className="card p-4 mt-6" style={{ backgroundColor: 'var(--card-bg)' }}>
                  <h3 className="font-semibold mb-2" style={{ color: 'var(--text-primary)' }}>
                    파일 정보
                  </h3>
                  <ul className="text-base space-y-1" style={{ color: 'var(--text-primary)' }}>
                    <li>총 페이지: {preview.pageCount}</li>
                    {detectTwoUp && preview.pages?.some((p) => p.isTwoUp) && <li className="text-orange-600 dark:text-orange-400">2-up 페이지 감지: {preview.pages?.filter((p) => p.isTwoUp).length}개</li>}
                    {detectTwoUp && preview.pages?.some((p) => p.isTwoUp) && <li className="text-green-600 dark:text-green-400">예상 출력 페이지: {preview.pageCount + (preview.pages?.filter((p) => p.isTwoUp).length || 0)}</li>}
                  </ul>
                </div>
              )}
            </div>

            {/* 오른쪽: 미리보기 */}
            <div>
              <h3 className="text-base font-medium mb-2" style={{ color: 'var(--text-primary)' }}>
                페이지 미리보기
              </h3>

              {/* 페이지 네비게이션 */}
              {preview && preview.pages && preview.pages.length > 0 && (
                <div className="flex items-center justify-between mb-4">
                  <button
                    onClick={() => setCurrentPage(Math.max(1, currentPage - 1))}
                    disabled={currentPage === 1}
                    className="px-3 py-1 rounded disabled:opacity-50"
                    style={{
                      backgroundColor: 'var(--card-bg)',
                      color: 'var(--text-primary)',
                      borderColor: 'var(--card-border)'
                    }}
                  >
                    이전
                  </button>
                  <span className="text-base" style={{ color: 'var(--text-primary)' }}>
                    {currentPage} / {preview.pages.length}
                  </span>
                  <button
                    onClick={() => setCurrentPage(Math.min(preview.pages.length, currentPage + 1))}
                    disabled={currentPage === preview.pages.length}
                    className="px-3 py-1 rounded disabled:opacity-50"
                    style={{
                      backgroundColor: 'var(--card-bg)',
                      color: 'var(--text-primary)',
                      borderColor: 'var(--card-border)'
                    }}
                  >
                    다음
                  </button>
                </div>
              )}

              {/* 캔버스 미리보기 */}
              <div className="border rounded-lg overflow-hidden" style={{ borderColor: 'var(--card-border)' }}>
                <canvas ref={canvasRef} className="w-full" style={{ maxHeight: '600px', objectFit: 'contain' }} />
              </div>

              {/* 범례 */}
              <div className="mt-4 text-xs space-y-1" style={{ color: 'var(--text-secondary)' }}>
                <div className="flex items-center gap-2">
                  <div className="w-4 h-0.5 bg-red-500"></div>
                  <span>2-up 분할선</span>
                </div>
                <div className="flex items-center gap-2">
                  <div className="w-4 h-0.5 bg-green-500" style={{ borderStyle: 'dashed', borderWidth: '1px 0' }}></div>
                  <span>크롭 영역</span>
                </div>
              </div>
            </div>
          </div>
        </div>

        {/* 푸터 */}
        <div
          className="flex justify-end gap-3 p-6 border-t"
          style={{
            backgroundColor: 'var(--sidebar-bg)',
            borderColor: 'var(--card-border)'
          }}
        >
          <button
            onClick={onClose}
            className="btn btn-secondary px-4 py-2"
            style={{
              backgroundColor: 'var(--btn-outline-bg)',
              color: 'var(--btn-outline-fg)',
              border: '1px solid var(--btn-outline-border)'
            }}
          >
            취소
          </button>
          <button
            onClick={handleProcess}
            disabled={!file || isProcessing}
            className="btn btn-primary px-4 py-2 disabled:opacity-50 disabled:cursor-not-allowed"
            style={{
              backgroundColor: 'var(--btn-primary-bg)',
              color: 'var(--btn-primary-fg)'
            }}
          >
            {isProcessing ? '처리 중...' : '전처리 시작'}
          </button>
        </div>
      </div>
    </div>
  );
}
