import { NextRequest, NextResponse } from 'next/server';

// 인증이 필요하지 않은 경로
const publicPaths = [
  '/',
  '/login',
  '/support',
  '/form',
  '/api/auth/login',
  '/api/auth/logout',
  '/api/formbuilder/form',
  '/favicon.ico',
  '/_next',
  '/images',
  '/lang',
];

// 차단할 악의적인 파일 확장자
const BLOCKED_EXTENSIONS = [
  '.php', '.php3', '.php4', '.php5', '.phtml',
  '.asp', '.aspx', '.jsp', '.jspx',
  '.cgi', '.pl', '.py', '.rb', '.sh',
  '.exe', '.dll', '.bat', '.cmd',
  '.sql', '.db', '.sqlite',
  '.env', '.config', '.ini', '.conf',
  '.bak', '.backup', '.old', '.tmp',
  '.git', '.svn', '.htaccess', '.htpasswd'
];

// 차단할 악의적인 경로 패턴
const MALICIOUS_PATTERNS = [
  /\.\./,                    // Path traversal
  /\/\.\./,                  // Path traversal
  /\0/,                      // Null byte injection
  /%00/,                     // Null byte (encoded)
  /%2e%2e/i,                 // Path traversal (encoded)
  /union.*select/i,          // SQL injection
  /select.*from/i,           // SQL injection
  /insert.*into/i,           // SQL injection
  /delete.*from/i,           // SQL injection
  /drop.*table/i,            // SQL injection
  /update.*set/i,            // SQL injection
  /\/etc\/passwd/i,          // System file access
  /\/proc\//i,               // System file access
  // /\/admin\//i,           // NOTE: Removed - this blocks legitimate app admin pages
  /\/phpmyadmin/i,           // Admin tool scanning
  /\/wp-admin/i,             // WordPress scanning
  /\/wp-content/i,           // WordPress scanning
  /\/wp-includes/i,          // WordPress scanning
  /\/xmlrpc\.php/i,          // WordPress vulnerability
  /\/setup\.php/i,           // Setup file scanning
  /\/install\.php/i,         // Install file scanning
  /\/config\.php/i,          // Config file scanning
  /eval\(/i,                 // Code injection
  /base64_decode/i,          // Suspicious function
  /system\(/i,               // Command injection
  /exec\(/i,                 // Command injection
  /passthru\(/i,             // Command injection
  /shell_exec/i,             // Command injection
  /<script/i,                // XSS attempt
  /javascript:/i,            // XSS attempt
  /onerror=/i,               // XSS attempt
  /onload=/i,                // XSS attempt
];

// Rate limiting용 간단한 인메모리 스토어 (프로덕션에서는 Redis 사용 권장)
const rateLimitStore = new Map<string, { count: number; resetTime: number }>();
const RATE_LIMIT_WINDOW = 60000; // 1분
const RATE_LIMIT_MAX_REQUESTS = 100; // 1분당 최대 100 요청

/**
 * 보안 위협 로깅 및 API 저장
 */
async function logSecurityThreat(
  type: string,
  pathname: string,
  ip: string,
  userAgent: string,
  details?: string
) {
  const timestamp = new Date().toISOString();

  // 콘솔 로깅
  console.warn(`[SECURITY ALERT] ${timestamp}`);
  console.warn(`  Type: ${type}`);
  console.warn(`  Path: ${pathname}`);
  console.warn(`  IP: ${ip}`);
  console.warn(`  User-Agent: ${userAgent}`);
  if (details) {
    console.warn(`  Details: ${details}`);
  }
  console.warn('---');

  // 보안 로그 저장 (비동기, 실패해도 요청 차단에는 영향 없음)
  try {
    // 동적 import로 securityLogStore 가져오기
    const { securityLogStore } = await import('./app/api/admin/security/route');
    securityLogStore.addLog({
      timestamp,
      type,
      path: pathname,
      ip,
      userAgent,
      reason: details,
    });
  } catch (error) {
    // 로그 저장 실패는 무시 (요청 차단이 우선)
    console.error('Failed to save security log:', error);
  }
}

/**
 * Rate limiting 체크
 */
function checkRateLimit(ip: string): boolean {
  const now = Date.now();
  const record = rateLimitStore.get(ip);

  if (!record || now > record.resetTime) {
    // 새로운 윈도우 시작
    rateLimitStore.set(ip, {
      count: 1,
      resetTime: now + RATE_LIMIT_WINDOW,
    });
    return true;
  }

  if (record.count >= RATE_LIMIT_MAX_REQUESTS) {
    return false;
  }

  record.count++;
  return true;
}

/**
 * 악의적인 요청인지 검사
 */
function isMaliciousRequest(pathname: string, searchParams: URLSearchParams): {
  isMalicious: boolean;
  reason?: string
} {
  // 1. 차단할 파일 확장자 체크
  const lowerPath = pathname.toLowerCase();
  for (const ext of BLOCKED_EXTENSIONS) {
    if (lowerPath.endsWith(ext) || lowerPath.includes(ext + '?')) {
      return { isMalicious: true, reason: `Blocked file extension: ${ext}` };
    }
  }

  // 2. 악의적인 패턴 체크 (경로)
  for (const pattern of MALICIOUS_PATTERNS) {
    if (pattern.test(pathname)) {
      return { isMalicious: true, reason: `Malicious pattern in path: ${pattern}` };
    }
  }

  // 3. 악의적인 패턴 체크 (쿼리 파라미터)
  const queryString = searchParams.toString();
  if (queryString) {
    for (const pattern of MALICIOUS_PATTERNS) {
      if (pattern.test(queryString)) {
        return { isMalicious: true, reason: `Malicious pattern in query: ${pattern}` };
      }
    }
  }

  return { isMalicious: false };
}

export function middleware(request: NextRequest) {
  const { pathname, searchParams } = request.nextUrl;
  const ip = request.ip || request.headers.get('x-forwarded-for') || 'unknown';
  const userAgent = request.headers.get('user-agent') || 'unknown';

  // 1. Rate limiting 체크
  if (!checkRateLimit(ip)) {
    logSecurityThreat('RATE_LIMIT_EXCEEDED', pathname, ip, userAgent);
    return new NextResponse('Too Many Requests', { status: 429 });
  }

  // 2. 악의적인 요청 체크
  const { isMalicious, reason } = isMaliciousRequest(pathname, searchParams);
  if (isMalicious) {
    logSecurityThreat('MALICIOUS_REQUEST_BLOCKED', pathname, ip, userAgent, reason);
    return new NextResponse('Forbidden', { status: 403 });
  }

  // 3. 공개 경로는 통과
  if (publicPaths.some(path => pathname.startsWith(path))) {
    return NextResponse.next();
  }

  // 4. 인증 상태 확인
  const authCookie = request.cookies.get('auth_token');
  const isAuthenticated = !!authCookie?.value;

  // 5. 로그인 상태에서 로그인 페이지에 접근하면 홈으로 리디렉션
  if (isAuthenticated && pathname === '/login') {
    return NextResponse.redirect(new URL('/', request.url));
  }

  // 6. 인증되지 않은 상태에서 보호된 경로에 접근하면 로그인 페이지로 리디렉션
  if (!isAuthenticated && pathname !== '/login') {
    const loginUrl = new URL('/login', request.url);
    loginUrl.searchParams.set('redirect', pathname);
    return NextResponse.redirect(loginUrl);
  }

  return NextResponse.next();
}

export const config = {
  matcher: [
    // 모든 경로에 적용 (API 경로, 정적 파일, 이미지는 제외)
    '/((?!api|_next/static|_next/image|favicon.ico|images).*)',
  ],
}; 