#!/bin/bash

# 서비스 등급 전환 스크립트
# 사용법: ./switch-service-tier.sh [basic|standard|premium|1|2|3]

SCRIPT_DIR=$(cd $(dirname $0) && pwd)
if [ "$NODE_ENV" = "production" ]; then
    ENV_FILE="$SCRIPT_DIR/.env.production.local"
else
    ENV_FILE="$SCRIPT_DIR/.env.development.local"
fi


# 색상 정의
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
PURPLE='\033[0;35m'
NC='\033[0m' # No Color

# 서비스 등급 정보 표시
show_tier_info() {
    echo ""
    echo -e "${PURPLE}🚀 하모나이즈 - 당신만의 소버린 AI 플랫폼${NC}"
    echo -e "${BLUE}======== 인프라 기반 AI 도입 로드맵 ========${NC}"
    echo ""
    echo -e "${GREEN}📝 BASIC 등급 (1) (AI 경험 축적):${NC}"
    echo "  - 인프라: GPU 없음 | 전략: External LLM + RAG"
    echo "  - 핵심 가치: 최소 투자로 AI 도입 효과 검증"
    echo "  - 제공 메뉴: 대화하기, 문서(RAG), 고객센터, RAG 관리, 사용자 관리, 설정, 모니터링"
    echo ""
    echo -e "${BLUE}⚡ STANDARD 등급 (2) (자체 AI 구축):${NC}"
    echo "  - 인프라: 추론용 GPU | 전략: Local LLM + 고도화 워크플로우"
    echo "  - 핵심 가치: 온프레미스 AI로 데이터 보안 + 성능 최적화"
    echo "  - 제공 메뉴: 대화하기, 코드, 자동화, 문서(RAG), 고객센터, RAG 관리, 평가, 사용자 관리, 설정, 모니터링"
    echo ""
    echo -e "${PURPLE}🎯 PREMIUM 등급 (3) (완전한 AI 주권):${NC}"
    echo "  - 인프라: 학습용 GPU | 전략: 자체 모델 개발 + 조직 특화 AI"
    echo "  - 핵심 가치: 완전한 AI 자립 + 차별화된 AI 경쟁력 확보"
    echo "  - 제공 메뉴: 모든 메뉴 + 모델 관리(파인튜닝)"
    echo ""
    echo -e "${BLUE}📝 BASIC(1) → ⚡ STANDARD(2) → 🎯 PREMIUM(3)${NC}"
    echo -e "${BLUE}===========================================${NC}"
    echo ""
}

# 숫자를 등급명으로 변환
convert_number_to_tier() {
    case $1 in
        1) echo "basic" ;;
        2) echo "standard" ;;
        3) echo "premium" ;;
        *) echo "$1" ;;
    esac
}

# 사용법 표시
show_usage() {
    echo ""
    echo -e "${YELLOW}사용법: $0 [basic|standard|premium|1|2|3]${NC}"
    echo ""
    echo "예시:"
    echo "  $0 basic     # Basic 등급으로 전환"
    echo "  $0 standard  # Standard 등급으로 전환"
    echo "  $0 premium   # Premium 등급으로 전환"
    echo "  $0 1         # Basic 등급으로 전환"
    echo "  $0 2         # Standard 등급으로 전환"
    echo "  $0 3         # Premium 등급으로 전환"
    echo ""
}

# 서비스 등급 전환
switch_tier() {
    local tier=$1
    local source_file="$SCRIPT_DIR/.env.$tier"
    
    if [ ! -f "$source_file" ]; then
        echo -e "${RED}❌ 오류: $source_file 파일이 존재하지 않습니다.${NC}"
        return 1
    fi
    

    
    # 새로운 설정 복사
    cp "$source_file" "$ENV_FILE"
    
    echo -e "${GREEN}✅ $tier 등급으로 전환 완료${NC}"
    echo -e "${BLUE}📁 설정 파일: $ENV_FILE${NC}"
    
    # 현재 설정 확인
    echo ""
    echo -e "${YELLOW}현재 설정:${NC}"
    echo "  서비스 등급: $(grep NEXT_PUBLIC_SERVICE_TIER "$ENV_FILE" | cut -d'=' -f2)"
    echo "  활성화된 메뉴:"
    grep "NEXT_PUBLIC_ENABLE_.*=true" "$ENV_FILE" | sed 's/NEXT_PUBLIC_ENABLE_//g' | sed 's/=true//g' | sed 's/^/    - /'
    echo ""
    echo -e "${BLUE}💡 변경사항을 적용하려면 개발 서버를 재시작하세요.${NC}"
    echo ""
}

# 현재 상태 확인
check_current_status() {
    if [ -f "$ENV_FILE" ]; then
        local current_tier=$(grep NEXT_PUBLIC_SERVICE_TIER "$ENV_FILE" | cut -d'=' -f2)
        echo -e "${BLUE}현재 서비스 등급: ${current_tier}${NC}"
    else
        echo -e "${YELLOW}현재 설정 파일이 없습니다.${NC}"
    fi
}

# 메인 로직
main() {
    clear
    echo -e "${PURPLE}🚀 AI.RUN 서비스 등급 전환 도구${NC}"
    echo ""
    
    if [ $# -eq 0 ]; then
        show_tier_info
        check_current_status
        show_usage
        exit 0
    fi
    
    local input_tier=$1
    local tier=$(convert_number_to_tier "$input_tier")
    
    case $tier in
        basic|standard|premium)
            show_tier_info
            check_current_status
            echo -e "${YELLOW}🔄 $tier 등급으로 전환 중...${NC}"
            echo ""
            switch_tier "$tier"
            ;;
        *)
            echo -e "${RED}❌ 잘못된 서비스 등급입니다: $input_tier${NC}"
            echo -e "${RED}   지원되는 값: basic, standard, premium, 1, 2, 3${NC}"
            show_usage
            exit 1
            ;;
    esac
}

# 스크립트 실행
main "$@" 