# AI.RUN API 가이드 (v1)

## 개요

AI Run API 서버는 AI 관련 다양한 기능을 제공하는 RESTful API 서버입니다. 이 서버는 코드 생성, 채팅, 에이전트 실행, 보고서 생성, RAG(Retrieval-Augmented Generation) 및 웹 검색 등 다양한 AI 기능을 API 형태로 제공합니다.

이 서버는 Swagger를 통해 API 문서화를 제공하며, API 사용량 제한 및 모니터링 기능을 포함하고 있습니다.

### API 서버 특징

- **Swagger 문서화**: 실시간 API 문서 및 테스트 환경 제공
- **API 사용량 제한**: 라이선스 기반 API 사용량 제한 기능
- **RAG 기능**: 문서 검색 및 AI 응답 생성에 활용
- **웹 검색**: 최신 정보를 검색하여 AI 응답에 활용
- **다중 세션 관리**: 사용자별 대화 세션 관리
- **보안 기능**: Helmet, Rate Limiting 등 보안 기능 적용

## API 문서 보기

- URL: http://localhost:5500/api/v1/docs
- 특징:
  - 인터랙티브한 API 문서 제공
  - API 엔드포인트 직접 테스트 가능
  - 요청/응답 스키마 자동 생성
  - 실시간 API 문서 동기화


## 빠른 시작 가이드

### 1. 클라이언트 환경 구성

AI.RUN API를 사용하기 위한 클라이언트 환경을 구성하는 방법입니다:

#### Linux/macOS 환경
```bash
# 설치 스크립트 다운로드
curl -o airun-setup.zip http://localhost:5500/api/v1/client/setup

# 압축 해제
unzip airun-setup.zip

# 설치 스크립트 실행 권한 부여
chmod +x client-setup.sh

# 설치 스크립트 실행
./client-setup.sh
```

#### Windows 환경
```powershell
# 설치 스크립트 다운로드
Invoke-WebRequest -Uri "http://localhost:5500/api/v1/client/setup" -OutFile "airun-setup.zip"

# 압축 해제
Expand-Archive -Path "airun-setup.zip" -DestinationPath "."

# 설치 스크립트 실행
.\client-setup.ps1
```

#### 설치 확인
설치가 완료되면 다음과 같은 구성 요소가 설치됩니다:
- Python 가상환경 (`~/.airun_venv`)
- 설정 파일 (`~/.airun/airun.conf`)
- 실행 스크립트 (`~/.airun/run.sh` 또는 `run.bat`)
- utils 패키지 및 user_tools.py

#### SMTP 설정
이메일 기능을 사용하려면 `~/.airun/airun.conf` 파일에서 SMTP 설정을 구성하세요:
```bash
# SMTP 설정 예시
export SMTP_SERVER="smtp.gmail.com"
export SMTP_PORT="587"
export SMTP_USERNAME="your-email@gmail.com"
export SMTP_PASSWORD="your-app-password"
export USE_SMTP="YES"
```

### 2. API 서버 설치 및 실행

AI.RUN API 서버는 다음과 같은 방식으로 실행할 수 있습니다:

#### 1) PM2를 사용한 실행 (기본 방식, 모든 운영체제 지원)
```bash
# 서버 백그라운드 실행 (PM2 사용)
airun api-server-start

# 서버 상태 확인
airun api-server-status

# 서버 중지
airun api-server-stop

# 포그라운드 모드로 실행 (개발 시):
airun api-server-start --foreground

# PM2 로그 확인
pm2 logs airun-api

# PM2 상태 확인
pm2 status airun-api
```

#### 2) systemd 서비스를 통한 실행 (Linux 전용)
> **참고**: 
> - 이 방식은 Linux 시스템에서만 사용 가능합니다.
> - Windows나 macOS에서는 PM2 방식을 사용해주세요.
> 
> **중요**: systemd 방식을 사용하려면 먼저 PM2로 실행 중인 서버를 중지해야 합니다:
> ```bash
> airun api-server-stop
> pm2 delete airun-api  # PM2 프로세스 완전 제거
> ```

systemd 서비스 설정 및 사용:
```bash
# 서비스 시작
sudo systemctl start airun-api

# 서비스 중지
sudo systemctl stop airun-api

# 서비스 상태 확인
sudo systemctl status airun-api

# 서비스 로그 확인
sudo journalctl -u airun-api -f

# 서버 자동 시작 설정
sudo systemctl enable airun-api

# 서버 자동 시작 해제
sudo systemctl disable airun-api
```

#### 실행 방식 비교

| 기능 | PM2 (기본) | systemd (Linux 전용) |
|------|------------|-------------------|
| 지원 OS | 모든 OS | Linux만 지원 |
| 자동 재시작 | ✓ | ✓ |
| 로그 관리 | ✓ | ✓ |
| 프로세스 모니터링 | ✓ | ✓ |
| 시스템 서비스 통합 | ✗ | ✓ |
| 의존성 관리 | 제한적 | ✓ |
| 리소스 제한 | 제한적 | ✓ |
| 사용자 권한 | 일반 사용자 | root 필요 |
| 설정 복잡도 | 낮음 | 높음 |

##### PM2 방식 (권장)
- 장점:
  - 모든 운영체제 지원 (Windows, macOS, Linux)
  - 설치와 사용이 간단
  - root 권한 불필요
  - `airun` CLI 명령어와 통합
  - 개발 환경에 적합
- 단점:
  - 시스템 서비스로 통합되지 않음
  - 리소스 제한 기능이 제한적

##### systemd 방식 (Linux 전용)
- 장점:
  - 시스템 서비스로 완벽한 통합
  - 강력한 의존성 관리
  - 세밀한 리소스 제한
  - 중앙 집중식 로그 관리
- 단점:
  - Linux 시스템에서만 사용 가능
  - root 권한 필요
  - 설정이 더 복잡
  - `airun` CLI와 통합되지 않음

> **권장사항**:
> - Windows/macOS: PM2 방식 사용
> - Linux 개발 환경: PM2 방식 사용
> - Linux 프로덕션 서버:
>   - 일반 사용자 환경: PM2 방식
>   - 엔터프라이즈 환경: systemd 방식

로그 파일 위치:
- PM2 방식: `~/.pm2/logs/airun-api-*.log`
- systemd 방식:
  - 일반 로그: `/var/log/airun/api.log`
  - 에러 로그: `/var/log/airun/api.error.log`
  - 시스템 로그: `journalctl -u airun-api`

### 3. Swagger UI를 통한 API 테스트

1. 브라우저에서 http://localhost:5500/api/v1/docs 접속
2. 테스트하고자 하는 API 엔드포인트 선택
3. "Try it out" 버튼 클릭
4. 요청 파라미터 입력
5. "Execute" 버튼으로 API 호출
6. 응답 결과 확인

### 4. API 호출 예제

#### 큐 시스템을 사용하는 API 호출
큐 시스템을 사용하는 API는 비동기로 처리되며, 작업 ID를 통해 상태를 확인할 수 있습니다.

##### 1. 요청 보내기
```javascript
// 채팅 API 호출 예시
async function sendChatRequest() {
  const response = await fetch('http://localhost:5500/api/v1/chat', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({
      prompt: "안녕하세요! AI.RUN API를 테스트 중입니다.",
      chat_id: "chat-123",  // 선택적: 세션 ID
      user_id: "user-456",  // 선택적: 사용자 ID
      rag: false,          // 선택적: RAG 사용 여부
      web: false          // 선택적: 웹 검색 사용 여부
    })
  });
  
  const data = await response.json();
  console.log('작업 ID:', data.jobId);
  return data.jobId;
}
```

##### 2. 작업 상태 확인
```javascript
// 작업 상태 확인 예시
async function checkJobStatus(jobId) {
  const response = await fetch(`http://localhost:5500/api/v1/chat/status/${jobId}`);
  const data = await response.json();
  console.log('작업 상태:', data.data.state);
  return data;
}
```

##### 3. 큐 시스템을 사용하는 엔드포인트 목록

| 엔드포인트 | 설명 | 상태 확인 엔드포인트 |
|-----------|------|-------------------|
| POST /api/v1/chat | 채팅 요청 | GET /api/v1/chat/status/{jobId} |
| POST /api/v1/code | 코드 생성 요청 | GET /api/v1/code/status/{jobId} |
| POST /api/v1/agent | 에이전트 실행 요청 | GET /api/v1/agent/status/{jobId} |
| POST /api/v1/report | 보고서 생성 요청 | GET /api/v1/report/status/{jobId} |
| POST /api/v1/rag/search | RAG 검색 요청 | GET /api/v1/rag/status/{jobId} |
| POST /api/v1/web/search | 웹 검색 요청 | GET /api/v1/web/status/{jobId} |

##### 4. 작업 상태 값
- `queued`: 작업이 큐에 추가됨
- `active`: 작업이 처리 중
- `completed`: 작업이 완료됨
- `failed`: 작업이 실패함
- `delayed`: 작업이 지연됨

##### 5. 작업 상태 확인 예시 코드
```javascript
// 작업 상태를 주기적으로 확인하는 예시
async function pollJobStatus(jobId, interval = 1000) {
  return new Promise((resolve, reject) => {
    const checkStatus = async () => {
      try {
        const status = await checkJobStatus(jobId);
        
        if (status.data.state === 'completed') {
          resolve(status.data);
        } else if (status.data.state === 'failed') {
          reject(new Error(status.data.message || '작업이 실패했습니다.'));
        } else {
          setTimeout(checkStatus, interval);
        }
      } catch (error) {
        reject(error);
      }
    };
    
    checkStatus();
  });
}

// 사용 예시
async function example() {
  try {
    const jobId = await sendChatRequest();
    const result = await pollJobStatus(jobId);
    console.log('작업 완료:', result);
  } catch (error) {
    console.error('오류:', error);
  }
}
```

##### 6. 실제 사용 예시

###### 채팅 API 사용 예시
```javascript
// 채팅 API 클라이언트 클래스
class ChatAPIClient {
  constructor(baseUrl = 'ws://localhost:5500') {
    this.baseUrl = baseUrl;
    this.ws = null;
    this.jobCallbacks = new Map();
  }

  // WebSocket 연결
  async connect() {
    return new Promise((resolve, reject) => {
      this.ws = new WebSocket(this.baseUrl + '/ws');
      
      this.ws.onopen = () => {
        console.log('WebSocket 연결됨');
        resolve();
      };

      this.ws.onerror = (error) => {
        console.error('WebSocket 오류:', error);
        reject(error);
      };

      this.ws.onmessage = (event) => {
        const data = JSON.parse(event.data);
        const callback = this.jobCallbacks.get(data.jobId);
        if (callback) {
          callback(data);
        }
      };
    });
  }

  // 채팅 요청 전송
  async sendMessage(prompt, options = {}) {
    const response = await fetch('http://localhost:5500/api/v1/chat', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        prompt,
        chat_id: options.chatId,
        user_id: options.userId,
        rag: options.useRag || false,
        web: options.useWeb || false,
        temperature: options.temperature || 0.7
      })
    });

    if (!response.ok) {
      throw new Error(`API 요청 실패: ${response.status}`);
    }

    const data = await response.json();
    return data.jobId;
  }

  // 작업 완료 대기
  async waitForCompletion(jobId) {
    return new Promise((resolve, reject) => {
      this.jobCallbacks.set(jobId, (data) => {
        if (data.state === 'completed') {
          this.jobCallbacks.delete(jobId);
          resolve(data);
        } else if (data.state === 'failed') {
          this.jobCallbacks.delete(jobId);
          reject(new Error(data.message || '작업이 실패했습니다.'));
        }
      });
    });
  }

  // 연결 종료
  disconnect() {
    if (this.ws) {
      this.ws.close();
      this.ws = null;
    }
  }
}

// 사용 예시
async function chatExample() {
  const client = new ChatAPIClient();
  
  try {
    // 1. WebSocket 연결
    console.log('WebSocket 연결 중...');
    await client.connect();

    // 2. 채팅 요청 전송
    console.log('채팅 요청 전송 중...');
    const jobId = await client.sendMessage(
      "파이썬으로 피보나치 수열을 계산하는 함수를 작성해주세요.",
      {
        chatId: "chat-" + Date.now(),
        userId: "user-123",
        useRag: true,
        temperature: 0.7
      }
    );
    console.log('작업 ID:', jobId);

    // 3. 작업 완료 대기
    console.log('작업 완료 대기 중...');
    const result = await client.waitForCompletion(jobId);
    
    // 4. 결과 처리
    console.log('응답:', result.choices[0].message.content);
    console.log('토큰 사용량:', result.usage);
    
  } catch (error) {
    console.error('오류 발생:', error);
  } finally {
    // 5. WebSocket 연결 종료
    client.disconnect();
  }
}

// 실행
chatExample();
```

###### 코드 생성 API 사용 예시
```