# Nginx Sub-path 설정 가이드

## 1. Flask 앱 준비

Sub-path를 지원하는 Flask 앱이 준비되었습니다:
```
/data/chandra/web_demo_subpath.py
```

## 2. Nginx 설정 추가

### 방법 A: 기존 hamonize.conf 파일 수정

```bash
sudo nano /etc/nginx/sites-available/hamonize.conf
```

**server { ... } 블록 안에 다음 내용 추가:**

```nginx
location /aiform/ {
    # Flask 앱이 실행되는 포트로 프록시
    proxy_pass http://127.0.0.1:5000/;

    # 프록시 헤더 설정
    proxy_set_header Host $host;
    proxy_set_header X-Real-IP $remote_addr;
    proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
    proxy_set_header X-Forwarded-Proto $scheme;
    proxy_set_header X-Script-Name /aiform;

    # 대용량 파일 업로드 지원 (PDF/이미지)
    client_max_body_size 100M;
    client_body_timeout 600s;

    # 긴 처리 시간 대응 (OCR 처리)
    proxy_connect_timeout 600s;
    proxy_send_timeout 600s;
    proxy_read_timeout 600s;
    send_timeout 700s;

    # WebSocket 지원 (필요시)
    proxy_http_version 1.1;
    proxy_set_header Upgrade $http_upgrade;
    proxy_set_header Connection "upgrade";
}
```

**추가 위치:**
```nginx
server {
    listen 443 ssl;
    server_name www.hamonize.com;

    # ... 기존 설정 ...

    location / {
        proxy_pass http://127.0.0.1:5700;
        # ...
    }

    # 여기에 /aiform 설정 추가 👇
    location /aiform/ {
        proxy_pass http://127.0.0.1:5000/;
        # ...
    }
}
```

### 방법 B: nginx_aiform.conf 내용 복사

준비된 설정 파일의 내용을 복사:
```bash
cat /data/chandra/nginx_aiform.conf
```

## 3. Nginx 설정 테스트

```bash
sudo nginx -t
```

오류가 없으면:
```bash
sudo systemctl reload nginx
```

## 4. Flask 앱 실행

### 방법 1: 직접 실행 (테스트용)

```bash
cd /data/chandra
source .venv/bin/activate
export APPLICATION_ROOT=/aiform
python web_demo_subpath.py
```

접속: `https://www.hamonize.com/aiform`

### 방법 2: Systemd 서비스로 실행 (운영용)

#### A. 로그 디렉토리 생성
```bash
sudo mkdir -p /var/log/chandra-ocr
sudo chown airun:airun /var/log/chandra-ocr
```

#### B. 서비스 파일 설치
```bash
sudo cp /data/chandra/chandra-ocr.service /etc/systemd/system/
sudo systemctl daemon-reload
```

#### C. 서비스 시작
```bash
sudo systemctl start chandra-ocr
sudo systemctl status chandra-ocr
```

#### D. 부팅 시 자동 시작
```bash
sudo systemctl enable chandra-ocr
```

## 5. 서비스 관리 명령어

### 상태 확인
```bash
sudo systemctl status chandra-ocr
```

### 재시작
```bash
sudo systemctl restart chandra-ocr
```

### 중지
```bash
sudo systemctl stop chandra-ocr
```

### 로그 확인
```bash
# 실시간 로그
sudo journalctl -u chandra-ocr -f

# 최근 100줄
sudo journalctl -u chandra-ocr -n 100

# 앱 로그 파일
tail -f /var/log/chandra-ocr/access.log
tail -f /var/log/chandra-ocr/error.log
```

## 6. 접속 테스트

서비스가 실행되면 다음 URL로 접속:

- **메인 페이지**: `https://www.hamonize.com/aiform`
- **헬스체크**: `https://www.hamonize.com/aiform/health`

## 7. 문제 해결

### Flask 앱이 시작되지 않는 경우

```bash
# 포트 5000이 이미 사용 중인지 확인
sudo lsof -i :5000

# 수동으로 실행해서 오류 확인
cd /data/chandra
source .venv/bin/activate
python web_demo_subpath.py
```

### Nginx 프록시 오류

```bash
# Nginx 오류 로그 확인
sudo tail -f /var/log/nginx/error.log

# Flask 앱이 실행 중인지 확인
curl http://localhost:5000/health
```

### 업로드 파일 크기 제한

nginx.conf에서 `client_max_body_size` 값을 확인하고 필요시 증가:
```nginx
client_max_body_size 100M;
```

## 8. 포트 변경이 필요한 경우

### 5000 포트가 이미 사용 중인 경우

#### Flask 앱 포트 변경
```bash
# web_demo_subpath.py 파일의 마지막 줄 수정
app.run(host='0.0.0.0', port=5001, debug=False)  # 5001로 변경
```

#### Nginx 설정 수정
```nginx
location /aiform/ {
    proxy_pass http://127.0.0.1:5001/;  # 5001로 변경
    # ...
}
```

#### Systemd 서비스 환경 변수 추가
```bash
# chandra-ocr.service 파일에 추가
Environment="FLASK_PORT=5001"
```

## 9. 보안 고려사항

### IP 제한 (선택사항)
특정 IP에서만 접근 허용:
```nginx
location /aiform/ {
    allow 192.168.1.0/24;
    allow 10.0.0.0/8;
    deny all;

    proxy_pass http://127.0.0.1:5000/;
    # ...
}
```

### 인증 추가 (선택사항)
기본 인증 추가:
```bash
# htpasswd 파일 생성
sudo apt-get install apache2-utils
sudo htpasswd -c /etc/nginx/.htpasswd admin
```

```nginx
location /aiform/ {
    auth_basic "Chandra OCR";
    auth_basic_user_file /etc/nginx/.htpasswd;

    proxy_pass http://127.0.0.1:5000/;
    # ...
}
```

## 요약

1. ✅ Flask 앱 준비: `web_demo_subpath.py`
2. ✅ Nginx 설정: `/aiform/` 경로 추가
3. ✅ Systemd 서비스: 자동 시작/관리
4. ✅ 접속: `https://www.hamonize.com/aiform`

모든 설정이 완료되면 안정적으로 운영 가능합니다!
