#!/usr/bin/env python3
"""
Chandra OCR 테스트 스크립트
PDF/이미지 파일을 구조화된 JSON으로 변환합니다.
"""

import json
from pathlib import Path
from bs4 import BeautifulSoup


def html_to_structured_json(html_content):
    """HTML을 구조화된 JSON으로 변환"""
    soup = BeautifulSoup(html_content, 'html.parser')

    result = {
        "title": "",
        "tables": [],
        "paragraphs": [],
        "lists": []
    }

    # 제목 추출
    title = soup.find(['h1', 'h2', 'h3', 'h4'])
    if title:
        result["title"] = title.get_text(strip=True)

    # 테이블 추출
    for table in soup.find_all('table'):
        table_data = {
            "headers": [],
            "rows": []
        }

        # 헤더 추출
        thead = table.find('thead')
        if thead:
            for tr in thead.find_all('tr'):
                header_row = []
                for th in tr.find_all('th'):
                    header_row.append({
                        "text": th.get_text(strip=True),
                        "rowspan": int(th.get('rowspan', 1)),
                        "colspan": int(th.get('colspan', 1))
                    })
                table_data["headers"].append(header_row)

        # 바디 추출
        tbody = table.find('tbody')
        if tbody:
            for tr in tbody.find_all('tr'):
                row = []
                for td in tr.find_all(['td', 'th']):
                    cell = {
                        "text": td.get_text(strip=True),
                        "rowspan": int(td.get('rowspan', 1)),
                        "colspan": int(td.get('colspan', 1))
                    }

                    # 체크박스 확인
                    checkbox = td.find('input', {'type': 'checkbox'})
                    if checkbox:
                        cell["has_checkbox"] = True
                        cell["text"] = td.get_text(strip=True)

                    row.append(cell)
                table_data["rows"].append(row)

        result["tables"].append(table_data)

    # 단락 추출
    for p in soup.find_all('p'):
        text = p.get_text(strip=True)
        if text:
            result["paragraphs"].append(text)

    # 리스트 추출
    for ol in soup.find_all(['ol', 'ul']):
        list_items = []
        for li in ol.find_all('li', recursive=False):
            list_items.append(li.get_text(strip=True))
        result["lists"].append({
            "type": "ordered" if ol.name == 'ol' else "unordered",
            "items": list_items
        })

    return result


def main():
    # 생성된 HTML 파일 읽기
    html_file = Path("output/리모델링신청서/리모델링신청서.html")

    if not html_file.exists():
        print(f"파일을 찾을 수 없습니다: {html_file}")
        return

    with open(html_file, 'r', encoding='utf-8') as f:
        html_content = f.read()

    # HTML을 구조화된 JSON으로 변환
    structured_data = html_to_structured_json(html_content)

    # JSON 파일로 저장
    output_file = html_file.parent / f"{html_file.stem}_structured.json"
    with open(output_file, 'w', encoding='utf-8') as f:
        json.dump(structured_data, f, ensure_ascii=False, indent=2)

    print(f"구조화된 JSON 파일이 생성되었습니다: {output_file}")
    print("\n=== 구조화된 데이터 미리보기 ===")
    print(json.dumps(structured_data, ensure_ascii=False, indent=2))


if __name__ == "__main__":
    main()
