#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
문서 생성을 위한 템플릿 관리 모듈
"""

from datetime import datetime
from typing import Dict, Any, List, Optional

class DocumentTemplateManager:
    """문서 템플릿 관리 클래스"""

    def __init__(self):
        self.templates = {
            'petition': self.get_petition_template(),
            'license': self.get_license_template(),
            'official': self.get_official_template(),
            'default': self.get_default_template()
        }

    def get_template(self, document_type: str) -> Dict[str, Any]:
        """문서 타입에 따른 템플릿 반환"""
        return self.templates.get(document_type, self.get_default_template())

    def get_available_templates(self) -> Dict[str, str]:
        """사용 가능한 템플릿 목록 반환"""
        return {
            'petition': '민원신청서 (표 형식)',
            'license': '허가신청서',
            'official': '공문',
            'default': '기본 문서'
        }

    def get_petition_template(self) -> Dict[str, Any]:
        """민원신청서 템플릿 (표 형식)"""
        return {
            'title': '민원신청서',
            'style': 'table',
            'table_structure': {
                'header_rows': [
                    {'label': '민원인 성명', 'colspan': 1},
                    {'label': '주민등록번호', 'colspan': 1}
                ],
                'content_rows': [
                    {
                        'label': '성명',
                        'field': 'applicant.name',
                        'type': 'text'
                    },
                    {
                        'label': '주소',
                        'field': 'applicant.address',
                        'type': 'text',
                        'rowspan': 2
                    },
                    {
                        'label': 'E-mail 주소',
                        'field': 'applicant.email',
                        'type': 'text'
                    },
                    {
                        'label': '전화 번호',
                        'field': 'applicant.phone',
                        'type': 'text'
                    },
                    {
                        'label': '민원내용',
                        'field': 'application.details',
                        'type': 'textarea',
                        'rowspan': 5
                    },
                    {
                        'label': '첨부',
                        'field': 'attachments',
                        'type': 'file_list',
                        'rowspan': 2
                    }
                ],
                'footer_section': {
                    'title': '개 소 서 류',
                    'content': 'large_textarea'
                }
            },
            'sections': [
                {
                    'section_title': '민원인 정보',
                    'style': 'table',
                    'fields': [
                        {'label': '민원인 성명', 'field': 'applicant.name', 'required': True},
                        {'label': '주민등록번호', 'field': 'applicant.id_number', 'required': False},
                        {'label': '주소', 'field': 'applicant.address', 'required': True},
                        {'label': 'E-mail 주소', 'field': 'applicant.email', 'required': False},
                        {'label': '전화번호', 'field': 'applicant.phone', 'required': True}
                    ]
                },
                {
                    'section_title': '민원내용',
                    'style': 'table',
                    'fields': [
                        {'label': '제목', 'field': 'application.title', 'required': True},
                        {'label': '목적', 'field': 'application.purpose', 'required': False},
                        {'label': '대상', 'field': 'application.target', 'required': False},
                        {'label': '기간', 'field': 'application.period', 'required': False},
                        {'label': '민원내용', 'field': 'application.details', 'required': True, 'type': 'textarea'}
                    ]
                },
                {
                    'section_title': '첨부서류',
                    'style': 'table',
                    'fields': [
                        {'label': '첨부서류 목록', 'field': 'attachments', 'required': False, 'type': 'hardcoded_list',
                         'default_list': [
                             '1. 신청서 1부',
                             '2. 사업계획서 1부 (해당자에 한함)',
                             '3. 위치도 및 배치도 1부 (해당자에 한함)',
                             '4. 토지사용승낙서 1부 (해당자에 한함)',
                             '5. 기타 관련 서류'
                         ]}
                    ]
                },
                {
                    'section_title': '참고사항',
                    'style': 'standard',
                    'fields': [
                        {'label': '처리안내', 'field': 'processing_info', 'required': False, 'type': 'hardcoded_text',
                         'default_text': '• 민원 처리기간: 접수일로부터 7일 이내 (관계법령에 따른 처리기간이 별도로 정해진 경우 제외)\n• 접수된 서류는 반환하지 않습니다.\n• 허위 또는 부정한 방법으로 신청한 경우 신청이 취소될 수 있습니다.\n• 문의처: 강진군 민원봉사과 (☎ 061-430-3200)'},
                        {'label': '개인정보 수집·이용 동의', 'field': 'privacy_consent', 'required': False, 'type': 'hardcoded_text',
                         'default_text': '「개인정보보호법」제15조에 따라 개인정보를 수집·이용합니다.\n▣ 수집·이용 목적: 민원사무 처리\n▣ 수집항목: 성명, 주소, 전화번호, 이메일 등\n▣ 보유·이용기간: 민원 처리 완료 후 3년\n▣ 동의 거부권: 동의를 거부할 권리가 있으나, 동의 거부 시 민원 처리가 제한됩니다.\n\n개인정보 수집·이용에 □ 동의함'}
                    ]
                }
            ],
            'footer': [
                {'label': '신청일', 'value': 'current_date'},
                {'label': '신청인', 'field': 'applicant.name', 'suffix': ' (서명)'}
            ],
            'validation_rules': {
                'applicant': ['name', 'address', 'phone'],
                'application': ['title', 'details']
            }
        }


    def get_license_template(self) -> Dict[str, Any]:
        """허가신청서 템플릿"""
        return {
            'title': '허가신청서',
            'style': 'standard',
            'sections': [
                {
                    'section_title': '1. 신청인 정보',
                    'fields': [
                        {'label': '신청인', 'field': 'applicant.name', 'required': True},
                        {'label': '주소', 'field': 'applicant.address', 'required': True},
                        {'label': '연락처', 'field': 'applicant.phone', 'required': True},
                        {'label': '이메일', 'field': 'applicant.email', 'required': False}
                    ]
                },
                {
                    'section_title': '2. 허가신청 내용',
                    'fields': [
                        {'label': '허가대상', 'field': 'application.title', 'required': True},
                        {'label': '허가목적', 'field': 'application.purpose', 'required': True},
                        {'label': '허가범위', 'field': 'application.target', 'required': True},
                        {'label': '허가기간', 'field': 'application.period', 'required': False},
                        {'label': '상세내용', 'field': 'application.details', 'required': True}
                    ]
                },
                {
                    'section_title': '3. 첨부서류',
                    'fields': [
                        {'label': '첨부서류 목록', 'field': 'attachments', 'required': False}
                    ]
                }
            ],
            'footer': [
                {'label': '신청일', 'value': 'current_date'},
                {'label': '신청인', 'field': 'applicant.name', 'suffix': ' (서명)'}
            ],
            'validation_rules': {
                'applicant': ['name', 'address', 'phone'],
                'application': ['title', 'purpose', 'target', 'details']
            }
        }

    def validate_data(self, document_type: str, data: Dict[str, Any]) -> Dict[str, Any]:
        """데이터 유효성 검증"""
        template = self.get_template(document_type)
        validation_result = {
            'valid': True,
            'missing_fields': [],
            'errors': []
        }

        # 필수 필드 검증
        validation_rules = template.get('validation_rules', {})

        for section_key, required_fields in validation_rules.items():
            if section_key not in data:
                validation_result['missing_fields'].append(f"{section_key} 섹션이 누락되었습니다")
                validation_result['valid'] = False
                continue

            section_data = data[section_key]
            for field in required_fields:
                if field not in section_data or not section_data[field]:
                    validation_result['missing_fields'].append(f"{section_key}.{field}")
                    validation_result['valid'] = False

        return validation_result

    def get_field_value(self, data: Dict[str, Any], field_path: str, default: str = '') -> str:
        """중첩된 딕셔너리에서 필드 값 추출"""
        try:
            keys = field_path.split('.')
            value = data
            for key in keys:
                value = value[key]
            return str(value) if value is not None else default
        except (KeyError, TypeError):
            return default

    def generate_review_comments(self, document_type: str, data: Dict[str, Any]) -> List[str]:
        """데이터 검토 의견 생성"""
        validation = self.validate_data(document_type, data)
        comments = []

        if not validation['valid']:
            comments.append("🔍 다음 필수 정보가 누락되었습니다:")
            for missing_field in validation['missing_fields']:
                comments.append(f"  - {missing_field}")

        # 추가 검토 의견
        template = self.get_template(document_type)

        # 신청자 정보 검토
        if 'applicant' in data:
            applicant = data['applicant']

            # 전화번호 형식 검사
            if 'phone' in applicant and applicant['phone']:
                phone = applicant['phone'].replace('-', '').replace(' ', '')
                if not phone.isdigit() or len(phone) < 10:
                    comments.append("📞 전화번호 형식을 확인해주세요 (예: 010-1234-5678)")

            # 이메일 형식 검사 (간단한 검증)
            if 'email' in applicant and applicant['email']:
                email = applicant['email']
                if '@' not in email or '.' not in email:
                    comments.append("📧 이메일 형식을 확인해주세요")

        # 신청 내용 검토
        if 'application' in data:
            application = data['application']

            # 상세내용 길이 검사
            if 'details' in application and application['details']:
                details_length = len(application['details'])
                if details_length < 50:
                    comments.append("📝 상세내용을 더 자세히 작성해주세요 (현재: {}자)".format(details_length))
                elif details_length > 1000:
                    comments.append("📝 상세내용이 너무 길어 보입니다 (현재: {}자)".format(details_length))

            # 기간 형식 검사
            if 'period' in application and application['period']:
                period = application['period']
                if '~' not in period and '-' not in period:
                    comments.append("📅 기간 형식을 확인해주세요 (예: 2024-01-01 ~ 2024-12-31)")

        # 첨부서류 검토
        if 'attachments' in data and data['attachments']:
            attachments = data['attachments']
            if len(attachments) == 0:
                comments.append("📎 필요한 첨부서류를 확인해주세요")
            elif len(attachments) > 10:
                comments.append("📎 첨부서류가 많습니다. 필요한 서류만 선별해주세요")

        if not comments:
            comments.append("✅ 모든 정보가 올바르게 입력되었습니다.")

        return comments

    def suggest_improvements(self, document_type: str, data: Dict[str, Any], rag_info: str = None) -> List[str]:
        """개선 제안 생성"""
        suggestions = []

        # RAG 정보 기반 제안
        if rag_info and len(rag_info.strip()) > 0:
            suggestions.append("💡 관련 문서에서 발견된 정보:")
            suggestions.append(f"   {rag_info[:200]}...")
            suggestions.append("   위 정보를 참고하여 신청 내용을 보완해보세요.")

        # 문서 타입별 맞춤 제안
        if document_type == 'business':
            suggestions.extend([
                "💼 사업신청서 작성 팁:",
                "   - 사업계획서나 수지계산서 첨부를 고려해보세요",
                "   - 사업 관련 허가증이나 자격증이 필요할 수 있습니다",
                "   - 사업장 임대차 계약서 확인이 필요할 수 있습니다"
            ])
        elif document_type == 'license':
            suggestions.extend([
                "📋 허가신청서 작성 팁:",
                "   - 관련 법령 및 시행규칙을 미리 확인하세요",
                "   - 기술기준이나 안전기준 관련 서류가 필요할 수 있습니다",
                "   - 관할 기관의 사전상담을 받아보세요"
            ])
        else:  # petition
            suggestions.extend([
                "📄 민원신청서 작성 팁:",
                "   - 민원 처리절차와 소요기간을 미리 확인하세요",
                "   - 관련 증빙서류를 빠짐없이 준비하세요",
                "   - 처리 담당자나 부서를 미리 확인하면 도움이 됩니다"
            ])

        return suggestions

    def get_official_template(self) -> Dict[str, Any]:
        """공공기관 공문 템플릿 - 원본 공문 형식 완전 복제"""
        return {
            'title': '강진군',  # 헤더에는 "강진군"만 표시
            'style': 'official_document',
            'header': {
                'office_name': '강진군',
                'slogan': '소통, 연결, 창조의 新 강진',  # 상단 슬로건
                'date_format': 'yyyy. mm. dd',
                'document_number_prefix': '민원봉사과-',
                'show_center_title_only': True
            },
            'sections': [
                {
                    'section_type': 'recipient_info',
                    'fields': [
                        {'label': '수신', 'field': 'recipient', 'default': '관계자 참조'},
                        {'label': '경유', 'field': 'via', 'default': '(경유)'}
                    ]
                },
                {
                    'section_type': 'official_title',
                    'fields': [
                        {'label': '제목', 'field': 'application.title', 'required': True, 'style': 'bold_underline'}
                    ]
                },
                {
                    'section_type': 'numbered_content',
                    'fields': [
                        {'field': 'application.details', 'required': True, 'type': 'numbered_paragraphs'}
                    ]
                },
                {
                    'section_type': 'structured_info',
                    'fields': [
                        {'label': '사업개요', 'field': 'project_overview', 'type': 'structured_table', 'required': False}
                    ]
                },
                {
                    'section_type': 'attachments_simple',
                    'fields': [
                        {'label': '붙임', 'field': 'attachments', 'type': 'simple_list', 'default': '관련 서류 1부. 끝.'}
                    ]
                }
            ],
            'footer': {
                'type': 'contact_table',
                'fields': [
                    {'label': '주무관', 'field': 'applicant.name', 'required': True},
                    {'label': '협조자', 'default': ''},
                    {'label': '팩스번호', 'default': '061-430-3433'},
                    {'label': '전화번호', 'field': 'applicant.phone', 'required': True},
                    {'label': '시행', 'default': '민원봉사과-6875'},
                    {'label': '접수', 'default': ''},
                    {'label': '우편번호', 'default': '59228'},
                    {'label': '주소', 'default': '전라남도 강진군 강진읍 탑진로 111'},
                    {'label': 'URL', 'default': 'http://www.gangjin.go.kr'}
                ]
            },
            'validation_rules': {
                'applicant': ['name', 'phone'],
                'application': ['title', 'details']
            }
        }

    def get_default_template(self) -> Dict[str, Any]:
        """기본 문서 템플릿 - 문서 유형을 식별할 수 없을 때 사용 (간단하고 유연한 형식)"""
        return {
            'title': '문서',
            'style': 'simple',  # 간단한 스타일
            'sections': [
                {
                    'section_title': '기본 정보',
                    'style': 'simple',
                    'fields': [
                        {'label': '작성자', 'field': 'applicant.name', 'required': False},
                        {'label': '연락처', 'field': 'applicant.phone', 'required': False},
                        {'label': '제목', 'field': 'application.title', 'required': False}
                    ]
                },
                {
                    'section_title': '내용',
                    'style': 'simple',
                    'fields': [
                        {'label': '문서내용', 'field': 'application.details', 'required': False, 'type': 'textarea'}
                    ]
                }
            ],
            'footer': [
                {'label': '작성일', 'value': 'current_date'},
                {'label': '작성자', 'field': 'applicant.name', 'suffix': ''}
            ],
            'validation_rules': {
                'applicant': [],  # 필수 필드 없음
                'application': []  # 필수 필드 없음
            }
        }


# 전역 템플릿 매니저 인스턴스
template_manager = DocumentTemplateManager()

def get_template_manager() -> DocumentTemplateManager:
    """템플릿 매니저 인스턴스 반환"""
    return template_manager