from fastapi import FastAPI, HTTPException, Depends
from pydantic import BaseModel
from typing import List, Optional
from datetime import datetime

app = FastAPI(title="Product Management API")

class Product(BaseModel):
    id: Optional[int]
    name: str
    description: str
    price: float
    category: str
    stock: int
    created_at: datetime = datetime.now()

class ProductCreate(BaseModel):
    name: str
    description: str
    price: float
    category: str
    stock: int

class ProductUpdate(BaseModel):
    name: Optional[str]
    description: Optional[str]
    price: Optional[float]
    category: Optional[str]
    stock: Optional[int]

@app.post("/products/", response_model=Product, tags=["products"])
async def create_product(product: ProductCreate):
    """
    제품 등록 API
    
    Parameters:
    - name: 제품명
    - description: 제품 설명
    - price: 가격
    - category: 카테고리
    - stock: 재고수량
    
    Returns:
    - Product 객체
    """
    # 제품 등록 로직
    return {
        "id": 1,
        "name": product.name,
        "description": product.description,
        "price": product.price,
        "category": product.category,
        "stock": product.stock,
        "created_at": datetime.now()
    }

@app.get("/products/{product_id}", response_model=Product, tags=["products"])
async def get_product(product_id: int):
    """
    제품 조회 API
    
    Parameters:
    - product_id: 제품 ID
    
    Returns:
    - Product 객체
    """
    # 제품 조회 로직
    return {
        "id": product_id,
        "name": "Sample Product",
        "description": "This is a sample product",
        "price": 10000.0,
        "category": "Electronics",
        "stock": 100,
        "created_at": datetime.now()
    }

@app.put("/products/{product_id}", response_model=Product, tags=["products"])
async def update_product(product_id: int, product: ProductUpdate):
    """
    제품 정보 수정 API
    
    Parameters:
    - product_id: 제품 ID
    - product: 수정할 제품 정보
    
    Returns:
    - 수정된 Product 객체
    """
    # 제품 정보 수정 로직
    return {
        "id": product_id,
        "name": product.name,
        "description": product.description,
        "price": product.price,
        "category": product.category,
        "stock": product.stock,
        "created_at": datetime.now()
    }

@app.delete("/products/{product_id}", tags=["products"])
async def delete_product(product_id: int):
    """
    제품 삭제 API
    
    Parameters:
    - product_id: 제품 ID
    
    Returns:
    - 성공 메시지
    """
    # 제품 삭제 로직
    return {"message": "Product deleted successfully"} 