#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
민원신청서 및 기타 정형화된 문서 생성 스크립트
"""

import os
import sys
import json
from datetime import datetime
from utils import HWPDocument, PDFDocument, DOCXDocument
from document_templates import get_template_manager

def parse_input_data(json_str):
    """입력 JSON 데이터 파싱"""
    try:
        return json.loads(json_str)
    except json.JSONDecodeError as e:
        raise ValueError(f"JSON 파싱 실패: {e}")

def validate_document_data(data):
    """문서 생성 데이터 검증"""
    required_fields = ['document_type', 'format', 'applicant', 'application']

    for field in required_fields:
        if field not in data:
            raise ValueError(f"필수 필드 누락: {field}")

    # 템플릿 매니저를 사용한 상세 검증
    template_manager = get_template_manager()
    document_type = data.get('document_type', 'petition')

    validation_result = template_manager.validate_data(document_type, data)

    if not validation_result['valid']:
        missing_fields = validation_result['missing_fields']
        raise ValueError(f"필수 필드 누락: {', '.join(missing_fields)}")

    return validation_result

def create_document_with_template(data, output_path):
    """템플릿 기반 문서 생성 (통합 함수)"""
    template_manager = get_template_manager()
    document_type = data.get('document_type', 'petition')
    format_type = data.get('format', 'pdf').lower()

    # 템플릿 가져오기
    template = template_manager.get_template(document_type)
    template_style = template.get('style', 'default')

    # 문서 객체 생성
    if format_type == 'pdf':
        doc = PDFDocument()
    elif format_type == 'docx':
        doc = DOCXDocument()
    elif format_type == 'hwpx':
        doc = HWPDocument()
    else:
        raise ValueError(f"지원하지 않는 문서 형식: {format_type}")

    # 스타일에 따른 문서 생성
    if template_style == 'official':
        return create_official_document(doc, template, data, output_path, format_type, template_manager)
    elif template_style == 'official_document':
        return create_official_memo_document(doc, template, data, output_path, format_type, template_manager)
    elif template_style == 'table':
        return create_table_document(doc, template, data, output_path, format_type, template_manager)
    elif template_style == 'simple':
        return create_simple_document(doc, template, data, output_path, format_type, template_manager)
    else:
        return create_standard_document(doc, template, data, output_path, format_type, template_manager)

def create_petition_document(data, output_path):
    """민원신청서 생성 (템플릿 기반으로 리팩토링됨 - 더 이상 사용하지 않음)"""
    # 템플릿 기반 문서 생성으로 대체
    return create_document_with_template(data, output_path)

def create_business_application(data, output_path):
    """사업신청서 생성 (템플릿 기반으로 리팩토링됨 - 더 이상 사용하지 않음)"""
    # 템플릿 기반 문서 생성으로 대체
    return create_document_with_template(data, output_path)

def create_license_application(data, output_path):
    """허가신청서 생성 (템플릿 기반으로 리팩토링됨 - 더 이상 사용하지 않음)"""
    # 템플릿 기반 문서 생성으로 대체
    return create_document_with_template(data, output_path)

def main():
    """메인 함수"""
    if len(sys.argv) != 3:
        print("사용법: python generate_document.py <JSON_DATA> <OUTPUT_PATH>", file=sys.stderr)
        sys.exit(1)

    try:
        # 입력 데이터 파싱
        json_data = sys.argv[1]
        output_path = sys.argv[2]

        data = parse_input_data(json_data)
        validate_document_data(data)

        # 템플릿 기반 문서 생성 (통합 방식)
        document_type = data['document_type']

        # 지원되는 문서 타입 확인
        supported_types = ['petition', 'license', 'official', 'default']
        if document_type not in supported_types:
            raise ValueError(f"지원하지 않는 문서 타입: {document_type}. 지원 타입: {', '.join(supported_types)}")

        result_path = create_document_with_template(data, output_path)

        # 성공 메시지 출력
        print(f"문서 생성 완료: {result_path}")

        # 파일 크기 정보
        if os.path.exists(result_path):
            file_size = os.path.getsize(result_path)
            print(f"파일 크기: {file_size} bytes")

        sys.exit(0)

    except Exception as e:
        print(f"문서 생성 실패: {str(e)}", file=sys.stderr)
        sys.exit(1)

def create_standard_document(doc, template, data, output_path, format_type, template_manager):
    """기본 스타일 문서 생성 - utils.py 메소드 활용"""
    # 헤더 이미지 추가 (기본값: assets/kangjin-header.png)
    header_image = data.get('header_image', 'assets/kangjin-header.png')
    if header_image and os.path.exists(header_image):
        try:
            doc.add_image(header_image)
            if format_type == 'pdf':
                doc.add_spacing(10)
            elif format_type == 'hwpx':
                # HWP에서는 빈 paragraph로 간격 구현
                doc.add_paragraph("")
        except Exception as e:
            print(f"[WARNING] 헤더 이미지 추가 실패: {str(e)}")

    # 제목 추가
    if format_type == 'pdf':
        doc.add_heading(template['title'], level=1, align='center')
    else:
        doc.add_heading(template['title'], level=1)

    if format_type == 'pdf':
        doc.add_spacing(20)
    elif format_type == 'hwpx':
        # HWP에서는 빈 paragraph로 간격 구현
        doc.add_paragraph("")
        doc.add_paragraph("")

    # 섹션별 내용 생성
    for section in template['sections']:
        section_title = section['section_title']
        doc.add_heading(section_title, level=2)

        # 섹션 내용을 표 데이터로 준비
        section_data = []
        for field in section['fields']:
            field_path = field['field']
            label = field['label']

            if field_path == 'attachments':
                attachments = data.get('attachments', [])
                if attachments:
                    attachment_text = "\n".join([f"{i}. {att}" for i, att in enumerate(attachments, 1)])
                    section_data.append([label, attachment_text])
            elif field_path == 'base_info':
                base_info = data.get('base_info')
                if base_info:
                    info_text = base_info[:500] + "..." if len(base_info) > 500 else base_info
                    section_data.append([label, info_text])
            else:
                value = template_manager.get_field_value(data, field_path)
                if value:
                    section_data.append([label, value])

        # 섹션에 데이터가 있으면 표로 추가
        if section_data and format_type == 'pdf':
            doc.add_table(section_data, narrow_first_col=True)
        else:
            # PDF가 아니거나 표 기능이 없는 경우 일반 텍스트로 추가
            for row in section_data:
                doc.add_paragraph(f"{row[0]}: {row[1]}")

        if format_type == 'pdf':
            doc.add_spacing(15)
        elif format_type == 'hwpx':
            # HWP에서는 빈 paragraph로 간격 구현
            doc.add_paragraph("")

    # 푸터 (서명 영역) 추가
    current_date = datetime.now().strftime("%Y년 %m월 %d일")

    for footer_item in template.get('footer', []):
        if footer_item.get('value') == 'current_date':
            if format_type == 'pdf':
                doc.add_paragraph(f"{footer_item['label']}: {current_date}", align='right')
            else:
                doc.add_paragraph(f"{footer_item['label']}: {current_date}")
        elif 'field' in footer_item:
            field_value = template_manager.get_field_value(data, footer_item['field'])
            suffix = footer_item.get('suffix', '')
            if field_value:
                if format_type == 'pdf':
                    doc.add_paragraph(f"{footer_item['label']}: {field_value}{suffix}", align='right')
                else:
                    doc.add_paragraph(f"{footer_item['label']}: {field_value}{suffix}")

    doc.save(output_path)
    return output_path

def create_official_document(doc, template, data, output_path, format_type, template_manager):
    """공공기관 공문 스타일 문서 생성 - utils.py 메소드 활용"""
    header = template.get('header', {})
    main_content = template.get('main_content', {})
    footer_items = template.get('footer_items', [])
    current_date = datetime.now().strftime("%Y년 %m월 %d일")

    # 헤더 이미지 추가 (기본값: assets/kangjin-header.png)
    header_image = data.get('header_image', 'assets/kangjin-header.png')
    if header_image and os.path.exists(header_image):
        try:
            doc.add_image(header_image)
            if format_type == 'pdf':
                doc.add_spacing(10)
            elif format_type == 'hwpx':
                # HWP에서는 빈 paragraph로 간격 구현
                doc.add_paragraph("")
        except Exception as e:
            print(f"[WARNING] 헤더 이미지 추가 실패: {str(e)}")

    # 헤더 섹션
    if format_type == 'pdf':
        doc.add_paragraph(f"작성 : 마을 / 민원담당자 ({current_date})", align='right')
    else:
        doc.add_paragraph(f"작성 : 마을 / 민원담당자 ({current_date})")

    if format_type == 'pdf':
        doc.add_spacing(20)
    elif format_type == 'hwpx':
        # HWP에서는 빈 paragraph로 간격 구현
        doc.add_paragraph("")
        doc.add_paragraph("")

    # 공문 번호 및 제목
    if header.get('document_number'):
        if format_type == 'pdf':
            doc.add_paragraph(header['document_number'], align='center')
        else:
            doc.add_paragraph(header['document_number'])

    if format_type == 'pdf':
        doc.add_heading(template['title'], level=1, align='center')
    else:
        doc.add_heading(template['title'], level=1)

    if format_type == 'pdf':
        doc.add_spacing(20)
    elif format_type == 'hwpx':
        # HWP에서는 빈 paragraph로 간격 구현
        doc.add_paragraph("")
        doc.add_paragraph("")

    # 수신자 정보
    applicant_name = template_manager.get_field_value(data, 'applicant.name')
    doc.add_paragraph(f"수신 : {applicant_name} 참조")
    doc.add_paragraph("(경유)")

    # 메인 내용을 표 형식으로 구성
    main_data = []

    if main_content.get('event_title'):
        main_data.append(["사업명", main_content['event_title']])

    schedule = main_content.get('schedule', {})
    if schedule.get('period'):
        main_data.append(["사업기간", schedule['period']])
    if schedule.get('target'):
        main_data.append(["사업량", schedule['target']])
    if schedule.get('content'):
        main_data.append(["사업내용", schedule['content']])

    # 상세 내용
    details_text = "\n".join([f"- {detail}" for detail in main_content.get('details', [])])
    if details_text:
        main_data.append(["상세내용", details_text])

    # 하이라이트 정보
    if main_content.get('highlight'):
        main_data.append(["특이사항", main_content['highlight']])

    # 표로 메인 내용 추가
    if main_data and format_type == 'pdf':
        doc.add_table(main_data, narrow_first_col=True)
    else:
        # 표가 지원되지 않는 경우 일반 텍스트로
        for row in main_data:
            doc.add_paragraph(f"{row[0]}: {row[1]}")

    if format_type == 'pdf':
        doc.add_spacing(20)
    elif format_type == 'hwpx':
        # HWP에서는 빈 paragraph로 간격 구현
        doc.add_paragraph("")
        doc.add_paragraph("")

    # 푸터
    if format_type == 'pdf':
        doc.add_paragraph("위 사항에 관하여 가목", align='right')
    else:
        doc.add_paragraph("위 사항에 관하여 가목")

    for item in footer_items:
        doc.add_paragraph(item)

    doc.save(output_path)
    return output_path

def create_table_document(doc, template, data, output_path, format_type, template_manager):
    """공공기관 스타일 표 형식 문서 생성 - 섹션별 표 구성"""
    if format_type != 'pdf':
        # PDF가 아닌 경우 기본 문서로 생성
        return create_standard_document(doc, template, data, output_path, format_type, template_manager)

    # 헤더 이미지 추가 (기본값: assets/kangjin-header.png)
    header_image = data.get('header_image', 'assets/kangjin-header.png')
    if header_image and os.path.exists(header_image):
        try:
            doc.add_image(header_image)
            if format_type == 'pdf':
                doc.add_spacing(10)
            elif format_type == 'hwpx':
                # HWP에서는 빈 paragraph로 간격 구현
                doc.add_paragraph("")
        except Exception as e:
            print(f"[WARNING] 헤더 이미지 추가 실패: {str(e)}")

    # 제목
    doc.add_heading(template['title'], level=1)
    if format_type == 'pdf':
        doc.add_spacing(20)
    elif format_type == 'hwpx':
        # HWP에서는 빈 paragraph로 간격 구현
        doc.add_paragraph("")
        doc.add_paragraph("")

    # 섹션별로 표 생성
    for section in template['sections']:
        section_style = section.get('style', 'table')

        if section_style == 'table':
            section_title = section['section_title']

            # 섹션 제목 추가
            doc.add_heading(section_title, level=2)
            if format_type == 'pdf':
                doc.add_spacing(10)
            elif format_type == 'hwpx':
                # HWP에서는 빈 paragraph로 간격 구현
                doc.add_paragraph("")

            # 해당 섹션의 표 데이터 생성
            section_table_data = []

            for field in section['fields']:
                label = field['label']
                field_path = field['field']
                field_type = field.get('type', 'text')

                if field_type == 'hardcoded_list':
                    # 하드코딩된 리스트 처리
                    default_list = field.get('default_list', [])
                    list_text = "\n".join(default_list)
                    section_table_data.append([label, list_text])
                elif field_type == 'hardcoded_text':
                    # 하드코딩된 텍스트 처리
                    default_text = field.get('default_text', '')
                    section_table_data.append([label, default_text])
                elif field_path == 'attachments':
                    attachments = data.get('attachments', [])
                    if attachments:
                        attachment_text = "\n".join([f"{i}. {att}" for i, att in enumerate(attachments, 1)])
                        section_table_data.append([label, attachment_text])
                    else:
                        section_table_data.append([label, ""])
                elif field_path == 'base_info':
                    base_info = data.get('base_info')
                    if base_info:
                        info_text = base_info[:300] + "..." if len(base_info) > 300 else base_info
                        section_table_data.append([label, info_text])
                    else:
                        section_table_data.append([label, ""])
                else:
                    value = template_manager.get_field_value(data, field_path)
                    section_table_data.append([label, value if value else ""])

            # 섹션별 표 생성 (데이터가 있는 경우만)
            if section_table_data:
                doc.add_table(section_table_data, header=["항목", "내용"], narrow_first_col=True)
                if format_type == 'pdf':
                    doc.add_spacing(15)

        elif section_style == 'standard':
            # 표준 섹션 (하드코딩된 텍스트 등)
            section_title = section['section_title']

            # 섹션 제목 추가
            doc.add_heading(section_title, level=2)
            if format_type == 'pdf':
                doc.add_spacing(10)
            elif format_type == 'hwpx':
                doc.add_paragraph("")

            # 필드별 처리
            for field in section['fields']:
                label = field['label']
                field_type = field.get('type', 'text')

                if field_type == 'hardcoded_text':
                    default_text = field.get('default_text', '')
                    if label:
                        doc.add_paragraph(f"◎ {label}")
                        if format_type == 'pdf':
                            doc.add_spacing(5)

                    # 텍스트를 줄바꿈으로 나누어 처리
                    lines = default_text.split('\n')
                    for line in lines:
                        if line.strip():
                            doc.add_paragraph(f"  {line.strip()}")

                    if format_type == 'pdf':
                        doc.add_spacing(10)
                    elif format_type == 'hwpx':
                        doc.add_paragraph("")

    # 푸터 (서명 영역) 추가
    current_date = datetime.now().strftime("%Y년 %m월 %d일")

    # 서명 섹션
    doc.add_heading("신청인", level=2)
    if format_type == 'pdf':
        doc.add_spacing(10)

    signature_data = [
        ["신청일", current_date],
        ["신청인", f"{template_manager.get_field_value(data, 'applicant.name')} (서명)"]
    ]

    doc.add_table(signature_data, header=["구분", "내용"], narrow_first_col=True)

    # 문서 저장
    doc.save(output_path)
    return output_path

def create_official_memo_document(doc, template, data, output_path, format_type, template_manager):
    """공문 문서 생성 - 원본 공문 형식 완전 복제"""
    header = template.get('header', {})
    sections = template.get('sections', [])
    footer = template.get('footer', {})
    current_date = datetime.now().strftime("작성 : 2025. 09. 25")

    # 헤더 이미지 추가 (기본값: assets/kangjin-header.png)
    header_image = data.get('header_image', 'assets/kangjin-header.png')
    if header_image and os.path.exists(header_image):
        try:
            doc.add_image(header_image)
            if format_type == 'pdf':
                doc.add_spacing(10)
            elif format_type == 'hwpx':
                doc.add_paragraph("")
        except Exception as e:
            print(f"[WARNING] 헤더 이미지 추가 실패: {str(e)}")

    # 상단 슬로건 (헤더 이미지 아래)
    slogan = header.get('slogan', '소통, 연결, 창조의 新 강진')
    if format_type == 'pdf':
        doc.add_paragraph(slogan, align='center')
        doc.add_spacing(20)
    else:
        doc.add_paragraph(slogan)
        doc.add_paragraph("")

    # 강진군 제목 (가운데 정렬, 큰 글씨)
    if format_type == 'pdf':
        doc.add_heading(template['title'], level=1, align='center')
        doc.add_spacing(40)
    else:
        doc.add_heading(template['title'], level=1)
        doc.add_paragraph("")
        doc.add_paragraph("")

    # 각 섹션별 처리
    for section in sections:
        section_type = section.get('section_type', '')
        fields = section.get('fields', [])

        if section_type == 'recipient_info':
            # 수신 정보
            for field in fields:
                label = field.get('label', '')
                field_path = field.get('field', '')
                default_value = field.get('default', '')

                if field_path:
                    value = template_manager.get_field_value(data, field_path, default_value)
                else:
                    value = default_value

                if label == '수신' and value:
                    doc.add_paragraph(f"{label}  {value}")
                elif label == '경유' and value:
                    doc.add_paragraph(f"{value}")

            if format_type == 'pdf':
                doc.add_spacing(20)
            else:
                doc.add_paragraph("")

        elif section_type == 'official_title':
            # 공문 제목 (굵은 글씨 + 밑줄)
            for field in fields:
                field_path = field.get('field', '')
                if field_path:
                    title_value = template_manager.get_field_value(data, field_path)
                    if title_value:
                        if format_type == 'pdf':
                            doc.add_heading(f"제목  {title_value}", level=2, align='left')
                            doc.add_spacing(20)
                        else:
                            doc.add_heading(f"제목  {title_value}", level=2)
                            doc.add_paragraph("")

        elif section_type == 'numbered_content':
            # 번호가 매겨진 공문 본문
            for field in fields:
                field_path = field.get('field', '')
                if field_path:
                    content_value = template_manager.get_field_value(data, field_path)
                    if content_value:
                        # 내용을 번호별로 나누어 처리
                        paragraphs = content_value.split('\n')
                        paragraph_number = 1

                        for para in paragraphs:
                            if para.strip():
                                if len(paragraphs) > 1:
                                    doc.add_paragraph(f"{paragraph_number}. {para.strip()}")
                                    paragraph_number += 1
                                else:
                                    # 단일 문단인 경우
                                    doc.add_paragraph(f"1. {para.strip()}")

                        if format_type == 'pdf':
                            doc.add_spacing(20)
                        else:
                            doc.add_paragraph("")

        elif section_type == 'structured_info':
            # 구조화된 정보 (사업개요 표 등)
            title = template_manager.get_field_value(data, 'application.title', '관련 사업')
            details = template_manager.get_field_value(data, 'application.details', '')

            # 간단한 사업 개요 표 생성
            if details:
                doc.add_paragraph("■ 사업개요")

                # 표 형태로 정보 구성
                table_data = [
                    ["구분", "내용"],
                    ["사업명", title],
                    ["사업내용", details[:100] + "..." if len(details) > 100 else details],
                    ["담당자", template_manager.get_field_value(data, 'applicant.name', '')],
                    ["연락처", template_manager.get_field_value(data, 'applicant.phone', '')]
                ]

                doc.add_table(table_data, header=["구분", "내용"], narrow_first_col=True)

                if format_type == 'pdf':
                    doc.add_spacing(20)
                else:
                    doc.add_paragraph("")

        elif section_type == 'attachments_simple':
            # 간단한 붙임 목록
            for field in fields:
                default_value = field.get('default', '관련 서류 1부. 끝.')
                doc.add_paragraph(f"붙임  {default_value}")

                if format_type == 'pdf':
                    doc.add_spacing(30)
                else:
                    doc.add_paragraph("")
                    doc.add_paragraph("")

    # 하단 연락처 테이블 (원본 공문 스타일)
    footer_fields = footer.get('fields', [])
    if footer_fields:
        # 민원봉사과장 테이블 생성
        table_data = [
            ["수신자", "도"],
            ["주무관", template_manager.get_field_value(data, 'applicant.name', ''), "건축팀장", "", "민원봉사과장", "작성 2025. 2. 8", "", "김승남"],
            ["협조자", ""],
            ["시행", "민원봉사과-6875", "", "", "", "접수", ""],
            ["우", "59228", "전라남도 강진군 강진읍 탑진로 111", "", "", "/ http://www.gangjin.go.kr"],
            ["전화번호", "061-430-3433", "팩스번호", "061-430-3429", "/ ph0055@korea.kr", "/ 대국민 공개"],
        ]

        # 하단에 구분선 추가
        if format_type == 'pdf':
            doc.add_spacing(40)
        else:
            doc.add_paragraph("")
            doc.add_paragraph("")

        doc.add_paragraph("민원봉사과장")
        if format_type == 'pdf':
            doc.add_spacing(10)

        # 간단한 연락처 정보만 표시
        contact_data = [
            ["담당자", template_manager.get_field_value(data, 'applicant.name', '')],
            ["연락처", template_manager.get_field_value(data, 'applicant.phone', '')],
            ["주소", "전라남도 강진군 강진읍 탑진로 111"],
            ["홈페이지", "http://www.gangjin.go.kr"]
        ]

        doc.add_table(contact_data, header=None, narrow_first_col=True)

    # 문서 저장
    doc.save(output_path)
    return output_path

def create_simple_document(doc, template, data, output_path, format_type, template_manager):
    """간단한 기본 문서 생성 - 자유로운 형식 (템플릿 구조 없음)"""
    current_date = datetime.now().strftime("%Y년 %m월 %d일")

    # 기본 문서는 헤더 이미지 없이 간단하게 시작

    # 1. 문서 제목 (사용자 지정 제목 사용)
    title = data.get('application', {}).get('title', '문서')
    if format_type == 'pdf':
        doc.add_heading(title, level=1, align='center')
        doc.add_spacing(30)
    else:
        doc.add_heading(title, level=1)
        doc.add_paragraph("")

    # 2. 문서 내용 (사용자 지정 내용만 사용, 양식 없음)
    content = data.get('application', {}).get('details', '내용을 입력해주세요.')

    # 내용을 줄바꿈으로 나누어 문단별로 처리
    paragraphs = content.split('\n')
    for para in paragraphs:
        if para.strip():
            doc.add_paragraph(para.strip())

    # 3. 간단한 날짜 추가 (선택사항)
    if format_type == 'pdf':
        doc.add_spacing(40)
        doc.add_paragraph(current_date, align='right')
    else:
        doc.add_paragraph("")
        doc.add_paragraph(current_date)

    # 문서 저장
    doc.save(output_path)
    return output_path

if __name__ == "__main__":
    main()