#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
Chandra vLLM 기반 문서 처리 모듈
페이지별로 처리하여 속도 및 메모리 문제 해결
"""

import os
import subprocess
import json
import tempfile
import shutil
from pathlib import Path
from typing import Dict, List, Optional, Callable, Union
import logging
import textwrap
from PIL import Image
import time
import re
import requests
import base64
from concurrent.futures import ThreadPoolExecutor, as_completed
import threading

# pypdfium2 (최우선), PyMuPDF, pdf2image 순서로 시도
try:
    import pypdfium2 as pdfium
    import pypdfium2.raw as pdfium_c
    HAS_PDFIUM = True
except ImportError:
    HAS_PDFIUM = False

try:
    import fitz  # PyMuPDF
    HAS_PYMUPDF = True
except ImportError:
    HAS_PYMUPDF = False

try:
    from pdf2image import convert_from_path
    HAS_PDF2IMAGE = True
except ImportError:
    HAS_PDF2IMAGE = False

logger = logging.getLogger(__name__)


def format_chandra_log(result: subprocess.CompletedProcess) -> str:
    divider = "─" * 60

    formatted = f"""
{divider}
🎉 Chandra OCR 실행 완료
{divider}
📌 명령어:
{textwrap.indent(" ".join(result.args), "  ")}

📎 리턴코드: {result.returncode}

📤 STDOUT:
{textwrap.indent(result.stdout.strip(), "  ") if result.stdout else "  (empty)"}

📥 STDERR:
{textwrap.indent(result.stderr.strip(), "  ") if result.stderr else "  (empty)"}
{divider}
"""
    return formatted


def flatten_pdf_page(page, flag=None):
    """PDF 페이지의 주석과 폼 필드를 평탄화 (pdfium 방식)"""
    if not HAS_PDFIUM:
        return

    if flag is None:
        flag = pdfium_c.FLAT_NORMALDISPLAY

    try:
        rc = pdfium_c.FPDFPage_Flatten(page, flag)
        if rc == pdfium_c.FLATTEN_FAIL:
            logger.warning(f"⚠️ 페이지 평탄화 실패")
    except Exception as e:
        logger.debug(f"⚠️ 페이지 평탄화 중 오류 (무시됨): {e}")


def translate_to_korean(english_text: str, ollama_url: str) -> str:
    """Translate English text to Korean using Ollama API"""
    
    # 빈 문자열, None, 공백만 있는 경우 번역하지 않음
    if not english_text or not english_text.strip():
        logger.debug("⏭️ 빈 텍스트는 번역하지 않음")
        return english_text or ""
    
    try:
        response = requests.post(
            f"{ollama_url}/api/generate",
            json={
                "model": "hamonize:latest",
                "prompt": f"다음 영어 이미지 설명을 자연스러운 한국어로 번역해주세요. 번역 결과만 출력하고 다른 설명은 하지 마세요.\n\n영어: {english_text}\n\n한국어:",
                "stream": False,
            },
            timeout=30,
        )

        if response.status_code == 200:
            korean_text = response.json().get('response', '').strip()
            logger.info(f"✅ 번역 완료: \"{english_text[:50]}...\" → \"{korean_text[:50]}...\"")
            return korean_text
        else:
            logger.warning(f"⚠️ Ollama API 오류: {response.status_code}")
            return english_text
    except Exception as e:
        logger.warning(f"⚠️ 번역 중 오류 발생: {e}")
        return english_text


def translate_image_alts_to_korean(content: str, ollama_url: str) -> str:
    """Translate all English alt attributes in HTML and Markdown to Korean"""

    # HTML img 태그의 alt 속성 추출
    html_img_regex = r'<img[^>]+alt="([^"]+)"[^>]*>'
    html_matches = list(re.finditer(html_img_regex, content))

    # Markdown 이미지 형식 추출 ![alt](src)
    # 더 견고한 정규식: 탐욕적이지 않은 매칭으로 모든 문자 허용 (이스케이프 문자 포함)
    md_img_regex = r'!\[(.*?)\]\([^)]+\)'
    md_matches = list(re.finditer(md_img_regex, content))

    logger.info(f"🔍 HTML 이미지 속성 매칭 수: {len(html_matches)}")
    logger.info(f"🔍 MD 이미지 속성 매칭 수: {len(md_matches)}")

    total_matches = len(html_matches) + len(md_matches)

    if total_matches == 0:
        logger.debug("⚠️ 번역할 이미지 alt 속성이 없습니다.")
        return content

    logger.info(f"🔄 {total_matches}개의 이미지 설명을 한국어로 번역 중...")

    translated_content = content

    # HTML 형식 번역
    for match in html_matches:
        full_img_tag = match.group(0)
        english_alt = match.group(1)

        # 빈 문자열이나 공백만 있으면 건너뛰기
        if not english_alt or not english_alt.strip():
            logger.debug(f"⏭️ 건너뜀 (빈 텍스트): HTML alt 속성")
            continue

        # 이미 한글이 포함되어 있으면 건너뛰기
        if re.search(r'[ㄱ-ㅎ|ㅏ-ㅣ|가-힣]', english_alt):
            logger.debug(f"⏭️ 건너뜀 (이미 한글 포함): \"{english_alt[:50]}...\"")
            continue

        # 영어 설명을 한국어로 번역
        korean_alt = translate_to_korean(english_alt, ollama_url)

        # HTML에서 영어 alt를 한국어 alt로 교체
        new_img_tag = full_img_tag.replace(f'alt="{english_alt}"', f'alt="{korean_alt}"')
        translated_content = translated_content.replace(full_img_tag, new_img_tag)

    # Markdown 형식 번역
    for match in md_matches:
        full_md_img = match.group(0)
        english_alt = match.group(1)

        # 빈 문자열이나 공백만 있으면 건너뛰기
        if not english_alt or not english_alt.strip():
            logger.info(f"⏭️ 건너뜀 (빈 텍스트): Markdown alt 속성")
            continue

        # 이미 한글이 포함되어 있으면 건너뛰기
        if re.search(r'[ㄱ-ㅎ|ㅏ-ㅣ|가-힣]', english_alt):
            logger.info(f"⏭️ 건너뜀 (이미 한글 포함): \"{english_alt[:50]}...\"")
            continue

        # 영어 설명을 한국어로 번역
        korean_alt = translate_to_korean(english_alt, ollama_url)

        # Markdown에서 영어 alt를 한국어 alt로 교체
        new_md_img = full_md_img.replace(f'[{english_alt}]', f'[{korean_alt}]')
        translated_content = translated_content.replace(full_md_img, new_md_img)

    logger.info("✅ 모든 이미지 설명 번역 완료")
    return translated_content


def encode_image_to_base64(image_path: Path) -> str:
    """이미지 파일을 base64로 인코딩"""
    try:
        with open(image_path, 'rb') as image_file:
            encoded = base64.b64encode(image_file.read()).decode('utf-8')
            return encoded
    except Exception as e:
        logger.error(f"❌ 이미지 base64 인코딩 실패 ({image_path}): {e}")
        return ""


def generate_image_description_with_vision(image_path: Path, ollama_url: str) -> str:
    """airun-vision 모델을 사용하여 이미지 설명 생성"""

    if not image_path.exists():
        logger.warning(f"⚠️ 이미지 파일이 존재하지 않음: {image_path}")
        return ""

    try:
        # 이미지를 base64로 인코딩
        base64_image = encode_image_to_base64(image_path)
        if not base64_image:
            return ""

        # airun-vision 모델로 이미지 설명 요청
        logger.info(f"🔍 airun-vision으로 이미지 설명 생성 중: {image_path.name}")

        response = requests.post(
            f"{ollama_url}/api/generate",
            json={
                "model": "airun-vision:latest",
                "prompt": "이 이미지를 100자 이내의 한 문장으로 간결하게 설명해주세요. 이미지의 핵심 내용만 포함하고, 줄바꿈 없이 한 줄로 작성하세요.",
                "images": [base64_image],
                "stream": False,
            },
            timeout=60,
        )

        if response.status_code == 200:
            description = response.json().get('response', '').strip()

            # 후처리: 줄바꿈 제거 및 공백 정리
            description = ' '.join(description.split())

            # 첫 문장만 추출 (마침표, 느낌표, 물음표로 끝나는 첫 문장)
            import re
            sentences = re.split(r'[.!?]\s+', description)
            if sentences:
                description = sentences[0]
                # 문장 끝에 마침표가 없으면 추가
                if not description.endswith(('.', '!', '?')):
                    description += '.'

            # 최대 길이 제한 (150자)
            if len(description) > 150:
                description = description[:147] + '...'

            logger.info(f"✅ 이미지 설명 생성 완료: \"{description}\"")
            return description
        else:
            logger.warning(f"⚠️ airun-vision API 오류: {response.status_code}")
            return ""

    except Exception as e:
        logger.warning(f"⚠️ 이미지 설명 생성 중 오류: {e}")
        return ""


def generate_page_visual_description(page_image_path: Path, ollama_url: str) -> str:
    """페이지 전체 이미지에 대한 시각적 설명 생성"""

    if not page_image_path.exists():
        logger.warning(f"⚠️ 페이지 이미지 파일이 존재하지 않음: {page_image_path}")
        return ""

    try:
        # 이미지를 base64로 인코딩
        base64_image = encode_image_to_base64(page_image_path)
        if not base64_image:
            return ""

        # airun-vision 모델로 페이지 전체 설명 요청
        logger.info(f"📄 airun-vision으로 페이지 전체 설명 생성 중: {page_image_path.name}")

        response = requests.post(
            f"{ollama_url}/api/generate",
            json={
                "model": "airun-vision:latest",
                "prompt": "이 페이지의 시각적 요소(이미지, 도표, 그림 등)가 무엇을 표현하고 있는지 설명해주세요.",
                # "prompt": "이 페이지의 시각적 요소(이미지, 도표, 그림 등)가 무엇을 표현하고 있는지 설명해주세요. 레이아웃 분석이나 구조적 설명은 생략하고, 이미지의 내용과 흐름만 3-5문장으로 서술하세요.",
                "images": [base64_image],
                "stream": False,
            },
            timeout=60,
        )

        if response.status_code == 200:
            description = response.json().get('response', '').strip()
            logger.info(f"✅ 페이지 설명 생성 완료: \"{description[:100]}...\"")
            return description
        else:
            logger.warning(f"⚠️ airun-vision API 오류: {response.status_code}")
            return ""

    except Exception as e:
        logger.warning(f"⚠️ 페이지 설명 생성 중 오류: {e}")
        return ""




def fill_missing_image_descriptions(content: str, image_dir: Path, ollama_url: str) -> str:
    """설명이 없는 이미지에 대해 airun-vision으로 설명 생성 및 채우기"""

    # 너무 길면 전체 content 로그는 피하는 편이 좋습니다.
    # logger.debug("content 미리보기: %s", content[:500])

    # 1) HTML <img> 태그 처리
    img_tag_regex = r'<img[^>]*>'
    html_matches: list[tuple[re.Match, str, str | None]] = []  # (match_obj, src, alt_text)

    for m in re.finditer(img_tag_regex, content):
        tag = m.group(0)

        # src 추출
        src_match = re.search(r'src="([^"]+)"', tag)
        if not src_match:
            continue
        src = src_match.group(1)

        # alt 추출 (없을 수도 있음)
        alt_match = re.search(r'alt="([^"]*)"', tag)
        alt_text = alt_match.group(1).strip() if alt_match else None

        # 재생성 대상 조건
        # - alt 없음
        # - alt가 빈 문자열
        # - alt 길이가 너무 짧은 경우(예: 3자 이하)
        if alt_text is None or alt_text == "":
            html_matches.append((m, src, alt_text))
            logger.info("🔍 HTML 이미지 alt 없음: src=%s -> 재생성 대상", src)
        elif len(alt_text) <= 3:
            html_matches.append((m, src, alt_text))
            logger.info("🔍 HTML 이미지 alt 불완전: alt='%s', src=%s -> 재생성 대상", alt_text, src)

    # 2) Markdown 이미지 처리
    md_img_regex = r'!\[([^\]]*)\]\(([^)]+)\)'
    md_matches_all = list(re.finditer(md_img_regex, content))
    logger.info("이미지 전체 매칭 수: %d", len(md_matches_all))

    md_matches: list[re.Match] = []
    for match in md_matches_all:
        alt_text = match.group(1).strip()
        if (
            not alt_text
            or len(alt_text) <= 5
            or alt_text in [ "한국어", "한국어:", "Korean", "Korean:", "English", "English:", "영어", "영어:", "이미지", "그림", "Image", ":", "-", ]
        ):
            md_matches.append(match)
            logger.info(
                "🔍 불완전한 이미지 설명 감지: alt='%s' -> 재생성 대상",
                alt_text,
            )

    if md_matches:
        md_src_samples = [m.group(2) for m in md_matches[:5]]
        logger.info("Markdown 재생성 대상 이미지 src 샘플 (최대 5개): %s", md_src_samples)

    total_matches = len(html_matches) + len(md_matches)

    if total_matches == 0:
        logger.info("💡 alt 재생성 대상 이미지 없음 (HTML/Markdown 모두 정상)")
        return content

    logger.info(
        "▶ alt 재생성 대상 이미지 총 %d개 (HTML=%d, Markdown=%d) - vision 호출 예정",
        total_matches,
        len(html_matches),
        len(md_matches),
    )
    logger.info("🔧 %d개의 설명 없는/불완전한 이미지에 대해 설명 생성 중...", total_matches)

    updated_content = content

    # 3) HTML 이미지 설명 채우기
    for match_obj, img_src, alt_text in html_matches:
        full_img_tag = match_obj.group(0)
        if not img_src:
            continue

        img_filename = Path(img_src).name
        img_path = image_dir / img_filename

        if not img_path.exists():
            logger.debug("⏭️ 이미지 파일 없음: %s", img_path)
            continue

        description = generate_image_description_with_vision(img_path, ollama_url)

        if not description:
            logger.debug("⏭️ vision 설명 생성 실패 또는 빈 값: %s", img_path)
            continue

        # alt 속성이 이미 있는 경우: alt="..." 전체를 교체
        if 'alt="' in full_img_tag:
            new_img_tag = re.sub(
                r'alt="[^"]*"',
                f'alt="{description}"',
                full_img_tag,
            )
        else:
            # alt 속성이 없는 경우: <img 뒤에 alt 추가
            new_img_tag = full_img_tag.replace(
                "<img",
                f'<img alt="{description}"',
                1,  # 첫 번째만 치환
            )

        updated_content = updated_content.replace(full_img_tag, new_img_tag)
        logger.info("✅ HTML 이미지 설명 추가: %s", img_filename)
        
    # Markdown 형식 처리
    for match in md_matches:
        full_md_img = match.group(0)
        img_src = match.group(2)  # 수정: group(2)가 이미지 소스 경로

        # 이미지 파일 경로 추출
        img_filename = Path(img_src).name
        img_path = image_dir / img_filename

        if not img_path.exists():
            logger.debug(f"⏭️ 이미지 파일 없음: {img_path}")
            continue

        # airun-vision으로 설명 생성
        description = generate_image_description_with_vision(img_path, ollama_url)

        if description:
            # alt 텍스트 추가
            new_md_img = full_md_img.replace('![]', f'![{description}]')
            updated_content = updated_content.replace(full_md_img, new_md_img)
            logger.info(f"✅ Markdown 이미지 설명 추가: {img_filename}")

    logger.info("✅ 모든 설명 없는 이미지에 설명 추가 완료")
    return updated_content



class ChandraDocumentProcessor:
    """
    Chandra vLLM을 사용한 문서 처리
    페이지별로 이미지 변환 → Chandra 분석 → 청크 생성
    """

    def __init__(self, config: Optional[Dict] = None):
        """
        Args:
            config: 설정 딕셔너리
                - chandra_path: Chandra 설치 경로
                - vllm_api_base: vLLM API 엔드포인트
                - vllm_model_name: vLLM 모델명
                - ollama_proxy: Ollama 프록시 서버
                - max_workers: 최대 워커 수
                - timeout: 처리 타임아웃 (초)
        """
        self.config = config or {}

        # Thread-safe를 위한 Lock 객체들
        self._file_write_lock = threading.Lock()  # 파일 쓰기용
        self._image_copy_lock = threading.Lock()  # 이미지 복사용

        # 환경변수 또는 설정에서 값 가져오기
        self.chandra_path = (
            self.config.get('chandra_path')
            or os.environ.get('CHANDRA_PATH')
            or self._read_chandra_path_from_conf()
            or self._default_chandra_path()
        )
        self.vllm_api_base = self.config.get('vllm_api_base') or os.environ.get(
            'VLLM_API_BASE', 'http://121.78.116.30:8000/v1'
        )
        self.vllm_model_name = self.config.get('vllm_model_name') or os.environ.get(
            'VLLM_MODEL_NAME', 'chandra'
        )
        self.ollama_proxy = self.config.get('ollama_proxy') or os.environ.get(
            'OLLAMA_PROXY_SERVER', 'https://api.hamonize.com/ollama'
        )

        # max_workers 설정: config > 환경변수 > RAG 설정 > 기본값(2)
        # 병렬 처리 최적화를 위한 워커 수 설정
        max_workers_value = (
            self.config.get('max_workers') or
            self.config.get('chandra_max_workers') or
            os.environ.get('RAG_CHANDRA_MAX_WORKERS') or
            '2'
        )
        try:
            self.max_workers = int(max_workers_value)
            logger.info(f"🔧 Chandra max_workers 설정: {self.max_workers}")
        except (ValueError, TypeError):
            self.max_workers = 2
            logger.warning(f"⚠️ max_workers 값 파싱 실패 ({max_workers_value}), 기본값 2 사용")

        # 타임아웃 설정: config > 환경변수 > 기본값(10분)
        # 실제 처리는 페이지당 10-20초 정도이지만 vLLM 서버 응답 지연을 고려하여 여유있게 설정
        timeout_value = (
            self.config.get('timeout') or
            self.config.get('chandra_timeout') or
            os.environ.get('RAG_CHANDRA_TIMEOUT') or
            os.environ.get('CHANDRA_TIMEOUT') or
            '600'
        )
        try:
            self.timeout = int(timeout_value)
        except (ValueError, TypeError):
            self.timeout = 600
            logger.warning(f"⚠️ timeout 값 파싱 실패 ({timeout_value}), 기본값 600 사용")

        # Chandra CLI가 있는지 확인
        if not os.path.exists(self.chandra_path):
            raise FileNotFoundError(f"Chandra path not found: {self.chandra_path}")

        self.chandra_cli = self._resolve_chandra_cli()

    @staticmethod
    def _read_chandra_path_from_conf() -> Optional[str]:
        """~/.airun/airun.conf에서 CHANDRA_PATH 값을 읽어온다."""
        home_dir = os.environ.get('HOME')
        if not home_dir:
            return None
        conf_path = Path(home_dir) / '.airun' / 'airun.conf'
        if not conf_path.exists():
            return None

        try:
            for line in conf_path.read_text().splitlines():
                line = line.strip()
                if line.startswith('export CHANDRA_PATH='):
                    value = line.split('=', 1)[1].strip().strip("'\"")
                    return value or None
        except Exception as error:
            logger.warning(f"airun.conf에서 CHANDRA_PATH 읽기 실패: {error}")
        return None

    @staticmethod
    def _default_chandra_path() -> str:
        """환경 변수와 설정이 모두 없을 때 사용할 기본 경로"""
        home_dir = os.environ.get('HOME')
        if home_dir:
            return str(Path(home_dir) / 'airun' / 'chandra')
        # HOME이 없을 때만 이전 기본 경로 사용
        return '/home/hamonikr/work/airun/chandra'

    def _resolve_chandra_cli(self) -> str:
        """
        Chandra CLI 실행 파일 경로를 결정한다.
        1) config.chandra_cli 또는 CHANDRA_CLI_PATH 환경변수
        2) ~/.airun_venv/bin/chandra
        3) PATH에 등록된 chandra
        """
        candidates = []

        explicit_cli = self.config.get('chandra_cli') or os.environ.get('CHANDRA_CLI_PATH')
        if explicit_cli:
            candidates.append(explicit_cli)

        home_dir = os.environ.get('HOME')
        if home_dir:
            candidates.append(os.path.join(home_dir, '.airun_venv', 'bin', 'chandra'))

        which_cli = shutil.which('chandra')
        if which_cli:
            candidates.append(which_cli)

        # 이전 코드와의 호환성을 위해 chandra_path/chandra 도 확인
        candidates.append(os.path.join(self.chandra_path, 'chandra'))

        for candidate in candidates:
            if candidate and os.path.isfile(candidate) and os.access(candidate, os.X_OK):
                return candidate

        raise FileNotFoundError(
            "Chandra CLI 실행 파일을 찾을 수 없습니다. "
            "install.sh를 다시 실행하거나 CHANDRA_CLI_PATH 환경변수를 설정하세요."
        )

    def process_document(
        self,
        file_path: str,
        progress_callback: Optional[Callable[[int, int, str], None]] = None,
        page_processed_callback: Optional[Callable[[Dict, int, int], None]] = None,
        return_format: str = 'dictionary'
    ) -> Union[Dict, List[Dict]]:
        """
        문서를 페이지별로 처리

        Args:
            file_path: 처리할 문서 경로 (PDF, 이미지 등)
            progress_callback: 진행상황 콜백 함수 (current_page, total_pages, message)
            page_processed_callback: 페이지 처리 완료 콜백 함수 (page_result, current_page, total_pages)
                                    각 페이지 처리 완료 즉시 호출됨
            return_format: 반환 형식 ('dictionary' 또는 'list')
                          - 'dictionary': 기존 RAG 형식과 호환되는 Dictionary 반환
                          - 'list': 페이지별 List[Dict] 반환

        Returns:
            Union[Dict, List[Dict]]: 처리 결과

            Dictionary format (return_format='dictionary'):
            {
                'status': 'success',
                'text': '전체 문서 텍스트',
                'pages': [...],
                'total_pages': 10,
                'successful_pages': 10,
                'failed_pages': [],
                'processed_by': 'chandra',
                'message': '처리 완료 메시지'
            }

            List format (return_format='list'):
            [
                {
                    'page_num': 0,
                    'markdown': '# 제목...',
                    'html': '<h1>제목</h1>...',
                    'metadata': {...},
                    'images': ['img1.webp', ...],
                    'error': None
                },
                ...
            ]
        """
        file_path = Path(file_path)

        if not file_path.exists():
            raise FileNotFoundError(f"File not found: {file_path}")

        # 파일 형식에 따라 처리
        ext = file_path.suffix.lower()

        if ext == '.pdf':
            page_results = self._process_pdf(file_path, progress_callback, page_processed_callback)
        elif ext in ['.png', '.jpg', '.jpeg', '.webp', '.gif', '.tiff', '.bmp']:
            page_results = self._process_image(file_path, progress_callback, page_processed_callback)
        else:
            raise ValueError(f"Unsupported file format: {ext}")

        # 반환 형식에 따라 변환
        if return_format == 'dictionary':
            return self._convert_pages_to_dictionary(page_results)
        else:
            return page_results

    def _copy_images_to_permanent_storage(
        self,
        pdf_path: Path,
        temp_dir_path: Path,
        results: List[Dict]
    ) -> None:
        """
        임시 디렉토리의 이미지를 영구 저장 위치로 복사 (Thread-Safe)

        기존 RAG 이미지 추출과 동일한 구조 사용:
        {PDF가 있는 디렉토리}/.extracts/{문서명}/

        Args:
            pdf_path: 원본 PDF 파일 경로
            temp_dir_path: Chandra가 이미지를 생성한 임시 디렉토리
            results: 페이지별 처리 결과 (images 리스트 업데이트됨)
        """
        # 🔒 Lock을 사용하여 이미지 복사 작업을 Thread-Safe하게 보호
        with self._image_copy_lock:
            try:
                # 문서 이름 추출 (확장자 제외)
                doc_name = pdf_path.stem

                # .extracts/{문서명}/ 디렉토리 생성
                extract_dir = pdf_path.parent / '.extracts' / doc_name
                extract_dir.mkdir(parents=True, exist_ok=True)

                logger.debug(f"📂 이미지 영구 저장 디렉토리: {extract_dir}")

                # Chandra는 page_{num}/{image_stem}/ 구조로 이미지 저장
                # 모든 페이지 디렉토리를 순회하며 이미지 수집
                copied_count = 0
                for page_result in results:
                    page_num = page_result['page_num']

                    # page_{num} 디렉토리 찾기
                    page_dir = temp_dir_path / f"page_{page_num}"

                    if not page_dir.exists():
                        continue

                    # page_{num} 안의 모든 서브디렉토리에서 .png 파일 찾기
                    for png_file in page_dir.rglob("*.png"):
                        dest_file = extract_dir / png_file.name

                        # 이미 존재하는 파일은 덮어쓰지 않고 건너뛰기 (중복 방지)
                        if not dest_file.exists():
                            shutil.copy2(png_file, dest_file)
                            copied_count += 1
                            logger.debug(f"  ✅ 복사: {png_file.name} → {extract_dir}")
                        else:
                            logger.debug(f"  ⏭️  건너뜀 (이미 존재): {png_file.name}")

                if copied_count > 0:
                    logger.info(f"🖼️ 페이지 {page_result['page_num']}: {copied_count}개 이미지 저장 완료")

            except Exception as e:
                logger.error(f"❌ 이미지 영구 저장 실패: {e}")

    def _convert_pages_to_dictionary(self, page_results: List[Dict]) -> Dict:
        """
        페이지별 List[Dict] 결과를 기존 RAG 형식의 Dictionary로 변환

        Args:
            page_results: 페이지별 처리 결과 List

        Returns:
            Dict: 기존 RAG 형식과 호환되는 Dictionary
            {
                'status': 'success',
                'text': '전체 문서 텍스트',
                'pages': [...],
                'total_pages': 10,
                'successful_pages': 10,
                'failed_pages': [],
                'processed_by': 'chandra',
                'message': '처리 완료 메시지'
            }
        """
        combined_text = ""
        successful_pages = 0
        failed_pages = []

        for page in page_results:
            page_num = page.get('page_num', 0)
            markdown = page.get('markdown', '')
            error = page.get('error')

            if error:
                failed_pages.append(f"페이지 {page_num}: {error}")
            elif markdown and len(markdown.strip()) > 0:
                combined_text += markdown + "\n\n"
                successful_pages += 1

        # 통합 결과 Dictionary 생성
        if successful_pages > 0:
            return {
                'status': 'success',
                'text': combined_text.strip(),
                'pages': page_results,
                'total_pages': len(page_results),
                'successful_pages': successful_pages,
                'failed_pages': failed_pages,
                'processed_by': 'chandra',
                'message': f'Chandra 처리 완료: {successful_pages}/{len(page_results)} 페이지 성공'
            }
        else:
            return {
                'status': 'failed',
                'text': '',
                'pages': page_results,
                'total_pages': len(page_results),
                'successful_pages': 0,
                'failed_pages': failed_pages,
                'processed_by': 'chandra',
                'message': f'Chandra 처리 실패: 모든 페이지 실패'
            }

    def _process_pdf(
        self,
        pdf_path: Path,
        progress_callback: Optional[Callable] = None,
        page_processed_callback: Optional[Callable] = None
    ) -> List[Dict]:
        """PDF를 페이지별로 처리 (병렬 처리 지원)"""

        # 1. PDF를 페이지별 이미지로 변환
        logger.info(f"📄 PDF 페이지 추출 중: {pdf_path.name}")
        page_images = self._pdf_to_images(pdf_path)
        total_pages = len(page_images)

        logger.info(f"📊 총 {total_pages}개 페이지 발견")

        # max_workers가 1보다 크면 병렬 처리, 아니면 순차 처리
        if self.max_workers > 1:
            logger.info(f"🚀 병렬 처리 모드: {self.max_workers}개 워커 사용")
            return self._process_pdf_parallel(
                pdf_path, page_images, total_pages,
                progress_callback, page_processed_callback
            )
        else:
            logger.info(f"🔄 순차 처리 모드")
            return self._process_pdf_sequential(
                pdf_path, page_images, total_pages,
                progress_callback, page_processed_callback
            )

    def _process_pdf_sequential(
        self,
        pdf_path: Path,
        page_images: List[Path],
        total_pages: int,
        progress_callback: Optional[Callable] = None,
        page_processed_callback: Optional[Callable] = None
    ) -> List[Dict]:
        """PDF를 순차적으로 처리 (기존 방식)"""

        results = []

        with tempfile.TemporaryDirectory() as temp_dir:
            temp_dir_path = Path(temp_dir)

            for page_num, page_image_path in enumerate(page_images, start=1):
                try:
                    if progress_callback:
                        progress_callback(
                            page_num,
                            total_pages,
                            f"페이지 {page_num}/{total_pages} 처리 중..."
                        )

                    logger.info(f"🔄 페이지 {page_num}/{total_pages} 처리 시작")

                    # Chandra로 단일 페이지 처리
                    page_result = self._process_single_page(
                        page_image_path,
                        page_num,
                        temp_dir_path
                    )

                    results.append(page_result)

                    # 처리된 페이지의 이미지를 즉시 영구 저장 위치로 복사
                    self._copy_images_to_permanent_storage(pdf_path, temp_dir_path, [page_result])

                    # 페이지 처리 완료 콜백 호출
                    if page_processed_callback:
                        page_processed_callback(page_result, page_num, total_pages)

                    logger.info(f"✅ 페이지 {page_num}/{total_pages} 처리 완료")

                except Exception as e:
                    logger.error(f"❌ 페이지 {page_num} 처리 실패: {e}")
                    results.append({
                        'page_num': page_num,
                        'markdown': '',
                        'html': '',
                        'metadata': {},
                        'images': [],
                        'error': str(e)
                    })

                finally:
                    # 임시 이미지 파일 삭제
                    if os.path.exists(page_image_path):
                        os.remove(page_image_path)

        return results

    def _process_pdf_parallel(
        self,
        pdf_path: Path,
        page_images: List[Path],
        total_pages: int,
        progress_callback: Optional[Callable] = None,
        page_processed_callback: Optional[Callable] = None
    ) -> List[Dict]:
        """PDF를 병렬로 처리 (여러 페이지 동시 처리)"""

        results = [None] * total_pages  # 페이지 순서 유지를 위한 리스트
        processed_count = 0

        with tempfile.TemporaryDirectory() as temp_dir:
            temp_dir_path = Path(temp_dir)

            # ThreadPoolExecutor로 병렬 처리
            with ThreadPoolExecutor(max_workers=self.max_workers) as executor:
                # 모든 페이지 작업 제출
                future_to_page = {
                    executor.submit(
                        self._process_single_page,
                        page_image_path,
                        page_num,
                        temp_dir_path
                    ): (page_num, page_image_path)
                    for page_num, page_image_path in enumerate(page_images, start=1)
                }

                # 완료된 작업 처리
                for future in as_completed(future_to_page):
                    page_num, page_image_path = future_to_page[future]
                    processed_count += 1

                    try:
                        page_result = future.result()
                        results[page_num - 1] = page_result

                        # 처리된 페이지의 이미지를 즉시 영구 저장 위치로 복사
                        self._copy_images_to_permanent_storage(pdf_path, temp_dir_path, [page_result])

                        # 페이지 처리 완료 콜백 호출
                        if page_processed_callback:
                            page_processed_callback(page_result, page_num, total_pages)

                        logger.info(f"✅ 페이지 {page_num}/{total_pages} 처리 완료 (진행률: {processed_count}/{total_pages})")

                    except Exception as e:
                        logger.error(f"❌ 페이지 {page_num} 처리 실패: {e}")
                        results[page_num - 1] = {
                            'page_num': page_num,
                            'markdown': '',
                            'html': '',
                            'metadata': {},
                            'images': [],
                            'error': str(e)
                        }

                    finally:
                        # 임시 이미지 파일 삭제
                        if os.path.exists(page_image_path):
                            os.remove(page_image_path)

                    # 진행상황 콜백 (비동기이므로 정확한 순서는 아님)
                    if progress_callback:
                        progress_callback(
                            processed_count,
                            total_pages,
                            f"페이지 {processed_count}/{total_pages} 처리 완료 (최근: {page_num})"
                        )

        return results

    def _process_image(
        self,
        image_path: Path,
        progress_callback: Optional[Callable] = None,
        page_processed_callback: Optional[Callable] = None
    ) -> List[Dict]:
        """단일 이미지 파일 처리"""

        if progress_callback:
            progress_callback(1, 1, "이미지 처리 중...")

        with tempfile.TemporaryDirectory() as temp_dir:
            result = self._process_single_page(image_path, 1, Path(temp_dir))

        # 페이지 처리 완료 콜백 호출
        if page_processed_callback:
            page_processed_callback(result, 1, 1)

        return [result]

    def _process_single_page(
        self,
        image_path: Path,
        page_num: int,
        output_dir: Path
    ) -> Dict:
        """
        단일 페이지를 Chandra로 처리

        Args:
            image_path: 페이지 이미지 경로
            page_num: 페이지 번호
            output_dir: 출력 디렉토리

        Returns:
            Dict: 페이지 처리 결과
        """
        # 전체 페이지 처리 시간 측정 시작
        page_total_start = time.time()
        logger.info(f"⏱️  ▶️  페이지 {page_num} 처리 시작...")

        # Chandra CLI 실행
        page_output_dir = output_dir / f"page_{page_num}"
        page_output_dir.mkdir(exist_ok=True)

        chandra_cli = self.chandra_cli
        python_path_entries = [self.chandra_path]
        existing_pythonpath = os.environ.get('PYTHONPATH')
        if existing_pythonpath:
            python_path_entries.append(existing_pythonpath)
        python_path = ":".join(filter(None, python_path_entries))

        logger.info(f"🚀 Chandra 실행: {chandra_cli} '{image_path}'")

        # Shell injection 방지: 명령어를 리스트로 구성하고 환경변수는 env로 전달
        cmd_list = [
            chandra_cli,
            str(image_path),
            str(page_output_dir),
            "--method", "vllm",
            "--max-output-tokens", "4096",
            "--max-workers", str(self.max_workers)
        ]

        env = {
            **os.environ,
            "PYTHONPATH": python_path,
            "OLLAMA_PROXY_SERVER": self.ollama_proxy,
            "VLLM_API_BASE": self.vllm_api_base,
            "VLLM_MODEL_NAME": self.vllm_model_name,
            "VLLM_API_KEY": "EMPTY"
        }

        # DEBUG: 환경변수 확인
        logger.info(f"🔍 DEBUG - VLLM_API_BASE: {self.vllm_api_base}")
        logger.info(f"🔍 DEBUG - VLLM_MODEL_NAME: {self.vllm_model_name}")
        logger.info(f"🔍 DEBUG - Chandra command: {' '.join(cmd_list)}")

        try:
            start_time = time.time()

            result = subprocess.run(
                cmd_list,
                env=env,
                capture_output=True,
                text=True,
                timeout=self.timeout
            )

            chandra_elapsed = time.time() - start_time
            logger.info(format_chandra_log(result))
            logger.info(f"⏱️  Chandra vLLM 처리 시간: {chandra_elapsed:.2f}초")

            # DEBUG: Chandra 출력 확인
            logger.info(f"🔍 DEBUG - Chandra stdout length: {len(result.stdout)}")
            logger.info(f"🔍 DEBUG - Chandra returncode: {result.returncode}")

            # stderr 출력이 있으면 로깅 (경고일 수 있음)
            if result.stderr:
                # FutureWarning 등의 경고는 무시하고 로그만 남김
                if 'FutureWarning' in result.stderr or 'DeprecationWarning' in result.stderr:
                    logger.debug(f"⚠️  Chandra 경고 메시지: {result.stderr[:200]}...")
                else:
                    logger.warning(f"⚠️  Chandra stderr: {result.stderr[:500]}")

            # returncode가 0이 아니고 실제 오류인 경우에만 실패 처리
            if result.returncode != 0:
                # 출력 파일이 생성되었는지 확인
                file_dir = page_output_dir / image_path.stem
                expected_md = file_dir / f"{image_path.stem}.md"

                if not expected_md.exists():
                    raise Exception(f"Chandra 처리 실패 (returncode={result.returncode}): {result.stderr}")
                else:
                    # 파일은 생성되었지만 returncode가 0이 아닌 경우 (경고만 있는 경우)
                    logger.warning(f"⚠️  Chandra가 경고와 함께 완료됨 (returncode={result.returncode})")

            # 결과 파일 로드 (페이지 이미지 경로도 전달)
            result = self._load_page_result(page_output_dir, image_path.stem, page_num, image_path)

            # 전체 페이지 처리 시간 로깅
            page_total_elapsed = time.time() - page_total_start
            logger.info(f"⏱️  ═══════════════════════════════════════════════════════")
            logger.info(f"⏱️  ✅ 페이지 {page_num} 처리 완료 - 총 소요 시간: {page_total_elapsed:.2f}초")
            logger.info(f"⏱️  ═══════════════════════════════════════════════════════")

            return result

        except subprocess.TimeoutExpired:
            raise Exception(f"Chandra 처리 시간 초과 ({self.timeout}초)")
        except Exception as e:
            # 파일 로드 실패 등의 경우 더 자세한 에러 메시지 제공
            if "FileNotFoundError" in str(type(e).__name__):
                logger.error(f"❌ Chandra 출력 파일을 찾을 수 없음: {e}")
            raise Exception(f"Chandra 처리 중 오류: {e}")

    def _load_page_result(
        self,
        result_dir: Path,
        base_name: str,
        page_num: int,
        page_image_path: Optional[Path] = None
    ) -> Dict:
        """Chandra 출력 결과 로드 및 후처리"""

        file_dir = result_dir / base_name
        markdown_file = file_dir / f"{base_name}.md"
        html_file = file_dir / f"{base_name}.html"
        json_file = file_dir / f"{base_name}.json"
        metadata_file = file_dir / f"{base_name}_metadata.json"

        # Markdown 읽기
        markdown = ""
        if markdown_file.exists():
            file_size = markdown_file.stat().st_size
            logger.info(f"🔍 DEBUG - Markdown 파일 존재: {markdown_file}, 크기: {file_size} bytes")
            with open(markdown_file, 'r', encoding='utf-8') as f:
                markdown = f.read()
            logger.info(f"🔍 DEBUG - Markdown 읽기 완료: {len(markdown)} chars")

            # Chandra가 생성한 원본 이미지 마크다운 확인
            import re
            img_matches = re.findall(r'!\[([^\]]*)\]\(([^)]+)\)', markdown)
            if img_matches:
                logger.info(f"🔍 Chandra 생성 원본 이미지 마크다운: {img_matches}")
        else:
            logger.warning(f"⚠️ Markdown 파일 없음: {markdown_file}")

        # HTML 읽기
        html = ""
        if html_file.exists():
            with open(html_file, 'r', encoding='utf-8') as f:
                html = f.read()

        # JSON 데이터 읽기
        structured_data = {}
        if json_file.exists():
            with open(json_file, 'r', encoding='utf-8') as f:
                structured_data = json.load(f)

        # Metadata 읽기
        metadata = {}
        if metadata_file.exists():
            with open(metadata_file, 'r', encoding='utf-8') as f:
                metadata = json.load(f)

        # 이미지 파일 목록
        images = []
        if file_dir.exists():
            images = [f.name for f in file_dir.glob("*.webp")]

        # 🌐 한국어 번역 후처리 (Chandra 패키지에서 번역 안된 경우 대비)
        translate_start = time.time()
        logger.info(f"📝 페이지 {page_num}의 이미지 번역 후처리 중...")
        if markdown:
            logger.info(f"🔍 DEBUG - 번역 전 Markdown content: {markdown}")
            markdown = translate_image_alts_to_korean(markdown, self.ollama_proxy)
        if html:
            logger.info(f"🔍 DEBUG - 번역 전 html content: {html}")
            html = translate_image_alts_to_korean(html, self.ollama_proxy)
        translate_elapsed = time.time() - translate_start
        if translate_elapsed > 0.1:  # 0.1초 이상 걸린 경우만 로깅
            logger.info(f"⏱️  번역 후처리 시간: {translate_elapsed:.2f}초")

        # 🔧 설명 없는 이미지 처리
        vision_start = time.time()
        if markdown and file_dir.exists():
            logger.info(f"🔧 [markdown] 페이지 {page_num}에서 설명 없는 이미지 확인 및 설명 추가 처리 중...")
            markdown = fill_missing_image_descriptions(markdown, file_dir, self.ollama_proxy)
            # 🔒 Thread-Safe 파일 쓰기
            if markdown_file.exists():
                with self._file_write_lock:
                    with open(markdown_file, 'w', encoding='utf-8') as f:
                        f.write(markdown)

        if html and file_dir.exists():
            logger.info(f"🔧 [html] 페이지 {page_num}에서 설명 없는 이미지 확인 및 설명 추가 처리 중...")
            html = fill_missing_image_descriptions(html, file_dir, self.ollama_proxy)
            # 🔒 Thread-Safe 파일 쓰기
            if html_file.exists():
                with self._file_write_lock:
                    with open(html_file, 'w', encoding='utf-8') as f:
                        f.write(html)
        vision_elapsed = time.time() - vision_start
        if vision_elapsed > 0.1:  # 0.1초 이상 걸린 경우만 로깅
            logger.info(f"⏱️  airun-vision 이미지 설명 생성 시간: {vision_elapsed:.2f}초")

        # 📄 페이지 전체 시각적 설명 생성 (설정에 따라 선택적으로 생성)
        page_visual_description = ""

        # get_rag_settings()를 통해 설정값 확인
        from rag_process import get_rag_settings
        rag_settings = get_rag_settings()
        add_page_desc = rag_settings.get('add_page_description', 'no').lower() in ('yes', 'true', '1')

        if add_page_desc and page_image_path and page_image_path.exists():
            page_desc_start = time.time()
            logger.info(f"📄 페이지 {page_num} 전체 시각적 설명 생성 중...")
            page_visual_description = generate_page_visual_description(
                page_image_path,
                self.ollama_proxy
            )
            page_desc_elapsed = time.time() - page_desc_start
            logger.info(f"⏱️  페이지 전체 설명 생성 시간: {page_desc_elapsed:.2f}초")

            # 메타데이터에 페이지 설명 추가
            if page_visual_description:
                metadata['page_visual_description'] = page_visual_description
                # 🔒 Thread-Safe 메타데이터 파일 업데이트
                if metadata_file.exists():
                    with self._file_write_lock:
                        with open(metadata_file, 'w', encoding='utf-8') as f:
                            json.dump(metadata, f, ensure_ascii=False, indent=2)
        elif not add_page_desc:
            logger.info(f"📄 페이지 {page_num}: RAG_ADD_PAGE_DESCRIPTION=no로 페이지 설명 생성 건너뜀")

        return {
            'page_num': page_num,
            'markdown': markdown,
            'html': html,
            'structured_data': structured_data,
            'metadata': metadata,
            'images': images,
            'page_visual_description': page_visual_description,
            'error': None
        }

    def _pdf_to_images(self, pdf_path: Path) -> List[Path]:
        """
        PDF를 페이지별 이미지로 변환
        우선순위: pypdfium2 > PyMuPDF > pdf2image

        Returns:
            List[Path]: 페이지 이미지 경로 리스트
        """
        if HAS_PDFIUM:
            logger.info("📄 pypdfium2를 사용하여 PDF를 이미지로 변환 (폼빌더 방식)")
            return self._pdf_to_images_pdfium(pdf_path)
        elif HAS_PYMUPDF:
            logger.info("📄 PyMuPDF를 사용하여 PDF를 이미지로 변환")
            return self._pdf_to_images_pymupdf(pdf_path)
        elif HAS_PDF2IMAGE:
            logger.info("📄 pdf2image를 사용하여 PDF를 이미지로 변환")
            return self._pdf_to_images_pdf2image(pdf_path)
        else:
            raise ImportError(
                "PDF 처리를 위해 pypdfium2, PyMuPDF 또는 pdf2image가 필요합니다.\n"
                "권장 설치: pip install pypdfium2"
            )

    def _pdf_to_images_pdfium(self, pdf_path: Path) -> List[Path]:
        """
        pypdfium2로 PDF를 이미지로 변환 (폼빌더와 동일한 방식)
        주석/폼 필드를 평탄화하여 전체 페이지 렌더링
        """
        if not HAS_PDFIUM:
            raise ImportError("pypdfium2가 설치되지 않음")

        # Chandra 설정값 (폼빌더와 동일)
        MIN_PDF_IMAGE_DIM = 1024
        IMAGE_DPI = 192

        doc = pdfium.PdfDocument(str(pdf_path))
        doc.init_forms()

        page_images = []
        temp_dir = tempfile.mkdtemp()

        total_pages = len(doc)
        logger.info(f"📄 PDF 문서 열기 완료: {total_pages}개 페이지")

        for page_num in range(total_pages):
            try:
                page_obj = doc[page_num]

                # 페이지 크기에 따라 동적 DPI 계산 (폼빌더 방식)
                min_page_dim = min(page_obj.get_width(), page_obj.get_height())
                scale_dpi = (MIN_PDF_IMAGE_DIM / min_page_dim) * 72
                scale_dpi = max(scale_dpi, IMAGE_DPI)  # 최소 192 DPI 보장

                logger.debug(f"  페이지 {page_num + 1}: 크기 {page_obj.get_width():.1f}x{page_obj.get_height():.1f}, DPI {scale_dpi:.1f}")

                # 주석/폼 필드 평탄화 (중요!)
                flatten_pdf_page(page_obj)

                # 페이지 재로드 (flatten 후)
                page_obj = doc[page_num]

                # 고해상도 렌더링
                pil_image = page_obj.render(scale=scale_dpi / 72).to_pil().convert("RGB")

                # 임시 파일로 저장
                image_path = Path(temp_dir) / f"page_{page_num + 1}.png"
                pil_image.save(str(image_path), 'PNG')
                page_images.append(image_path)

                logger.debug(f"  ✅ 페이지 {page_num + 1} 이미지 생성: {image_path.name} ({pil_image.width}x{pil_image.height})")

            except Exception as e:
                logger.error(f"  ❌ 페이지 {page_num + 1} 이미지 변환 실패: {e}")
                # 실패한 페이지는 건너뛰고 계속 진행
                continue

        doc.close()

        logger.info(f"✅ PDF → 이미지 변환 완료: {len(page_images)}/{total_pages}개 페이지")
        return page_images

    def _pdf_to_images_pymupdf(self, pdf_path: Path) -> List[Path]:
        """PyMuPDF로 PDF를 이미지로 변환"""
        import fitz

        doc = fitz.open(pdf_path)
        page_images = []

        temp_dir = tempfile.mkdtemp()

        for page_num in range(1, len(doc) + 1):
            page = doc[page_num - 1]

            # 고해상도로 렌더링 (DPI 300)
            mat = fitz.Matrix(300/72, 300/72)
            pix = page.get_pixmap(matrix=mat)

            # 임시 파일로 저장
            image_path = Path(temp_dir) / f"page_{page_num}.png"
            pix.save(str(image_path))
            page_images.append(image_path)

        doc.close()
        return page_images

    def _pdf_to_images_pdf2image(self, pdf_path: Path) -> List[Path]:
        """pdf2image로 PDF를 이미지로 변환"""
        from pdf2image import convert_from_path

        temp_dir = tempfile.mkdtemp()

        # PDF를 이미지로 변환 (DPI 300)
        images = convert_from_path(str(pdf_path), dpi=300)

        page_images = []
        for page_num, image in enumerate(images, start=1):
            image_path = Path(temp_dir) / f"page_{page_num}.png"
            image.save(str(image_path), 'PNG')
            page_images.append(image_path)

        return page_images


def test_chandra_processor():
    """테스트 함수"""

    def progress(current, total, message):
        print(f"[{current}/{total}] {message}")

    processor = ChandraDocumentProcessor()

    # 테스트 PDF 경로 목록
    test_pdfs = [
        "/home/infoshare/invesume_app/ragdocs/admin/01. [SAT 완료] 원자로보호계통(SSPS) 개요.pdf",
        # "/path/to/another.pdf",
    ]

    for test_pdf in test_pdfs:
        if not os.path.exists(test_pdf):
            print(f"⚠️  테스트 파일 없음: {test_pdf}")
            continue

        print(f"\n🧪 테스트 시작: {test_pdf}")

        try:
            results = processor.process_document(
                test_pdf,
                progress_callback=progress,
                return_format='list'
            )

            print(f"✅ 처리 완료! 총 {len(results)}개 페이지")

            for result in results:
                page_num = result['page_num']
                markdown_len = len(result['markdown'])
                error = result.get('error')

                if error:
                    print(f"  ❌ 페이지 {page_num}: {error}")
                else:
                    print(f"  ✅ 페이지 {page_num}: {markdown_len} chars")

        except Exception as e:
            print(f"❌ 테스트 실패 ({test_pdf}): {e}")



if __name__ == "__main__":
    # 로깅 설정
    logging.basicConfig(
        level=logging.INFO,
        format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
    )

    test_chandra_processor()
