#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
PDF 페이지 썸네일 생성 유틸리티
지연 생성 + 캐싱 방식 (Lazy + Cache Hybrid Approach)

사용 예시:
    from thumbnail_generator import ThumbnailGenerator

    generator = ThumbnailGenerator('~/.airun/rag_docs')

    # 단일 페이지 썸네일 생성
    thumbnail_path = generator.generate_thumbnail(pdf_path, page_num=1)

    # 여러 페이지 일괄 생성
    results = generator.generate_thumbnails_batch(pdf_path, [1, 3, 5])

    # 썸네일 URL 생성
    url = generator.get_thumbnail_url('admin', 'document.pdf', 1)
"""

import os
import io
import hashlib
from pathlib import Path
from typing import Optional, List, Dict, Tuple
from PIL import Image
import logging
import time
import tempfile
import subprocess
import shutil

# PyMuPDF 체크
try:
    import fitz  # PyMuPDF
    HAS_PYMUPDF = True
except ImportError:
    HAS_PYMUPDF = False
    print("⚠️ PyMuPDF가 설치되지 않았습니다. pip install PyMuPDF")

logger = logging.getLogger(__name__)

# 지원되는 문서 형식
SUPPORTED_DOCUMENT_FORMATS = {
    '.pdf': 'PDF',
    '.hwp': 'HWP',
    '.hwpx': 'HWPX',
    '.doc': 'DOC',
    '.docx': 'DOCX',
    '.ppt': 'PPT',
    '.pptx': 'PPTX',
    '.xls': 'XLS',
    '.xlsx': 'XLSX',
    '.odt': 'ODT',
    '.ods': 'ODS',
    '.odp': 'ODP'
}


class ThumbnailGenerator:
    """
    PDF 페이지 썸네일 생성 및 캐싱

    Features:
        - 지연 생성: 실제 요청 시점에 썸네일 생성
        - 캐싱: 한 번 생성된 썸네일 재사용
        - WebP 포맷: 높은 압축률로 저장 공간 절약
        - 자동 정리: 오래된 썸네일 자동 삭제 기능
    """

    def __init__(
        self,
        rag_docs_path: str,
        thumbnail_size: Tuple[int, int] = (800, 1200),
        cache_subdir: str = "thumbnails",
        webp_quality: int = 95
    ):
        """
        Args:
            rag_docs_path: RAG 문서 저장 경로 (예: ~/.airun/rag_docs)
            thumbnail_size: 썸네일 크기 (너비, 높이) - 기본 800x1200px
            cache_subdir: 썸네일 캐시 서브디렉토리명
            webp_quality: WebP 압축 품질 (1-100, 높을수록 고품질)
        """
        self.rag_docs_path = Path(os.path.expanduser(rag_docs_path))
        self.thumbnail_size = thumbnail_size
        self.cache_subdir = cache_subdir
        self.webp_quality = webp_quality
        self._temp_converted_files = []  # 임시 변환된 PDF 파일 추적

        logger.debug(f"ThumbnailGenerator 초기화: {self.rag_docs_path}")

    def _convert_to_pdf(self, file_path: Path, timeout: int = 120) -> Optional[Path]:
        """
        문서를 PDF로 변환 (LibreOffice 사용)

        Args:
            file_path: 원본 문서 파일 경로
            timeout: 변환 타임아웃 (초)

        Returns:
            Path: 변환된 PDF 파일 경로 또는 None (실패 시)
        """
        if not file_path.exists():
            logger.error(f"❌ 파일을 찾을 수 없음: {file_path}")
            return None

        file_ext = file_path.suffix.lower()

        # 이미 PDF인 경우
        if file_ext == '.pdf':
            return file_path

        # 지원되는 형식인지 확인
        if file_ext not in SUPPORTED_DOCUMENT_FORMATS:
            logger.error(f"❌ 지원되지 않는 파일 형식: {file_ext}")
            return None

        # 문서 유형에 따른 PDF 필터 선택
        # Writer 문서: DOC, DOCX, HWP, HWPX, ODT
        # Calc 문서: XLS, XLSX, ODS
        # Impress 문서: PPT, PPTX, ODP
        if file_ext in ['.xls', '.xlsx', '.ods']:
            pdf_filter = 'pdf:calc_pdf_Export'
            doc_type = 'Calc'
        elif file_ext in ['.ppt', '.pptx', '.odp']:
            pdf_filter = 'pdf:impress_pdf_Export'
            doc_type = 'Impress'
        else:
            # Writer 문서 (HWP, HWPX, DOC, DOCX, ODT 등)
            pdf_filter = 'pdf:writer_pdf_Export'
            doc_type = 'Writer'

        logger.info(f"📄 문서를 PDF로 변환 중: {file_path.name} ({SUPPORTED_DOCUMENT_FORMATS[file_ext]} → {doc_type})")

        # 임시 디렉토리 생성
        temp_dir = tempfile.mkdtemp(prefix='thumbnail_convert_')
        output_dir = Path(temp_dir)

        try:
            # LibreOffice 명령어 구성
            cmd = [
                'libreoffice',
                '--headless',
                '--invisible',
                '--nodefault',
                '--nolockcheck',
                '--nologo',
                '--norestore',
                '--convert-to', pdf_filter,
                '--outdir', str(output_dir),
                str(file_path)
            ]

            logger.debug(f"LibreOffice 명령: {' '.join(cmd)}")

            # 변환 실행
            result = subprocess.run(
                cmd,
                timeout=timeout,
                capture_output=True,
                text=True,
                env={**os.environ, 'HOME': os.path.expanduser('~')}
            )

            # 변환된 PDF 찾기
            pdf_name = file_path.stem + '.pdf'
            pdf_path = output_dir / pdf_name

            if pdf_path.exists() and pdf_path.stat().st_size > 0:
                logger.info(f"✅ PDF 변환 성공: {pdf_path} ({pdf_path.stat().st_size / 1024:.1f}KB)")
                self._temp_converted_files.append(pdf_path)  # 추적 목록에 추가
                return pdf_path
            else:
                logger.error(f"❌ PDF 변환 실패: 출력 파일 없음")
                logger.error(f"stdout: {result.stdout}")
                logger.error(f"stderr: {result.stderr}")
                return None

        except subprocess.TimeoutExpired:
            logger.error(f"❌ PDF 변환 타임아웃 ({timeout}초)")
            return None
        except Exception as e:
            logger.error(f"❌ PDF 변환 중 오류: {e}")
            return None

    def _cleanup_temp_files(self):
        """임시로 변환된 PDF 파일들 정리"""
        for temp_file in self._temp_converted_files:
            try:
                if temp_file.exists():
                    # 임시 디렉토리 전체 삭제
                    temp_dir = temp_file.parent
                    shutil.rmtree(temp_dir, ignore_errors=True)
                    logger.debug(f"🗑️ 임시 파일 정리: {temp_file}")
            except Exception as e:
                logger.debug(f"⚠️ 임시 파일 정리 실패: {e}")

        self._temp_converted_files.clear()

    def get_thumbnail_cache_path(self, pdf_path: Path, page_num: int) -> Path:
        """
        썸네일 캐시 저장 경로 생성

        Args:
            pdf_path: PDF 파일 경로
            page_num: 페이지 번호 (1부터 시작)

        Returns:
            Path: 썸네일 파일 경로
            예: ~/.airun/rag_docs/admin/document.pdf/.extracts/document/thumbnails/page_1.webp
        """
        doc_name = pdf_path.stem
        extract_dir = pdf_path.parent / '.extracts' / doc_name / self.cache_subdir
        extract_dir.mkdir(parents=True, exist_ok=True)

        return extract_dir / f"page_{page_num}.webp"

    def is_thumbnail_cached(self, pdf_path: Path, page_num: int) -> bool:
        """
        썸네일이 캐시에 존재하는지 확인

        Args:
            pdf_path: PDF 파일 경로
            page_num: 페이지 번호

        Returns:
            bool: 캐시 존재 여부
        """
        thumbnail_path = self.get_thumbnail_cache_path(pdf_path, page_num)
        return thumbnail_path.exists()

    def generate_thumbnail(
        self,
        pdf_path: Path,
        page_num: int,
        force_regenerate: bool = False
    ) -> Optional[Path]:
        """
        페이지 썸네일 생성 (캐시 우선)

        프로세스:
            0. PDF가 아닌 경우 → PDF로 변환
            1. 캐시 확인 → 있으면 즉시 반환
            2. 캐시 없음 → PDF에서 페이지 추출
            3. 썸네일 크기로 리사이즈
            4. WebP 형식으로 저장
            5. 캐시 경로 반환

        Args:
            pdf_path: 문서 파일 경로 (PDF 또는 HWP, DOC, DOCX, PPT, PPTX 등)
            page_num: 페이지 번호 (1부터 시작)
            force_regenerate: 캐시 무시하고 재생성 여부

        Returns:
            Path: 썸네일 파일 경로 또는 None (실패 시)
        """
        if not HAS_PYMUPDF:
            logger.error("❌ PyMuPDF가 설치되지 않음. pip install PyMuPDF")
            return None

        if not pdf_path.exists():
            logger.error(f"❌ 파일을 찾을 수 없음: {pdf_path}")
            return None

        # 원본 파일 경로 저장 (캐시 키로 사용)
        original_path = pdf_path

        # PDF가 아닌 경우 PDF로 변환
        file_ext = pdf_path.suffix.lower()
        if file_ext != '.pdf':
            logger.info(f"📄 {SUPPORTED_DOCUMENT_FORMATS.get(file_ext, file_ext)} 문서를 PDF로 변환합니다...")
            converted_pdf = self._convert_to_pdf(pdf_path)

            if not converted_pdf:
                logger.error(f"❌ PDF 변환 실패: {pdf_path.name}")
                return None

            pdf_path = converted_pdf  # 변환된 PDF로 경로 변경
            logger.info(f"✅ PDF 변환 완료: {pdf_path.name}")

        # 캐시 확인 (원본 파일 경로 기준)
        thumbnail_path = self.get_thumbnail_cache_path(original_path, page_num)

        if not force_regenerate and thumbnail_path.exists():
            logger.debug(f"✅ 썸네일 캐시 사용: {thumbnail_path}")
            self._cleanup_temp_files()  # 임시 파일 정리
            return thumbnail_path

        # 썸네일 생성
        start_time = time.time()
        try:
            logger.info(f"🖼️ 썸네일 생성 시작: {original_path.name} 페이지 {page_num}")

            doc = fitz.open(str(pdf_path))

            if page_num < 1 or page_num > len(doc):
                logger.error(f"❌ 유효하지 않은 페이지 번호: {page_num} (총 {len(doc)} 페이지)")
                doc.close()
                return None

            page = doc[page_num - 1]  # 0-based index

            # 고해상도 렌더링을 위해 zoom 계산
            # 목표 너비 기준으로 zoom 설정 (2배 크기로 렌더링 후 축소하여 선명도 향상)
            target_width = self.thumbnail_size[0]
            zoom = (target_width * 2) / page.rect.width  # 2배 해상도로 렌더링
            mat = fitz.Matrix(zoom, zoom)
            pix = page.get_pixmap(matrix=mat, alpha=False)

            # PIL Image로 변환
            img_data = pix.tobytes("png")
            img = Image.open(io.BytesIO(img_data))

            # 원본 비율을 유지하면서 최대 너비에 맞춰 리사이즈
            original_width, original_height = img.size
            aspect_ratio = original_height / original_width
            new_width = min(original_width, target_width)
            new_height = int(new_width * aspect_ratio)

            # 고품질 리샘플링으로 리사이즈
            img = img.resize((new_width, new_height), Image.Resampling.LANCZOS)

            # WebP 형식으로 저장 (고품질)
            img.save(thumbnail_path, "WEBP", quality=self.webp_quality, method=6)

            doc.close()

            file_size_kb = os.path.getsize(thumbnail_path) / 1024
            elapsed = time.time() - start_time

            logger.info(
                f"✅ 썸네일 생성 완료: {thumbnail_path.name} "
                f"크기: {new_width}×{new_height}px, {file_size_kb:.1f}KB, {elapsed:.2f}초"
            )

            # 임시 변환 파일 정리
            self._cleanup_temp_files()

            return thumbnail_path

        except Exception as e:
            logger.error(f"❌ 썸네일 생성 실패: {e}")
            self._cleanup_temp_files()  # 실패 시에도 임시 파일 정리
            return None

    def generate_thumbnails_batch(
        self,
        pdf_path: Path,
        page_nums: List[int],
        force_regenerate: bool = False
    ) -> Dict[int, Optional[Path]]:
        """
        여러 페이지 썸네일 일괄 생성

        Args:
            pdf_path: 문서 파일 경로 (PDF 또는 HWP, DOC, DOCX, PPT, PPTX 등)
            page_nums: 페이지 번호 리스트
            force_regenerate: 캐시 무시하고 재생성 여부

        Returns:
            dict: {페이지번호: 썸네일경로} 매핑
            예: {1: Path('/path/to/page_1.webp'), 3: Path('/path/to/page_3.webp')}

        Note:
            PDF가 아닌 문서는 자동으로 PDF로 변환 후 썸네일 생성
        """
        results = {}

        logger.info(f"📚 일괄 썸네일 생성: {pdf_path.name} - {len(page_nums)}개 페이지")
        start_time = time.time()

        cached_count = 0
        generated_count = 0

        for page_num in page_nums:
            # 캐시 먼저 확인
            thumbnail_path = self.get_thumbnail_cache_path(pdf_path, page_num)

            if not force_regenerate and thumbnail_path.exists():
                results[page_num] = thumbnail_path
                cached_count += 1
                logger.debug(f"  ✅ 캐시 사용: 페이지 {page_num}")
            else:
                # 캐시 없으면 생성
                thumbnail_path = self.generate_thumbnail(pdf_path, page_num, force_regenerate)
                results[page_num] = thumbnail_path
                if thumbnail_path:
                    generated_count += 1

        elapsed = time.time() - start_time
        logger.info(
            f"✅ 일괄 생성 완료: {len(page_nums)}개 "
            f"(캐시: {cached_count}, 생성: {generated_count}, {elapsed:.2f}초)"
        )

        return results

    def get_thumbnail_url(
        self,
        user_id: str,
        filename: str,
        page_num: int
    ) -> str:
        """
        썸네일 API URL 생성

        Args:
            user_id: 사용자 ID
            filename: 문서 파일명
            page_num: 페이지 번호

        Returns:
            str: 썸네일 API URL
            예: /api/v1/rag/thumbnail?user_id=admin&filename=document.pdf&page=1
        """
        from urllib.parse import urlencode

        params = {
            'user_id': user_id,
            'filename': filename,
            'page': page_num
        }

        return f"/api/v1/rag/thumbnail?{urlencode(params)}"

    def cleanup_old_thumbnails(self, days: int = 30) -> int:
        """
        오래된 썸네일 정리 (사용되지 않은 지 N일 경과)

        Args:
            days: 삭제 기준 일수 (기본 30일)

        Returns:
            int: 삭제된 썸네일 개수
        """
        import time

        current_time = time.time()
        cutoff_time = current_time - (days * 86400)  # 초 단위

        deleted_count = 0
        freed_space = 0

        logger.info(f"🗑️ {days}일 이상 사용되지 않은 썸네일 정리 시작...")

        for thumbnail_dir in self.rag_docs_path.rglob(f"*/{self.cache_subdir}"):
            for thumbnail_file in thumbnail_dir.glob("*.webp"):
                # 최근 접근 시간 (atime) 확인
                if thumbnail_file.stat().st_atime < cutoff_time:
                    file_size = thumbnail_file.stat().st_size
                    thumbnail_file.unlink()
                    deleted_count += 1
                    freed_space += file_size
                    logger.debug(f"  🗑️ 삭제: {thumbnail_file.name}")

        freed_space_mb = freed_space / (1024 * 1024)
        logger.info(
            f"✅ 썸네일 정리 완료: {deleted_count}개 삭제, "
            f"{freed_space_mb:.2f}MB 확보"
        )

        return deleted_count

    def get_cache_statistics(self) -> Dict[str, any]:
        """
        썸네일 캐시 통계 조회

        Returns:
            dict: 캐시 통계 정보
            {
                'total_thumbnails': 총 썸네일 개수,
                'total_size_mb': 총 크기 (MB),
                'documents': 문서별 썸네일 개수
            }
        """
        total_count = 0
        total_size = 0
        documents = {}

        for thumbnail_dir in self.rag_docs_path.rglob(f"*/{self.cache_subdir}"):
            doc_name = thumbnail_dir.parent.name
            thumbnail_files = list(thumbnail_dir.glob("*.webp"))
            doc_count = len(thumbnail_files)
            doc_size = sum(f.stat().st_size for f in thumbnail_files)

            total_count += doc_count
            total_size += doc_size

            documents[doc_name] = {
                'count': doc_count,
                'size_kb': doc_size / 1024
            }

        return {
            'total_thumbnails': total_count,
            'total_size_mb': total_size / (1024 * 1024),
            'documents': documents
        }


# 테스트 코드
if __name__ == "__main__":
    # 로깅 설정
    logging.basicConfig(
        level=logging.INFO,
        format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
    )

    # 테스트
    rag_docs_path = os.path.expanduser('~/.airun/rag_docs')
    generator = ThumbnailGenerator(rag_docs_path)

    # 캐시 통계 출력
    stats = generator.get_cache_statistics()
    print("\n📊 썸네일 캐시 통계:")
    print(f"  총 썸네일: {stats['total_thumbnails']}개")
    print(f"  총 크기: {stats['total_size_mb']:.2f}MB")
    print(f"  문서 수: {len(stats['documents'])}개")

    # 샘플 PDF 테스트 (실제 파일 경로로 변경 필요)
    # sample_pdf = Path(rag_docs_path) / 'admin' / 'sample.pdf'
    # if sample_pdf.exists():
    #     print(f"\n🧪 테스트: {sample_pdf.name}")
    #
    #     # 단일 페이지 생성
    #     thumbnail_path = generator.generate_thumbnail(sample_pdf, 1)
    #     print(f"  단일 페이지: {thumbnail_path}")
    #
    #     # 여러 페이지 일괄 생성
    #     results = generator.generate_thumbnails_batch(sample_pdf, [1, 2, 3])
    #     print(f"  일괄 생성: {len(results)}개")
