/**
 * AI 프로바이더 및 모델 설정
 * 모든 AI 프로바이더 관련 설정을 중앙 집중화하여 관리
 */

// Llama 계열 모델 목록
export const LLAMA_FAMILY = [
    "llama-3.1-405b-reasoning",
    "llama-3.1-70b-versatile",
    "llama3-8b-8192",
    "llama3-70b-8192",
    "llama3:latest",
    "llama3:8b-instruct-q8_0",
    "llama3:70b",
];

async function getVLLMModels(server) {
    try {
        const response = await fetch(`${server || 'http://localhost:11400'}/v1/models`);
        if (!response.ok) {
            throw new Error(`Failed to fetch models: ${response.statusText}`);
        }
        const data = await response.json();
        return data.data.map(model => model.id);
    } catch (error) {
        console.error('Error fetching vLLM models:', error);
        return [];
    }
}

export const AI_PROVIDERS = {
    openai: {
        displayName: 'OpenAI',
        keyName: 'OPENAI_API_KEY',
        modelVar: 'OPENAI_MODEL',
        defaultModel: 'gpt-4o-mini',
        apiKeyDisplayName: 'OpenAI API Key',
        models: [
            // GPT-5 시리즈
            'gpt-5',
            'gpt-5-mini',
            'gpt-5-nano',
            // GPT-4.1 시리즈
            'gpt-4.1',
            'gpt-4.1-mini',
            'gpt-4.1-nano',
            // GPT-4o 시리즈
            'gpt-4o',
            'gpt-4o-mini',
            // 기존 모델
            'gpt-3.5-turbo'
        ],
        apiKeyInstructions: [
            'Go to https://platform.openai.com and sign in',
            'Click on your profile icon in the top right and select "View API keys"',
            'Click "Create new secret key" to generate a new API key'
        ],
        apiEndpoint: 'https://api.openai.com/v1/chat/completions',
        headers: (apiKey) => ({
            'Authorization': `Bearer ${apiKey}`,
            'Content-Type': 'application/json'
        })
    },
    anthropic: {
        displayName: 'Anthropic',
        keyName: 'ANTHROPIC_API_KEY',
        modelVar: 'ANTHROPIC_MODEL',
        defaultModel: 'claude-3-5-sonnet-20241022',
        apiKeyDisplayName: 'Anthropic API Key',
        models: [
            // Claude 4 시리즈 (최신)
            'claude-sonnet-4-20250514',
            'claude-opus-4-20250514',
            // Claude 3.7 시리즈
            'claude-3-7-sonnet-20250219',
            'claude-3-7-sonnet-latest',
            // Claude 3.5 시리즈 (최신)
            'claude-3-5-sonnet-20241022',
            'claude-3-5-haiku-20241022',
            // Claude 3 기본 시리즈
            'claude-3-sonnet-20240229',
            'claude-3-opus-20240229',
            'claude-3-haiku-20240307'
        ],
        apiKeyInstructions: [
            'Go to https://console.anthropic.com and sign in',
            'Generate a new API key in the "API Keys" section'
        ],
        apiEndpoint: 'https://api.anthropic.com/v1/messages',
        apiVersion: '2023-06-01',
        headers: (apiKey) => ({
            'x-api-key': apiKey,
            'anthropic-version': '2023-06-01',
            'content-type': 'application/json'
        }),
        maxTokens: 8192
    },
    gemini: {
        displayName: 'Gemini',
        keyName: 'GEMINI_API_KEY',
        modelVar: 'GEMINI_MODEL',
        defaultModel: 'gemini-2.5-flash',
        apiKeyDisplayName: 'Google API Key',
        models: [
            // Gemini 2.5 시리즈 (최신)
            'gemini-2.5-flash',
            'gemini-2.5-pro',
            'gemini-2.5-pro-preview-05-06',
            // Gemini 2.0 시리즈
            'gemini-2.0-flash',
            'gemini-2.0-flash-thinking-exp',
            'gemini-2.0-pro-exp',
            // Gemini 1.5 시리즈
            'gemini-1.5-flash',
            'gemini-1.5-flash-8b',
            'gemini-1.5-pro',
            // Gemma 3 시리즈
            'gemma-3-27b-it',
            'gemma-3-12b-it',
            'gemma-3-4b-it',
            'gemma-3-1b-it'
        ],
        apiKeyInstructions: [
            'Go to https://makersuite.google.com/app/apikeys',
            'Click "Create API Key" to generate a new key'
        ]
    },
    groq: {
        displayName: 'Groq',
        keyName: 'GROQ_API_KEY',
        modelVar: 'GROQ_MODEL',
        defaultModel: 'meta-llama/llama-4-maverick-17b-128e-instruct',
        apiKeyDisplayName: 'Groq API Key',
        models: [
            // Llama 4 시리즈 (최신)
            'meta-llama/llama-4-maverick-17b-128e-instruct',
            'meta-llama/llama-4-scout-17b-16e-instruct',
            // Llama 3.x 시리즈
            'llama-3.3-70b-versatile',
            'llama-3.1-8b-instant',
            'llama3-70b-8192',
            'llama3-8b-8192',
            // DeepSeek R1
            'deepseek-r1-distill-llama-70b',
            // Qwen 시리즈
            'qwen-qwq-32b',
            'qwen/qwen3-32b',
            // Mistral 시리즈
            'mistral-saba-24b',
            // Gemma 시리즈
            'gemma2-9b-it',
            // 실험적 모델
            'compound-beta',
            'compound-beta-mini',
            'allam-2-7b'
        ],
        apiKeyInstructions: [
            'Go to https://console.groq.com and sign in',
            'Generate a new API key in the "API Keys" section'
        ],
        apiEndpoint: 'https://api.groq.com/openai/v1/chat/completions',
        headers: (apiKey) => ({
            'Authorization': `Bearer ${apiKey}`,
            'Content-Type': 'application/json'
        })
    },
    ollama: {
        displayName: 'Ollama',
        modelVar: 'OLLAMA_MODEL',
        defaultModel: 'airun-chat',
        models: [],
        isDynamic: true,
        defaultEndpoint: 'http://127.0.0.1:11434',
        apiEndpoint: (server) => `${server || 'http://127.0.0.1:11434'}/api/chat`,
        headers: {
            'Content-Type': 'application/json'
        }
    },
    vllm: {
        displayName: 'vLLM',
        modelVar: 'VLLM_MODEL',
        defaultModel: 'unsloth/gemma-3-12b-it-unsloth-bnb-4bit',
        models: [],
        isDynamic: true,
        defaultEndpoint: 'http://localhost:11400',
        apiEndpoint: (server) => `${server || 'http://localhost:11400'}/v1/chat/completions`,
        headers: {
            'Content-Type': 'application/json'
        },
        requiresServer: true,
        serverVar: 'VLLM_SERVER',
        getModels: getVLLMModels
    }
};

/**
 * 모델별 컨텍스트 윈도우 크기 설정
 */
export const MODEL_CONTEXT_WINDOWS = {
    // OpenAI 모델
    // GPT-4.1 시리즈
    'gpt-4.1': 128000,
    'gpt-4.1-mini': 128000,
    'gpt-4.1-nano': 128000,
    // GPT-5 시리즈
    'gpt-5': 400000,
    'gpt-5-mini': 400000,
    'gpt-5-nano': 400000,
    // GPT-4o 시리즈
    'gpt-4o': 128000,
    'gpt-4o-mini': 128000,
    'gpt-3.5-turbo': 16385,
    
    // Anthropic 모델
    // Claude 4 시리즈
    'claude-opus-4-20250514': 200000,
    'claude-sonnet-4-20250514': 200000,
    // Claude 3.7 시리즈
    'claude-3-7-sonnet-20250219': 200000,
    'claude-3-7-sonnet-latest': 200000,
    // Claude 3.5 시리즈
    'claude-3-5-sonnet-20241022': 200000,
    'claude-3-5-haiku-20241022': 200000,
    // Claude 3 기본 시리즈
    'claude-3-opus-20240229': 200000,
    'claude-3-sonnet-20240229': 200000,
    'claude-3-haiku-20240307': 200000,
    
    // Gemini 모델
    // Gemini 2.5 시리즈
    'gemini-2.5-flash': 1048576,
    'gemini-2.5-pro': 2097152, // 2M context
    'gemini-2.5-pro-preview-05-06': 1048576,
    // Gemini 2.0 시리즈
    'gemini-2.0-flash': 1048576,
    'gemini-2.0-flash-thinking-exp': 1048576,
    'gemini-2.0-pro-exp': 2097152, // 2M context
    // Gemini 1.5 시리즈
    'gemini-1.5-flash': 1048576,
    'gemini-1.5-flash-8b': 1048576,
    'gemini-1.5-pro': 1048576,
    // Gemma 3 시리즈
    'gemma-3-27b-it': 1048576,
    'gemma-3-12b-it': 1048576,
    'gemma-3-4b-it': 1048576,
    'gemma-3-1b-it': 1048576,
    
    // Groq 모델
    'mixtral-8x7b-32768': 32768,
    // Llama 4 시리즈
    'meta-llama/llama-4-maverick-17b-128e-instruct': 131072, // 128k context
    'meta-llama/llama-4-scout-17b-16e-instruct': 16384,     // 16k context  
    // Llama 3.x 시리즈
    'llama-3.3-70b-versatile': 32768,
    'llama-3.1-8b-instant': 8192,
    'llama3-70b-8192': 8192,
    'llama3-8b-8192': 8192,
    // DeepSeek R1
    'deepseek-r1-distill-llama-70b': 131072, // 128k context
    // Qwen 시리즈
    'qwen-qwq-32b': 32768,
    'qwen/qwen3-32b': 32768,
    // Mistral 시리즈
    'mistral-saba-24b': 32768,
    // Gemma 시리즈
    'gemma2-9b-it': 8192,
    // 실험적 모델
    'compound-beta': 32768,
    'compound-beta-mini': 8192,
    'allam-2-7b': 8192,

    // vLLM 모델
    'unsloth/gemma-3-12b-it-unsloth-bnb-4bit': 8192,
    'llama2': 8192,
    'llama3-8b-8192': 8192,
    'llama3-70b-8192': 8192,
    'llama3:latest': 8192,
    'llama3:8b-instruct-q8_0': 8192,
    'llama3:70b': 8192,
    'airun-chat': 128000,
    'airun-think': 8192,
    'airun-vision': 8192,
    'gemma3:27b': 128000,
    'gemma3:12b': 128000,
    'devstral': 8192,
    'mistral-small3.1': 8192,
    'hyperclova-1.5b': 8192,
    'phi4': 16384,
    'phi4:14b': 16384,
    'phi4:latest': 16384
};

