/**
 * 견적서 API 라우트
 *
 * 견적서 CRUD 및 AI 생성 기능을 제공합니다.
 */

import express from 'express';
import { debugLog, logger } from '../utils/logger.js';
import quotationAgent from '../services/quotation/quotation-agent.js';
import dbManager from '../services/database/index.js';

const router = express.Router();

// 데이터베이스 연결 가져오기
const getDBPool = async () => {
    return dbManager;
};

// 에러 핸들러 헬퍼
const asyncHandler = (fn) => (req, res, next) => {
    Promise.resolve(fn(req, res, next)).catch(next);
};

// 응답 포맷 헬퍼
const formatResponse = (success, data = null, error = null) => ({
    success,
    data,
    error: error ? { message: error.message || error, code: error.code } : null,
    timestamp: new Date().toISOString()
});

// 견적번호 생성 헬퍼
const generateQuotationNumber = () => {
    const date = new Date();
    const year = date.getFullYear();
    const month = String(date.getMonth() + 1).padStart(2, '0');
    const day = String(date.getDate()).padStart(2, '0');
    const random = Math.random().toString(36).substring(2, 6).toUpperCase();
    return `QT-${year}${month}${day}-${random}`;
};

/**
 * AI 견적서 생성
 * POST /api/v1/quotations/generate
 *
 * 자연어 요청으로 견적서를 자동 생성합니다.
 */
router.post('/generate', asyncHandler(async (req, res) => {
    const { prompt, options = {} } = req.body;
    const username = req.user?.id || req.body.username || 'admin';

    if (!prompt) {
        return res.status(400).json(formatResponse(false, null, { message: '견적서 생성 요청이 필요합니다.' }));
    }

    logger.info(`[QuotationAPI] AI 견적서 생성 요청: ${prompt.substring(0, 100)}...`);

    try {
        const result = await quotationAgent.generateQuotation(prompt, {
            username,
            ...options
        });

        if (result.success) {
            res.json(formatResponse(true, result.data));
        } else {
            res.status(400).json(formatResponse(false, null, { message: result.error || result.message }));
        }
    } catch (error) {
        logger.error('[QuotationAPI] 견적서 생성 실패:', error);
        res.status(500).json(formatResponse(false, null, { message: error.message }));
    }
}));

/**
 * 견적서 목록 조회
 * GET /api/v1/quotations
 */
router.get('/', asyncHandler(async (req, res) => {
    const { status, clientCompany, page = 1, limit = 20 } = req.query;
    const username = req.user?.id || req.query.username || 'admin';

    try {
        const db = await getDBPool();

        let sql = `
            SELECT id, quotation_number, client_company, client_name, client_email,
                   client_phone, project_name, supply_amount, vat_amount, total_amount,
                   validity_period, notes, status, created_at, updated_at, user_id
            FROM quotations
            WHERE user_id = $1
        `;
        const params = [username];
        let paramIndex = 2;

        if (status) {
            sql += ` AND status = $${paramIndex}`;
            params.push(status);
            paramIndex++;
        }

        if (clientCompany) {
            sql += ` AND client_company ILIKE $${paramIndex}`;
            params.push(`%${clientCompany}%`);
            paramIndex++;
        }

        const offset = (parseInt(page) - 1) * parseInt(limit);
        sql += ` ORDER BY created_at DESC LIMIT $${paramIndex} OFFSET $${paramIndex + 1}`;
        params.push(parseInt(limit), offset);

        const result = await db.query(sql, params);

        // 총 개수 조회
        let countSql = 'SELECT COUNT(*) FROM quotations WHERE user_id = $1';
        const countParams = [username];
        let countParamIndex = 2;

        if (status) {
            countSql += ` AND status = $${countParamIndex}`;
            countParams.push(status);
            countParamIndex++;
        }
        if (clientCompany) {
            countSql += ` AND client_company ILIKE $${countParamIndex}`;
            countParams.push(`%${clientCompany}%`);
        }

        const countResult = await db.query(countSql, countParams);
        const totalCount = parseInt(countResult.rows[0].count);

        res.json(formatResponse(true, {
            quotations: result.rows,
            pagination: {
                page: parseInt(page),
                limit: parseInt(limit),
                totalCount,
                totalPages: Math.ceil(totalCount / parseInt(limit))
            }
        }));
    } catch (error) {
        logger.error('[QuotationAPI] 견적서 목록 조회 실패:', error);
        res.status(500).json(formatResponse(false, null, { message: error.message }));
    }
}));

/**
 * 견적서 상세 조회
 * GET /api/v1/quotations/:id
 *
 * 주의: 이 라우트는 /company/profile, /price-catalog 등 구체적인 경로보다
 * 뒤에 위치해야 합니다. Express는 라우트를 순서대로 매칭하기 때문입니다.
 * 하지만 현재 파일 구조상 여기에 위치하므로, ID가 숫자인지 검증합니다.
 */
router.get('/:id', asyncHandler(async (req, res, next) => {
    const { id } = req.params;

    // ID가 숫자가 아니면 다음 라우트로 패스 (company, price-catalog 등 처리)
    if (!/^\d+$/.test(id)) {
        return next('route');
    }

    const username = req.user?.id || req.query.username || 'admin';

    try {
        const db = await getDBPool();

        // 견적서 기본 정보 조회
        const quotationResult = await db.query(
            `SELECT * FROM quotations WHERE id = $1 AND user_id = $2`,
            [id, username]
        );

        if (quotationResult.rows.length === 0) {
            return res.status(404).json(formatResponse(false, null, { message: '견적서를 찾을 수 없습니다.' }));
        }

        // 견적서 항목 조회
        const itemsResult = await db.query(
            `SELECT * FROM quotation_items WHERE quotation_id = $1 ORDER BY item_order`,
            [id]
        );

        res.json(formatResponse(true, {
            ...quotationResult.rows[0],
            items: itemsResult.rows
        }));
    } catch (error) {
        logger.error('[QuotationAPI] 견적서 상세 조회 실패:', error);
        res.status(500).json(formatResponse(false, null, { message: error.message }));
    }
}));

/**
 * 견적서 생성 (수동)
 * POST /api/v1/quotations
 */
router.post('/', asyncHandler(async (req, res) => {
    const {
        clientCompany, clientName, clientEmail, clientPhone,
        projectName, supplyAmount, vatAmount, totalAmount,
        validityPeriod, notes, items
    } = req.body;
    const username = req.user?.id || req.body.username || 'admin';

    if (!clientCompany || !supplyAmount) {
        return res.status(400).json(formatResponse(false, null, { message: '고객사와 공급가액은 필수입니다.' }));
    }

    try {
        const db = await getDBPool();

        const quotationNumber = generateQuotationNumber();

        // 견적서 생성
        const insertResult = await db.query(
            `INSERT INTO quotations (
                user_id, quotation_number, client_company, client_name,
                client_email, client_phone, project_name, supply_amount,
                vat_amount, total_amount, validity_period, notes, status
            ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, 'draft')
            RETURNING *`,
            [
                username, quotationNumber, clientCompany, clientName || null,
                clientEmail || null, clientPhone || null, projectName || null,
                supplyAmount, vatAmount || 0, totalAmount || supplyAmount,
                validityPeriod || 30, notes || null
            ]
        );

        const quotation = insertResult.rows[0];

        // 견적서 항목 생성
        if (items && items.length > 0) {
            for (let i = 0; i < items.length; i++) {
                const item = items[i];
                await db.query(
                    `INSERT INTO quotation_items (
                        quotation_id, item_order, item_name, specification,
                        unit, quantity, unit_price, amount, notes
                    ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9)`,
                    [
                        quotation.id, i + 1, item.itemName, item.specification || null,
                        item.unit || null, item.quantity || 1, item.unitPrice || 0,
                        item.amount || 0, item.notes || null
                    ]
                );
            }
        }

        // 생성된 견적서 조회
        const resultQuery = await db.query(
            'SELECT * FROM quotations WHERE id = $1',
            [quotation.id]
        );
        const itemsQuery = await db.query(
            'SELECT * FROM quotation_items WHERE quotation_id = $1 ORDER BY item_order',
            [quotation.id]
        );

        res.status(201).json(formatResponse(true, {
            ...resultQuery.rows[0],
            items: itemsQuery.rows
        }));
    } catch (error) {
        logger.error('[QuotationAPI] 견적서 생성 실패:', error);
        res.status(500).json(formatResponse(false, null, { message: error.message }));
    }
}));

/**
 * 견적서 수정
 * PUT /api/v1/quotations/:id
 */
router.put('/:id', asyncHandler(async (req, res) => {
    const { id } = req.params;
    const {
        clientCompany, clientName, clientEmail, clientPhone,
        projectName, supplyAmount, vatAmount, totalAmount,
        validityPeriod, notes, items
    } = req.body;
    const username = req.user?.id || req.body.username || 'admin';

    try {
        const db = await getDBPool();

        // 기존 견적서 확인
        const existingResult = await db.query(
            'SELECT * FROM quotations WHERE id = $1 AND user_id = $2',
            [id, username]
        );

        if (existingResult.rows.length === 0) {
            return res.status(404).json(formatResponse(false, null, { message: '견적서를 찾을 수 없습니다.' }));
        }

        // 견적서 수정
        await db.query(
            `UPDATE quotations SET
                client_company = COALESCE($1, client_company),
                client_name = COALESCE($2, client_name),
                client_email = COALESCE($3, client_email),
                client_phone = COALESCE($4, client_phone),
                project_name = COALESCE($5, project_name),
                supply_amount = COALESCE($6, supply_amount),
                vat_amount = COALESCE($7, vat_amount),
                total_amount = COALESCE($8, total_amount),
                validity_period = COALESCE($9, validity_period),
                notes = COALESCE($10, notes),
                updated_at = CURRENT_TIMESTAMP
            WHERE id = $11`,
            [
                clientCompany, clientName, clientEmail, clientPhone,
                projectName, supplyAmount, vatAmount, totalAmount,
                validityPeriod, notes, id
            ]
        );

        // 항목 수정 (기존 항목 삭제 후 새로 삽입)
        if (items && items.length > 0) {
            await db.query('DELETE FROM quotation_items WHERE quotation_id = $1', [id]);

            for (let i = 0; i < items.length; i++) {
                const item = items[i];
                await db.query(
                    `INSERT INTO quotation_items (
                        quotation_id, item_order, item_name, specification,
                        unit, quantity, unit_price, amount, notes
                    ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9)`,
                    [
                        id, i + 1, item.itemName, item.specification || null,
                        item.unit || null, item.quantity || 1, item.unitPrice || 0,
                        item.amount || 0, item.notes || null
                    ]
                );
            }
        }

        // 수정된 견적서 조회
        const resultQuery = await db.query(
            'SELECT * FROM quotations WHERE id = $1',
            [id]
        );
        const itemsQuery = await db.query(
            'SELECT * FROM quotation_items WHERE quotation_id = $1 ORDER BY item_order',
            [id]
        );

        res.json(formatResponse(true, {
            ...resultQuery.rows[0],
            items: itemsQuery.rows
        }));
    } catch (error) {
        logger.error('[QuotationAPI] 견적서 수정 실패:', error);
        res.status(500).json(formatResponse(false, null, { message: error.message }));
    }
}));

/**
 * 견적서 삭제
 * DELETE /api/v1/quotations/:id
 */
router.delete('/:id', asyncHandler(async (req, res) => {
    const { id } = req.params;
    const username = req.user?.id || req.query.username || 'admin';

    try {
        const db = await getDBPool();

        // 기존 견적서 확인
        const existingResult = await db.query(
            'SELECT * FROM quotations WHERE id = $1 AND user_id = $2',
            [id, username]
        );

        if (existingResult.rows.length === 0) {
            return res.status(404).json(formatResponse(false, null, { message: '견적서를 찾을 수 없습니다.' }));
        }

        // 견적서 삭제 (CASCADE로 항목도 삭제됨)
        await db.query('DELETE FROM quotations WHERE id = $1', [id]);

        res.json(formatResponse(true, { message: '견적서가 삭제되었습니다.' }));
    } catch (error) {
        logger.error('[QuotationAPI] 견적서 삭제 실패:', error);
        res.status(500).json(formatResponse(false, null, { message: error.message }));
    }
}));

/**
 * 견적서 상태 변경
 * PATCH /api/v1/quotations/:id/status
 */
router.patch('/:id/status', asyncHandler(async (req, res) => {
    const { id } = req.params;
    const { status } = req.body;
    const username = req.user?.id || req.body.username || 'admin';

    const validStatuses = ['draft', 'sent', 'accepted', 'rejected', 'expired'];
    if (!status || !validStatuses.includes(status)) {
        return res.status(400).json(formatResponse(false, null, {
            message: `유효하지 않은 상태입니다. 사용 가능: ${validStatuses.join(', ')}`
        }));
    }

    try {
        const db = await getDBPool();

        const result = await db.query(
            `UPDATE quotations SET status = $1, updated_at = CURRENT_TIMESTAMP
             WHERE id = $2 AND user_id = $3 RETURNING *`,
            [status, id, username]
        );

        if (result.rows.length === 0) {
            return res.status(404).json(formatResponse(false, null, { message: '견적서를 찾을 수 없습니다.' }));
        }

        res.json(formatResponse(true, result.rows[0]));
    } catch (error) {
        logger.error('[QuotationAPI] 견적서 상태 변경 실패:', error);
        res.status(500).json(formatResponse(false, null, { message: error.message }));
    }
}));

/**
 * 회사 프로필 조회
 * GET /api/v1/quotations/company/profile
 */
router.get('/company/profile', asyncHandler(async (req, res) => {
    const username = req.user?.id || req.query.username || 'admin';

    try {
        const db = await getDBPool();

        const result = await db.query(
            'SELECT * FROM company_profiles WHERE user_id = $1',
            [username]
        );

        if (result.rows.length === 0) {
            return res.json(formatResponse(true, null));
        }

        res.json(formatResponse(true, result.rows[0]));
    } catch (error) {
        logger.error('[QuotationAPI] 회사 프로필 조회 실패:', error);
        res.status(500).json(formatResponse(false, null, { message: error.message }));
    }
}));

/**
 * 회사 프로필 생성/수정
 * POST /api/v1/quotations/company/profile
 */
router.post('/company/profile', asyncHandler(async (req, res) => {
    const {
        companyName, businessNumber, ceoName, address,
        phone, fax, email, website,
        logoPath, logoUrl, sealPath, sealUrl, // 신구 필드 모두 지원
        bankName, bankAccount, bankHolder, bankInfo,
        defaultValidityDays, defaultPaymentTerms, defaultNotes
    } = req.body;
    const username = req.user?.id || req.body.username || 'admin';

    if (!companyName) {
        return res.status(400).json(formatResponse(false, null, { message: '회사명은 필수입니다.' }));
    }

    // 새 필드 우선, 구 필드 폴백
    const finalLogoPath = logoPath || logoUrl || null;
    const finalSealPath = sealPath || sealUrl || null;

    try {
        const db = await getDBPool();

        // 기존 프로필 확인
        const existingResult = await db.query(
            'SELECT * FROM company_profiles WHERE user_id = $1',
            [username]
        );

        if (existingResult.rows.length > 0) {
            // 수정
            const result = await db.query(
                `UPDATE company_profiles SET
                    company_name = $1, business_number = $2, ceo_name = $3,
                    address = $4, phone = $5, fax = $6, email = $7, website = $8,
                    logo_path = $9, seal_path = $10,
                    bank_name = $11, bank_account = $12, bank_holder = $13,
                    default_validity_days = $14, default_payment_terms = $15, default_notes = $16,
                    updated_at = CURRENT_TIMESTAMP
                WHERE user_id = $17 RETURNING *`,
                [
                    companyName, businessNumber || null, ceoName || null,
                    address || null, phone || null, fax || null, email || null, website || null,
                    finalLogoPath, finalSealPath,
                    bankName || null, bankAccount || null, bankHolder || null,
                    defaultValidityDays || 30, defaultPaymentTerms || null, defaultNotes || null,
                    username
                ]
            );
            res.json(formatResponse(true, result.rows[0]));
        } else {
            // 생성
            const result = await db.query(
                `INSERT INTO company_profiles (
                    user_id, company_name, business_number, ceo_name,
                    address, phone, fax, email, website,
                    logo_path, seal_path,
                    bank_name, bank_account, bank_holder,
                    default_validity_days, default_payment_terms, default_notes
                ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16, $17) RETURNING *`,
                [
                    username, companyName, businessNumber || null, ceoName || null,
                    address || null, phone || null, fax || null, email || null, website || null,
                    finalLogoPath, finalSealPath,
                    bankName || null, bankAccount || null, bankHolder || null,
                    defaultValidityDays || 30, defaultPaymentTerms || null, defaultNotes || null
                ]
            );
            res.status(201).json(formatResponse(true, result.rows[0]));
        }
    } catch (error) {
        logger.error('[QuotationAPI] 회사 프로필 저장 실패:', error);
        res.status(500).json(formatResponse(false, null, { message: error.message }));
    }
}));

/**
 * 단가표 목록 조회
 * GET /api/v1/quotations/price-catalog
 */
router.get('/price-catalog', asyncHandler(async (req, res) => {
    const { category, search, page = 1, limit = 50 } = req.query;
    const username = req.user?.id || req.query.username || 'admin';

    try {
        const db = await getDBPool();

        let sql = `
            SELECT id, category, item_name, unit, unit_price, description, is_active, created_at, updated_at
            FROM price_catalog
            WHERE user_id = $1 AND is_active = true
        `;
        const params = [username];
        let paramIndex = 2;

        if (category) {
            sql += ` AND category = $${paramIndex}`;
            params.push(category);
            paramIndex++;
        }

        if (search) {
            sql += ` AND (item_name ILIKE $${paramIndex} OR description ILIKE $${paramIndex})`;
            params.push(`%${search}%`);
            paramIndex++;
        }

        const offset = (parseInt(page) - 1) * parseInt(limit);
        sql += ` ORDER BY category, item_name LIMIT $${paramIndex} OFFSET $${paramIndex + 1}`;
        params.push(parseInt(limit), offset);

        const result = await db.query(sql, params);

        // 카테고리 목록 조회
        const categoriesResult = await db.query(
            'SELECT DISTINCT category FROM price_catalog WHERE user_id = $1 AND is_active = true ORDER BY category',
            [username]
        );

        res.json(formatResponse(true, {
            items: result.rows,
            categories: categoriesResult.rows.map(r => r.category)
        }));
    } catch (error) {
        logger.error('[QuotationAPI] 단가표 조회 실패:', error);
        res.status(500).json(formatResponse(false, null, { message: error.message }));
    }
}));

/**
 * 단가표 항목 추가
 * POST /api/v1/quotations/price-catalog
 */
router.post('/price-catalog', asyncHandler(async (req, res) => {
    const { category, itemName, unit, unitPrice, description } = req.body;
    const username = req.user?.id || req.body.username || 'admin';

    if (!category || !itemName || unitPrice === undefined) {
        return res.status(400).json(formatResponse(false, null, {
            message: '카테고리, 품목명, 단가는 필수입니다.'
        }));
    }

    try {
        const db = await getDBPool();

        const result = await db.query(
            `INSERT INTO price_catalog (user_id, category, item_name, unit, unit_price, description)
             VALUES ($1, $2, $3, $4, $5, $6) RETURNING *`,
            [username, category, itemName, unit || null, unitPrice, description || null]
        );

        res.status(201).json(formatResponse(true, result.rows[0]));
    } catch (error) {
        logger.error('[QuotationAPI] 단가표 항목 추가 실패:', error);
        res.status(500).json(formatResponse(false, null, { message: error.message }));
    }
}));

/**
 * 단가표 항목 수정
 * PUT /api/v1/quotations/price-catalog/:id
 */
router.put('/price-catalog/:id', asyncHandler(async (req, res) => {
    const { id } = req.params;
    const { category, itemName, unit, unitPrice, description } = req.body;
    const username = req.user?.id || req.body.username || 'admin';

    try {
        const db = await getDBPool();

        const result = await db.query(
            `UPDATE price_catalog SET
                category = COALESCE($1, category),
                item_name = COALESCE($2, item_name),
                unit = COALESCE($3, unit),
                unit_price = COALESCE($4, unit_price),
                description = COALESCE($5, description),
                updated_at = CURRENT_TIMESTAMP
            WHERE id = $6 AND user_id = $7 RETURNING *`,
            [category, itemName, unit, unitPrice, description, id, username]
        );

        if (result.rows.length === 0) {
            return res.status(404).json(formatResponse(false, null, { message: '항목을 찾을 수 없습니다.' }));
        }

        res.json(formatResponse(true, result.rows[0]));
    } catch (error) {
        logger.error('[QuotationAPI] 단가표 항목 수정 실패:', error);
        res.status(500).json(formatResponse(false, null, { message: error.message }));
    }
}));

/**
 * 단가표 항목 삭제
 * DELETE /api/v1/quotations/price-catalog/:id
 */
router.delete('/price-catalog/:id', asyncHandler(async (req, res) => {
    const { id } = req.params;
    const username = req.user?.id || req.query.username || 'admin';

    try {
        const db = await getDBPool();

        // 소프트 삭제 (is_active = false)
        const result = await db.query(
            `UPDATE price_catalog SET is_active = false, updated_at = CURRENT_TIMESTAMP
             WHERE id = $1 AND user_id = $2 RETURNING *`,
            [id, username]
        );

        if (result.rows.length === 0) {
            return res.status(404).json(formatResponse(false, null, { message: '항목을 찾을 수 없습니다.' }));
        }

        res.json(formatResponse(true, { message: '항목이 삭제되었습니다.' }));
    } catch (error) {
        logger.error('[QuotationAPI] 단가표 항목 삭제 실패:', error);
        res.status(500).json(formatResponse(false, null, { message: error.message }));
    }
}));

export default router;
