#!/bin/bash

#############################################
# Offline Package Repository Creator
# Creates a local APT repository for offline installation
#############################################

set -e

# Color codes
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Progress indicator
show_info() {
    echo -e "${BLUE}ℹ️  $1${NC}"
}

show_success() {
    echo -e "${GREEN}✅ $1${NC}"
}

show_warning() {
    echo -e "${YELLOW}⚠️  $1${NC}"
}

show_error() {
    echo -e "${RED}❌ $1${NC}"
}

handle_error() {
    echo -e "${RED}❌ 오류 발생: $1${NC}"
    echo -e "${RED}작업을 중단합니다.${NC}"
    exit 1
}

# Get script directory
SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"
REPO_BASE_DIR="$SCRIPT_DIR/../offline-repo"
PACKAGES_DIR="$REPO_BASE_DIR/packages"
CACHE_DIR="$REPO_BASE_DIR/cache"

echo ""
echo -e "${GREEN}╔═══════════════════════════════════════════════════════╗${NC}"
echo -e "${GREEN}║   오프라인 패키지 저장소 생성 스크립트              ║${NC}"
echo -e "${GREEN}╚═══════════════════════════════════════════════════════╝${NC}"
echo ""

# Check if running as root
if [ "$EUID" -ne 0 ]; then
    show_error "이 스크립트는 root 권한이 필요합니다"
    show_info "실행 방법: sudo $0"
    exit 1
fi

# Create directories
show_info "디렉토리 구조 생성 중..."
mkdir -p "$PACKAGES_DIR"
mkdir -p "$CACHE_DIR"
mkdir -p "$REPO_BASE_DIR/conf"

show_success "디렉토리 생성 완료: $REPO_BASE_DIR"

# Install required tools
show_info "필수 도구 설치 중..."
apt-get update -qq
apt-get install -y dpkg-dev apt-rdepends

# List of all required packages from install.sh
PACKAGES=(
    # Build essentials
    "build-essential"
    "python3-dev"
    "python3-pip"
    "python3-venv"

    # Basic tools
    "git"
    "curl"
    "wget"

    # Services
    "redis-server"
    "postgresql-client"
    "libpq-dev"

    # OCR and document processing
    "tesseract-ocr"
    "tesseract-ocr-kor"
    "poppler-utils"
    "libmagic1"
    "ffmpeg"

    # Development libraries
    "libssl-dev"
    "libffi-dev"
    "libxml2-dev"
    "libxslt1-dev"
    "libjpeg-dev"
    "zlib1g-dev"

    # Docker (will be handled separately)
    "docker.io"
    "docker-compose"

    # Browser support
    "chromium"

    # Additional Python dependencies
    "python3.12"
    "python3.12-dev"
    "python3.12-venv"
)

# Function to download package and all dependencies
download_package_with_deps() {
    local package=$1

    show_info "패키지 다운로드 중: $package"

    cd "$PACKAGES_DIR"

    # Download package and dependencies
    apt-get download "$package" 2>/dev/null || {
        show_warning "$package 다운로드 실패 (선택적 패키지일 수 있음)"
        return 0
    }

    # Get all dependencies
    apt-rdepends "$package" 2>/dev/null | grep -v "^ " | grep -v "^$package$" | while read -r dep; do
        if [ ! -z "$dep" ] && [ "$dep" != "Depends:" ]; then
            # Download if not already downloaded
            if ! ls "${dep}_"*.deb 1> /dev/null 2>&1; then
                apt-get download "$dep" 2>/dev/null || true
            fi
        fi
    done
}

# Download all packages
show_info "패키지 및 의존성 다운로드 시작..."
echo ""

TOTAL_PACKAGES=${#PACKAGES[@]}
CURRENT=0

for package in "${PACKAGES[@]}"; do
    CURRENT=$((CURRENT + 1))
    echo -e "${BLUE}[$CURRENT/$TOTAL_PACKAGES]${NC} $package"
    download_package_with_deps "$package"
done

show_success "모든 패키지 다운로드 완료"

# Count downloaded packages
PACKAGE_COUNT=$(ls -1 "$PACKAGES_DIR"/*.deb 2>/dev/null | wc -l)
show_info "총 다운로드된 패키지 수: $PACKAGE_COUNT"

# Create Packages.gz index
show_info "패키지 인덱스 생성 중..."
cd "$REPO_BASE_DIR"
dpkg-scanpackages packages /dev/null | gzip -9c > packages/Packages.gz
dpkg-scanpackages packages /dev/null > packages/Packages

show_success "패키지 인덱스 생성 완료"

# Create repository configuration
show_info "저장소 설정 파일 생성 중..."

cat > "$REPO_BASE_DIR/conf/distributions" << EOF
Origin: AIRUN Offline Repository
Label: AIRUN
Codename: offline
Architectures: amd64 i386
Components: main
Description: AIRUN offline package repository
EOF

# Create installation guide
cat > "$REPO_BASE_DIR/INSTALL.md" << 'EOF'
# AIRUN 오프라인 저장소 설치 가이드

## 1. 저장소 파일 복사

오프라인 환경으로 `offline-repo` 디렉토리 전체를 복사합니다:

```bash
# USB 또는 네트워크를 통해 복사
cp -r offline-repo /path/to/target/system/
```

## 2. 로컬 저장소 설정

```bash
# 저장소 디렉토리로 이동
cd /path/to/offline-repo

# APT 소스 추가
echo "deb [trusted=yes] file://$(pwd) packages/" | sudo tee /etc/apt/sources.list.d/airun-offline.list

# 패키지 목록 업데이트
sudo apt-get update
```

## 3. 설치 스크립트 실행

이제 일반적인 방법으로 install.sh를 실행할 수 있습니다:

```bash
cd /path/to/airun
./install.sh
```

## 수동 패키지 설치 (필요시)

로컬 저장소가 작동하지 않는 경우, 직접 패키지를 설치할 수 있습니다:

```bash
cd offline-repo/packages
sudo dpkg -i *.deb
sudo apt-get install -f  # 의존성 해결
```

## 저장소 제거

설치 완료 후 로컬 저장소를 제거하려면:

```bash
sudo rm /etc/apt/sources.list.d/airun-offline.list
sudo apt-get update
```
EOF

show_success "설치 가이드 생성 완료: $REPO_BASE_DIR/INSTALL.md"

# Create a setup script for offline installation
cat > "$REPO_BASE_DIR/setup-offline-repo.sh" << 'SETUP_SCRIPT'
#!/bin/bash

set -e

REPO_DIR="$(cd "$(dirname "$0")" && pwd)"

echo "🔧 오프라인 APT 저장소 설정 중..."

# Add local repository to APT sources
echo "deb [trusted=yes] file://$REPO_DIR packages/" | sudo tee /etc/apt/sources.list.d/airun-offline.list

echo "📦 패키지 목록 업데이트 중..."
sudo apt-get update

echo "✅ 오프라인 저장소 설정 완료!"
echo ""
echo "이제 install.sh를 실행할 수 있습니다."
SETUP_SCRIPT

chmod +x "$REPO_BASE_DIR/setup-offline-repo.sh"

# Calculate total size
TOTAL_SIZE=$(du -sh "$REPO_BASE_DIR" | cut -f1)

echo ""
echo -e "${GREEN}╔═══════════════════════════════════════════════════════╗${NC}"
echo -e "${GREEN}║           오프라인 저장소 생성 완료!                 ║${NC}"
echo -e "${GREEN}╚═══════════════════════════════════════════════════════╝${NC}"
echo ""
echo -e "${BLUE}저장소 위치:${NC} $REPO_BASE_DIR"
echo -e "${BLUE}패키지 수:${NC} $PACKAGE_COUNT"
echo -e "${BLUE}총 크기:${NC} $TOTAL_SIZE"
echo ""
echo -e "${YELLOW}다음 단계:${NC}"
echo "1. offline-repo 디렉토리를 오프라인 시스템으로 복사"
echo "2. 대상 시스템에서 setup-offline-repo.sh 실행"
echo "3. install.sh 실행"
echo ""
echo -e "${BLUE}자세한 내용은 INSTALL.md를 참조하세요${NC}"
echo ""
