#!/usr/bin/env python3
"""
ESG 분석을 위한 문서 텍스트 추출 스크립트
utils.py의 extract_text_from_document 함수를 사용합니다.
"""

import sys
import json
import os
import warnings

# matplotlib 경고 억제
warnings.filterwarnings('ignore', category=UserWarning, module='matplotlib')

# airun 프로젝트 루트를 Python path에 추가
project_root = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
sys.path.insert(0, project_root)

# utils.py의 print 문을 stderr로 리다이렉트
import builtins
original_print = builtins.print
def stderr_print(*args, **kwargs):
    """모든 print를 stderr로 리다이렉트"""
    kwargs['file'] = sys.stderr
    original_print(*args, **kwargs)
builtins.print = stderr_print

from utils import extract_text_from_document

# JSON 출력을 위해 원본 print 복원
builtins.print = original_print

def main():
    if len(sys.argv) < 2:
        print(json.dumps({
            "success": False,
            "error": "문서 파일 경로가 필요합니다."
        }))
        sys.exit(1)

    document_path = sys.argv[1]
    use_ocr = sys.argv[2].lower() == 'true' if len(sys.argv) > 2 else False
    lang = sys.argv[3] if len(sys.argv) > 3 else 'kor'

    try:
        # utils.py의 extract_text_from_document 함수 사용
        text = extract_text_from_document(document_path, use_ocr=use_ocr, lang=lang)

        print(json.dumps({
            "success": True,
            "text": text,
            "length": len(text)
        }, ensure_ascii=False))

    except FileNotFoundError as e:
        print(json.dumps({
            "success": False,
            "error": f"파일을 찾을 수 없습니다: {str(e)}"
        }))
        sys.exit(1)

    except ValueError as e:
        print(json.dumps({
            "success": False,
            "error": f"지원하지 않는 파일 형식이거나 추출 실패: {str(e)}"
        }))
        sys.exit(1)

    except Exception as e:
        print(json.dumps({
            "success": False,
            "error": f"텍스트 추출 중 오류 발생: {str(e)}"
        }))
        sys.exit(1)

if __name__ == '__main__':
    main()
