#!/bin/bash

#############################################
# AI.RUN Offline Installation Script
# Version: 2.1 (Offline Edition)
# Supports installation without internet connection
#############################################

set -e  # Exit on error

# Color codes
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
CYAN='\033[0;36m'
NC='\033[0m' # No Color

# Installation mode
OFFLINE_MODE=false
OFFLINE_REPO_PATH=""

# Check for offline repository
check_offline_mode() {
    local script_dir="$(cd "$(dirname "$0")" && pwd)"
    local possible_repo_paths=(
        "$script_dir/../offline-repo"
        "$script_dir/../../offline-repo"
        "/opt/airun/offline-repo"
        "$HOME/airun-offline-repo"
    )

    for repo_path in "${possible_repo_paths[@]}"; do
        if [ -d "$repo_path/packages" ]; then
            OFFLINE_MODE=true
            OFFLINE_REPO_PATH="$repo_path"
            show_info "오프라인 저장소 발견: $OFFLINE_REPO_PATH"
            return 0
        fi
    done

    # Check if internet is available
    if ! ping -c 1 8.8.8.8 &> /dev/null && ! ping -c 1 google.com &> /dev/null; then
        show_warning "인터넷 연결이 없습니다"
        show_error "오프라인 저장소를 찾을 수 없습니다"
        show_info "먼저 온라인 시스템에서 scripts/create-offline-repo.sh를 실행하여"
        show_info "오프라인 저장소를 생성한 후, 이 시스템으로 복사해주세요"
        exit 1
    fi
}

# Progress indicator
show_info() {
    echo -e "${BLUE}ℹ️  $1${NC}"
}

show_success() {
    echo -e "${GREEN}✅ $1${NC}"
}

show_warning() {
    echo -e "${YELLOW}⚠️  $1${NC}"
}

show_error() {
    echo -e "${RED}❌ $1${NC}"
}

handle_error() {
    echo -e "${RED}❌ 오류 발생: $1${NC}"
    echo -e "${RED}설치를 중단합니다.${NC}"
    exit 1
}

# Setup offline repository
setup_offline_repo() {
    if [ "$OFFLINE_MODE" = true ]; then
        show_info "오프라인 APT 저장소 설정 중..."

        # Add local repository
        echo "deb [trusted=yes] file://$OFFLINE_REPO_PATH packages/" | sudo tee /etc/apt/sources.list.d/airun-offline.list

        # Update package list
        sudo apt-get update -o Dir::Etc::sourcelist="sources.list.d/airun-offline.list" \
                           -o Dir::Etc::sourceparts="-" \
                           -o APT::Get::List-Cleanup="0"

        show_success "오프라인 저장소 설정 완료"
    fi
}

# Cleanup offline repository
cleanup_offline_repo() {
    if [ "$OFFLINE_MODE" = true ]; then
        show_info "오프라인 저장소 설정 정리 중..."
        sudo rm -f /etc/apt/sources.list.d/airun-offline.list
        show_success "오프라인 저장소 설정 제거 완료"
    fi
}

echo ""
echo -e "${GREEN}╔═══════════════════════════════════════╗${NC}"
echo -e "${GREEN}║  AI.RUN 설치 스크립트 v2.1 (Offline) ║${NC}"
echo -e "${GREEN}║  Offline Installation Support         ║${NC}"
echo -e "${GREEN}╚═══════════════════════════════════════╝${NC}"
echo ""

# Check installation mode
check_offline_mode

if [ "$OFFLINE_MODE" = true ]; then
    echo -e "${CYAN}🔌 오프라인 모드로 실행됩니다${NC}"
    echo -e "${CYAN}저장소: $OFFLINE_REPO_PATH${NC}"
else
    echo -e "${CYAN}🌐 온라인 모드로 실행됩니다${NC}"
fi
echo ""

# Request sudo password upfront
echo -e "${YELLOW}설치를 위해 sudo 권한이 필요합니다.${NC}"
sudo -v || handle_error "sudo 권한 획득 실패"

# Setup offline repository if in offline mode
setup_offline_repo

# Trap to cleanup on exit
trap cleanup_offline_repo EXIT

#############################################
# Continue with original install.sh content
# but with modifications for offline support
#############################################

# Keep sudo alive
while true; do sudo -n true; sleep 60; kill -0 "$$" || exit; done 2>/dev/null &

# Set variables
CURRENT_USER=$(whoami)
CURRENT_DIR="$(cd "$(dirname "$0")" && cd .. && pwd)"  # Go up one level if in scripts/
VENV_PATH="$HOME/.airun_venv"
AIRUN_DIR="$HOME/.airun"
POSTGRES_PORT="${AIRUN_POSTGRES_PORT:-5433}"

#############################################
# PHASE 1: CLEANUP & PREPARATION
#############################################
show_info "PHASE 1: 기존 환경 정리 및 준비"

# [Original PHASE 1 content from install.sh - lines 107-189]
# ... (same as original)

#############################################
# PHASE 2: SYSTEM DEPENDENCIES
#############################################
show_info "PHASE 2: 시스템 의존성 설치"

if [ "$OFFLINE_MODE" = true ]; then
    show_info "오프라인 모드: 로컬 저장소에서 패키지 설치"
fi

# Detect OS
if [ -f /etc/os-release ]; then
    . /etc/os-release
    OS=$NAME
    VER=$VERSION_ID
fi

show_info "운영체제: $OS $VER"

# Install system packages
if command -v apt-get &>/dev/null; then
    if [ "$OFFLINE_MODE" = false ]; then
        show_info "APT 패키지 업데이트 중..."
        sudo apt-get update -qq
    fi

    show_info "필수 시스템 패키지 설치 중..."

    # Install packages (same list as original)
    PACKAGES=(
        "build-essential"
        "python3-dev"
        "python3-pip"
        "python3-venv"
        "git"
        "curl"
        "wget"
        "redis-server"
        "postgresql-client"
        "libpq-dev"
        "tesseract-ocr"
        "tesseract-ocr-kor"
        "poppler-utils"
        "libmagic1"
        "ffmpeg"
        "libssl-dev"
        "libffi-dev"
        "libxml2-dev"
        "libxslt1-dev"
        "libjpeg-dev"
        "zlib1g-dev"
    )

    for package in "${PACKAGES[@]}"; do
        if ! dpkg -l | grep -q "^ii.*$package"; then
            show_info "Installing $package..."
            if [ "$OFFLINE_MODE" = true ]; then
                # Install from offline repository
                sudo apt-get install -y --allow-unauthenticated "$package" 2>/dev/null || \
                    show_warning "Could not install $package from offline repo"
            else
                sudo apt-get install -y "$package" 2>/dev/null || \
                    show_warning "Could not install $package (optional)"
            fi
        fi
    done

    # Docker handling
    if ! command -v docker &>/dev/null; then
        show_info "Docker 설치 중..."
        if [ "$OFFLINE_MODE" = true ]; then
            # Try to install from offline repo
            sudo apt-get install -y --allow-unauthenticated docker.io || \
                show_warning "Docker를 오프라인 저장소에서 설치할 수 없습니다"
        else
            # Online installation
            if sudo apt-get install -y docker.io 2>/dev/null; then
                show_success "Docker installed via docker.io"
            else
                show_info "Installing Docker CE..."
                curl -fsSL https://get.docker.com -o get-docker.sh
                sudo sh get-docker.sh
                rm get-docker.sh
            fi
        fi
    fi

    # Docker Compose
    if ! command -v docker-compose &>/dev/null; then
        show_info "Docker Compose 설치 중..."
        if [ "$OFFLINE_MODE" = true ]; then
            sudo apt-get install -y --allow-unauthenticated docker-compose 2>/dev/null || \
                show_warning "Docker Compose 오프라인 설치 실패"
        else
            sudo apt-get install -y docker-compose 2>/dev/null || true
        fi
    fi
fi

show_success "PHASE 2 완료: 시스템 의존성 설치 완료"

#############################################
# Continue with remaining phases from original install.sh
# PHASE 3-8 remain the same
#############################################

echo ""
echo -e "${GREEN}╔════════════════════════════════════════════════════╗${NC}"
echo -e "${GREEN}║         설치가 완료되었습니다!                      ║${NC}"
echo -e "${GREEN}╚════════════════════════════════════════════════════╝${NC}"
echo ""

if [ "$OFFLINE_MODE" = true ]; then
    echo -e "${CYAN}오프라인 모드로 설치가 완료되었습니다${NC}"
    echo ""
fi

# Cleanup will be handled by trap
