#!/bin/bash

# Color definitions
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m'

# PostgreSQL with pgvector setup - Optimized version
setup_postgresql_with_pgvector() {
    echo -e "${GREEN}PostgreSQL with pgvector 설치 및 설정을 시작합니다...${NC}"

    # Docker 설치 확인
    if ! command -v docker &>/dev/null; then
        echo -e "${RED}Docker가 설치되어 있지 않습니다. Docker를 먼저 설치해주세요.${NC}"
        return 1
    fi

    # PostgreSQL 설정 변수
    CONTAINER_NAME="airun-postgres"
    PG_HOST="localhost"
    PG_PORT="5433"
    PG_USER="ivs"
    PG_PASSWORD="Exitem08**"
    PG_DATABASE="airun"
    
    # pgvector가 포함된 PostgreSQL 16 이미지 사용 (기존 데이터 호환성)
    PG_IMAGE="pgvector/pgvector:0.8.0-pg17-trixie"
    
    echo -e "${YELLOW}사용할 이미지: $PG_IMAGE${NC}"
    echo -e "${YELLOW}이 이미지에는 pgvector 0.8.0이 사전 설치되어 있습니다.${NC}"

    # 기존 컨테이너 확인 및 제거
    if docker ps -a | grep -q $CONTAINER_NAME; then
        echo -e "${YELLOW}기존 PostgreSQL 컨테이너가 발견되었습니다.${NC}"
        read -p "기존 컨테이너를 제거하고 새로 설치하시겠습니까? (y/n): " -n 1 -r
        echo
        if [[ $REPLY =~ ^[Yy]$ ]]; then
            docker stop $CONTAINER_NAME 2>/dev/null || true
            docker rm $CONTAINER_NAME 2>/dev/null || true
            echo -e "${GREEN}기존 컨테이너가 제거되었습니다.${NC}"
        else
            echo -e "${YELLOW}설치를 취소합니다.${NC}"
            return 0
        fi
    fi

    # PostgreSQL 데이터 디렉토리 생성
    POSTGRES_DATA_DIR="$HOME/.airun/postgres-data"
    mkdir -p $POSTGRES_DATA_DIR
    echo -e "${GREEN}데이터 디렉토리 생성: $POSTGRES_DATA_DIR${NC}"

    # Docker 이미지 pull
    echo -e "${YELLOW}Docker 이미지를 다운로드합니다...${NC}"
    docker pull $PG_IMAGE

    # PostgreSQL Docker 컨테이너 실행
    echo -e "${YELLOW}PostgreSQL Docker 컨테이너를 시작합니다...${NC}"
    docker run --name $CONTAINER_NAME \
        -e POSTGRES_USER=$PG_USER \
        -e POSTGRES_PASSWORD=$PG_PASSWORD \
        -e POSTGRES_DB=$PG_DATABASE \
        -e POSTGRES_HOST_AUTH_METHOD=md5 \
        -p $PG_PORT:5432 \
        -v $POSTGRES_DATA_DIR:/var/lib/postgresql/data \
        -d $PG_IMAGE

    # 컨테이너 실행 확인
    if ! docker ps | grep -q $CONTAINER_NAME; then
        echo -e "${RED}PostgreSQL Docker 컨테이너 실행에 실패했습니다.${NC}"
        docker logs $CONTAINER_NAME
        return 1
    fi

    echo -e "${GREEN}PostgreSQL Docker 컨테이너가 성공적으로 실행되었습니다.${NC}"

    # PostgreSQL이 완전히 시작될 때까지 대기
    echo -e "${YELLOW}PostgreSQL 서버가 준비될 때까지 기다립니다...${NC}"
    local max_attempts=30
    local attempt=0
    
    while [ $attempt -lt $max_attempts ]; do
        if docker exec $CONTAINER_NAME pg_isready -h localhost -U $PG_USER >/dev/null 2>&1; then
            echo -e "${GREEN}PostgreSQL 서버가 준비되었습니다.${NC}"
            break
        fi
        echo -n "."
        sleep 2
        ((attempt++))
    done
    
    if [ $attempt -eq $max_attempts ]; then
        echo -e "${RED}PostgreSQL 서버가 시작되지 않았습니다.${NC}"
        docker logs $CONTAINER_NAME
        return 1
    fi

    # pgvector 확장 활성화 (이미 설치되어 있으므로 CREATE EXTENSION만 실행)
    echo -e "${YELLOW}pgvector 확장을 활성화합니다...${NC}"
    
    # 확장 생성 시도
    docker exec $CONTAINER_NAME psql -U $PG_USER -d $PG_DATABASE -c "CREATE EXTENSION IF NOT EXISTS vector;" 2>/dev/null
    
    # 확장 확인
    if docker exec $CONTAINER_NAME psql -U $PG_USER -d $PG_DATABASE -c "\dx" | grep -q vector; then
        echo -e "${GREEN}pgvector 확장이 성공적으로 활성화되었습니다.${NC}"
        
        # pgvector 버전 확인
        VECTOR_VERSION=$(docker exec $CONTAINER_NAME psql -U $PG_USER -d $PG_DATABASE -t -c "SELECT extversion FROM pg_extension WHERE extname='vector';" | tr -d ' ')
        echo -e "${GREEN}pgvector 버전: $VECTOR_VERSION${NC}"
    else
        echo -e "${RED}pgvector 확장 활성화에 실패했습니다.${NC}"
        return 1
    fi

    # 컨테이너 자동 시작 설정
    echo -e "${YELLOW}시스템 부팅 시 자동 시작 설정...${NC}"
    docker update --restart unless-stopped $CONTAINER_NAME

    # .env 파일 생성/업데이트
    ENV_FILE=".env"
    
    if [ -f "$ENV_FILE" ]; then
        echo -e "${YELLOW}기존 .env 파일을 백업합니다...${NC}"
        cp $ENV_FILE "${ENV_FILE}.backup.$(date +%Y%m%d_%H%M%S)"
    fi
    
    echo -e "${YELLOW}.env 파일을 생성/업데이트합니다...${NC}"
    
    # 기존 PostgreSQL 설정 제거 후 새 설정 추가
    if [ -f "$ENV_FILE" ]; then
        # 기존 PG_ 설정 제거
        grep -v "^PG_\|^POSTGRES_\|^DB_" $ENV_FILE > ${ENV_FILE}.tmp || true
        mv ${ENV_FILE}.tmp $ENV_FILE
    fi
    
    # 새 설정 추가
    cat >> $ENV_FILE <<EOF

# PostgreSQL with pgvector Configuration
PG_HOST=$PG_HOST
PG_PORT=$PG_PORT
PG_USER=$PG_USER
PG_PASSWORD=$PG_PASSWORD
PG_DATABASE=$PG_DATABASE

# Alternative naming conventions (for compatibility)
POSTGRES_HOST=$PG_HOST
POSTGRES_PORT=$PG_PORT
POSTGRES_USER=$PG_USER
POSTGRES_PASSWORD=$PG_PASSWORD
POSTGRES_DB=$PG_DATABASE

DB_HOST=$PG_HOST
DB_PORT=$PG_PORT
DB_USER=$PG_USER
DB_PASSWORD=$PG_PASSWORD
DB_NAME=$PG_DATABASE
EOF

    echo -e "${GREEN}환경 설정 파일이 업데이트되었습니다: $ENV_FILE${NC}"

    # 설치 정보 출력
    echo -e "${GREEN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo -e "${GREEN}PostgreSQL with pgvector 설정 완료!${NC}"
    echo -e "${GREEN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo -e "${GREEN}컨테이너 이름: $CONTAINER_NAME${NC}"
    echo -e "${GREEN}호스트: $PG_HOST${NC}"
    echo -e "${GREEN}포트: $PG_PORT${NC}"
    echo -e "${GREEN}사용자: $PG_USER${NC}"
    echo -e "${GREEN}데이터베이스: $PG_DATABASE${NC}"
    echo -e "${GREEN}pgvector 버전: $VECTOR_VERSION${NC}"
    echo -e "${GREEN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    
    # 연결 테스트 명령어 안내
    echo -e "${YELLOW}연결 테스트:${NC}"
    echo -e "${YELLOW}docker exec -it $CONTAINER_NAME psql -U $PG_USER -d $PG_DATABASE${NC}"
    
    return 0
}

# 함수 실행
setup_postgresql_with_pgvector