#!/bin/bash

# Health Monitoring 최적화 스크립트

set -e

echo "=== RAG Health Monitoring 최적화 ==="

# 기존 타이머 중지
echo "exitem08" | sudo -S systemctl stop airun-rag-healthcheck.timer

# 최적화된 타이머 설정 (15분마다)
echo "exitem08" | sudo -S tee /etc/systemd/system/airun-rag-healthcheck.timer > /dev/null << 'EOF'
[Unit]
Description=AI.RUN RAG Health Check Timer (Optimized)
Requires=airun-rag-healthcheck.service

[Timer]
# 서비스 시작 후 5분 뒤 첫 실행
OnStartupSec=300

# 이후 15분마다 실행 (성능 영향 최소화)
OnUnitInactiveSec=900

# 정확도 1분 (시스템 부하 분산)
AccuracySec=60

[Install]
WantedBy=timers.target
EOF

# Health Check 스크립트도 최적화
cat > /home/hamonikr/work/airun/scripts/airun-rag-healthcheck.sh << 'EOF'
#!/bin/bash

# 최적화된 AI.RUN RAG 서비스 Health Check 스크립트

LOG_FILE="/var/log/airun/airun-rag-healthcheck.log"
RAG_HEALTH_URL="http://localhost:5600/health"
MAX_RETRIES=2  # 3 -> 2로 줄임
RETRY_DELAY=3  # 5 -> 3초로 줄임

# 로그 함수 (타임스탬프 간소화)
log() {
    echo "[$(date '+%H:%M:%S')] $1" | tee -a "$LOG_FILE"
}

# Health Check 수행 (최적화됨)
check_rag_health() {
    local attempt=1
    
    while [ $attempt -le $MAX_RETRIES ]; do
        # HTTP Health Check (타임아웃 단축)
        response=$(curl -s -w "%{http_code}" --connect-timeout 5 --max-time 15 "$RAG_HEALTH_URL" 2>/dev/null)
        http_code="${response: -3}"
        
        if [ "$http_code" = "200" ]; then
            body="${response%???}"
            if echo "$body" | grep -q '"model_loaded":true'; then
                # 정상 상태는 간단히 로깅
                [ $attempt -eq 1 ] && log "✅ 정상"
                return 0
            elif echo "$body" | grep -q '"model_loaded":false'; then
                log "⚠️ 모델 미로드 (자동복구 대기중)"
                return 2
            fi
        elif [ "$http_code" = "000" ]; then
            log "❌ 연결실패 (시도 $attempt/$MAX_RETRIES)"
        else
            log "❌ HTTP $http_code (시도 $attempt/$MAX_RETRIES)"
        fi
        
        if [ $attempt -lt $MAX_RETRIES ]; then
            sleep $RETRY_DELAY
        fi
        
        ((attempt++))
    done
    
    return 1
}

# 메인 로직 (간소화됨)
main() {
    # 로그 디렉토리 확인
    mkdir -p "$(dirname "$LOG_FILE")"
    
    # Health Check 수행
    check_rag_health
    result=$?
    
    case $result in
        0) exit 0 ;;           # 정상
        1) 
            log "Health check 실패 - 서비스 재시작 필요"
            exit 1 ;;          # systemd 재시작
        2) exit 0 ;;           # 부분 실패 (자동복구 대기)
    esac
}

main "$@"
EOF

# 스크립트 권한 설정
chmod +x /home/hamonikr/work/airun/scripts/airun-rag-healthcheck.sh

# systemd 재로드 및 타이머 재시작
echo "exitem08" | sudo -S systemctl daemon-reload
echo "exitem08" | sudo -S systemctl start airun-rag-healthcheck.timer

echo ""
echo "=== 최적화 완료 ✅ ==="
echo "주요 변경사항:"
echo "• 실행 주기: 5분 → 15분"
echo "• 첫 실행 대기: 1분 → 5분"  
echo "• 재시도 횟수: 3회 → 2회"
echo "• 재시도 간격: 5초 → 3초"
echo "• 연결 타임아웃: 10초 → 5초"
echo "• 응답 타임아웃: 30초 → 15초"
echo ""
echo "예상 성능 영향: 거의 없음 (15분마다 ~6ms)"

# 상태 확인
echo ""
echo "=== 타이머 상태 ==="
echo "exitem08" | sudo -S systemctl list-timers airun-rag-healthcheck.timer