#!/bin/bash

#############################################
# AIRUN Offline Package Preparation Script
# All-in-one script to prepare complete offline installation package
#############################################

set -e

# Color codes
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
CYAN='\033[0;36m'
NC='\033[0m'

# Progress functions
show_info() { echo -e "${BLUE}ℹ️  $1${NC}"; }
show_success() { echo -e "${GREEN}✅ $1${NC}"; }
show_warning() { echo -e "${YELLOW}⚠️  $1${NC}"; }
show_error() { echo -e "${RED}❌ $1${NC}"; }
handle_error() { echo -e "${RED}❌ 오류: $1${NC}"; exit 1; }

echo ""
echo -e "${CYAN}╔═══════════════════════════════════════════════════════════╗${NC}"
echo -e "${CYAN}║                                                           ║${NC}"
echo -e "${CYAN}║       AIRUN 오프라인 패키지 준비 스크립트                ║${NC}"
echo -e "${CYAN}║       Complete Offline Installation Package              ║${NC}"
echo -e "${CYAN}║                                                           ║${NC}"
echo -e "${CYAN}╚═══════════════════════════════════════════════════════════╝${NC}"
echo ""

# Check if running with sudo
if [ "$EUID" -ne 0 ]; then
    show_error "이 스크립트는 sudo 권한이 필요합니다"
    echo "실행 방법: sudo $0"
    exit 1
fi

# Check internet connection
show_info "인터넷 연결 확인 중..."
if ! ping -c 1 8.8.8.8 &> /dev/null; then
    show_error "인터넷 연결이 필요합니다"
    exit 1
fi
show_success "인터넷 연결 확인됨"

# Get directories
SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"
AIRUN_ROOT="$(cd "$SCRIPT_DIR/.." && pwd)"
OFFLINE_DIR="$AIRUN_ROOT/offline-package-$(date +%Y%m%d-%H%M%S)"

show_info "오프라인 패키지 디렉토리: $OFFLINE_DIR"

# Create directory structure
show_info "디렉토리 구조 생성 중..."
mkdir -p "$OFFLINE_DIR"/{packages,python-wheels,models,docker,nodejs,scripts,docs}
show_success "디렉토리 생성 완료"

#############################################
# Step 1: Download Debian Packages
#############################################
echo ""
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo -e "${CYAN}  Step 1/6: Debian 패키지 다운로드                        ${NC}"
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"

show_info "APT 패키지 캐시 업데이트 중..."
apt-get update -qq

show_info "dpkg-dev 및 apt-rdepends 설치 중..."
apt-get install -y dpkg-dev apt-rdepends >/dev/null 2>&1

# Package list
PACKAGES=(
    "build-essential" "python3-dev" "python3-pip" "python3-venv"
    "git" "curl" "wget" "redis-server" "postgresql-client" "libpq-dev"
    "tesseract-ocr" "tesseract-ocr-kor" "poppler-utils" "libmagic1"
    "ffmpeg" "libssl-dev" "libffi-dev" "libxml2-dev" "libxslt1-dev"
    "libjpeg-dev" "zlib1g-dev" "docker.io" "docker-compose"
    "python3.12" "python3.12-dev" "python3.12-venv"
)

cd "$OFFLINE_DIR/packages"

show_info "패키지 다운로드 시작 (총 ${#PACKAGES[@]}개)..."
for i in "${!PACKAGES[@]}"; do
    package="${PACKAGES[$i]}"
    echo -ne "${BLUE}[$((i+1))/${#PACKAGES[@]}]${NC} $package\r"

    # Download package
    apt-get download "$package" 2>/dev/null || true

    # Download dependencies
    apt-rdepends "$package" 2>/dev/null | grep -v "^ " | grep -v "^$package$" | while read -r dep; do
        if [ ! -z "$dep" ] && [ "$dep" != "Depends:" ]; then
            apt-get download "$dep" 2>/dev/null || true
        fi
    done
done
echo ""

# Create Packages.gz
show_info "패키지 인덱스 생성 중..."
cd "$OFFLINE_DIR"
dpkg-scanpackages packages /dev/null | gzip -9c > packages/Packages.gz
dpkg-scanpackages packages /dev/null > packages/Packages

PACKAGE_COUNT=$(ls -1 packages/*.deb 2>/dev/null | wc -l)
show_success "Debian 패키지 다운로드 완료 ($PACKAGE_COUNT 개)"

#############################################
# Step 2: Download Python Wheels
#############################################
echo ""
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo -e "${CYAN}  Step 2/6: Python Wheels 다운로드                        ${NC}"
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"

# Create temp venv
TEMP_VENV="$OFFLINE_DIR/.temp_venv"
show_info "임시 가상환경 생성 중..."
python3 -m venv "$TEMP_VENV"
source "$TEMP_VENV/bin/activate"

pip install --upgrade pip wheel >/dev/null 2>&1

# Download from requirements files
if [ -f "$AIRUN_ROOT/requirements-linux.txt" ]; then
    show_info "requirements-linux.txt 다운로드 중..."
    pip download \
        -r "$AIRUN_ROOT/requirements-linux.txt" \
        --dest "$OFFLINE_DIR/python-wheels" \
        --index-url https://download.pytorch.org/whl/cu129 \
        --extra-index-url https://pypi.org/simple \
        2>/dev/null || show_warning "일부 패키지 다운로드 실패"
fi

if [ -f "$AIRUN_ROOT/requirement-system.txt" ]; then
    show_info "requirement-system.txt 다운로드 중..."
    pip download \
        -r "$AIRUN_ROOT/requirement-system.txt" \
        --dest "$OFFLINE_DIR/python-wheels" \
        2>/dev/null || show_warning "일부 시스템 패키지 다운로드 실패"
fi

# Download additional essentials
show_info "추가 필수 패키지 다운로드 중..."
for pkg in Cython numpy setuptools wheel pip huggingface-hub sentence-transformers; do
    pip download "$pkg" --dest "$OFFLINE_DIR/python-wheels" 2>/dev/null || true
done

deactivate
rm -rf "$TEMP_VENV"

WHEEL_COUNT=$(ls -1 "$OFFLINE_DIR/python-wheels"/*.whl 2>/dev/null | wc -l)
show_success "Python Wheels 다운로드 완료 ($WHEEL_COUNT 개)"

#############################################
# Step 3: Download AI Models
#############################################
echo ""
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo -e "${CYAN}  Step 3/6: AI 모델 다운로드                              ${NC}"
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"

# Create temp venv for model download
python3 -m venv "$TEMP_VENV"
source "$TEMP_VENV/bin/activate"
pip install --upgrade pip huggingface-hub >/dev/null 2>&1

# Download models
show_info "모델 다운로드 중 (1/3): KURE-v1..."
python3 << PYTHON_SCRIPT
from huggingface_hub import snapshot_download
import os
try:
    snapshot_download(
        'nlpai-lab/KURE-v1',
        local_dir='$OFFLINE_DIR/models/KURE-v1',
        ignore_patterns=['*.bin']
    )
    print('✅ KURE-v1 다운로드 완료')
except Exception as e:
    print(f'⚠️ KURE-v1 다운로드 실패: {e}')
PYTHON_SCRIPT

show_info "모델 다운로드 중 (2/3): KR-SBERT..."
python3 << PYTHON_SCRIPT
from huggingface_hub import snapshot_download
try:
    snapshot_download(
        'snunlp/KR-SBERT-V40K-klueNLI-augSTS',
        local_dir='$OFFLINE_DIR/models/KR-SBERT',
        ignore_patterns=['*.bin']
    )
    print('✅ KR-SBERT 다운로드 완료')
except Exception as e:
    print(f'⚠️ KR-SBERT 다운로드 실패: {e}')
PYTHON_SCRIPT

show_info "모델 다운로드 중 (3/3): CLIP-ViT..."
python3 << PYTHON_SCRIPT
from huggingface_hub import snapshot_download
try:
    snapshot_download(
        'Bingsu/clip-vit-base-patch32-ko',
        local_dir='$OFFLINE_DIR/models/clip-vit',
        ignore_patterns=['*.bin']
    )
    print('✅ CLIP-ViT 다운로드 완료')
except Exception as e:
    print(f'⚠️ CLIP-ViT 다운로드 실패: {e}')
PYTHON_SCRIPT

deactivate
rm -rf "$TEMP_VENV"

show_success "AI 모델 다운로드 완료"

#############################################
# Step 4: Download Docker Images
#############################################
echo ""
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo -e "${CYAN}  Step 4/6: Docker 이미지 다운로드                        ${NC}"
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"

if command -v docker &>/dev/null; then
    show_info "PostgreSQL+pgvector 이미지 다운로드 중..."
    docker pull pgvector/pgvector:pg17 >/dev/null 2>&1 || show_warning "이미지 다운로드 실패"

    show_info "Docker 이미지 저장 중..."
    docker save pgvector/pgvector:pg17 -o "$OFFLINE_DIR/docker/pgvector-pg17.tar"

    show_success "Docker 이미지 저장 완료"
else
    show_warning "Docker가 설치되지 않아 이미지를 다운로드하지 못했습니다"
fi

#############################################
# Step 5: Download Node.js
#############################################
echo ""
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo -e "${CYAN}  Step 5/6: Node.js 다운로드                              ${NC}"
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"

show_info "NVM 설치 스크립트 다운로드 중..."
curl -sS https://raw.githubusercontent.com/nvm-sh/nvm/v0.40.1/install.sh \
    -o "$OFFLINE_DIR/nodejs/nvm-install.sh"

show_info "Node.js v22.18.0 바이너리 다운로드 중..."
wget -q https://nodejs.org/dist/v22.18.0/node-v22.18.0-linux-x64.tar.xz \
    -O "$OFFLINE_DIR/nodejs/node-v22.18.0-linux-x64.tar.xz"

show_success "Node.js 다운로드 완료"

#############################################
# Step 6: Download Ollama
#############################################
echo ""
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo -e "${CYAN}  Step 6/7: Ollama 다운로드                               ${NC}"
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"

# Create ollama directory
mkdir -p "$OFFLINE_DIR/ollama"

show_info "Ollama 설치 스크립트 다운로드 중..."
wget -q https://ollama.com/install.sh -O "$OFFLINE_DIR/ollama/install.sh"

show_info "Ollama 바이너리 다운로드 중..."
wget -q --show-progress https://ollama.com/download/ollama-linux-amd64.tgz \
    -O "$OFFLINE_DIR/ollama/ollama-linux-amd64.tgz"

# Create Ollama installation script
cat > "$OFFLINE_DIR/ollama/install-ollama-offline.sh" << 'OLLAMA_INSTALL_EOF'
#!/bin/bash
set -e

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
OLLAMA_BINARY="$SCRIPT_DIR/ollama-linux-amd64.tgz"
BIN_DIR="/usr/local/bin"
MODELS_DIR="$SCRIPT_DIR/models"

echo "🚀 Ollama 오프라인 설치"
echo ""

if [[ $EUID -ne 0 ]]; then
   echo "❌ root 권한이 필요합니다. sudo로 실행하세요."
   exit 1
fi

echo "📦 Ollama 바이너리 압축 해제 중..."
tar -xzf "$OLLAMA_BINARY" -C /tmp/

echo "📦 Ollama 설치 중..."
if [ -f "/tmp/bin/ollama" ]; then
    install -o0 -g0 -m755 /tmp/bin/ollama "$BIN_DIR/ollama"
elif [ -f "/tmp/ollama" ]; then
    install -o0 -g0 -m755 /tmp/ollama "$BIN_DIR/ollama"
fi
rm -rf /tmp/bin /tmp/ollama 2>/dev/null || true

echo "👤 Ollama 사용자 생성 중..."
if ! id -u ollama >/dev/null 2>&1; then
    useradd -r -s /bin/false -U -m -d /usr/share/ollama ollama
fi

echo "⚙️  systemd 서비스 설정 중..."
cat > /etc/systemd/system/ollama.service << EOF
[Unit]
Description=Ollama Service
After=network-online.target

[Service]
ExecStart=$BIN_DIR/ollama serve
User=ollama
Group=ollama
Restart=always
RestartSec=3
Environment="PATH=$PATH"

[Install]
WantedBy=default.target
EOF

systemctl daemon-reload
systemctl enable ollama
systemctl start ollama

sleep 3

if systemctl is-active --quiet ollama; then
    echo "✅ Ollama 서비스 시작됨!"
else
    echo "⚠️  Ollama 서비스 시작 실패"
    systemctl status ollama --no-pager
fi

if [ -d "$MODELS_DIR" ]; then
    echo "📦 사전 다운로드된 모델 설치 중..."
    mkdir -p /usr/share/ollama/.ollama/models

    if [ -d "$MODELS_DIR/blobs" ]; then
        cp -r "$MODELS_DIR/blobs" /usr/share/ollama/.ollama/models/
    fi

    if [ -d "$MODELS_DIR/manifests" ]; then
        cp -r "$MODELS_DIR/manifests" /usr/share/ollama/.ollama/models/
    fi

    chown -R ollama:ollama /usr/share/ollama/.ollama
    echo "✅ 모델 설치 완료!"
else
    echo "ℹ️  모델 디렉토리가 없습니다. 온라인에서 모델을 다운로드하세요."
fi

echo ""
echo "✅ Ollama 설치 완료!"
echo "   - 서비스 상태: systemctl status ollama"
echo "   - 모델 목록: ollama list"
OLLAMA_INSTALL_EOF

chmod +x "$OFFLINE_DIR/ollama/install-ollama-offline.sh"

# Create model download instructions
cat > "$OFFLINE_DIR/ollama/README.md" << 'OLLAMA_README_EOF'
# Ollama 오프라인 설치

## 온라인 환경 - 모델 다운로드

모델을 사전에 다운로드하려면:

```bash
# Ollama 설치 (아직 없는 경우)
curl -fsSL https://ollama.com/install.sh | sh

# 베이스 모델 다운로드
ollama pull devstral
ollama pull deepseek-r1:latest
ollama pull gpt-oss:20b
ollama pull gemma3:12b

# 모델 내보내기
# Ollama 모델 디렉토리 복사
cp -r ~/.ollama/models/* ./models/
```

## 오프라인 환경 - 설치

```bash
sudo ./install-ollama-offline.sh
```

## AIRUN 통합

설정 파일 (`~/.airun/airun.conf`)에 추가:

```bash
export OLLAMA_PROXY_SERVER=http://localhost:11434
export OLLAMA_MODEL=airun-chat
```
OLLAMA_README_EOF

show_success "Ollama 다운로드 완료"
show_warning "모델은 온라인 환경에서 별도로 다운로드해야 합니다 (README 참조)"

#############################################
# Create Installation Scripts
#############################################
echo ""
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo -e "${CYAN}  설치 스크립트 생성                                      ${NC}"
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"

# Copy installation scripts
cp "$SCRIPT_DIR/install-offline.sh" "$OFFLINE_DIR/scripts/" 2>/dev/null || true

# Create setup script
cat > "$OFFLINE_DIR/setup.sh" << 'SETUP_EOF'
#!/bin/bash
set -e

PACKAGE_DIR="$(cd "$(dirname "$0")" && pwd)"

echo "🚀 AIRUN 오프라인 설치 시작"
echo ""

# Setup APT repository
echo "📦 로컬 APT 저장소 설정 중..."
echo "deb [trusted=yes] file://$PACKAGE_DIR packages/" | sudo tee /etc/apt/sources.list.d/airun-offline.list
sudo apt-get update

# Load Docker image
if [ -f "$PACKAGE_DIR/docker/pgvector-pg17.tar" ]; then
    echo "🐳 Docker 이미지 로드 중..."
    sudo docker load -i "$PACKAGE_DIR/docker/pgvector-pg17.tar"
fi

echo ""
echo "✅ 준비 완료!"
echo ""
echo "다음 단계:"
echo "1. cd scripts"
echo "2. sudo ./install-offline.sh"
SETUP_EOF

chmod +x "$OFFLINE_DIR/setup.sh"

# Copy documentation
cp "$AIRUN_ROOT/docs/OFFLINE_INSTALLATION.md" "$OFFLINE_DIR/docs/" 2>/dev/null || true

# Create README
cat > "$OFFLINE_DIR/README.md" << 'README_EOF'
# AIRUN 오프라인 설치 패키지

## 빠른 시작

1. 전체 디렉토리를 오프라인 시스템으로 복사
2. 설치 스크립트 실행:

```bash
cd offline-package-YYYYMMDD-HHMMSS
sudo ./setup.sh
cd scripts
sudo ./install-offline.sh
```

## 포함된 내용

- Debian 시스템 패키지 및 의존성
- Python wheels (PyTorch CUDA 12.9 포함)
- AI 임베딩 모델 (KURE-v1, KR-SBERT, CLIP)
- PostgreSQL+pgvector Docker 이미지
- Node.js v22.18.0

## 자세한 설명

`docs/OFFLINE_INSTALLATION.md` 참조

## 시스템 요구사항

- Ubuntu 22.04/24.04 LTS
- 최소 30GB 디스크 공간
- sudo 권한
README_EOF

show_success "설치 스크립트 생성 완료"

#############################################
# Step 7: Download NVIDIA Driver Packages
#############################################
echo ""
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo -e "${CYAN}  Step 7/7: NVIDIA Driver 580 다운로드                   ${NC}"
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"

cd "$OFFLINE_DIR/packages"

# NVIDIA 드라이버 패키지 목록
NVIDIA_PACKAGES=(
    "nvidia-driver-580-open"
    "nvidia-dkms-580-open"
    "nvidia-kernel-common-580"
    "nvidia-kernel-source-580-open"
    "libnvidia-compute-580"
    "libnvidia-extra-580"
    "libnvidia-decode-580"
    "libnvidia-encode-580"
    "libnvidia-fbc1-580"
    "libnvidia-gl-580"
    "xserver-xorg-video-nvidia-580"
    "nvidia-settings"
    "nvidia-utils-580"
    "dkms"
    "linux-headers-generic"
)

show_info "NVIDIA Driver 580 패키지 다운로드 중..."
NVIDIA_COUNT=0

for package in "${NVIDIA_PACKAGES[@]}"; do
    if ! ls ${package}_*.deb 1> /dev/null 2>&1; then
        show_info "  - $package 다운로드 중..."
        if apt-get download "$package" 2>/dev/null; then
            ((NVIDIA_COUNT++))
        else
            show_warning "  - $package 다운로드 실패 (선택 사항)"
        fi
    fi
done

show_success "NVIDIA 패키지 다운로드 완료 (새로 다운로드: $NVIDIA_COUNT 개)"

# Repository 인덱스 재생성
show_info "저장소 인덱스 재생성 중..."
dpkg-scanpackages . /dev/null | gzip -9c > Packages.gz
show_success "Packages.gz 업데이트 완료"

#############################################
# Calculate Statistics and Create Archive
#############################################
echo ""
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
echo -e "${CYAN}  통계 및 압축                                            ${NC}"
echo -e "${CYAN}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"

# Calculate sizes
TOTAL_SIZE=$(du -sh "$OFFLINE_DIR" | cut -f1)
PACKAGES_SIZE=$(du -sh "$OFFLINE_DIR/packages" | cut -f1)
WHEELS_SIZE=$(du -sh "$OFFLINE_DIR/python-wheels" | cut -f1)
MODELS_SIZE=$(du -sh "$OFFLINE_DIR/models" | cut -f1)

echo ""
echo -e "${BLUE}📊 패키지 통계${NC}"
echo -e "  Debian 패키지: $PACKAGES_SIZE ($PACKAGE_COUNT 개)"
echo -e "  Python Wheels: $WHEELS_SIZE ($WHEEL_COUNT 개)"
echo -e "  AI 모델: $MODELS_SIZE"
echo -e "  총 크기: $TOTAL_SIZE"
echo ""

# Create tarball
read -p "압축 파일을 생성하시겠습니까? (y/n) " -n 1 -r
echo
if [[ $REPLY =~ ^[Yy]$ ]]; then
    TARBALL="$AIRUN_ROOT/airun-offline-$(date +%Y%m%d).tar.gz"
    show_info "압축 파일 생성 중: $(basename $TARBALL)"
    tar -czf "$TARBALL" -C "$(dirname $OFFLINE_DIR)" "$(basename $OFFLINE_DIR)"
    TARBALL_SIZE=$(du -sh "$TARBALL" | cut -f1)
    show_success "압축 완료: $TARBALL ($TARBALL_SIZE)"
fi

#############################################
# Final Summary
#############################################
echo ""
echo -e "${GREEN}╔═══════════════════════════════════════════════════════════╗${NC}"
echo -e "${GREEN}║                                                           ║${NC}"
echo -e "${GREEN}║           오프라인 패키지 준비 완료! 🎉                  ║${NC}"
echo -e "${GREEN}║                                                           ║${NC}"
echo -e "${GREEN}╚═══════════════════════════════════════════════════════════╝${NC}"
echo ""
echo -e "${CYAN}📁 패키지 위치:${NC} $OFFLINE_DIR"
if [ -f "$TARBALL" ]; then
    echo -e "${CYAN}📦 압축 파일:${NC} $TARBALL"
fi
echo ""
echo -e "${YELLOW}다음 단계:${NC}"
echo "1. 오프라인 시스템으로 패키지 전송"
echo "2. 압축 해제 (필요시): tar -xzf airun-offline-*.tar.gz"
echo "3. 설치 실행: sudo ./setup.sh"
echo "4. 상세 가이드: docs/OFFLINE_INSTALLATION.md 참조"
echo ""
