﻿# RAG 기능에 필요한 Python 패키지들을 설치하는 PowerShell 스크립트

# Set output encoding to UTF-8
[Console]::OutputEncoding = [System.Text.Encoding]::UTF8
$env:PYTHONIOENCODING = "utf-8"

# PowerShell 프로필 설정 함수
function Setup-PowerShellProfile {
    Write-ColorOutput "Setting up PowerShell profile for UTF-8 encoding..." "INFO"
    
    # Create PowerShell profile if it doesn't exist
    if (-not (Test-Path $PROFILE)) {
        Write-ColorOutput "Creating PowerShell profile..." "INFO"
        New-Item -Path $PROFILE -Type File -Force | Out-Null
    }
    
    # Check if encoding setting already exists in profile
    $profileContent = Get-Content $PROFILE -ErrorAction SilentlyContinue
    $encodingCommand = '$OutputEncoding = [Console]::OutputEncoding = [System.Text.Encoding]::UTF8'
    
    if ($profileContent -notcontains $encodingCommand) {
        Write-ColorOutput "Adding UTF-8 encoding configuration to PowerShell profile..." "INFO"
        Add-Content -Path $PROFILE -Value "`n# Set console output encoding to UTF-8"
        Add-Content -Path $PROFILE -Value $encodingCommand
        Write-ColorOutput "PowerShell profile updated successfully" "INFO"
        Write-ColorOutput "Note: You'll need to restart PowerShell for the changes to take effect" "INFO"
    } else {
        Write-ColorOutput "UTF-8 encoding configuration already exists in PowerShell profile" "INFO"
    }
}

# 로그 함수
function Write-ColorOutput {
    param(
        [Parameter(Mandatory = $true)]
        [string]$Message,
        [Parameter(Mandatory = $true)]
        [string]$Type
    )
    
    $Color = switch ($Type) {
        "INFO" { "Green" }
        "WARN" { "Yellow" }
        "ERROR" { "Red" }
    }
    
    Write-Host "[$Type] " -ForegroundColor $Color -NoNewline
    Write-Host $Message
}

# Visual C++ Build Tools check
function Check-BuildTools {
    Write-ColorOutput "Checking Visual C++ Build Tools..." "INFO"
    
    # Check VS Build Tools installation
    $vsWhere = "${env:ProgramFiles(x86)}\Microsoft Visual Studio\Installer\vswhere.exe"
    if (-not (Test-Path $vsWhere)) {
        Write-ColorOutput "Visual C++ Build Tools are not installed." "ERROR"
        Write-ColorOutput "Please install 'Microsoft C++ Build Tools' from:" "INFO"
        Write-ColorOutput "https://visualstudio.microsoft.com/visual-cpp-build-tools/" "INFO"
        Write-ColorOutput "During installation, select 'Desktop development with C++'" "INFO"
        Write-ColorOutput "After installation is complete, run this script again." "INFO"
        exit 1
    }
    
    # Check MSVC compiler
    $hasMSVC = $false
    & $vsWhere -latest -products * -requires Microsoft.VisualStudio.Component.VC.Tools.x86.x64 -property installationPath | ForEach-Object {
        if (Test-Path "$_\VC\Tools\MSVC") {
            $hasMSVC = $true
        }
    }
    
    if (-not $hasMSVC) {
        Write-ColorOutput "Visual C++ Build Tools are not properly installed." "ERROR"
        Write-ColorOutput "Please run Visual Studio Installer and install 'Desktop development with C++'" "INFO"
        exit 1
    }
    
    Write-ColorOutput "Visual C++ Build Tools are properly installed." "INFO"
}

# Python 버전 체크 함수
function Check-PythonVersion {
    Write-ColorOutput "Checking Python version..." "INFO"
    
    $venvPath = "$env:USERPROFILE\.airun_venv"
    $pythonPath = "$venvPath\Scripts\python.exe"

    # Python 실행 파일 존재 여부 확인
    if (-not (Test-Path $pythonPath)) {
        Write-ColorOutput "Python executable not found at: $pythonPath" "ERROR"
        exit 1
    }
    
    # 가상환경의 Python 버전 확인
    try {
        $pythonVersion = & "$pythonPath" --version 2>&1
        if ($pythonVersion -match 'Python (\d+\.\d+\.\d+)') {
            $version = $matches[1]
            Write-ColorOutput "Detected Python version: $version" "INFO"
            
            if ($version -match '^3\.(8|9|10|11)\.') {
                Write-ColorOutput "Python version is compatible with PyTorch" "INFO"
            } else {
                Write-ColorOutput "Warning: Your Python version ($version) might not be compatible with PyTorch" "WARN"
                Write-ColorOutput "PyTorch supports Python versions 3.8 to 3.11" "WARN"
                Write-ColorOutput "Consider using Python 3.10 for best compatibility" "WARN"
                
                $continue = Read-Host "Do you want to continue anyway? (y/N)"
                if ($continue -ne "y" -and $continue -ne "Y") {
                    Write-ColorOutput "Installation aborted. Please install a compatible Python version (3.8-3.11)" "ERROR"
                    exit 1
                }
            }
        } else {
            Write-ColorOutput "Failed to parse Python version: $pythonVersion" "ERROR"
            exit 1
        }
    }
    catch {
        Write-ColorOutput "Error checking Python version: $_" "ERROR"
        exit 1
    }
}

# Clean up numpy package function
function Clean-NumpyPackage {
    param(
        [string]$VenvPath
    )
    
    Write-ColorOutput "Cleaning up numpy package..." "INFO"
    
    # 가상환경 비활성화
    if (Test-Path Function:\deactivate) {
        deactivate
    }
    
    # site-packages 디렉토리 정리
    $sitePackagesPath = "$VenvPath\Lib\site-packages"
    
    # pip로 numpy 제거 (가상환경의 pip 직접 사용)
    Write-ColorOutput "Uninstalling numpy..." "INFO"
    & "$VenvPath\Scripts\pip.exe" uninstall numpy -y
    Start-Sleep -Seconds 2  # 파일 시스템 작업이 완료될 때까지 대기
    
    try {
        # numpy 패키지 디렉토리만 제거
        $numpyPaths = @(
            "$sitePackagesPath\numpy",
            "$sitePackagesPath\numpy-*.dist-info",
            "$sitePackagesPath\numpy-*.egg-info",
            "$sitePackagesPath\-umpy"
        )
        
        foreach ($path in $numpyPaths) {
            if (Test-Path $path) {
                Write-ColorOutput "Removing $path" "INFO"
                Remove-Item -Path $path -Force -Recurse -ErrorAction SilentlyContinue
            }
        }
        
        # -umpy 폴더가 여전히 존재하면 cmd.exe로 제거 시도
        if (Test-Path "$sitePackagesPath\-umpy") {
            Write-ColorOutput "Attempting to remove -umpy directory using cmd..." "INFO"
            cmd /c "rd /s /q `"$sitePackagesPath\-umpy`""
        }
    }
    catch {
        Write-ColorOutput "Warning: Some files could not be removed. Continuing anyway..." "WARN"
    }
    
    # 가상환경 재활성화
    Write-ColorOutput "Reactivating virtual environment..." "INFO"
    & "$VenvPath\Scripts\Activate.ps1"
    
    # numpy 재설치
    Write-ColorOutput "Reinstalling numpy..." "INFO"
    & "$VenvPath\Scripts\pip.exe" install numpy==1.24.4 --no-cache-dir --no-deps
    if ($LASTEXITCODE -ne 0) {
        Write-ColorOutput "Failed to install numpy" "ERROR"
        exit 1
    }
    
    # 의존성 패키지 설치
    Write-ColorOutput "Installing numpy dependencies..." "INFO"
    & "$VenvPath\Scripts\pip.exe" install --no-cache-dir --upgrade pip setuptools wheel
}

# Clean up broken packages function
function Clean-BrokenPackages {
    param(
        [string]$VenvPath
    )
    
    Write-ColorOutput "Cleaning up broken packages..." "INFO"
    
    $sitePackagesPath = "$VenvPath\Lib\site-packages"
    
    # 가상환경 비활성화
    if (Test-Path Function:\deactivate) {
        deactivate
    }
    
    # 손상된 패키지들 제거
    $brokenPackages = @(
        "Pillow",
        "numpy",
        "torch",
        "transformers"
    )
    
    foreach ($package in $brokenPackages) {
        Write-ColorOutput "Uninstalling $package..." "INFO"
        & "$VenvPath\Scripts\pip.exe" uninstall -y $package 2>$null
    }
    
    # 손상된 디렉토리 패턴들
    $brokenPatterns = @(
        "-illow*",
        "-umpy*",
        "-orch*",
        "-ransformers*"
    )
    
    # 손상된 디렉토리 강제 삭제
    foreach ($pattern in $brokenPatterns) {
        $paths = Get-ChildItem -Path $sitePackagesPath -Directory -Filter $pattern -ErrorAction SilentlyContinue
        foreach ($path in $paths) {
            Write-ColorOutput "Removing broken directory: $($path.FullName)" "INFO"
            try {
                # 먼저 PowerShell로 삭제 시도
                Remove-Item -Path $path.FullName -Force -Recurse -ErrorAction Stop
            } catch {
                # PowerShell 삭제 실패시 cmd.exe로 시도
                Write-ColorOutput "Attempting to remove using cmd..." "INFO"
                cmd /c "rd /s /q `"$($path.FullName)`"" 2>$null
                
                # 여전히 존재하는지 확인
                if (Test-Path $path.FullName) {
                    # 파일 시스템 권한으로 시도
                    Write-ColorOutput "Attempting to remove with takeown..." "INFO"
                    cmd /c "takeown /f `"$($path.FullName)`" /r /d y" 2>$null
                    cmd /c "icacls `"$($path.FullName)`" /grant administrators:F /t" 2>$null
                    cmd /c "rd /s /q `"$($path.FullName)`"" 2>$null
                }
            }
        }
    }
    
    # dist-info 및 egg-info 파일들도 정리
    Get-ChildItem -Path $sitePackagesPath -Include "*.dist-info","*.egg-info" -Directory | 
    Where-Object { $_.Name -match "^-" } | 
    ForEach-Object {
        Write-ColorOutput "Removing broken info directory: $($_.FullName)" "INFO"
        try {
            Remove-Item -Path $_.FullName -Force -Recurse -ErrorAction Stop
        } catch {
            cmd /c "rd /s /q `"$($_.FullName)`"" 2>$null
        }
    }
    
    # 가상환경 재활성화
    & "$VenvPath\Scripts\Activate.ps1"
}

# 가상환경 재생성 함수
function Recreate-VirtualEnvironment {
    param(
        [string]$VenvPath
    )
    
    Write-ColorOutput "Current virtual environment seems to have issues." "WARN"
    Write-ColorOutput "Would you like to recreate the virtual environment? This will remove all installed packages." "WARN"
    $answer = Read-Host "Enter 'y' to continue, any other key to skip"
    
    if ($answer -ne "y") {
        Write-ColorOutput "Skipping virtual environment recreation" "INFO"
        return $false
    }
    
    Write-ColorOutput "Recreating virtual environment..." "INFO"
    
    # 가상환경 비활성화
    if (Test-Path Function:\deactivate) {
        deactivate
    }
    
    # 기존 가상환경 제거
    if (Test-Path $VenvPath) {
        Write-ColorOutput "Removing existing virtual environment..." "INFO"
        Remove-Item -Force -Recurse $VenvPath
    }
    
    # 새 가상환경 생성
    Write-ColorOutput "Creating new virtual environment..." "INFO"
    python -m venv $VenvPath
    if ($LASTEXITCODE -ne 0) {
        Write-ColorOutput "Failed to create virtual environment" "ERROR"
        exit 1
    }
    
    # 가상환경 활성화
    Write-ColorOutput "Activating new virtual environment..." "INFO"
    & "$VenvPath\Scripts\Activate.ps1"
    if ($LASTEXITCODE -ne 0) {
        Write-ColorOutput "Failed to activate virtual environment" "ERROR"
        exit 1
    }
    
    # pip 업그레이드
    Write-ColorOutput "Upgrading pip..." "INFO"
    python -m pip install --upgrade pip --no-cache-dir
    
    return $true
}

# pip 업그레이드 함수 추가
function Upgrade-Pip {
    param(
        [string]$VenvPath
    )
    
    Write-ColorOutput "Upgrading pip..." "INFO"
    
    # Python 실행 파일 경로
    $pythonPath = "$VenvPath\Scripts\python.exe"
    
    # pip 업그레이드 명령 실행
    & $pythonPath -m pip install --no-cache-dir --upgrade pip setuptools wheel --quiet
    if ($LASTEXITCODE -ne 0) {
        Write-ColorOutput "Warning: Failed to upgrade pip" "WARN"
        return $false
    }
    return $true
}

# Python package installation
function Install-PythonPackages {
    Write-ColorOutput "Starting RAG-related Python package installation..." "INFO"
    
    # Virtual environment path
    $VENV_PATH = "$env:USERPROFILE\.airun_venv"
    
    # Check virtual environment
    if (-not (Test-Path $VENV_PATH)) {
        Write-ColorOutput "AI.RUN virtual environment ($VENV_PATH) not found." "ERROR"
        Write-ColorOutput "Please run 'airun' command first to create the virtual environment." "INFO"
        exit 1
    }

    # 가상환경 활성화
    Write-ColorOutput "Activating virtual environment..." "INFO"
    & "$VENV_PATH\Scripts\Activate.ps1"
    if ($LASTEXITCODE -ne 0) {
        Write-ColorOutput "Failed to activate virtual environment" "ERROR"
        exit 1
    }

    # pip 업그레이드
    Write-ColorOutput "Upgrading pip..." "INFO"
    & "$VENV_PATH\Scripts\python.exe" -m pip install --no-cache-dir --upgrade pip setuptools wheel
    
    # 기존 패키지 정리
    Clean-BrokenPackages -VenvPath $VENV_PATH
    
    # numpy 특별 처리
    Write-ColorOutput "Installing numpy..." "INFO"
    & "$VENV_PATH\Scripts\pip.exe" install --no-cache-dir --upgrade "numpy==1.24.4"
    
    # PyTorch 설치 (다른 패키지들의 의존성)
    Write-ColorOutput "Installing PyTorch..." "INFO"
    & "$VENV_PATH\Scripts\pip.exe" install --no-cache-dir --upgrade torch==2.5.1 torchvision torchaudio

    # 특별 처리가 필요한 패키지들 먼저 설치
    $specialPackages = @(
        "numpy==1.24.4",
        "wheel",
        "setuptools",
        "--only-binary :all: langdetect"  # wheel 빌드 없이 바이너리로만 설치
    )

    foreach ($package in $specialPackages) {
        Write-ColorOutput "Installing special package: $package" "INFO"
        & "$VENV_PATH\Scripts\pip.exe" install --no-cache-dir $package
        if ($LASTEXITCODE -ne 0) {
            Write-ColorOutput "Warning: Failed to install $package" "WARN"
        }
    }

    # 나머지 패키지들 업그레이드 방식으로 설치
    $packages = @(
        "transformers>=4.41.0,<5.0.0",
        "sentence-transformers",
        "Pillow>=10.0.0,<11.0.0",
        "scikit-image>=0.25.0",
        "matplotlib>=3.4,!=3.6.1",
        "seaborn>=0.13.2",
        "pandas>=2.0.0",
        "PyPDF2>=3.0.0",
        "python-docx>=0.8.11",
        "python-pptx>=0.6.21",
        "openpyxl>=3.1.0",
        "beautifulsoup4>=4.12.0",
        "markdown>=3.4.0",
        "python-magic-bin>=0.4.14",
        "chardet>=5.0.0",
        "langchain==0.3.14",
        "onnxruntime>=1.15.0",
        "imageio>=2.31.0",
        "scikit-learn>=1.3.0",
        "tqdm>=4.65.0",
        "requests>=2.31.0",
        "typing-extensions>=4.5.0",
        "urllib3>=2.0.0",
        "regex>=2023.0",
        "joblib>=1.3.0",
        "tokenizers>=0.15.0",
        "filelock>=3.12.0",
        "packaging>=23.0",
        "huggingface-hub>=0.19.0",
        "safetensors>=0.4.0",
        # "chromadb>=0.4.24",  # Replaced with PostgreSQL
        "langchain-community>=0.0.1",
        "langchain-huggingface",
        "pyhwp",
        "lxml",
        "selenium",
        "webdriver_manager",
        "cairosvg",
        "trafilatura",
        "fpdf",
        "reportlab",
        "olefile"
    )

    foreach ($package in $packages) {
        Write-ColorOutput "Installing/Upgrading package: $package" "INFO"
        & "$VENV_PATH\Scripts\pip.exe" install --no-cache-dir --upgrade $package
        if ($LASTEXITCODE -ne 0) {
            Write-ColorOutput "Warning: Failed to install/upgrade $package" "WARN"
            # 계속 진행 (치명적 오류가 아닌 경우)
        }
    }

    # 의존성 충돌 해결을 위한 최종 업데이트
    Write-ColorOutput "Resolving dependencies..." "INFO"
    & "$VENV_PATH\Scripts\pip.exe" install --no-cache-dir --upgrade pip setuptools wheel

    Write-ColorOutput "RAG package installation completed" "INFO"

    # Set up transformers cache directory
    $TRANSFORMERS_CACHE_DIR = "$env:APPDATA\npm\node_modules\airun\node_modules\@xenova\transformers\.cache"
    if (Test-Path "$env:APPDATA\npm\node_modules\airun") {
        Write-ColorOutput "Global installation detected, creating transformers cache directory..." "INFO"
        New-Item -ItemType Directory -Force -Path $TRANSFORMERS_CACHE_DIR | Out-Null
        Write-ColorOutput "Cache directory creation completed" "INFO"
    }

    # Deactivate virtual environment
    deactivate
}

# Main execution
Write-ColorOutput "Starting RAG system setup" "INFO"
Setup-PowerShellProfile
Check-BuildTools
Check-PythonVersion
Install-PythonPackages
Write-ColorOutput "RAG system setup completed" "INFO"
Write-ColorOutput "You can now use 'airun rag' command to activate RAG functionality" "INFO"
Write-ColorOutput "Please restart PowerShell to apply encoding settings" "INFO" 