/**
 * RAG Agent
 *
 * 내부 문서 검색을 전담하는 에이전트
 * - RAG 서비스 연동 (port 5600)
 * - 쿼리 최적화
 * - 결과 포맷팅
 *
 * @module services/agent-system/agents/rag-agent
 */

import axios from 'axios';
import { logger } from '../../../utils/logger.js';

class RAGAgent {
  /**
   * @param {Object} options - 에이전트 옵션
   * @param {string} options.ragServerUrl - RAG 서버 URL
   * @param {number} options.topK - 반환할 최대 결과 수
   * @param {number} options.timeout - 요청 타임아웃 (ms)
   */
  constructor(options = {}) {
    this.ragServerUrl =
      options.ragServerUrl ||
      process.env.RAG_SERVER_URL ||
      'http://localhost:5600';
    this.topK = options.topK || 5;
    this.timeout = options.timeout || 30000;

    // Axios 클라이언트 생성
    this.client = axios.create({
      baseURL: this.ragServerUrl,
      timeout: this.timeout,
      headers: {
        'Content-Type': 'application/json',
      },
    });

    logger.info(`[RAGAgent] 초기화 완료: ${this.ragServerUrl}`);
  }

  /**
   * RAG 검색 실행
   *
   * @param {Object} context - 실행 컨텍스트
   * @param {string} context.query - 검색 쿼리
   * @param {string} context.username - 사용자명
   * @param {Object} context.options - 추가 옵션
   * @returns {Promise<Object>} 검색 결과
   */
  async execute(context) {
    try {
      const { query, username, options = {} } = context;

      logger.info(`[RAGAgent] 검색 시작: query="${query}", user="${username}"`);

      // RAG 서버에 검색 요청
      const response = await this.client.post('/search', {
        query,
        top_k: options.top_k || this.topK,
        user_id: username || 'admin',
      });

      if (!response.data || !response.data.results) {
        throw new Error('RAG 서버 응답 형식 오류');
      }

      const results = response.data.results;

      logger.info(`[RAGAgent] 검색 완료: ${results.length}개 문서 발견`);

      // 결과 포맷팅
      const formattedResults = this._formatResults(results);

      return {
        success: true,
        results: formattedResults,
        metadata: {
          source: 'internal_documents',
          totalResults: results.length,
          query: query,
          username: username,
        },
      };
    } catch (error) {
      logger.error(`[RAGAgent] 검색 실패: ${error.message}`);

      return {
        success: false,
        error: error.message,
        results: [],
        metadata: {
          source: 'internal_documents',
          error: error.message,
        },
      };
    }
  }

  /**
   * 결과 포맷팅
   *
   * @private
   * @param {Array} rawResults - RAG 서버 원본 결과
   * @returns {Array} 포맷팅된 결과
   */
  _formatResults(rawResults) {
    return rawResults.map((result, index) => ({
      rank: index + 1,
      content: result.content || result.text || '',
      metadata: {
        filename: result.metadata?.filename || result.metadata?.source || '알 수 없음',
        chunk_index: result.metadata?.chunk_index || 0,
        page: result.metadata?.page || result.metadata?.page_number || null,
      },
      scores: {
        keyword_precision: result.scores?.keyword_precision || 0,
        semantic_relevance: result.scores?.semantic_relevance || 0,
        content_quality: result.scores?.content_quality || 0,
        recency_score: result.scores?.recency_score || 0,
      },
    }));
  }

  /**
   * RAG 서버 상태 확인
   *
   * @returns {Promise<Object>} 서버 상태
   */
  async checkHealth() {
    try {
      const response = await this.client.get('/health');

      return {
        success: true,
        status: response.data.status || 'healthy',
        serverUrl: this.ragServerUrl,
      };
    } catch (error) {
      logger.error(`[RAGAgent] 상태 확인 실패: ${error.message}`);

      return {
        success: false,
        status: 'unhealthy',
        error: error.message,
        serverUrl: this.ragServerUrl,
      };
    }
  }

  /**
   * 사용자 문서 목록 조회
   *
   * @param {string} username - 사용자명
   * @returns {Promise<Array>} 문서 목록
   */
  async listDocuments(username) {
    try {
      const response = await this.client.get('/list', {
        params: { user_id: username || 'admin' },
      });

      return {
        success: true,
        documents: response.data.documents || [],
      };
    } catch (error) {
      logger.error(`[RAGAgent] 문서 목록 조회 실패: ${error.message}`);

      return {
        success: false,
        error: error.message,
        documents: [],
      };
    }
  }
}

export default RAGAgent;
