/**
 * Web Agent
 *
 * 웹 검색을 전담하는 에이전트
 * - Web Search 서비스 연동 (port 5610)
 * - 검색 엔진 선택 로직
 * - 결과 필터링
 *
 * @module services/agent-system/agents/web-agent
 */

import axios from 'axios';
import { logger } from '../../../utils/logger.js';

class WebAgent {
  /**
   * @param {Object} options - 에이전트 옵션
   * @param {string} options.webSearchUrl - Web Search 서버 URL
   * @param {number} options.numResults - 반환할 최대 결과 수
   * @param {number} options.timeout - 요청 타임아웃 (ms)
   * @param {string} options.defaultEngine - 기본 검색 엔진
   */
  constructor(options = {}) {
    this.webSearchUrl =
      options.webSearchUrl ||
      process.env.WEB_SEARCH_URL ||
      'http://localhost:5610';
    this.numResults = options.numResults || 10;
    this.timeout = options.timeout || 30000;
    this.defaultEngine = options.defaultEngine || 'google';

    // Axios 클라이언트 생성
    this.client = axios.create({
      baseURL: this.webSearchUrl,
      timeout: this.timeout,
      headers: {
        'Content-Type': 'application/json',
      },
    });

    logger.info(`[WebAgent] 초기화 완료: ${this.webSearchUrl}`);
  }

  /**
   * 웹 검색 실행
   *
   * @param {Object} context - 실행 컨텍스트
   * @param {string} context.query - 검색 쿼리
   * @param {Object} context.options - 추가 옵션
   * @returns {Promise<Object>} 검색 결과
   */
  async execute(context) {
    try {
      const { query, options = {} } = context;

      logger.info(`[WebAgent] 검색 시작: query="${query}"`);

      // 검색 엔진 선택
      const engine = options.engine || this.defaultEngine;

      // Web Search 서버에 검색 요청
      const response = await this.client.post('/search', {
        query,
        num_results: options.num_results || this.numResults,
        engine: engine,
      });

      if (!response.data || !response.data.results) {
        throw new Error('Web Search 서버 응답 형식 오류');
      }

      const results = response.data.results;

      logger.info(`[WebAgent] 검색 완료: ${results.length}개 결과 발견`);

      // 결과 포맷팅 및 필터링
      const formattedResults = this._formatResults(results);
      const filteredResults = this._filterResults(formattedResults, query);

      return {
        success: true,
        results: filteredResults,
        metadata: {
          source: 'web_search',
          engine: engine,
          totalResults: filteredResults.length,
          query: query,
        },
      };
    } catch (error) {
      logger.error(`[WebAgent] 검색 실패: ${error.message}`);

      return {
        success: false,
        error: error.message,
        results: [],
        metadata: {
          source: 'web_search',
          error: error.message,
        },
      };
    }
  }

  /**
   * 결과 포맷팅
   *
   * @private
   * @param {Array} rawResults - Web Search 서버 원본 결과
   * @returns {Array} 포맷팅된 결과
   */
  _formatResults(rawResults) {
    return rawResults.map((result, index) => ({
      rank: index + 1,
      title: result.title || '제목 없음',
      url: result.url || result.link || '',
      snippet: result.snippet || result.description || '',
      publishedDate: result.published_date || result.date || null,
      relevanceScore: result.relevance_score || 0,
    }));
  }

  /**
   * 결과 필터링 (품질 기준)
   *
   * @private
   * @param {Array} results - 포맷팅된 결과
   * @param {string} query - 검색 쿼리
   * @returns {Array} 필터링된 결과
   */
  _filterResults(results, query) {
    return results.filter((result) => {
      // 1. URL이 있어야 함
      if (!result.url) {
        return false;
      }

      // 2. 제목 또는 snippet에 최소한의 내용이 있어야 함
      const hasContent =
        (result.title && result.title.length > 5) ||
        (result.snippet && result.snippet.length > 20);

      if (!hasContent) {
        return false;
      }

      // 3. 스팸/광고성 필터링 (간단한 휴리스틱)
      const spamKeywords = ['광고', '홍보', '이벤트', 'AD', 'sponsored'];
      const isSpam = spamKeywords.some(
        (keyword) =>
          result.title.includes(keyword) || result.snippet.includes(keyword)
      );

      if (isSpam) {
        logger.debug(`[WebAgent] 스팸 필터링: ${result.title}`);
        return false;
      }

      return true;
    });
  }

  /**
   * Web Search 서버 상태 확인
   *
   * @returns {Promise<Object>} 서버 상태
   */
  async checkHealth() {
    try {
      const response = await this.client.get('/health');

      return {
        success: true,
        status: response.data.status || 'healthy',
        serverUrl: this.webSearchUrl,
      };
    } catch (error) {
      logger.error(`[WebAgent] 상태 확인 실패: ${error.message}`);

      return {
        success: false,
        status: 'unhealthy',
        error: error.message,
        serverUrl: this.webSearchUrl,
      };
    }
  }

  /**
   * 검색 엔진 선택 로직
   *
   * @param {string} query - 검색 쿼리
   * @param {Object} context - 컨텍스트
   * @returns {string} 선택된 검색 엔진
   */
  selectEngine(query, context = {}) {
    // 간단한 휴리스틱 기반 엔진 선택
    const lowerQuery = query.toLowerCase();

    // 뉴스 관련 검색
    if (
      lowerQuery.includes('뉴스') ||
      lowerQuery.includes('최신') ||
      lowerQuery.includes('오늘')
    ) {
      return 'google_news';
    }

    // 학술 검색
    if (
      lowerQuery.includes('논문') ||
      lowerQuery.includes('연구') ||
      lowerQuery.includes('학술')
    ) {
      return 'google_scholar';
    }

    // 기본: Google
    return 'google';
  }
}

export default WebAgent;
