/**
 * Document Grader for CRAG
 *
 * 문서의 관련성을 평가하고 등급을 부여하는 컴포넌트
 *
 * @module services/agent-system/crag/document-grader
 */

import { GRADE_ACTION, DOCUMENT_GRADE } from './constants.js';

class DocumentGrader {
  constructor(options = {}) {
    this.correctThreshold = options.correctThreshold || 0.7;
    this.incorrectThreshold = options.incorrectThreshold || 0.3;
    this.modelManager = options.modelManager;
  }

  /**
   * 문서들의 관련성을 평가하고 등급을 부여
   *
   * @param {string} query - 사용자 쿼리
   * @param {Array} documents - 평가할 문서들
   * @returns {Promise<Object>} 평가 결과
   */
  async gradeDocuments(query, documents) {
    try {
      const grades = [];

      // 각 문서에 대해 관련성 평가
      for (const doc of documents) {
        const relevanceScore = await this._assessRelevance(query, doc);
        const grade = this._assignGrade(relevanceScore);

        grades.push({
          document: doc,
          relevanceScore,
          grade,
          relevanceScore: relevanceScore
        });
      }

      // 통계 계산
      const correctDocs = grades.filter(g => g.grade === DOCUMENT_GRADE.CORRECT);
      const ambiguousDocs = grades.filter(g => g.grade === DOCUMENT_GRADE.AMBIGUOUS);
      const incorrectDocs = grades.filter(g => g.grade === DOCUMENT_GRADE.INCORRECT);

      const correctCount = correctDocs.length;
      const ambiguousCount = ambiguousDocs.length;
      const incorrectCount = incorrectDocs.length;

      // 액션 결정
      let action;
      if (correctCount >= 2) {
        // 충분한 관련 문서가 있는 경우
        action = {
          type: GRADE_ACTION.PROCEED,
          correctCount,
          ambiguousCount,
          incorrectCount,
          filteredDocuments: correctDocs.map(g => g.document),
          reason: '충분한 관련 문서 확보'
        };
      } else if (incorrectCount > documents.length / 2) {
        // 대부분의 문서가 관련 없는 경우
        action = {
          type: GRADE_ACTION.REWRITE,
          correctCount,
          ambiguousCount,
          incorrectCount,
          filteredDocuments: correctDocs.map(g => g.document),
          reason: '문서들의 관련성이 낮아 쿼리 재작성 필요'
        };
      } else {
        // 일부 문서만 관련 있는 경우
        action = {
          type: GRADE_ACTION.FALLBACK,
          correctCount,
          ambiguousCount,
          incorrectCount,
          filteredDocuments: correctDocs.map(g => g.document),
          reason: '관련 문서 부족하여 웹 검색 보강 필요'
        };
      }

      return {
        success: true,
        action,
        grades,
        summary: {
          total: documents.length,
          correct: correctCount,
          ambiguous: ambiguousCount,
          incorrect: incorrectCount,
          averageScore: grades.reduce((sum, g) => sum + g.relevanceScore, 0) / grades.length
        }
      };

    } catch (error) {
      return {
        success: false,
        error: error.message,
        action: {
          type: GRADE_ACTION.PROCEED,
          correctCount: 0,
          ambiguousCount: 0,
          incorrectCount: documents.length,
          filteredDocuments: documents,
          reason: '평가 실패로 원본 문서 사용'
        }
      };
    }
  }

  /**
   * 단일 문서의 관련성 평가
   *
   * @private
   * @param {string} query - 사용자 쿼리
   * @param {Object} document - 평가할 문서
   * @returns {Promise<number>} 관련성 점수 (0-1)
   */
  async _assessRelevance(query, document) {
    const content = document.content || document.text || '';

    // 간단한 키워드 기반 관련성 평가 (실제로는 AI 모델 사용 가능)
    const queryKeywords = query.toLowerCase().split(/\s+/);
    const contentLower = content.toLowerCase();

    let matchCount = 0;
    for (const keyword of queryKeywords) {
      if (contentLower.includes(keyword)) {
        matchCount++;
      }
    }

    const relevanceScore = Math.min(matchCount / queryKeywords.length, 1);

    // 문서 길이에 따른 페널티/보너스
    let lengthScore = 1;
    if (content.length < 50) {
      lengthScore = 0.5; // 너무 짧은 문서
    } else if (content.length > 2000) {
      lengthScore = 0.8; // 너무 긴 문서
    }

    return relevanceScore * lengthScore;
  }

  /**
   * 관련성 점수에 따라 등급 부여
   *
   * @private
   * @param {number} score - 관련성 점수
   * @returns {string} 문서 등급
   */
  _assignGrade(score) {
    if (score >= this.correctThreshold) {
      return DOCUMENT_GRADE.CORRECT;
    } else if (score >= this.incorrectThreshold) {
      return DOCUMENT_GRADE.AMBIGUOUS;
    } else {
      return DOCUMENT_GRADE.INCORRECT;
    }
  }

  /**
   * 평가 결과 요약 생성
   *
   * @param {Object} result - 평가 결과
   * @returns {string} 요약 텍스트
   */
  generateSummary(result) {
    const { action, summary } = result;
    return `[문서 평가] 전체: ${summary.total}, 관련: ${summary.correct}, 모호: ${summary.ambiguous}, 미관련: ${summary.incorrect}, 평균: ${(summary.averageScore * 100).toFixed(1)}% - 액션: ${action.type}`;
  }
}

export default DocumentGrader;