/**
 * Query Rewriter for CRAG
 *
 * 검색 쿼리를 개선하고 재작성하는 컴포넌트
 *
 * @module services/agent-system/crag/query-rewriter
 */

import ModelManager from '../ollama/model-manager.js';

class QueryRewriter {
  constructor(options = {}) {
    this.modelManager = new ModelManager({
      ollamaUrl: options.ollamaUrl || process.env.OLLAMA_PROXY_SERVER || 'https://api.hamonize.com/ollama',
      defaultModel: 'airun-chat',
      fallbackModels: ['hamonize:latest', 'airun-chat']
    });
  }

  /**
   * 쿼리 재작성
   *
   * @param {string} originalQuery - 원본 쿼리
   * @param {Object} context - 재작성 컨텍스트
   * @param {Object} context.gradeResult - 문서 평가 결과
   * @param {string} context.failureReason - 실패 이유
   * @returns {Promise<Object>} 재작성 결과
   */
  async rewriteQuery(originalQuery, context = {}) {
    try {
      const { gradeResult, failureReason } = context;

      const rewritePrompt = `
원본 질문: "${originalQuery}"

문서 평가 결과:
- 관련 문서: ${gradeResult?.action?.correctCount || 0}개
- 모호 문서: ${gradeResult?.action?.ambiguousCount || 0}개
- 미관련 문서: ${gradeResult?.action?.incorrectCount || 0}개
- 실패 이유: ${failureReason || '관련 문서 부족'}

쿼리 재작성 지침:
1. 원본 의도를 유지하면서 더 구체적인 키워드 추가
2. 관련 문서를 더 잘 찾을 수 있도록 검색어 최적화
3. 동의어나 유사 표현을 포함하여 검색 범위 확장
4. 질문의 핵심 개념을 명확하게 표현

아래 JSON 형식으로 응답해주세요:
{
  "rewrittenQueries": {
    "rag": "RAG 검색용 재작성된 쿼리",
    "web": "웹 검색용 재작성된 쿼리"
  },
  "reasoning": "재작성 이유와 전략",
  "improvements": ["개선된 부분들"]
}`;

      const response = await this.modelManager.chatWithJSON(
        this.modelManager.selectModel('queryRewrite'),
        [{ role: 'user', content: rewritePrompt }],
        {
          rewrittenQueries: {
            rag: 'string',
            web: 'string'
          },
          reasoning: 'string',
          improvements: 'array'
        }
      );

      const result = response.content;

      return {
        success: true,
        rewrittenQueries: result.rewrittenQueries,
        reasoning: result.reasoning,
        improvements: result.improvements,
        originalQuery
      };

    } catch (error) {
      // 실패 시 원본 쿼리 반환
      return {
        success: false,
        rewrittenQueries: {
          rag: originalQuery,
          web: originalQuery
        },
        reasoning: `재작성 실패: ${error.message}`,
        improvements: [],
        originalQuery,
        error: error.message
      };
    }
  }

  /**
   * 쿼리 확장 (동의어 추가)
   *
   * @param {string} query - 원본 쿼리
   * @returns {Promise<Array>} 확장된 쿼리 배열
   */
  async expandQuery(query) {
    try {
      const expandPrompt = `
다음 쿼리에 대한 동의어나 유사 표현을 생성해주세요:

원본 쿼리: "${query}"

요구사항:
1. 원본 의미를 유지
2. 다양한 표현 방식 제안
3. 기술 용어와 일반 용어 모두 포함
4. 최대 5개의 확장 쿼리 생성

JSON 형식으로 응답해주세요:
{
  "expandedQueries": ["확장된 쿼리 1", "확장된 쿼리 2", ...],
  "synonyms": ["동의어 1", "동의어 2", ...],
  "reasoning": "확장 전략"
}`;

      const response = await this.modelManager.chatWithJSON(
        this.modelManager.selectModel('queryExpand'),
        [{ role: 'user', content: expandPrompt }],
        {
          expandedQueries: 'array',
          synonyms: 'array',
          reasoning: 'string'
        }
      );

      return response.content.expandedQueries;

    } catch (error) {
      // 실패 시 원본 쿼리만 반환
      return [query];
    }
  }

  /**
   * 재작성 결과 요약 생성
   *
   * @param {Object} result - 재작성 결과
   * @returns {string} 요약 텍스트
   */
  generateSummary(result) {
    if (!result.success) {
      return `[쿼리 재작성] 실패: ${result.reasoning}`;
    }

    const { rag, web } = result.rewrittenQueries;
    const improvements = result.improvements?.length || 0;

    return `[쿼리 재작성] 성공 - RAG: "${rag.substring(0, 30)}...", Web: "${web.substring(0, 30)}...", 개선: ${improvements}개 항목`;
  }
}

export default QueryRewriter;