/**
 * Document Grader
 *
 * RAG 검색 결과의 개별 문서에 대한 관련성을 평가하는 모듈
 * Corrective RAG (CRAG) 패턴의 핵심 컴포넌트
 *
 * @module services/agent-system/graders/document-grader
 */

import ModelManager from '../ollama/model-manager.js';
import { logger } from '../../../utils/logger.js';

// 문서 등급 상수
const GRADE = {
  CORRECT: 'CORRECT',       // 관련성 높음 (≥0.7)
  AMBIGUOUS: 'AMBIGUOUS',   // 관련성 불확실 (0.3-0.7)
  INCORRECT: 'INCORRECT'    // 관련성 없음 (<0.3)
};

// 액션 타입 상수
const ACTION = {
  PROCEED: 'PROCEED',       // 결과 사용 진행
  REWRITE: 'REWRITE',       // 쿼리 재작성 후 재검색
  FALLBACK: 'FALLBACK'      // 웹 검색으로 대체
};

class DocumentGrader {
  /**
   * @param {Object} options - 그레이더 옵션
   * @param {Object} options.modelManager - 기존 ModelManager 인스턴스 (선택 사항)
   * @param {string} options.ollamaUrl - Ollama 서버 URL (modelManager 없을 때만 사용)
   * @param {number} options.correctThreshold - CORRECT 판정 임계값 (기본: 0.7)
   * @param {number} options.incorrectThreshold - INCORRECT 판정 임계값 (기본: 0.3)
   * @param {number} options.minCorrectRatio - 진행에 필요한 최소 CORRECT 비율 (기본: 0.5)
   */
  constructor(options = {}) {
    // 기존 ModelManager를 재사용하거나 새로 생성
    if (options.modelManager) {
      this.modelManager = options.modelManager;
      logger.info('[DocumentGrader] 기존 ModelManager 재사용');
    } else {
      this.modelManager = new ModelManager({
        ollamaUrl: options.ollamaUrl || process.env.OLLAMA_PROXY_SERVER,
        defaultModel: 'hamonize:latest',
      });
      logger.info('[DocumentGrader] 새로운 ModelManager 생성');
    }

    this.correctThreshold = options.correctThreshold || 0.7;
    this.incorrectThreshold = options.incorrectThreshold || 0.3;
    this.minCorrectRatio = options.minCorrectRatio || 0.5;

    logger.info('[DocumentGrader] 초기화 완료');
  }

  /**
   * 문서 배열에 대한 일괄 평가
   *
   * @param {string} query - 사용자 쿼리
   * @param {Array} documents - 평가할 문서 배열
   * @returns {Promise<Object>} 평가 결과 및 권장 액션
   */
  async gradeDocuments(query, documents) {
    try {
      if (!documents || documents.length === 0) {
        logger.warn('[DocumentGrader] 평가할 문서가 없음');
        return {
          gradedDocuments: [],
          action: {
            type: ACTION.FALLBACK,
            reason: '검색 결과 없음',
            correctCount: 0,
            ambiguousCount: 0,
            incorrectCount: 0
          },
          summary: {
            totalDocuments: 0,
            avgScore: 0,
            processingTime: 0
          }
        };
      }

      const startTime = Date.now();
      logger.info(`[DocumentGrader] ${documents.length}개 문서 평가 시작`);

      // 병렬로 모든 문서 평가
      const gradePromises = documents.map((doc, index) =>
        this._gradeDocument(query, doc, index)
      );

      const gradedDocuments = await Promise.all(gradePromises);

      // 등급별 카운트
      const counts = this._countGrades(gradedDocuments);

      // 평균 점수 계산
      const avgScore = gradedDocuments.reduce((sum, d) => sum + d.score, 0) / gradedDocuments.length;

      // 액션 결정
      const action = this._determineAction(gradedDocuments, counts);

      const processingTime = Date.now() - startTime;

      logger.info(`[DocumentGrader] 평가 완료: CORRECT=${counts.correct}, AMBIGUOUS=${counts.ambiguous}, INCORRECT=${counts.incorrect}, 액션=${action.type}`);

      return {
        gradedDocuments,
        action,
        summary: {
          totalDocuments: documents.length,
          avgScore: Math.round(avgScore * 100) / 100,
          processingTime
        }
      };

    } catch (error) {
      logger.error(`[DocumentGrader] 문서 평가 실패: ${error.message}`);
      throw error;
    }
  }

  /**
   * 개별 문서 평가
   *
   * @private
   * @param {string} query - 사용자 쿼리
   * @param {Object} document - 평가할 문서
   * @param {number} index - 문서 인덱스
   * @returns {Promise<Object>} 평가 결과
   */
  async _gradeDocument(query, document, index) {
    try {
      const content = document.content || document.text || '';
      const metadata = document.metadata || {};

      // 문서 내용이 너무 짧은 경우
      if (content.length < 20) {
        return {
          index,
          document,
          grade: GRADE.INCORRECT,
          score: 0.1,
          reasoning: '문서 내용이 너무 짧아 평가 불가'
        };
      }

      // LLM을 사용한 관련성 평가
      const prompt = this._buildGradingPrompt(query, content, metadata);

      const response = await this.modelManager.chatWithJSON(
        this.modelManager.selectModel('queryGeneration'), // 빠른 모델 사용
        [{ role: 'user', content: prompt }],
        {
          score: 'number',
          reasoning: 'string',
          keyMatches: 'array'
        }
      );

      const result = response.content;
      const score = Math.max(0, Math.min(1, result.score || 0));

      // 점수에 따른 등급 결정
      let grade;
      if (score >= this.correctThreshold) {
        grade = GRADE.CORRECT;
      } else if (score < this.incorrectThreshold) {
        grade = GRADE.INCORRECT;
      } else {
        grade = GRADE.AMBIGUOUS;
      }

      return {
        index,
        document,
        grade,
        score,
        reasoning: result.reasoning || '',
        keyMatches: result.keyMatches || []
      };

    } catch (error) {
      logger.error(`[DocumentGrader] 문서 ${index} 평가 실패: ${error.message}`);

      // 오류 발생 시 기존 점수 기반 평가
      return this._fallbackGrading(document, index);
    }
  }

  /**
   * LLM 호출 실패 시 폴백 평가
   *
   * @private
   * @param {Object} document - 평가할 문서
   * @param {number} index - 문서 인덱스
   * @returns {Object} 평가 결과
   */
  _fallbackGrading(document, index) {
    const scores = document.scores || {};

    // 기존 RAG 점수 기반 평가
    const keywordScore = scores.keyword_precision || 0;
    const semanticScore = scores.semantic_relevance || 0;
    const qualityScore = scores.content_quality || 0;

    // 가중 평균
    const avgScore = (keywordScore * 0.3 + semanticScore * 0.5 + qualityScore * 0.2);

    let grade;
    if (avgScore >= this.correctThreshold) {
      grade = GRADE.CORRECT;
    } else if (avgScore < this.incorrectThreshold) {
      grade = GRADE.INCORRECT;
    } else {
      grade = GRADE.AMBIGUOUS;
    }

    return {
      index,
      document,
      grade,
      score: Math.round(avgScore * 100) / 100,
      reasoning: '기존 RAG 점수 기반 평가 (LLM 평가 실패)',
      keyMatches: []
    };
  }

  /**
   * 평가 프롬프트 생성
   *
   * @private
   * @param {string} query - 사용자 쿼리
   * @param {string} content - 문서 내용
   * @param {Object} metadata - 문서 메타데이터
   * @returns {string} 프롬프트
   */
  _buildGradingPrompt(query, content, metadata) {
    const truncatedContent = content.length > 1000
      ? content.substring(0, 1000) + '...'
      : content;

    return `당신은 검색된 문서가 사용자 질문에 얼마나 관련있는지 평가하는 전문가입니다.

사용자 질문: ${query}

검색된 문서:
${truncatedContent}

문서 출처: ${metadata.filename || metadata.source || '알 수 없음'}

다음 기준으로 문서의 관련성을 평가하세요:
1. 질문의 핵심 키워드가 문서에 포함되어 있는가?
2. 문서 내용이 질문에 대한 답변을 제공하는가?
3. 문서의 맥락이 질문의 의도와 일치하는가?

다음 형식의 JSON으로 응답하세요:
{
  "score": 0.0-1.0,
  "reasoning": "평가 근거 (50자 이내)",
  "keyMatches": ["일치하는 핵심 키워드1", "키워드2"]
}`;
  }

  /**
   * 등급별 카운트 집계
   *
   * @private
   * @param {Array} gradedDocuments - 평가된 문서 배열
   * @returns {Object} 등급별 카운트
   */
  _countGrades(gradedDocuments) {
    return gradedDocuments.reduce((counts, doc) => {
      switch (doc.grade) {
        case GRADE.CORRECT:
          counts.correct++;
          break;
        case GRADE.AMBIGUOUS:
          counts.ambiguous++;
          break;
        case GRADE.INCORRECT:
          counts.incorrect++;
          break;
      }
      return counts;
    }, { correct: 0, ambiguous: 0, incorrect: 0 });
  }

  /**
   * 평가 결과에 따른 액션 결정
   *
   * @private
   * @param {Array} gradedDocuments - 평가된 문서 배열
   * @param {Object} counts - 등급별 카운트
   * @returns {Object} 권장 액션
   */
  _determineAction(gradedDocuments, counts) {
    const total = gradedDocuments.length;
    const correctRatio = counts.correct / total;
    const incorrectRatio = counts.incorrect / total;

    // CORRECT 비율이 충분하면 진행
    if (correctRatio >= this.minCorrectRatio) {
      return {
        type: ACTION.PROCEED,
        reason: `충분한 관련 문서 발견 (${counts.correct}/${total})`,
        correctCount: counts.correct,
        ambiguousCount: counts.ambiguous,
        incorrectCount: counts.incorrect,
        filteredDocuments: gradedDocuments
          .filter(d => d.grade !== GRADE.INCORRECT)
          .map(d => d.document)
      };
    }

    // INCORRECT가 대부분이면 웹 검색으로 대체
    if (incorrectRatio > 0.7) {
      return {
        type: ACTION.FALLBACK,
        reason: `관련 문서 부족 - 웹 검색 필요 (INCORRECT: ${counts.incorrect}/${total})`,
        correctCount: counts.correct,
        ambiguousCount: counts.ambiguous,
        incorrectCount: counts.incorrect
      };
    }

    // AMBIGUOUS가 많으면 쿼리 재작성
    return {
      type: ACTION.REWRITE,
      reason: `관련성 불확실 - 쿼리 재작성 필요 (AMBIGUOUS: ${counts.ambiguous}/${total})`,
      correctCount: counts.correct,
      ambiguousCount: counts.ambiguous,
      incorrectCount: counts.incorrect,
      filteredDocuments: gradedDocuments
        .filter(d => d.grade === GRADE.CORRECT)
        .map(d => d.document)
    };
  }

  /**
   * CORRECT 등급 문서만 필터링
   *
   * @param {Array} gradedDocuments - 평가된 문서 배열
   * @returns {Array} CORRECT 문서 배열
   */
  filterCorrectDocuments(gradedDocuments) {
    return gradedDocuments
      .filter(d => d.grade === GRADE.CORRECT)
      .sort((a, b) => b.score - a.score)
      .map(d => d.document);
  }

  /**
   * 평가 결과 요약 생성
   *
   * @param {Object} gradeResult - gradeDocuments() 반환 결과
   * @returns {string} 요약 텍스트
   */
  generateSummary(gradeResult) {
    const { action, summary } = gradeResult;

    return `[문서 평가 결과]
- 총 문서 수: ${summary.totalDocuments}
- 평균 점수: ${summary.avgScore}
- CORRECT: ${action.correctCount}
- AMBIGUOUS: ${action.ambiguousCount}
- INCORRECT: ${action.incorrectCount}
- 권장 액션: ${action.type}
- 사유: ${action.reason}
- 처리 시간: ${summary.processingTime}ms`;
  }
}

// 상수 내보내기
export { GRADE, ACTION };
export default DocumentGrader;
