/**
 * Supervisor Multi-Agent System Integration
 *
 * 기존 AIRUN 시스템과의 통합 레이어
 *
 * @module services/agent-system/index
 */

import SupervisorOrchestrator from './supervisor/supervisor-orchestrator.js';
import RAGAgent from './agents/rag-agent.js';
import WebAgent from './agents/web-agent.js';
import { logger } from '../../utils/logger.js';

// 전역 Supervisor 인스턴스
let supervisorInstance = null;

/**
 * Supervisor 시스템 초기화
 *
 * @param {Object} options - 초기화 옵션
 * @returns {SupervisorOrchestrator} Supervisor 인스턴스
 */
export function initializeSupervisor(options = {}) {
  if (supervisorInstance) {
    logger.info('[SupervisorSystem] 이미 초기화됨, 기존 인스턴스 반환');
    return supervisorInstance;
  }

  try {
    logger.info('[SupervisorSystem] 초기화 시작...');

    // 환경변수 또는 기본값 결정
    const ollamaUrl = options.ollamaUrl
      || process.env.OLLAMA_PROXY_SERVER
      || 'https://api.hamonize.com/ollama';

    const ragServerUrl = options.ragServerUrl
      || process.env.RAG_SERVER_URL
      || 'http://localhost:5600';

    const webSearchUrl = options.webSearchUrl
      || process.env.WEB_SEARCH_URL
      || 'http://localhost:5610';

    logger.info(`[SupervisorSystem] Ollama URL: ${ollamaUrl}`);
    logger.info(`[SupervisorSystem] RAG URL: ${ragServerUrl}`);
    logger.info(`[SupervisorSystem] Web Search URL: ${webSearchUrl}`);

    // RAG Agent 생성
    const ragAgent = new RAGAgent({
      ragServerUrl: ragServerUrl,
    });

    // Web Agent 생성
    const webAgent = new WebAgent({
      webSearchUrl: webSearchUrl,
    });

    // Supervisor 생성
    supervisorInstance = new SupervisorOrchestrator({
      ollamaUrl: ollamaUrl,
      maxRetries: options.maxRetries || 2,
      qualityThreshold: options.qualityThreshold || 0.65,
    });

    // 에이전트 등록
    supervisorInstance.registerAgent('rag', ragAgent);
    supervisorInstance.registerAgent('web', webAgent);

    logger.info('[SupervisorSystem] 초기화 완료');

    return supervisorInstance;
  } catch (error) {
    logger.error(`[SupervisorSystem] 초기화 실패: ${error.message}`);
    throw error;
  }
}

/**
 * Supervisor 인스턴스 가져오기
 *
 * @returns {SupervisorOrchestrator|null} Supervisor 인스턴스 또는 null
 */
export function getSupervisor() {
  return supervisorInstance;
}

/**
 * Supervisor 요청 처리 (Express 핸들러용)
 *
 * @param {Object} req - Express request
 * @param {Object} res - Express response
 */
export async function handleSupervisorRequest(req, res) {
  try {
    const { prompt, sessionId, username, options = {} } = req.body;

    if (!prompt) {
      return res.status(400).json({
        success: false,
        error: {
          code: 'INVALID_REQUEST',
          message: '질문(prompt)이 필요합니다',
        },
        timestamp: new Date().toISOString(),
      });
    }

    // Supervisor 초기화 (필요시)
    const supervisor = supervisorInstance || initializeSupervisor();

    logger.info(`[SupervisorSystem] 요청 처리: user=${username}, session=${sessionId}`);

    // 요청 처리
    const result = await supervisor.process({
      query: prompt,
      sessionId: sessionId || `supervisor-${Date.now()}`,
      username: username || 'anonymous',
      options,
    });

    // 성공 응답
    if (result.success) {
      return res.json({
        success: true,
        data: {
          content: result.content,
          metadata: result.metadata,
        },
        timestamp: new Date().toISOString(),
      });
    } else {
      // 품질 미달 응답
      return res.status(200).json({
        success: false,
        data: {
          content: result.content,
          metadata: result.metadata,
        },
        error: {
          code: 'QUALITY_INSUFFICIENT',
          message: result.error,
        },
        timestamp: new Date().toISOString(),
      });
    }
  } catch (error) {
    logger.error(`[SupervisorSystem] 요청 처리 실패: ${error.message}`);

    return res.status(500).json({
      success: false,
      error: {
        code: 'INTERNAL_SERVER_ERROR',
        message: error.message,
      },
      timestamp: new Date().toISOString(),
    });
  }
}

/**
 * Supervisor 설정 업데이트
 *
 * @param {Object} config - 새 설정
 */
export function updateSupervisorConfig(config) {
  if (!supervisorInstance) {
    logger.warn('[SupervisorSystem] Supervisor가 초기화되지 않음');
    return;
  }

  supervisorInstance.updateConfig(config);
  logger.info('[SupervisorSystem] 설정 업데이트 완료');
}

/**
 * Supervisor 정리
 */
export function shutdownSupervisor() {
  if (supervisorInstance) {
    logger.info('[SupervisorSystem] Supervisor 종료');
    supervisorInstance = null;
  }
}

export default {
  initializeSupervisor,
  getSupervisor,
  handleSupervisorRequest,
  updateSupervisorConfig,
  shutdownSupervisor,
};
