/**
 * Ollama 모델 관리자
 *
 * 모델 선택, 상태 모니터링, 폴백 전략을 관리
 *
 * @module services/agent-system/ollama/model-manager
 */

import OllamaClient from './ollama-client.js';
import { logger } from '../../../utils/logger.js';
import axios from 'axios';

class ModelManager {
  /**
   * @param {Object} options - 관리자 옵션
   * @param {string} options.ollamaUrl - Ollama 서버 URL
   * @param {string} options.defaultModel - 기본 모델
   * @param {Array} options.fallbackModels - 폴백 모델 목록
   * @param {number} options.healthCheckInterval - 헬스체크 간격 (ms)
   */
  constructor(options = {}) {
    this.client = new OllamaClient({
      baseURL: options.ollamaUrl,
    });

    this.defaultModel = options.defaultModel || 'hamonize:latest';
    this.fallbackModels = options.fallbackModels || ['airun-chat', 'hamonize:latest'];
    this.healthCheckInterval = options.healthCheckInterval || 300000; // 5분

    // 모델 상태 캐시
    this.modelHealth = new Map();

    // 작업별 추천 모델
    // 품질 테스트 결과: hamonize 모델이 모든 작업에서 100% 통과 (2025-11-17)
    this.taskModels = {
      supervisor: 'hamonize:latest',      // Supervisor 판단
      taskAnalysis: 'hamonize:latest',    // 작업 분석
      queryGeneration: 'hamonize:latest', // 쿼리 생성 (테스트 통과: JSON 유효성 ✓)
      qualityValidation: 'hamonize:latest', // 품질 검증
      resultAggregation: 'hamonize:latest', // 결과 취합 (테스트 통과: 1811자 응답)
      general: 'hamonize:latest',         // 일반 작업
    };

    logger.info(`[ModelManager] 초기화 완료: 기본 모델=${this.defaultModel}`);

    // 초기 모델 상태 설정 (기본값은 건강 상태)
    this.modelHealth.set(this.defaultModel, {
      healthy: true,
      lastCheck: new Date(),
      lastError: null,
    });

    // healthCheckInterval이 0이 아닌 경우에만 헬스체크 시작
    if (this.healthCheckInterval > 0) {
      // 주기적 헬스체크 시작
      this._startHealthCheck();
    }
  }

  /**
   * 작업 유형에 따라 최적 모델 선택
   *
   * @param {string} taskType - 작업 유형 (supervisor, taskAnalysis, etc.)
   * @param {Object} options - 추가 옵션
   * @returns {string} 선택된 모델 이름
   */
  selectModel(taskType = 'general', options = {}) {
    // 옵션에서 명시적 모델 지정 시 우선 사용
    if (options.model) {
      logger.info(`[ModelManager] 명시적 모델 선택: ${options.model}`);
      return options.model;
    }

    // 작업 유형에 따른 추천 모델
    const recommendedModel = this.taskModels[taskType] || this.defaultModel;

    // 모델 상태 확인
    const health = this.modelHealth.get(recommendedModel);

    // 초기에는 모델이 건강하다고 가정 (상태 확인 전까지)
    if (!health || health.healthy) {
      logger.info(`[ModelManager] 추천 모델 선택: ${recommendedModel} (작업: ${taskType})`);
      return recommendedModel;
    } else {
      // 모델이 unhealthy로 기록된 경우에만 폴백
      logger.warn(`[ModelManager] 추천 모델 ${recommendedModel} 비정상, 폴백 시도`);
      return this._selectFallbackModel(recommendedModel);
    }
  }

  /**
   * 모델로 채팅 수행 (폴백 지원, 스트리밍 지원)
   *
   * @param {string} model - 모델 이름
   * @param {Array} messages - 메시지 배열
   * @param {Object} options - 추가 옵션
   * @param {boolean} options.stream - 스트리밍 활성화 여부
   * @param {Function} options.onContent - 스트리밍 콜백 함수
   * @returns {Promise<Object>} 응답 객체
   */
  async chat(model, messages, options = {}) {
    try {
      // 스트리밍 모드 감지 (aiFeatures.js와 동일한 패턴)
      const shouldStream = options.stream === true && typeof options.onContent === 'function';

      if (shouldStream) {
        logger.info(`[ModelManager] 스트리밍 채팅 시작: model=${model}`);

        const response = await this.client.chatStream(model, messages, (chunk) => {
          // OllamaClient의 청크 형식을 aiFeatures.js 형식으로 변환
          options.onContent(chunk.content);
        }, options);

        // 성공 시 모델 상태 업데이트
        this.modelHealth.set(model, {
          healthy: true,
          lastCheck: new Date(),
          lastError: null,
        });

        return response;
      }

      // 일반 채팅 (스트리밍 없음)
      logger.info(`[ModelManager] 채팅 시작: model=${model}`);

      const response = await this.client.chatWithRetry(model, messages, 3, options);

      // 성공 시 모델 상태 업데이트
      this.modelHealth.set(model, {
        healthy: true,
        lastCheck: new Date(),
        lastError: null,
      });

      return response;
    } catch (error) {
      logger.error(`[ModelManager] 채팅 실패 (model=${model}): ${error.message}`);

      // 실패 시 모델 상태 업데이트
      this.modelHealth.set(model, {
        healthy: false,
        lastCheck: new Date(),
        lastError: error.message,
      });

      throw error;
    }
  }


  /**
   * JSON 출력 채팅 (폴백 지원)
   *
   * @param {string} model - 모델 이름
   * @param {Array} messages - 메시지 배열
   * @param {Object} schema - JSON 스키마
   * @param {Object} options - 추가 옵션
   * @returns {Promise<Object>} 파싱된 JSON 응답
   */
  async chatWithJSON(model, messages, schema = null, options = {}) {
    try {
      logger.info(`[ModelManager] JSON 채팅 시작: model=${model}`);

      const response = await this.client.chatWithJSON(model, messages, schema, options);

      // JSON 파싱 실패 시 재시도
      if (!response.isValidJSON && options.retryOnInvalidJSON !== false) {
        logger.warn(`[ModelManager] JSON 파싱 실패, 재시도...`);

        // 프롬프트 강화하여 재시도
        const enhancedMessages = this._enhanceJSONPrompt(messages);
        const retryResponse = await this.client.chatWithJSON(model, enhancedMessages, schema, {
          ...options,
          retryOnInvalidJSON: false, // 무한 재귀 방지
        });

        if (retryResponse.isValidJSON) {
          logger.info(`[ModelManager] 재시도 후 JSON 파싱 성공`);
          return retryResponse;
        }
      }

      return response;
    } catch (error) {
      logger.error(`[ModelManager] JSON 채팅 실패: ${error.message}`);
      throw error;
    }
  }

  /**
   * 모델 상태 확인
   *
   * @param {string} model - 모델 이름
   * @returns {Promise<Object>} 모델 상태
   */
  async checkModelHealth(model) {
    try {
      const health = await this.client.checkModelHealth(model);

      this.modelHealth.set(model, {
        healthy: health.healthy,
        lastCheck: new Date(),
        lastError: health.error || null,
        responseTime: health.responseTime,
      });

      return health;
    } catch (error) {
      logger.error(`[ModelManager] 모델 상태 확인 실패: ${error.message}`);

      this.modelHealth.set(model, {
        healthy: false,
        lastCheck: new Date(),
        lastError: error.message,
      });

      return {
        success: false,
        model,
        healthy: false,
        error: error.message,
      };
    }
  }

  /**
   * 모든 모델 상태 조회
   *
   * @returns {Object} 모델 상태 맵
   */
  getModelsHealth() {
    const healthMap = {};

    for (const [model, health] of this.modelHealth.entries()) {
      healthMap[model] = {
        ...health,
        lastCheckAgo: Date.now() - health.lastCheck.getTime(),
      };
    }

    return healthMap;
  }

  /**
   * 사용 가능한 모델 목록 조회
   *
   * @returns {Promise<Array>} 모델 목록
   */
  async listAvailableModels() {
    try {
      const models = await this.client.listModels();

      logger.info(`[ModelManager] 사용 가능한 모델: ${models.map(m => m.name).join(', ')}`);

      return models;
    } catch (error) {
      logger.error(`[ModelManager] 모델 목록 조회 실패: ${error.message}`);
      return [];
    }
  }

  /**
   * 폴백 모델 선택
   *
   * @private
   * @param {string} failedModel - 실패한 모델
   * @returns {string|null} 폴백 모델 또는 null
   */
  _selectFallbackModel(failedModel) {
    // 폴백 목록에서 실패한 모델 제외
    const candidates = this.fallbackModels.filter(m => m !== failedModel);

    // 건강한 모델 우선 선택
    for (const candidate of candidates) {
      const health = this.modelHealth.get(candidate);
      if (health && health.healthy) {
        logger.info(`[ModelManager] 폴백 모델 선택: ${candidate}`);
        return candidate;
      }
    }

    // 건강한 모델이 없으면 첫 번째 후보 반환
    if (candidates.length > 0) {
      logger.warn(`[ModelManager] 건강한 폴백 모델 없음, 첫 번째 후보 사용: ${candidates[0]}`);
      return candidates[0];
    }

    logger.error(`[ModelManager] 사용 가능한 폴백 모델 없음`);
    return null;
  }

  /**
   * JSON 프롬프트 강화
   *
   * @private
   * @param {Array} messages - 원본 메시지
   * @returns {Array} 강화된 메시지
   */
  _enhanceJSONPrompt(messages) {
    return messages.map(m => {
      if (m.role === 'user') {
        return {
          ...m,
          content: `${m.content}\n\n중요: 응답은 반드시 유효한 JSON 형식이어야 합니다. 어떠한 설명이나 주석도 포함하지 마세요.`,
        };
      }
      return m;
    });
  }

  /**
   * 주기적 헬스체크 시작
   *
   * @private
   */
  _startHealthCheck() {
    // 초기 헬스체크
    this._performHealthCheck();

    // 주기적 헬스체크
    setInterval(() => {
      this._performHealthCheck();
    }, this.healthCheckInterval);

    logger.info(`[ModelManager] 주기적 헬스체크 시작 (간격: ${this.healthCheckInterval}ms)`);
  }

  /**
   * 헬스체크 수행
   *
   * @private
   */
  async _performHealthCheck() {
    logger.debug(`[ModelManager] 헬스체크 수행 중...`);

    const modelsToCheck = [
      this.defaultModel,
      ...this.fallbackModels,
      ...Object.values(this.taskModels),
    ];

    // 중복 제거
    const uniqueModels = [...new Set(modelsToCheck)];

    for (const model of uniqueModels) {
      try {
        await this.checkModelHealth(model);
      } catch (error) {
        logger.error(`[ModelManager] ${model} 헬스체크 실패: ${error.message}`);
      }
    }

    logger.debug(`[ModelManager] 헬스체크 완료`);
  }

  /**
   * 정리 (리소스 해제)
   */
  cleanup() {
    // 헬스체크 중단
    if (this.healthCheckTimer) {
      clearInterval(this.healthCheckTimer);
    }

    logger.info(`[ModelManager] 리소스 정리 완료`);
  }
}

export default ModelManager;
