/**
 * Ollama API 클라이언트
 *
 * AIRUN Supervisor Multi-Agent 시스템의 Ollama 연동 클라이언트
 * - OpenAI-compatible API 사용
 * - 재시도 로직 및 폴백 지원
 * - JSON 출력 강제
 * - 스트리밍 지원
 *
 * @module services/agent-system/ollama/ollama-client
 */

import axios from 'axios';
import { logger } from '../../../utils/logger.js';

class OllamaClient {
  /**
   * @param {Object} options - 클라이언트 옵션
   * @param {string} options.baseURL - Ollama 서버 URL
   * @param {number} options.timeout - 요청 타임아웃 (ms)
   * @param {number} options.maxRetries - 최대 재시도 횟수
   * @param {number} options.retryDelay - 재시도 간격 (ms)
   */
  constructor(options = {}) {
    this.baseURL = options.baseURL || process.env.OLLAMA_PROXY_SERVER || 'http://localhost:11434';
    this.timeout = options.timeout || parseInt(process.env.OLLAMA_TIMEOUT || '60000');
    this.maxRetries = options.maxRetries || 3;
    this.retryDelay = options.retryDelay || 1000;

    // Axios 인스턴스 생성
    this.client = axios.create({
      baseURL: this.baseURL,
      timeout: this.timeout,
      headers: {
        'Content-Type': 'application/json',
      },
    });

    // 요청 인터셉터: 로깅
    this.client.interceptors.request.use((config) => {
      logger.debug(`[OllamaClient] 요청: ${config.method?.toUpperCase()} ${config.url}`);
      return config;
    });

    // 응답 인터셉터: 로깅 및 에러 처리
    this.client.interceptors.response.use(
      (response) => {
        logger.debug(`[OllamaClient] 응답 성공: ${response.status}`);
        return response;
      },
      (error) => {
        logger.error(`[OllamaClient] 응답 실패: ${error.message}`);
        return Promise.reject(error);
      }
    );

    logger.info(`[OllamaClient] 초기화 완료: ${this.baseURL}`);
  }

  /**
   * 기본 채팅 완료 API
   *
   * @param {string} model - 모델 이름 (hamonize:latest, airun-chat)
   * @param {Array} messages - 메시지 배열 [{role, content}]
   * @param {Object} options - 추가 옵션
   * @returns {Promise<Object>} 응답 객체
   */
  async chat(model, messages, options = {}) {
    try {
      logger.info(`[OllamaClient] chat 시작: model=${model}, messages=${messages.length}개`);

      const response = await this.client.post('/api/chat', {
        model,
        messages,
        stream: false,
        options: {
          temperature: options.temperature || 0.7,
          top_p: options.top_p || 0.9,
          top_k: options.top_k || 40,
          ...options.modelOptions,
        },
      });

      if (!response.data || !response.data.message) {
        throw new Error('Ollama API 응답 형식 오류: message 필드 없음');
      }

      logger.info(`[OllamaClient] chat 완료`);

      return {
        success: true,
        content: response.data.message.content,
        model: response.data.model,
        done: response.data.done,
        total_duration: response.data.total_duration,
        load_duration: response.data.load_duration,
        prompt_eval_count: response.data.prompt_eval_count,
        eval_count: response.data.eval_count,
      };
    } catch (error) {
      logger.error(`[OllamaClient] chat 실패: ${error.message}`);
      throw this._handleError(error);
    }
  }

  /**
   * JSON 출력 강제 채팅 API
   *
   * @param {string} model - 모델 이름
   * @param {Array} messages - 메시지 배열
   * @param {Object} schema - JSON 스키마 (optional)
   * @param {Object} options - 추가 옵션
   * @returns {Promise<Object>} 파싱된 JSON 응답
   */
  async chatWithJSON(model, messages, schema = null, options = {}) {
    try {
      logger.info(`[OllamaClient] chatWithJSON 시작: model=${model}`);

      // 시스템 메시지에 JSON 출력 지시 추가
      const enhancedMessages = this._enforceJSONOutput(messages, schema);

      const response = await this.chat(model, enhancedMessages, {
        ...options,
        temperature: 0.1, // JSON 출력은 낮은 temperature 사용
      });

      // JSON 파싱 시도
      let parsedContent;
      try {
        // 코드 블록 제거 (```json ... ```)
        const cleanedContent = response.content
          .replace(/```json\n?/g, '')
          .replace(/```\n?/g, '')
          .trim();

        parsedContent = JSON.parse(cleanedContent);
        logger.info(`[OllamaClient] JSON 파싱 성공`);
      } catch (parseError) {
        logger.warn(`[OllamaClient] JSON 파싱 실패, 원본 반환: ${parseError.message}`);

        // 파싱 실패 시 원본 반환
        parsedContent = {
          raw: response.content,
          parseError: parseError.message,
        };
      }

      return {
        ...response,
        content: parsedContent,
        isValidJSON: typeof parsedContent === 'object' && !parsedContent.parseError,
      };
    } catch (error) {
      logger.error(`[OllamaClient] chatWithJSON 실패: ${error.message}`);
      throw this._handleError(error);
    }
  }

  /**
   * 재시도 로직이 포함된 채팅 API
   *
   * @param {string} model - 모델 이름
   * @param {Array} messages - 메시지 배열
   * @param {number} maxRetries - 최대 재시도 횟수 (기본값: this.maxRetries)
   * @param {Object} options - 추가 옵션
   * @returns {Promise<Object>} 응답 객체
   */
  async chatWithRetry(model, messages, maxRetries = null, options = {}) {
    const retries = maxRetries !== null ? maxRetries : this.maxRetries;
    let lastError;

    for (let attempt = 1; attempt <= retries; attempt++) {
      try {
        logger.info(`[OllamaClient] chatWithRetry 시도 ${attempt}/${retries}`);

        const response = await this.chat(model, messages, options);

        logger.info(`[OllamaClient] chatWithRetry 성공 (시도 ${attempt})`);
        return response;
      } catch (error) {
        lastError = error;
        logger.warn(`[OllamaClient] chatWithRetry 실패 (시도 ${attempt}): ${error.message}`);

        if (attempt < retries) {
          // 지수 백오프
          const delay = this.retryDelay * Math.pow(2, attempt - 1);
          logger.info(`[OllamaClient] ${delay}ms 후 재시도...`);
          await this._sleep(delay);
        }
      }
    }

    logger.error(`[OllamaClient] chatWithRetry 최종 실패 (${retries}회 시도)`);
    throw lastError;
  }

  /**
   * 스트리밍 채팅 API
   *
   * @param {string} model - 모델 이름
   * @param {Array} messages - 메시지 배열
   * @param {Function} onChunk - 청크 수신 콜백
   * @param {Object} options - 추가 옵션
   * @returns {Promise<Object>} 완료 응답
   */
  async chatStream(model, messages, onChunk, options = {}) {
    try {
      logger.info(`[OllamaClient] chatStream 시작: model=${model}`);

      const response = await this.client.post('/api/chat', {
        model,
        messages,
        stream: true,
        options: {
          temperature: options.temperature || 0.7,
          ...options.modelOptions,
        },
      }, {
        responseType: 'stream',
      });

      let fullContent = '';
      let chunkCount = 0;

      return new Promise((resolve, reject) => {
        response.data.on('data', (chunk) => {
          try {
            const lines = chunk.toString().split('\n').filter(line => line.trim());

            for (const line of lines) {
              const data = JSON.parse(line);

              if (data.message && data.message.content) {
                fullContent += data.message.content;
                chunkCount++;

                // 콜백 실행
                if (onChunk) {
                  onChunk({
                    content: data.message.content,
                    fullContent: fullContent,
                    done: data.done,
                  });
                }
              }

              // 스트림 완료
              if (data.done) {
                logger.info(`[OllamaClient] chatStream 완료: ${chunkCount}개 청크`);
                resolve({
                  success: true,
                  content: fullContent,
                  model: data.model,
                  chunkCount,
                });
              }
            }
          } catch (parseError) {
            logger.error(`[OllamaClient] 청크 파싱 실패: ${parseError.message}`);
          }
        });

        response.data.on('error', (error) => {
          logger.error(`[OllamaClient] 스트림 오류: ${error.message}`);
          reject(error);
        });
      });
    } catch (error) {
      logger.error(`[OllamaClient] chatStream 실패: ${error.message}`);
      throw this._handleError(error);
    }
  }

  /**
   * 모델 상태 확인
   *
   * @param {string} model - 모델 이름
   * @returns {Promise<Object>} 모델 정보
   */
  async checkModelHealth(model) {
    try {
      logger.info(`[OllamaClient] 모델 상태 확인: ${model}`);

      // 간단한 테스트 메시지로 모델 확인
      const response = await this.chat(model, [
        { role: 'user', content: 'ping' }
      ], {
        temperature: 0,
      });

      logger.info(`[OllamaClient] 모델 상태: 정상`);

      return {
        success: true,
        model,
        healthy: true,
        responseTime: response.total_duration,
      };
    } catch (error) {
      logger.error(`[OllamaClient] 모델 상태 확인 실패: ${error.message}`);

      return {
        success: false,
        model,
        healthy: false,
        error: error.message,
      };
    }
  }

  /**
   * 사용 가능한 모델 목록 조회
   *
   * @returns {Promise<Array>} 모델 목록
   */
  async listModels() {
    try {
      logger.info(`[OllamaClient] 모델 목록 조회`);

      const response = await this.client.get('/api/tags');

      if (!response.data || !response.data.models) {
        throw new Error('Ollama API 응답 형식 오류: models 필드 없음');
      }

      const models = response.data.models.map(m => ({
        name: m.name,
        size: m.size,
        modified_at: m.modified_at,
      }));

      logger.info(`[OllamaClient] 모델 ${models.length}개 발견`);

      return models;
    } catch (error) {
      logger.error(`[OllamaClient] 모델 목록 조회 실패: ${error.message}`);
      throw this._handleError(error);
    }
  }

  /**
   * JSON 출력 강제를 위한 메시지 변환
   *
   * @private
   * @param {Array} messages - 원본 메시지
   * @param {Object} schema - JSON 스키마
   * @returns {Array} 변환된 메시지
   */
  _enforceJSONOutput(messages, schema) {
    const systemMessage = {
      role: 'system',
      content: `당신은 항상 유효한 JSON 형식으로만 응답해야 합니다.
어떠한 설명도 추가하지 말고, 순수한 JSON 객체만 출력하세요.
코드 블록을 사용하지 마세요.
${schema ? `\n다음 스키마를 따르세요:\n${JSON.stringify(schema, null, 2)}` : ''}`,
    };

    // 시스템 메시지가 이미 있으면 병합, 없으면 추가
    const hasSystemMessage = messages.some(m => m.role === 'system');

    if (hasSystemMessage) {
      return messages.map(m =>
        m.role === 'system'
          ? { ...m, content: `${systemMessage.content}\n\n${m.content}` }
          : m
      );
    } else {
      return [systemMessage, ...messages];
    }
  }

  /**
   * 에러 핸들링
   *
   * @private
   * @param {Error} error - 원본 에러
   * @returns {Error} 처리된 에러
   */
  _handleError(error) {
    if (error.code === 'ECONNREFUSED') {
      return new Error(`Ollama 서버에 연결할 수 없습니다: ${this.baseURL}`);
    } else if (error.code === 'ETIMEDOUT') {
      return new Error(`Ollama 요청 타임아웃 (${this.timeout}ms)`);
    } else if (error.response) {
      return new Error(`Ollama API 오류: ${error.response.status} - ${error.response.statusText}`);
    } else {
      return error;
    }
  }

  /**
   * Sleep 유틸리티
   *
   * @private
   * @param {number} ms - 대기 시간 (ms)
   * @returns {Promise<void>}
   */
  _sleep(ms) {
    return new Promise(resolve => setTimeout(resolve, ms));
  }
}

export default OllamaClient;
