/**
 * Ollama 프롬프트 템플릿 관리자
 *
 * 각 작업별 프롬프트 템플릿 및 Few-shot 예시 관리
 *
 * @module services/agent-system/ollama/prompt-templates
 */

import { logger } from '../../../utils/logger.js';

class PromptTemplates {
  constructor() {
    // 작업 분석 템플릿
    this.taskAnalysis = {
      template: `당신은 사용자 질문을 분석하여 필요한 에이전트를 선정하는 전문가입니다.

**중요: 이 시스템은 고객의 문서가 RAG 데이터베이스에 저장되어 있는 환경에서 운영됩니다.**
**따라서 문서검색(rag)을 항상 우선적으로 선택해야 합니다.**

사용 가능한 에이전트:
- rag: 업로드된 내부 문서에서 정보 검색 (회사 자료, 기술 문서, 매뉴얼 등) - **우선 선택**
- web: 웹에서 정보 검색 - **보조적 사용**

에이전트 선정 원칙:
1. 모든 질문에 대해 rag를 기본으로 선택하세요
2. rag만으로 충분한 경우 web은 선택하지 마세요
3. web은 rag 결과가 부족할 때만 보조적으로 사용합니다
4. "최신", "트렌드", "동향" 등의 단어가 있어도 먼저 문서에서 검색하세요

사용자 질문: {query}

다음 형식의 JSON으로 응답하세요:
{
  "selectedAgents": ["agent1", "agent2"],
  "reasoning": "선정 이유 (50자 이내)",
  "confidence": 0.0-1.0
}`,
      examples: [
        {
          input: "우리 회사 AI 전략이 뭐야?",
          output: {
            selectedAgents: ["rag"],
            reasoning: "내부 문서에서 회사 전략 검색",
            confidence: 0.95
          }
        },
        {
          input: "최신 AI 트렌드를 알려줘",
          output: {
            selectedAgents: ["rag"],
            reasoning: "문서에서 AI 관련 정보 우선 검색",
            confidence: 0.85
          }
        },
        {
          input: "인공지능의 기본 개념을 설명해줘",
          output: {
            selectedAgents: ["rag"],
            reasoning: "문서에서 AI 개념 설명 검색",
            confidence: 0.9
          }
        },
        {
          input: "우리 회사 AI 전략과 업계 동향을 비교해줘",
          output: {
            selectedAgents: ["rag", "web"],
            reasoning: "내부 문서 우선, 웹 검색 보조",
            confidence: 0.9
          }
        }
      ]
    };

    // 쿼리 생성 템플릿
    this.queryGeneration = {
      template: `당신은 사용자 질문을 각 에이전트에게 전달할 구체적인 쿼리로 변환하는 전문가입니다.

원래 질문: {originalQuery}
대상 에이전트: {targetAgents}

각 에이전트에게 전달할 최적화된 쿼리를 생성하세요.
- rag 에이전트: 내부 문서 검색에 최적화된 키워드
- web 에이전트: 웹 검색에 적합한 쿼리

다음 형식의 JSON으로 응답하세요:
{
  "queries": {
    "agent1": "에이전트1용 쿼리",
    "agent2": "에이전트2용 쿼리"
  },
  "reasoning": "쿼리 생성 이유"
}`,
      examples: [
        {
          input: {
            originalQuery: "우리 회사 AI 전략과 최신 트렌드 비교",
            targetAgents: ["rag", "web"]
          },
          output: {
            queries: {
              rag: "AI 전략 인공지능 사업 계획",
              web: "2025 AI trends artificial intelligence latest"
            },
            reasoning: "RAG는 키워드 중심, Web은 영문 검색어 최적화"
          }
        }
      ]
    };

    // 품질 검증 템플릿
    this.qualityValidation = {
      template: `당신은 검색 결과의 품질을 평가하는 전문가입니다.

사용자 질문: {query}
검색 결과: {results}

다음 기준으로 결과의 품질을 평가하세요:
1. 관련성 (relevance): 질문과 얼마나 관련있는가? (0.0-1.0)
2. 완전성 (completeness): 질문에 완전히 답변했는가? (0.0-1.0)
3. 정확성 (accuracy): 정보가 정확하고 신뢰할 수 있는가? (0.0-1.0)
4. 최신성 (freshness): 정보가 최신인가? (0.0-1.0)

다음 형식의 JSON으로 응답하세요:
{
  "score": 0.0-1.0,
  "relevance": 0.0-1.0,
  "completeness": 0.0-1.0,
  "accuracy": 0.0-1.0,
  "freshness": 0.0-1.0,
  "reasoning": "평가 이유 (100자 이내)",
  "improvements": ["개선 제안1", "개선 제안2"]
}`,
      examples: [
        {
          input: {
            query: "파이썬 웹 프레임워크 추천",
            results: "Django와 Flask가 가장 인기있는 파이썬 웹 프레임워크입니다."
          },
          output: {
            score: 0.7,
            relevance: 0.9,
            completeness: 0.6,
            accuracy: 0.8,
            freshness: 0.5,
            reasoning: "관련성은 높으나 구체적 비교 및 최신 정보 부족",
            improvements: ["각 프레임워크 장단점 추가", "2025년 트렌드 반영"]
          }
        }
      ]
    };

    // 결과 취합 템플릿
    this.resultAggregation = {
      template: `당신은 다양한 소스의 정보를 종합하여 사용자에게 최적의 답변을 생성하는 전문가입니다.

사용자 질문: {query}

수집된 정보:
{agentResults}

다음 지침을 따라 종합 답변을 생성하세요:
1. 중복 정보는 통합하세요
2. 상충되는 정보는 출처를 명시하세요 (문서명, 웹사이트 제목 등)
3. 정보의 신뢰도를 평가하세요
4. 답변이 불완전하면 명확히 밝히세요
5. 한국어로 자연스럽게 작성하세요
6. 참고한 문서나 웹사이트가 있다면 응답 내용에서 명시하세요

답변:`,
      examples: [
        {
          input: {
            query: "파이썬 웹 개발 프레임워크 추천",
            agentResults: {
              rag: "내부 문서: Django 사용 중, 안정성 좋음",
              web: "최신 트렌드: FastAPI 인기 상승, 고성능"
            }
          },
          output: "파이썬 웹 개발 프레임워크로는 Django와 FastAPI를 추천드립니다.\n\n현재 회사에서는 Django를 사용하고 있으며 안정성이 검증되었습니다. 반면 최신 트렌드를 보면 FastAPI가 빠르게 성장하고 있으며 고성능을 제공합니다.\n\n프로젝트 요구사항에 따라:\n- 안정성과 완성도: Django\n- 성능과 최신 기술: FastAPI\n를 선택하시면 좋습니다."
        }
      ]
    };

    // 재시도 전략 템플릿
    this.retryStrategy = {
      template: `당신은 검색 결과가 부족할 때 더 나은 검색 전략을 수립하는 전문가입니다.

원래 질문: {originalQuery}
이전 검색 결과: {previousResults}
품질 점수: {qualityScore}

더 나은 결과를 얻기 위해:
1. 다른 각도로 질문을 재구성하세요
2. 더 구체적이거나 더 일반적인 검색어를 생성하세요
3. 누락된 측면을 탐색할 새로운 검색어를 제안하세요
4. 추가로 필요한 에이전트를 선정하세요

다음 형식의 JSON으로 응답하세요:
{
  "newQueries": {
    "agent1": "새 쿼리1",
    "agent2": "새 쿼리2"
  },
  "addedAgents": ["추가할 에이전트"],
  "reasoning": "재시도 전략 설명",
  "focusAreas": ["집중할 영역1", "집중할 영역2"]
}`,
      examples: [
        {
          input: {
            originalQuery: "AI 활용 방안",
            previousResults: "일반적인 AI 설명만 제공",
            qualityScore: 0.4
          },
          output: {
            newQueries: {
              rag: "AI 활용 사례 실무 적용 방안",
              web: "AI business applications 2025 실제 사례"
            },
            addedAgents: [],
            reasoning: "구체적 활용 사례와 실무 적용에 초점",
            focusAreas: ["실제 활용 사례", "ROI 측정", "도입 전략"]
          }
        }
      ]
    };

    logger.info('[PromptTemplates] 초기화 완료');
  }

  /**
   * 템플릿 렌더링
   *
   * @param {string} templateName - 템플릿 이름
   * @param {Object} variables - 변수 맵
   * @param {boolean} includeFewShot - Few-shot 예시 포함 여부
   * @returns {string} 렌더링된 프롬프트
   */
  render(templateName, variables = {}, includeFewShot = true) {
    const template = this[templateName];

    if (!template) {
      throw new Error(`템플릿을 찾을 수 없습니다: ${templateName}`);
    }

    // 변수 치환
    let prompt = template.template;
    for (const [key, value] of Object.entries(variables)) {
      const placeholder = `{${key}}`;
      const replacement = typeof value === 'object' ? JSON.stringify(value, null, 2) : String(value);
      prompt = prompt.replace(new RegExp(placeholder, 'g'), replacement);
    }

    // Few-shot 예시 추가
    if (includeFewShot && template.examples && template.examples.length > 0) {
      const examplesText = this._formatExamples(template.examples);
      prompt = `${prompt}\n\n예시:\n${examplesText}`;
    }

    logger.debug(`[PromptTemplates] 템플릿 렌더링: ${templateName}`);

    return prompt;
  }

  /**
   * Few-shot 예시 포맷팅
   *
   * @private
   * @param {Array} examples - 예시 배열
   * @returns {string} 포맷팅된 예시 텍스트
   */
  _formatExamples(examples) {
    return examples.map((example, index) => {
      const input = typeof example.input === 'object'
        ? JSON.stringify(example.input, null, 2)
        : example.input;
      const output = typeof example.output === 'object'
        ? JSON.stringify(example.output, null, 2)
        : example.output;

      return `예시 ${index + 1}:\n입력: ${input}\n출력: ${output}`;
    }).join('\n\n');
  }

  /**
   * 사용자 정의 템플릿 추가
   *
   * @param {string} name - 템플릿 이름
   * @param {Object} template - 템플릿 객체 {template, examples}
   */
  addTemplate(name, template) {
    this[name] = template;
    logger.info(`[PromptTemplates] 사용자 정의 템플릿 추가: ${name}`);
  }

  /**
   * 템플릿 목록 조회
   *
   * @returns {Array} 템플릿 이름 목록
   */
  listTemplates() {
    return Object.keys(this).filter(key =>
      typeof this[key] === 'object' && this[key].template
    );
  }
}

export default PromptTemplates;
