/**
 * Ollama 클라이언트 테스트 스크립트
 *
 * 기본 동작 검증:
 * 1. Ollama 서버 연결
 * 2. 모델 목록 조회
 * 3. 기본 채팅
 * 4. JSON 출력
 */

import OllamaClient from './ollama-client.js';
import ModelManager from './model-manager.js';
import PromptTemplates from './prompt-templates.js';

async function testOllamaClient() {
  console.log('=== Ollama Client 테스트 시작 ===\n');

  try {
    // 1. OllamaClient 초기화
    console.log('1. OllamaClient 초기화...');
    const client = new OllamaClient({
      baseURL: process.env.OLLAMA_PROXY_SERVER || 'https://api.hamonize.com/ollama',
    });
    console.log('✅ OllamaClient 초기화 성공\n');

    // 2. 모델 목록 조회
    console.log('2. 모델 목록 조회...');
    const models = await client.listModels();
    console.log(`✅ 발견된 모델: ${models.map(m => m.name).join(', ')}\n`);

    // 3. 기본 채팅 테스트
    console.log('3. 기본 채팅 테스트...');
    const chatResponse = await client.chat('hamonize:latest', [
      { role: 'user', content: '안녕하세요. 간단히 인사해주세요.' }
    ]);
    console.log('✅ 채팅 응답:', chatResponse.content.slice(0, 100), '...\n');

    // 4. JSON 출력 테스트
    console.log('4. JSON 출력 테스트...');
    const jsonResponse = await client.chatWithJSON('hamonize:latest', [
      { role: 'user', content: '다음 형식으로 응답하세요: {"message": "안녕하세요", "status": "success"}' }
    ]);
    console.log('✅ JSON 응답:', JSON.stringify(jsonResponse.content, null, 2));
    console.log('✅ JSON 유효성:', jsonResponse.isValidJSON ? '유효' : '무효', '\n');

    // 5. 모델 상태 확인
    console.log('5. 모델 상태 확인...');
    const health = await client.checkModelHealth('hamonize:latest');
    console.log('✅ 모델 상태:', health.healthy ? '정상' : '비정상', '\n');

    console.log('=== 모든 테스트 통과 ===');
    return true;
  } catch (error) {
    console.error('❌ 테스트 실패:', error.message);
    console.error(error.stack);
    return false;
  }
}

async function testModelManager() {
  console.log('\n=== ModelManager 테스트 시작 ===\n');

  try {
    // 1. ModelManager 초기화
    console.log('1. ModelManager 초기화...');
    const manager = new ModelManager({
      ollamaUrl: process.env.OLLAMA_PROXY_SERVER || 'https://api.hamonize.com/ollama',
      defaultModel: 'hamonize:latest',
    });
    console.log('✅ ModelManager 초기화 성공\n');

    // 2. 작업별 모델 선택
    console.log('2. 작업별 모델 선택...');
    const supervisorModel = manager.selectModel('supervisor');
    const queryModel = manager.selectModel('queryGeneration');
    console.log(`✅ Supervisor 모델: ${supervisorModel}`);
    console.log(`✅ Query 생성 모델: ${queryModel}\n`);

    // 3. 채팅 with 폴백
    console.log('3. 채팅 (폴백 지원)...');
    const response = await manager.chat('hamonize:latest', [
      { role: 'user', content: '파이썬이란?' }
    ]);
    console.log('✅ 응답:', response.content.slice(0, 100), '...\n');

    // 4. JSON 채팅
    console.log('4. JSON 채팅...');
    const jsonResponse = await manager.chatWithJSON('hamonize:latest', [
      { role: 'user', content: '다음 JSON으로 답하세요: {"language": "Python", "type": "programming"}' }
    ]);
    console.log('✅ JSON 응답:', JSON.stringify(jsonResponse.content, null, 2), '\n');

    // 5. 모델 상태 확인
    console.log('5. 모든 모델 상태 확인...');
    const healthMap = manager.getModelsHealth();
    console.log('✅ 모델 상태:', JSON.stringify(healthMap, null, 2), '\n');

    console.log('=== 모든 테스트 통과 ===');
    return true;
  } catch (error) {
    console.error('❌ 테스트 실패:', error.message);
    console.error(error.stack);
    return false;
  }
}

async function testPromptTemplates() {
  console.log('\n=== PromptTemplates 테스트 시작 ===\n');

  try {
    // 1. PromptTemplates 초기화
    console.log('1. PromptTemplates 초기화...');
    const templates = new PromptTemplates();
    console.log('✅ PromptTemplates 초기화 성공\n');

    // 2. 템플릿 목록 조회
    console.log('2. 템플릿 목록 조회...');
    const templateList = templates.listTemplates();
    console.log('✅ 사용 가능한 템플릿:', templateList.join(', '), '\n');

    // 3. Task Analysis 템플릿 렌더링
    console.log('3. Task Analysis 템플릿 렌더링...');
    const taskAnalysisPrompt = templates.render('taskAnalysis', {
      query: '우리 회사 AI 전략과 최신 트렌드 비교해줘'
    }, false); // Few-shot 제외
    console.log('✅ 렌더링된 프롬프트:\n', taskAnalysisPrompt.slice(0, 200), '...\n');

    // 4. Query Generation 템플릿 렌더링
    console.log('4. Query Generation 템플릿 렌더링...');
    const queryGenPrompt = templates.render('queryGeneration', {
      originalQuery: '파이썬 웹 프레임워크 추천',
      targetAgents: ['rag', 'web']
    });
    console.log('✅ 렌더링된 프롬프트:\n', queryGenPrompt.slice(0, 200), '...\n');

    console.log('=== 모든 테스트 통과 ===');
    return true;
  } catch (error) {
    console.error('❌ 테스트 실패:', error.message);
    console.error(error.stack);
    return false;
  }
}

async function runAllTests() {
  console.log('╔═══════════════════════════════════════════╗');
  console.log('║  AIRUN Ollama Integration Test Suite     ║');
  console.log('╚═══════════════════════════════════════════╝\n');

  const results = {
    ollamaClient: false,
    modelManager: false,
    promptTemplates: false,
  };

  // OllamaClient 테스트
  results.ollamaClient = await testOllamaClient();

  // ModelManager 테스트
  results.modelManager = await testModelManager();

  // PromptTemplates 테스트
  results.promptTemplates = await testPromptTemplates();

  // 결과 요약
  console.log('\n╔═══════════════════════════════════════════╗');
  console.log('║           테스트 결과 요약                ║');
  console.log('╚═══════════════════════════════════════════╝\n');
  console.log(`OllamaClient:     ${results.ollamaClient ? '✅ 통과' : '❌ 실패'}`);
  console.log(`ModelManager:     ${results.modelManager ? '✅ 통과' : '❌ 실패'}`);
  console.log(`PromptTemplates:  ${results.promptTemplates ? '✅ 통과' : '❌ 실패'}`);

  const allPassed = Object.values(results).every(r => r === true);
  console.log(`\n전체 결과: ${allPassed ? '✅ 모든 테스트 통과' : '❌ 일부 테스트 실패'}`);

  process.exit(allPassed ? 0 : 1);
}

// 스크립트 직접 실행 시
if (import.meta.url === `file://${process.argv[1]}`) {
  runAllTests().catch(error => {
    console.error('치명적 오류:', error);
    process.exit(1);
  });
}

export {
  testOllamaClient,
  testModelManager,
  testPromptTemplates,
};
