/**
 * Self-Reflector for Self-Reflective RAG
 *
 * RAG 결과의 품질을 평가하고 스스스로 개선하는 메커니즘
 *
 * @module services/agent-system/reflection/self-reflector
 */

import { logger } from '../../../utils/logger.js';
import ModelManager from '../ollama/model-manager.js';

class SelfReflector {
    constructor(options = {}) {
        this.modelManager = new ModelManager({
            ollamaUrl: options.ollamaUrl || process.env.OLLAMA_PROXY_SERVER,
            defaultModel: 'hamonize:latest',
        });

        this.maxReflections = options.maxReflections || 2;
        this.confidenceThreshold = options.confidenceThreshold || 0.7;
        this.improvementThreshold = options.improvementThreshold || 0.2;

        this.reflectionPrompts = {
            // 질문 이해도 평가
            questionClarity: `
다음 사용자 질문과 RAG 검색 결과를 분석하고, 질문의 명확성과 구체성을 평가하세요:

원본 질문: {originalQuery}
검색 결과: {searchResults}

평가 기준:
1. 질문의 명확성 (1-10)
2. 검색 결과와의 관련성 (1-10)
3. 질문의 구체성 (1-10)
4. 애매함 정도 (1-10)

결과 형식:
JSON 형식으로 {{
  "question_clarity_score": 8.5,
  "relevance_score": 7.2,
  "specificity_score": 9.0,
  "ambiguity_score": 3.5,
  "overall_confidence": 0.75,
  "issues": ["검색어가 너무 일반적임", "구체적인 맥락 제시 필요"],
  "suggestions": ["더 구체적인 검색어 사용", "시간이나 장소 명시"]
}}
            `,

            // 답변 품질 평가
            answerQuality: `
다음 답변의 품질을 다양한 차원에서 평가하세요:

사용자 질문: {originalQuery}
검색된 문서: {retrievedDocs}
생성된 답변: {generatedAnswer}

평가 차원:
1. 정확성 (Accuracy): 사실 기반 정확도 (1-10)
2. 완전성 (Completeness): 질문에 대한 충분한 답변 (1-10)
3. 일관성 (Consistency): 문서들과의 일관성 (1-10)
4. 유용성 (Usefulness): 실용적인 정보 제공 (1-10)
5. 명확성 (Clarity): 이해하기 쉬운 표현 (1-10)

결과 형식:
JSON 형식으로 {{
  "accuracy_score": 8.0,
  "completeness_score": 7.5,
  "consistency_score": 8.2,
  "usefulness_score": 9.0,
  "clarity_score": 8.5,
  "overall_quality": 0.82,
  "strengths": ["정확한 정보 제공", "명확한 구조"],
  "weaknesses": ["일부 정보 누락", "출처 명시 부족"],
  "improvement_suggestions": ["출처 정보 추가", "누락된 내용 보완"]
}}
            `,

            // 자기 수정 프롬프트
            selfCorrection: `
이전 답변과 피드백을 바탕으로 개선된 답변을 생성하세요:

원본 질문: {originalQuery}
이전 답변: {previousAnswer}
개선 제안: {feedback}
검색된 문서: {retrievedDocs}

지침:
1. 피드백의 약점을 모두 보완하세요
2. 새로운 정보나 관점을 추가할 수 있지만 문서 기반으로 해야 합니다
3. 답변을 더 구체적이고 유용하게 만드세요
4. 출처를 명확히 표기하세요
5. 이전 답변의 장점은 유지하세요

개선된 답변:
            `
        };
    }

    /**
     * 질문의 명확성을 평가
     * @param {string} query - 원본 질문
     * @param {Array} searchResults - 검색 결과
     * @returns {Object} 평가 결과
     */
    async evaluateQuestionClarity(query, searchResults) {
        try {
            const prompt = this.reflectionPrompts.questionClarity
                .replace('{originalQuery}', query)
                .replace('{searchResults}', JSON.stringify(searchResults, null, 2));

            const response = await this.modelManager.generateResponse(prompt, {
                temperature: 0.3,
                maxTokens: 500
            });

            // JSON 파싱
            const evaluation = this.parseJSONResponse(response);

            if (evaluation) {
                logger.info(`[SelfReflector] 질문 명확성 평가: ${evaluation.overall_confidence}`);
                return evaluation;
            }
        } catch (error) {
            logger.error(`[SelfReflector] 질문 평가 실패: ${error.message}`);
        }

        return {
            overall_confidence: 0.5,
            issues: ["평가 실패"],
            suggestions: ["질문을 더 구체적으로 작성"]
        };
    }

    /**
     * 답변의 품질을 평가
     * @param {string} query - 원본 질문
     * @param {string} answer - 생성된 답변
     * @param {Array} retrievedDocs - 검색된 문서
     * @returns {Object} 평가 결과
     */
    async evaluateAnswerQuality(query, answer, retrievedDocs) {
        try {
            const prompt = this.reflectionPrompts.answerQuality
                .replace('{originalQuery}', query)
                .replace('{generatedAnswer}', answer)
                .replace('{retrievedDocs}', JSON.stringify(retrievedDocs, null, 2));

            const response = await this.modelManager.generateResponse(prompt, {
                temperature: 0.3,
                maxTokens: 500
            });

            const evaluation = this.parseJSONResponse(response);

            if (evaluation) {
                logger.info(`[SelfReflector] 답변 품질 평가: ${evaluation.overall_quality}`);
                return evaluation;
            }
        } catch (error) {
            logger.error(`[SelfReflector] 답변 평가 실패: ${error.message}`);
        }

        return {
            overall_quality: 0.5,
            strengths: ["기본 답변 제공"],
            weaknesses: ["평가 실패"],
            improvement_suggestions: ["내용 보완 필요"]
        };
    }

    /**
     * 자기 수정을 통해 답변 개선
     * @param {string} query - 원본 질문
     * @param {string} previousAnswer - 이전 답변
     * @param {Object} feedback - 개선 제안
     * @param {Array} retrievedDocs - 검색된 문서
     * @returns {string} 개선된 답변
     */
    async improveAnswer(query, previousAnswer, feedback, retrievedDocs) {
        try {
            const prompt = this.reflectionPrompts.selfCorrection
                .replace('{originalQuery}', query)
                .replace('{previousAnswer}', previousAnswer)
                .replace('{feedback}', JSON.stringify(feedback, null, 2))
                .replace('{retrievedDocs}', JSON.stringify(retrievedDocs, null, 2));

            const improvedAnswer = await this.modelManager.generateResponse(prompt, {
                temperature: 0.4,
                maxTokens: 1000
            });

            logger.info('[SelfReflector] 자기 수정 완료');
            return improvedAnswer;
        } catch (error) {
            logger.error(`[SelfReflector] 자기 수정 실패: ${error.message}`);
            return previousAnswer;
        }
    }

    /**
     * 피드백 루프 실행
     * @param {Object} context - 검색 컨텍스트
     * @returns {Object} 최종 결과
     */
    async executeFeedbackLoop(context) {
        let currentAnswer = context.generatedAnswer;
        let currentQuality = 0;
        let improvementHistory = [];

        for (let iteration = 0; iteration < this.maxReflections; iteration++) {
            // 1. 답변 품질 평가
            const qualityEvaluation = await this.evaluateAnswerQuality(
                context.query,
                currentAnswer,
                context.retrievedDocuments
            );

            currentQuality = qualityEvaluation.overall_quality;

            // 2. 품질 기준 충족 시 종료
            if (currentQuality >= this.confidenceThreshold) {
                logger.info(`[SelfReflector] 품질 기준 충족 (${currentQuality}), 루프 종료`);
                break;
            }

            // 3. 개선 제안이 없는 경우 종료
            if (!qualityEvaluation.improvement_suggestions ||
                qualityEvaluation.improvement_suggestions.length === 0) {
                logger.info('[SelfReflector] 개선 제안 없음, 루프 종료');
                break;
            }

            // 4. 이전 개선과 비교하여 의미 있는 개선인지 확인
            if (iteration > 0) {
                const improvementAmount = currentQuality -
                    (improvementHistory[improvementHistory.length - 1]?.quality || 0);

                if (improvementAmount < this.improvementThreshold) {
                    logger.info(`[SelfReflector] 개선 폭 부족 (${improvementAmount}), 루프 종료`);
                    break;
                }
            }

            // 5. 자기 수정 실행
            logger.info(`[SelfReflector] ${iteration + 1}번째 자기 수정 실행`);
            currentAnswer = await this.improveAnswer(
                context.query,
                currentAnswer,
                {
                    weaknesses: qualityEvaluation.weaknesses,
                    improvement_suggestions: qualityEvaluation.improvement_suggestions
                },
                context.retrievedDocuments
            );

            // 6. 개선 이력 기록
            improvementHistory.push({
                iteration: iteration,
                quality: currentQuality,
                feedback: qualityEvaluation
            });
        }

        return {
            finalAnswer: currentAnswer,
            finalQuality: currentQuality,
            iterations: improvementHistory.length,
            improvementHistory: improvementHistory,
            confidence: Math.min(currentQuality, 0.95)
        };
    }

    /**
     * JSON 응답을 파싱
     * @param {string} response - 모델 응답
     * @returns {Object|null} 파싱된 객체
     */
    parseJSONResponse(response) {
        try {
            // ```json``` 블록 추출
            const jsonMatch = response.match(/```json\s*([\s\S]*?)\s*```/);
            if (jsonMatch) {
                return JSON.parse(jsonMatch[1]);
            }

            // 중괄호로 묶인 JSON 객체 추출
            const braceMatch = response.match(/\{[\s\S]*\}/);
            if (braceMatch) {
                return JSON.parse(braceMatch[0]);
            }
        } catch (error) {
            logger.debug(`[SelfReflector] JSON 파싱 실패: ${error.message}`);
        }
        return null;
    }
}

export default SelfReflector;