/**
 * Query Rewriter
 *
 * 검색 결과 품질 개선을 위한 쿼리 자동 재작성 모듈
 * Corrective RAG (CRAG) 패턴의 핵심 컴포넌트
 *
 * @module services/agent-system/rewriters/query-rewriter
 */

import ModelManager from '../ollama/model-manager.js';
import { logger } from '../../../utils/logger.js';

// 재작성 전략 상수
const STRATEGY = {
  EXPAND: 'EXPAND',         // 동의어/관련어 추가로 검색 범위 확대
  NARROW: 'NARROW',         // 핵심 키워드로 검색 범위 축소
  TRANSFORM: 'TRANSFORM',   // 다른 관점에서 쿼리 재구성
  DECOMPOSE: 'DECOMPOSE'    // 복잡한 쿼리를 하위 쿼리로 분해
};

class QueryRewriter {
  /**
   * @param {Object} options - 리라이터 옵션
   * @param {Object} options.modelManager - 기존 ModelManager 인스턴스 (선택 사항)
   * @param {string} options.ollamaUrl - Ollama 서버 URL (modelManager 없을 때만 사용)
   * @param {number} options.maxAlternatives - 최대 대안 쿼리 수 (기본: 3)
   */
  constructor(options = {}) {
    // 기존 ModelManager를 재사용하거나 새로 생성
    if (options.modelManager) {
      this.modelManager = options.modelManager;
      logger.info('[QueryRewriter] 기존 ModelManager 재사용');
    } else {
      this.modelManager = new ModelManager({
        ollamaUrl: options.ollamaUrl || process.env.OLLAMA_PROXY_SERVER,
        defaultModel: 'hamonize:latest',
      });
      logger.info('[QueryRewriter] 새로운 ModelManager 생성');
    }

    this.maxAlternatives = options.maxAlternatives || 3;

    logger.info('[QueryRewriter] 초기화 완료');
  }

  /**
   * 쿼리 재작성 메인 메서드
   *
   * @param {string} originalQuery - 원본 쿼리
   * @param {Object} context - 재작성 컨텍스트
   * @param {Object} context.gradeResult - Document Grader 결과
   * @param {Array} context.failedDocuments - 관련성 낮은 문서들
   * @param {string} context.failureReason - 검색 실패 이유
   * @returns {Promise<Object>} 재작성 결과
   */
  async rewriteQuery(originalQuery, context = {}) {
    try {
      const startTime = Date.now();
      logger.info(`[QueryRewriter] 쿼리 재작성 시작: "${originalQuery}"`);

      // 1. 쿼리 분석 및 실패 원인 파악
      const analysis = await this._analyzeQuery(originalQuery, context);

      // 2. 재작성 전략 선택
      const strategy = this._selectStrategy(analysis, context);
      logger.info(`[QueryRewriter] 선택된 전략: ${strategy}`);

      // 3. 전략에 따른 쿼리 생성
      const rewrittenQueries = await this._generateQueries(
        originalQuery,
        strategy,
        analysis,
        context
      );

      const processingTime = Date.now() - startTime;

      logger.info(`[QueryRewriter] 재작성 완료: ${Object.keys(rewrittenQueries.queries).length}개 쿼리 생성`);

      return {
        success: true,
        originalQuery,
        strategy,
        rewrittenQueries: rewrittenQueries.queries,
        alternatives: rewrittenQueries.alternatives || [],
        reasoning: rewrittenQueries.reasoning,
        analysis,
        processingTime
      };

    } catch (error) {
      logger.error(`[QueryRewriter] 쿼리 재작성 실패: ${error.message}`);

      // 폴백: 기본 재작성 적용
      return this._fallbackRewrite(originalQuery);
    }
  }

  /**
   * 쿼리 분석
   *
   * @private
   * @param {string} query - 원본 쿼리
   * @param {Object} context - 컨텍스트
   * @returns {Promise<Object>} 분석 결과
   */
  async _analyzeQuery(query, context) {
    const prompt = `당신은 검색 쿼리를 분석하는 전문가입니다.

쿼리: ${query}

${context.failureReason ? `검색 실패 이유: ${context.failureReason}` : ''}

다음을 분석하세요:
1. 쿼리의 핵심 의도
2. 핵심 키워드 추출
3. 쿼리의 복잡도 (simple/medium/complex)
4. 잠재적 모호성
5. 검색 개선을 위한 제안

다음 형식의 JSON으로 응답하세요:
{
  "intent": "쿼리의 핵심 의도",
  "keywords": ["핵심", "키워드", "목록"],
  "complexity": "simple|medium|complex",
  "ambiguities": ["모호한 부분1", "모호한 부분2"],
  "suggestions": ["개선 제안1", "개선 제안2"]
}`;

    try {
      const response = await this.modelManager.chatWithJSON(
        this.modelManager.selectModel('queryGeneration'),
        [{ role: 'user', content: prompt }],
        {
          intent: 'string',
          keywords: 'array',
          complexity: 'string',
          ambiguities: 'array',
          suggestions: 'array'
        }
      );

      return response.content;

    } catch (error) {
      logger.error(`[QueryRewriter] 쿼리 분석 실패: ${error.message}`);

      // 기본 분석 반환
      return {
        intent: query,
        keywords: query.split(' ').filter(w => w.length > 1),
        complexity: 'medium',
        ambiguities: [],
        suggestions: []
      };
    }
  }

  /**
   * 재작성 전략 선택
   *
   * @private
   * @param {Object} analysis - 쿼리 분석 결과
   * @param {Object} context - 컨텍스트
   * @returns {string} 선택된 전략
   */
  _selectStrategy(analysis, context) {
    const { gradeResult } = context;

    // 복잡한 쿼리는 분해
    if (analysis.complexity === 'complex' || analysis.keywords.length > 5) {
      return STRATEGY.DECOMPOSE;
    }

    // INCORRECT가 많으면 변환
    if (gradeResult && gradeResult.action) {
      const { incorrectCount, totalDocuments } = gradeResult.action;
      if (incorrectCount / totalDocuments > 0.7) {
        return STRATEGY.TRANSFORM;
      }
    }

    // 모호성이 있으면 구체화
    if (analysis.ambiguities && analysis.ambiguities.length > 0) {
      return STRATEGY.NARROW;
    }

    // 키워드가 적으면 확장
    if (analysis.keywords.length < 3) {
      return STRATEGY.EXPAND;
    }

    // 기본: 변환
    return STRATEGY.TRANSFORM;
  }

  /**
   * 전략에 따른 쿼리 생성
   *
   * @private
   * @param {string} originalQuery - 원본 쿼리
   * @param {string} strategy - 재작성 전략
   * @param {Object} analysis - 쿼리 분석 결과
   * @param {Object} context - 컨텍스트
   * @returns {Promise<Object>} 생성된 쿼리
   */
  async _generateQueries(originalQuery, strategy, analysis, context) {
    const prompt = this._buildGenerationPrompt(originalQuery, strategy, analysis);

    try {
      const response = await this.modelManager.chatWithJSON(
        this.modelManager.selectModel('queryGeneration'),
        [{ role: 'user', content: prompt }],
        {
          queries: 'object',
          alternatives: 'array',
          reasoning: 'string'
        }
      );

      return response.content;

    } catch (error) {
      logger.error(`[QueryRewriter] 쿼리 생성 실패: ${error.message}`);

      // 폴백 쿼리 생성
      return {
        queries: {
          rag: analysis.keywords.join(' '),
          web: `${originalQuery} 최신 정보`
        },
        alternatives: [],
        reasoning: '기본 키워드 기반 재작성'
      };
    }
  }

  /**
   * 쿼리 생성 프롬프트 빌드
   *
   * @private
   * @param {string} originalQuery - 원본 쿼리
   * @param {string} strategy - 재작성 전략
   * @param {Object} analysis - 쿼리 분석 결과
   * @returns {string} 프롬프트
   */
  _buildGenerationPrompt(originalQuery, strategy, analysis) {
    const strategyInstructions = {
      [STRATEGY.EXPAND]: `검색 범위를 확대하세요:
- 동의어와 관련 용어 추가
- 더 넓은 개념으로 확장
- 영문/한글 혼용 검색어 생성`,

      [STRATEGY.NARROW]: `검색 범위를 축소하세요:
- 핵심 키워드만 추출
- 구체적인 용어 사용
- 불필요한 단어 제거`,

      [STRATEGY.TRANSFORM]: `다른 관점에서 재구성하세요:
- 질문을 다른 방식으로 표현
- 다른 키워드 조합 시도
- 의도는 유지하되 표현 변경`,

      [STRATEGY.DECOMPOSE]: `하위 쿼리로 분해하세요:
- 복합 질문을 개별 질문으로 분리
- 각 측면을 별도로 검색
- 단계별 정보 수집`
    };

    return `당신은 검색 쿼리를 최적화하는 전문가입니다.

원본 쿼리: ${originalQuery}
쿼리 의도: ${analysis.intent}
핵심 키워드: ${analysis.keywords.join(', ')}

전략: ${strategy}
${strategyInstructions[strategy]}

각 에이전트에 최적화된 쿼리를 생성하세요:
- rag: 내부 문서 검색용 (한국어 키워드 중심)
- web: 웹 검색용 (영문 포함, 최신 정보 중심)

다음 형식의 JSON으로 응답하세요:
{
  "queries": {
    "rag": "RAG 검색용 쿼리",
    "web": "웹 검색용 쿼리"
  },
  "alternatives": [
    "대안 쿼리 1",
    "대안 쿼리 2"
  ],
  "reasoning": "재작성 근거 (50자 이내)"
}`;
  }

  /**
   * 폴백 재작성
   *
   * @private
   * @param {string} originalQuery - 원본 쿼리
   * @returns {Object} 폴백 재작성 결과
   */
  _fallbackRewrite(originalQuery) {
    const words = originalQuery.split(' ').filter(w => w.length > 1);
    const keywordQuery = words.slice(0, 5).join(' ');

    return {
      success: false,
      originalQuery,
      strategy: STRATEGY.TRANSFORM,
      rewrittenQueries: {
        rag: keywordQuery,
        web: `${originalQuery} 정보 검색`
      },
      alternatives: [
        keywordQuery,
        `${words[0]} ${words[words.length - 1] || ''} 관련`
      ],
      reasoning: '폴백 재작성 적용',
      analysis: null,
      processingTime: 0
    };
  }

  /**
   * 쿼리 품질 평가
   *
   * @param {string} query - 평가할 쿼리
   * @returns {Object} 품질 평가 결과
   */
  evaluateQueryQuality(query) {
    const words = query.split(' ').filter(w => w.length > 1);

    let score = 0;
    const issues = [];

    // 길이 체크
    if (words.length < 2) {
      score -= 0.2;
      issues.push('쿼리가 너무 짧음');
    } else if (words.length > 10) {
      score -= 0.1;
      issues.push('쿼리가 너무 김');
    } else {
      score += 0.3;
    }

    // 특수문자 체크
    if (/[!@#$%^&*()+=\[\]{};':"\\|<>\/?]+/.test(query)) {
      score -= 0.1;
      issues.push('불필요한 특수문자 포함');
    } else {
      score += 0.2;
    }

    // 의미있는 단어 체크
    const stopwords = ['의', '을', '를', '이', '가', '은', '는', '에', '에서', '로', '으로'];
    const meaningfulWords = words.filter(w => !stopwords.includes(w));
    if (meaningfulWords.length / words.length > 0.7) {
      score += 0.3;
    } else {
      score -= 0.1;
      issues.push('조사/접속사 비율이 높음');
    }

    // 정규화
    score = Math.max(0, Math.min(1, score + 0.5));

    return {
      score: Math.round(score * 100) / 100,
      issues,
      wordCount: words.length,
      meaningfulWordCount: meaningfulWords.length
    };
  }

  /**
   * 재작성 결과 요약 생성
   *
   * @param {Object} rewriteResult - rewriteQuery() 반환 결과
   * @returns {string} 요약 텍스트
   */
  generateSummary(rewriteResult) {
    const { originalQuery, strategy, rewrittenQueries, reasoning, processingTime } = rewriteResult;

    return `[쿼리 재작성 결과]
- 원본 쿼리: ${originalQuery}
- 전략: ${strategy}
- RAG 쿼리: ${rewrittenQueries.rag}
- Web 쿼리: ${rewrittenQueries.web}
- 근거: ${reasoning}
- 처리 시간: ${processingTime}ms`;
  }
}

// 상수 내보내기
export { STRATEGY };
export default QueryRewriter;
