/**
 * Supervisor Streaming Wrapper
 *
 * Supervisor의 결과를 WebSocket 스트리밍으로 전송하기 위한 래퍼
 *
 * @module services/agent-system/supervisor-stream
 */

import { getSupervisor, initializeSupervisor } from './index.js';
import { logger } from '../../utils/logger.js';

/**
 * Supervisor 결과를 스트리밍으로 전송
 *
 * @param {WebSocket} ws - WebSocket 연결
 * @param {string} prompt - 사용자 질문
 * @param {Object} options - 옵션
 * @param {Object} callbacks - 콜백 함수들
 * @returns {Promise<Object>} 최종 결과
 */
export async function streamSupervisorResult(ws, prompt, options, callbacks = {}) {
  try {
    logger.info('[SupervisorStream] Deep Research 시작');

    // Supervisor 초기화
    const supervisor = getSupervisor() || initializeSupervisor();

    // 진행 상태 메시지 전송
    const sendProgress = (message, step, totalSteps) => {
      if (callbacks.onProgress && ws.readyState === 1) {
        // 다른 도구들과 동일한 형식으로 전송
        const percent = totalSteps ? Math.round((step / totalSteps) * 100) : 0;
        callbacks.onProgress({
          type: 'progress',
          message,
          percent,
          stage: 'deep_research',
          timestamp: new Date().toISOString()
        });
      }
    };

    // 1단계: 작업 분석 시작
    sendProgress('🔍 질문을 분석하고 있습니다...', 1, 6);

    const context = {
      query: prompt,
      sessionId: options.sessionId || `supervisor-${Date.now()}`,
      username: options.username || 'anonymous',
      options: options,
      // 스트리밍 콜백 추가
      onStream: (chunk) => {
        if (callbacks.onStream && ws.readyState === 1) {
          callbacks.onStream({
            type: 'stream_content',
            content: chunk,
            timestamp: new Date().toISOString()
          });
        }
      }
    };

    // Supervisor 처리 시작
    const startTime = Date.now();

    // 진행 상태를 추적하기 위한 변수
    let currentStep = 1;

    // Supervisor 처리 전에 비동기적으로 진행 메시지 전송
    // 실제 처리는 동시에 진행
    const progressPromise = (async () => {
      // 2단계: 작업 분석
      sendProgress('🎯 필요한 에이전트를 선정하고 있습니다...', 2, 6);
      await new Promise(resolve => setTimeout(resolve, 100));

      // 3단계: 쿼리 생성 및 에이전트 실행
      sendProgress('🤖 검색 쿼리를 생성하고 에이전트를 실행합니다...', 3, 6);
      await new Promise(resolve => setTimeout(resolve, 100));

      // 4단계: 결과 수집
      sendProgress('📊 검색 결과를 수집하고 있습니다...', 4, 6);
      await new Promise(resolve => setTimeout(resolve, 100));

      // 5단계: 결과 종합
      sendProgress('✨ 결과를 종합하고 있습니다...', 5, 6);
      await new Promise(resolve => setTimeout(resolve, 100));

      // 6단계: 품질 검증
      sendProgress('✅ 품질을 검증하고 있습니다...', 6, 6);
    })();

    // Supervisor 처리 (실제 작업)
    const supervisorResult = await supervisor.process(context);

    // 진행 메시지 전송 완료 대기
    await progressPromise;

    const elapsedTime = ((Date.now() - startTime) / 1000).toFixed(1);

    // ReferenceManager 패턴 적용: 에이전트 결과에서 출처 정보 수집
    const documentReferences = new Map(); // filename -> Set<pageNumbers>
    const webReferences = [];

    // 에이전트 결과에서 출처 정보 추출 (mainLogic.js의 ReferenceManager 패턴 재사용)
    if (supervisorResult.metadata?.agentResults) {
      const agentResults = supervisorResult.metadata.agentResults;

      // RAG 에이전트 결과 처리
      if (agentResults.rag && agentResults.rag.success && Array.isArray(agentResults.rag.results)) {
        for (const doc of agentResults.rag.results) {
          const filename = doc.metadata?.filename || doc.filename;
          if (!filename) continue;

          if (!documentReferences.has(filename)) {
            documentReferences.set(filename, new Set());
          }

          const page = doc.metadata?.page;
          if (page) {
            documentReferences.get(filename).add(page);
          }
        }
        logger.info(`[SupervisorStream] RAG 출처 수집: ${documentReferences.size}개 문서`);
      }

      // Web 에이전트 결과 처리
      if (agentResults.web && agentResults.web.success && Array.isArray(agentResults.web.results)) {
        for (const webResult of agentResults.web.results) {
          const url = webResult.url || webResult.link;
          if (!url) continue;

          webReferences.push({
            title: webResult.title || '제목 없음',
            url: url,
            snippet: webResult.snippet || webResult.description || ''
          });
        }
        logger.info(`[SupervisorStream] 웹 출처 수집: ${webReferences.length}개`);
      }
    }

    // AI가 생성한 cited_documents JSON에서 실제 인용된 문서 추출
    let downloadInfo = [];
    const citedDocsPattern = /\{"cited_documents"[^}]*\[.*?\]\}/;
    const citedDocsMatch = supervisorResult.content.match(citedDocsPattern);
    
    if (citedDocsMatch) {
      try {
        const citedDocsJson = JSON.parse(citedDocsMatch[0]);
        const citedDocuments = citedDocsJson.cited_documents;
        
        if (Array.isArray(citedDocuments)) {
          // mainLogic.js의 filterActuallyCitedDocuments 로직 재사용
          for (const citedDoc of citedDocuments) {
            if (typeof citedDoc !== 'string') continue;
            
            const trimmedDoc = citedDoc.trim();
            if (!trimmedDoc) continue;

            // 문서명 매칭 (정확한 매칭 또는 부분 매칭)
            for (const [filename, pages] of documentReferences) {
              const match = filename === trimmedDoc || 
                           filename.includes(trimmedDoc) || 
                           trimmedDoc.includes(filename);
              
              if (match) {
                downloadInfo.push({
                  filename: filename,
                  source: 'rag',
                  pages: pages.size > 0 ? Array.from(pages).sort((a, b) => a - b) : []
                });
                logger.debug(`[SupervisorStream] RAG 출처 매칭: ${filename}, 페이지: ${Array.from(pages).join(', ')}`);
                break;
              }
            }
          }
          
          logger.info(`[SupervisorStream] cited_documents에서 ${downloadInfo.length}개 필터링됨`);
        }
      } catch (e) {
        logger.error(`[SupervisorStream] cited_documents 파싱 실패: ${e.message}`);
      }
    }

    // AI 응답에서 명시적 언급 확인 (fallback: cited_documents가 없는 경우)
    if (downloadInfo.length === 0 && documentReferences.size > 0) {
      logger.warn('[SupervisorStream] cited_documents 없음, 응답 내용에서 직접 확인');
      
      const normalizedContent = supervisorResult.content
        .replace(/[\u2010-\u2015]/g, '-')
        .replace(/\u00A0/g, ' ')
        .replace(/\u202F/g, ' ');

      for (const [filename, pages] of documentReferences) {
        const filenameWithoutExt = filename.replace(/\.[^/.]+$/, '');
        
        if (normalizedContent.includes(filename) || normalizedContent.includes(filenameWithoutExt)) {
          downloadInfo.push({
            filename: filename,
            source: 'rag',
            pages: pages.size > 0 ? Array.from(pages).sort((a, b) => a - b) : []
          });
        }
      }
    }

    // 웹 출처 추가
    for (const webRef of webReferences) {
      downloadInfo.push({
        title: webRef.title,
        url: webRef.url,
        source: 'web',
        snippet: webRef.snippet
      });
    }

    logger.info(`[SupervisorStream] 최종 출처 정보: ${downloadInfo.length}개 (RAG: ${downloadInfo.filter(d => d.source === 'rag').length}, Web: ${downloadInfo.filter(d => d.source === 'web').length})`)

    // 최종 결과 스트리밍
    if (supervisorResult.success) {
      logger.info(
        `[SupervisorStream] Deep Research 성공 (${elapsedTime}s, ${supervisorResult.metadata.attempts}회 시도)`
      );

      // 메타데이터 포함 응답
      const responseWithMetadata = `${supervisorResult.content}

---

**Deep Research 정보:**
- 사용된 에이전트: ${supervisorResult.metadata.selectedAgents.join(', ')}
- 품질 점수: ${(supervisorResult.metadata.qualityScore * 100).toFixed(0)}%
- 시도 횟수: ${supervisorResult.metadata.attempts}회
- 처리 시간: ${elapsedTime}초`;

      return {
        success: true,
        content: responseWithMetadata,
        downloadInfo: downloadInfo, // 출처 정보 추가 (기존 방식과 동일)
        metadata: {
          ...supervisorResult.metadata,
          processingTime: elapsedTime,
          deepResearch: true,
        },
      };
    } else {
      logger.warn(
        `[SupervisorStream] Deep Research 품질 미달 (${elapsedTime}s)`
      );

      // 품질 미달 시에도 결과 반환
      const responseWithWarning = `${supervisorResult.content}

---

⚠️ **참고:** 검색 결과의 품질이 기대에 미치지 못할 수 있습니다.
- 품질 점수: ${(supervisorResult.metadata.qualityScore * 100).toFixed(0)}%
- 시도 횟수: ${supervisorResult.metadata.attempts}회

더 정확한 정보가 필요하시면 질문을 구체적으로 다시 해주세요.`;

      return {
        success: false,
        content: responseWithWarning,
        metadata: {
          ...supervisorResult.metadata,
          processingTime: elapsedTime,
          deepResearch: true,
          qualityWarning: true,
        },
      };
    }
  } catch (error) {
    logger.error(`[SupervisorStream] Deep Research 실패: ${error.message}`);

    return {
      success: false,
      content: `Deep Research 처리 중 오류가 발생했습니다: ${error.message}`,
      error: error.message,
      metadata: {
        deepResearch: true,
        error: true,
      },
    };
  }
}

export default streamSupervisorResult;
