/**
 * Supervisor Multi-Agent 시스템 오케스트레이터
 *
 * 여러 전문화된 에이전트를 조율하여 복잡한 질문에 대한 종합적인 답변 생성
 * - 작업 분석 및 에이전트 선정
 * - 병렬 에이전트 실행
 * - 결과 취합 및 품질 검증
 * - CRAG (Corrective RAG) 기반 지능형 재시도
 *
 * @module services/agent-system/supervisor/supervisor-orchestrator
 */

import { logger } from '../../../utils/logger.js';
import ModelManager from '../ollama/model-manager.js';
import PromptTemplates from '../ollama/prompt-templates.js';

// CRAG 관련 import
import DocumentGrader from '../graders/document-grader.js';
import QueryRewriter from '../rewriters/query-rewriter.js';
import { GRADE_ACTION } from '../crag/constants.js';

class SupervisorOrchestrator {
  /**
   * Supervisor Multi-Agent 오케스트레이터
   *
   * 설정 우선순위 (높은 순):
   * 1. options에 전달된 값
   * 2. process.env.OLLAMA_PROXY_SERVER 환경변수
   * 3. 기본값 (https://api.hamonize.com/ollama)
   *
   * @param {Object} options - 오케스트레이터 옵션
   * @param {string} options.ollamaUrl - Ollama 서버 URL (우선순위 1)
   * @param {number} options.maxRetries - 최대 재시도 횟수 (기본값: 2)
   * @param {number} options.qualityThreshold - 품질 임계값 (기본값: 0.65)
   */
  constructor(options = {}) {
    // Ollama URL 설정 우선순위: 전달된 옵션 > 환경변수 > 기본값
    this.ollamaUrl = options.ollamaUrl || process.env.OLLAMA_PROXY_SERVER || 'https://api.hamonize.com/ollama';

    logger.info(`[SupervisorOrchestrator] Ollama URL: ${this.ollamaUrl}`);

    this.modelManager = new ModelManager({
      ollamaUrl: this.ollamaUrl,
      defaultModel: 'hamonize:latest',
      fallbackModels: ['airun-chat', 'hamonize:latest'],
      healthCheckInterval: 0  // CRAG에서는 헬스체크 비활성화 (불요한 성능 지연 방지)
    });

    this.templates = new PromptTemplates();
    this.maxRetries = options.maxRetries || 2;
    this.qualityThreshold = options.qualityThreshold || 0.65;
    this.agents = new Map();
    this.agentResults = null;  // 에이전트 실행 결과 저장을 위한 초기화

    // CRAG 관련 설정
    this.enableCRAG = options.enableCRAG !== false; // 기본 활성화
    this.maxCRAGRetries = options.maxCRAGRetries || 1;

    // CRAG 컴포넌트 초기화 - ModelManager 공유
    this.documentGrader = new DocumentGrader({
      modelManager: this.modelManager,  // 기존 ModelManager 재사용
      correctThreshold: options.documentGrader?.correctThreshold || 0.7,
      incorrectThreshold: options.documentGrader?.incorrectThreshold || 0.3,
    });

    // QueryRewriter도 ModelManager 공유하도록 수정 필요
    this.queryRewriter = new QueryRewriter({
      modelManager: this.modelManager,  // 기존 ModelManager 재사용
    });
  }

  /**
   * Self-Reflection 기반 결과 개선
   *
   * @private
   * @param {string} query - 원본 질문
   * @param {string} currentResult - 현재 결과
   * @param {Object} qualityEvaluation - 품질 평가 결과
   * @param {Array} retrievedDocuments - 검색된 문서들
   * @returns {Promise<Object>} 개선 결과
   */
  async _improveResultWithReflection(query, currentResult, qualityEvaluation, retrievedDocuments) {
    try {
      const reflectionPrompt = `
다음 답변의 품질을 개선해주세요:

원본 질문: ${query}
현재 답변: ${currentResult}

품질 평가:
- 전체 품질 점수: ${(qualityEvaluation.score * 100).toFixed(1)}%
- 약점: ${qualityEvaluation.improvements?.join(', ') || '없음'}

검색된 문서들:
${Array.isArray(retrievedDocuments) ? retrievedDocuments.slice(0, 5).map((doc, i) => `${i+1}. ${doc.content?.substring(0, 200)}...`).join('\n') : '검색된 문서 없음'}

개선 지침:
1. 품질 평가의 약점을 모두 보완하세요
2. 검색된 문서들의 정보를 더 적극적으로 활용하세요
3. 답변을 더 구체적이고 유용하게 만드세요
4. 출처를 명확히 표기하세요

개선된 답변:
`;

      const response = await this.modelManager.chat(
        this.modelManager.selectModel('resultAggregation'),
        [{ role: 'user', content: reflectionPrompt }]
      );

      // 개선된 답변 품질 재평가
      const improvedQualityPrompt = this.templates.render('qualityValidation', {
        query,
        results: response.content,
      });

      const qualityRecheck = await this.modelManager.chatWithJSON(
        this.modelManager.selectModel('qualityValidation'),
        [{ role: 'user', content: improvedQualityPrompt }],
        {
          score: 'number',
          reasoning: 'string',
        }
      );

      const newQuality = qualityRecheck.content.score;
      const improvement = newQuality - qualityEvaluation.score;

      if (improvement > 0.1) { // 최소 10% 이상 개선
        return {
          improved: true,
          newResult: response.content,
          newQuality: newQuality,
          improvement: improvement,
          reasoning: `품질 ${(improvement * 100).toFixed(1)}% 개선`
        };
      }

      return { improved: false, reasoning: '개선 폭 부족' };

    } catch (error) {
      logger.error(`[SupervisorOrchestrator] Self-Reflection 개선 실패: ${error.message}`);
      return { improved: false, reasoning: error.message };
    }
  }

  /**
   * 결과 품질 검증 (Self-Reflection 통합)
   *
   * @private
   * @param {string} query - 사용자 질문
   * @param {string} results - 취합된 결과
   * @param {Object} agentResults - 에이전트 실행 결과 (Self-Reflection용)
   * @returns {Promise<Object>} 품질 평가
   */
  async _validateQuality(query, results, agentResults = null) {
    const prompt = this.templates.render('qualityValidation', {
      query,
      results,
    });

    const response = await this.modelManager.chatWithJSON(
      this.modelManager.selectModel('qualityValidation'),
      [{ role: 'user', content: prompt }],
      {
        score: 'number',
        relevance: 'number',
        completeness: 'number',
        accuracy: 'number',
        freshness: 'number',
        reasoning: 'string',
        improvements: 'array',
      }
    );

    const qualityEvaluation = response.content;

    // Self-Reflection 기반 품질 개선 (활성화된 경우)
    if (this.enableSelfReflection && agentResults && qualityEvaluation.score < this.qualityThreshold) {
      logger.info(`[SupervisorOrchestrator] Self-Reflection 기반 품질 개선 시작 (현재 품질: ${(qualityEvaluation.score * 100).toFixed(1)}%)`);
      
      try {
        // 검색된 문서들 추출
        const retrievedDocuments = this._extractRetrievedDocuments(agentResults);
        
        // Self-Reflection 실행 (간소화 버전)
        const improvedResult = await this._improveResultWithReflection(
          query, 
          results, 
          qualityEvaluation, 
          retrievedDocuments
        );

        if (improvedResult.improved) {
          logger.info(`[SupervisorOrchestrator] Self-Reflection 개선 완료 (개선 품질: ${(improvedResult.newQuality * 100).toFixed(1)}%)`);
          
          return {
            ...qualityEvaluation,
            score: improvedResult.newQuality,
            selfReflectionApplied: true,
            originalScore: qualityEvaluation.score,
            improvements: [
              ...qualityEvaluation.improvements,
              `Self-Reflection 적용: ${improvedResult.reasoning}`
            ]
          };
        }
      } catch (error) {
        logger.error(`[SupervisorOrchestrator] Self-Reflection 품질 개선 실패: ${error.message}`);
      }
    }

    return qualityEvaluation;
  }

  /**
   * 재시도 전략 수립
   *
   * @private
   * @param {string} originalQuery - 원본 질문
   * @param {string} previousResults - 이전 결과
   * @param {Object} qualityScore - 품질 평가
   * @returns {Promise<Object>} 재시도 전략
   */
  async _planRetry(originalQuery, previousResults, qualityScore) {
    const prompt = this.templates.render('retryStrategy', {
      originalQuery,
      previousResults,
      qualityScore: qualityScore.score,
    });

    const response = await this.modelManager.chatWithJSON(
      this.modelManager.selectModel('supervisor'),
      [{ role: 'user', content: prompt }],
      {
        newQueries: 'object',
        addedAgents: 'array',
        reasoning: 'string',
        focusAreas: 'array',
      }
    );

    return response.content;
  }



  
  /**
   * 설정 업데이트
   *
   * @param {Object} config - 새 설정
   */
  updateConfig(config) {
    if (config.maxRetries !== undefined) {
      this.maxRetries = config.maxRetries;
    }
    if (config.qualityThreshold !== undefined) {
      this.qualityThreshold = config.qualityThreshold;
    }
    if (config.enableCRAG !== undefined) {
      this.enableCRAG = config.enableCRAG;
    }

    logger.info('[SupervisorOrchestrator] 설정 업데이트 완료');
  }

  /**
   * CRAG (Corrective RAG) 패턴 적용
   *
   * @private
   * @param {string} query - 사용자 쿼리
   * @param {Object} agentResults - 에이전트 실행 결과
   * @param {Object} queries - 에이전트별 쿼리
   * @param {Object} context - 실행 컨텍스트
   * @returns {Promise<Object>} CRAG 적용 결과
   */
  async _applyCRAG(query, agentResults, queries, context, updateProgress = null) {
    // 진행 상태 헬퍼
    const progress = async (message, percent) => {
      if (updateProgress) await updateProgress(message, percent);
    };

    try {
      logger.info('[SupervisorOrchestrator] CRAG 평가 시작');

      // RAG 결과에서 문서 추출
      const ragResults = agentResults.rag.results || [];

      if (ragResults.length === 0) {
        logger.info('[SupervisorOrchestrator] RAG 결과 없음 - CRAG 스킵');
        return { modified: false, agentResults };
      }

      // 문서 평가
      await progress(`📋 CRAG: ${ragResults.length}개 문서의 관련성을 평가하고 있습니다...`, 48);
      const gradeResult = await this.documentGrader.gradeDocuments(query, ragResults);

      logger.info(`[SupervisorOrchestrator] 문서 평가 완료: ${gradeResult.action.type}`);
      logger.debug(this.documentGrader.generateSummary(gradeResult));

      await progress(`📊 CRAG: 문서 평가 완료 (관련: ${gradeResult.action.correctCount || 0}, 모호: ${gradeResult.action.ambiguousCount || 0}, 미관련: ${gradeResult.action.incorrectCount || 0})`, 50);

      // 액션에 따른 처리
      switch (gradeResult.action.type) {
        case GRADE_ACTION.PROCEED:
          // 관련 문서 충분 - 필터링된 문서만 사용
          await progress('✅ CRAG: 관련 문서가 충분합니다', 52);
          return {
            modified: true,
            action: 'PROCEED_WITH_FILTERED',
            agentResults: {
              ...agentResults,
              rag: {
                ...agentResults.rag,
                results: gradeResult.action.filteredDocuments || ragResults,
                cragApplied: true,
                gradeResult: {
                  correctCount: gradeResult.action.correctCount,
                  ambiguousCount: gradeResult.action.ambiguousCount,
                  incorrectCount: gradeResult.action.incorrectCount
                }
              }
            }
          };

        case GRADE_ACTION.REWRITE:
          // 쿼리 재작성 후 재검색
          await progress('✏️ CRAG: 쿼리를 재작성하여 다시 검색합니다...', 52);
          return await this._handleCRAGRewrite(
            query,
            agentResults,
            gradeResult,
            context,
            progress
          );

        case GRADE_ACTION.FALLBACK:
          // 웹 검색으로 대체/보강
          await progress('🌐 CRAG: 웹 검색으로 정보를 보완합니다...', 52);
          return await this._handleCRAGFallback(
            query,
            agentResults,
            gradeResult,
            context,
            progress
          );

        default:
          return { modified: false, agentResults };
      }

    } catch (error) {
      logger.error(`[SupervisorOrchestrator] CRAG 적용 실패: ${error.message}`);
      // 오류 발생 시 원본 결과 유지
      return { modified: false, agentResults };
    }
  }

  /**
   * CRAG 쿼리 재작성 처리
   *
   * @private
   * @param {string} query - 원본 쿼리
   * @param {Object} agentResults - 에이전트 실행 결과
   * @param {Object} gradeResult - 문서 평가 결과
   * @param {Object} context - 실행 컨텍스트
   * @param {Function} progress - 진행 상태 콜백
   * @returns {Promise<Object>} 처리 결과
   */
  async _handleCRAGRewrite(query, agentResults, gradeResult, context, progress = null) {
    logger.info('[SupervisorOrchestrator] CRAG 쿼리 재작성 시작');

    // 쿼리 재작성
    if (progress) await progress('📝 CRAG: 더 나은 검색을 위해 쿼리를 재작성하고 있습니다...', 54);
    const rewriteResult = await this.queryRewriter.rewriteQuery(query, {
      gradeResult,
      failureReason: gradeResult.action.reason
    });

    logger.debug(this.queryRewriter.generateSummary(rewriteResult));

    if (!rewriteResult.success && !rewriteResult.rewrittenQueries) {
      logger.warn('[SupervisorOrchestrator] 쿼리 재작성 실패 - 원본 결과 유지');
      return { modified: false, agentResults };
    }

    if (progress) await progress(`🔄 CRAG: 재작성된 쿼리로 다시 검색합니다: "${rewriteResult.rewrittenQueries.rag?.substring(0, 30)}..."`, 56);

    // 재작성된 쿼리로 RAG 재검색
    const ragAgent = this.agents['rag'];
    if (!ragAgent) {
      return { modified: false, agentResults };
    }

    try {
      const reSearchResult = await ragAgent.execute({
        ...context,
        query: rewriteResult.rewrittenQueries.rag
      });

      if (reSearchResult.success && reSearchResult.results.length > 0) {
        // 기존 CORRECT 문서 + 새로운 검색 결과 병합
        const existingCorrect = gradeResult.action.filteredDocuments || [];
        const mergedResults = this._mergeResults(existingCorrect, reSearchResult.results);

        if (progress) await progress(`✅ CRAG: 재검색 완료! ${mergedResults.length}개 문서 확보`, 58);

        return {
          modified: true,
          action: 'REWRITE_AND_MERGED',
          agentResults: {
            ...agentResults,
            rag: {
              ...agentResults.rag,
              results: mergedResults,
              cragApplied: true,
              rewrittenQuery: rewriteResult.rewrittenQueries.rag,
              gradeResult: {
                correctCount: gradeResult.action.correctCount,
                ambiguousCount: gradeResult.action.ambiguousCount,
                incorrectCount: gradeResult.action.incorrectCount
              }
            }
          }
        };
      }

    } catch (error) {
      logger.error(`[SupervisorOrchestrator] CRAG 재검색 실패: ${error.message}`);
    }

    // 재검색 실패 시 기존 CORRECT 문서만 사용
    if (progress) await progress('⚠️ CRAG: 재검색 결과가 없어 기존 관련 문서를 사용합니다', 58);
    return {
      modified: true,
      action: 'REWRITE_PARTIAL',
      agentResults: {
        ...agentResults,
        rag: {
          ...agentResults.rag,
          results: gradeResult.action.filteredDocuments || [],
          cragApplied: true
        }
      }
    };
  }

  /**
   * CRAG 웹 검색 폴백 처리
   *
   * @private
   * @param {string} query - 원본 쿼리
   * @param {Object} agentResults - 에이전트 실행 결과
   * @param {Object} gradeResult - 문서 평가 결과
   * @param {Object} context - 실행 컨텍스트
   * @param {Function} progress - 진행 상태 콜백
   * @returns {Promise<Object>} 처리 결과
   */
  async _handleCRAGFallback(query, agentResults, gradeResult, context, progress = null) {
    logger.info('[SupervisorOrchestrator] CRAG 웹 검색 폴백 시작');

    // 웹 에이전트가 아직 실행되지 않았다면 실행
    const webAgent = this.agents['web'];
    if (!webAgent) {
      logger.warn('[SupervisorOrchestrator] 웹 에이전트 없음 - 폴백 불가');
      return { modified: false, agentResults };
    }

    // 이미 웹 결과가 있는지 확인
    if (agentResults.web && agentResults.web.success && agentResults.web.results?.length > 0) {
      // 이미 웹 결과가 있으면 RAG 결과만 축소
      if (progress) await progress('📚 CRAG: 기존 웹 검색 결과를 활용합니다', 56);
      return {
        modified: true,
        action: 'FALLBACK_USE_EXISTING_WEB',
        agentResults: {
          ...agentResults,
          rag: {
            ...agentResults.rag,
            results: [], // RAG 결과 제외
            cragApplied: true,
            fallbackReason: gradeResult.action.reason
          }
        }
      };
    }

    // 웹 검색 실행
    try {
      if (progress) await progress('🔍 CRAG: 문서 검색 결과가 부족하여 웹 검색을 수행합니다...', 54);
      const webResult = await webAgent.execute({
        ...context,
        query: query
      });

      if (progress) await progress(`🌐 CRAG: 웹 검색 완료! ${webResult.results?.length || 0}개 결과 확보`, 58);

      return {
        modified: true,
        action: 'FALLBACK_WEB_SEARCH',
        agentResults: {
          ...agentResults,
          rag: {
            ...agentResults.rag,
            results: [], // RAG 결과 제외
            cragApplied: true,
            fallbackReason: gradeResult.action.reason
          },
          web: webResult
        }
      };

    } catch (error) {
      logger.error(`[SupervisorOrchestrator] CRAG 웹 폴백 실패: ${error.message}`);
      if (progress) await progress('⚠️ CRAG: 웹 검색에 실패했습니다', 58);
      return { modified: false, agentResults };
    }
  }

  /**
   * 검색 결과 병합 (중복 제거)
   *
   * @private
   * @param {Array} existing - 기존 결과
   * @param {Array} newResults - 새 검색 결과
   * @returns {Array} 병합된 결과
   */
  _mergeResults(existing, newResults) {
    const seen = new Set();
    const merged = [];

    // 기존 결과 추가
    for (const doc of existing) {
      const key = this._getDocumentKey(doc);
      if (!seen.has(key)) {
        seen.add(key);
        merged.push(doc);
      }
    }

    // 새 결과 추가 (중복 제외)
    for (const doc of newResults) {
      const key = this._getDocumentKey(doc);
      if (!seen.has(key)) {
        seen.add(key);
        merged.push(doc);
      }
    }

    return merged;
  }

  /**
   * 문서 식별 키 생성
   *
   * @private
   * @param {Object} doc - 문서
   * @returns {string} 식별 키
   */
/**
   * 에이전트 결과에서 검색된 문서들 추출
   *
   * @private
   * @param {Object} agentResults - 에이전트 실행 결과
   * @returns {Array} 검색된 문서 배열
   */
  _extractRetrievedDocuments(agentResults) {
    const documents = [];
    
    try {
      // RAG 에이전트 결과에서 문서 추출
      if (agentResults.rag && agentResults.rag.success && agentResults.rag.results) {
        for (const doc of agentResults.rag.results) {
          documents.push({
            content: doc.content || doc.text || '',
            metadata: doc.metadata || {},
            source: 'rag',
            score: doc.score || 0
          });
        }
      }
      
      // Web 에이전트 결과에서 문서 추출
      if (agentResults.web && agentResults.web.success && agentResults.web.results) {
        for (const doc of agentResults.web.results) {
          documents.push({
            content: doc.content || doc.snippet || doc.description || '',
            metadata: {
              title: doc.title,
              url: doc.url,
              source: 'web'
            },
            source: 'web',
            score: doc.score || 0
          });
        }
      }
      
      logger.debug(`[SupervisorOrchestrator] ${documents.length}개의 검색된 문서 추출 완료`);
      return documents;
      
    } catch (error) {
      logger.error(`[SupervisorOrchestrator] 문서 추출 실패: ${error.message}`);
      return [];
    }
  }

  /**
   * 에이전트 등록
   *
   * @param {string} name - 에이전트 이름
   * @param {Object} agent - 에이전트 인스턴스
   */
  registerAgent(name, agent) {
    this.agents.set(name, agent);
    logger.info(`[SupervisorOrchestrator] 에이전트 등록: ${name}`);
  }

  /**
   * 설정 업데이트
   *
   * @param {Object} newConfig - 새로운 설정
   */
  updateConfig(newConfig) {
    if (newConfig.maxRetries !== undefined) {
      this.maxRetries = newConfig.maxRetries;
    }
    if (newConfig.qualityThreshold !== undefined) {
      this.qualityThreshold = newConfig.qualityThreshold;
    }
    if (newConfig.enableCRAG !== undefined) {
      this.enableCRAG = newConfig.enableCRAG;
    }
    logger.info('[SupervisorOrchestrator] 설정 업데이트 완료');
  }

  /**
   * 요청 처리 메인 함수
   *
   * @param {Object} request - 사용자 요청
   * @param {string} request.query - 사용자 질문
   * @param {string} request.sessionId - 세션 ID
   * @param {string} request.username - 사용자명
   * @param {Object} request.options - 추가 옵션
   * @returns {Promise<Object>} 처리 결과
   */
  async process(request) {
    const { query, sessionId, username, options = {} } = request;
    const { updateProgress } = options;

    logger.info(`[SupervisorOrchestrator] 요청 처리 시작: user=${username}, session=${sessionId}`);

    let attempts = 0;
    let bestResult = null;
    let bestQuality = 0;
    let finalSelectedAgents = [];

    while (attempts <= this.maxRetries) {
      attempts++;
      logger.info(`[SupervisorOrchestrator] 시도 ${attempts}/${this.maxRetries + 1}`);

      try {
        // 1. 작업 분석 및 에이전트 선정
        if (updateProgress) {
          await updateProgress('🔍 분석 중... 적절한 에이전트를 선택합니다', 10);
        }

        const taskAnalysis = await this._analyzeTask(query);
        const selectedAgents = taskAnalysis.agents;
        finalSelectedAgents = selectedAgents; // 최종 에이전트 목록 저장

        if (updateProgress) {
          const agentNames = selectedAgents.join(', ');
          await updateProgress(`📋 ${agentNames} 에이전트가 선택되었습니다`, 20);
        }

        // 2. 쿼리 생성 (각 에이전트에 최적화)
        if (updateProgress) {
          await updateProgress('✏️ 각 에이전트에 최적화된 검색 쿼리를 생성합니다', 30);
        }

        const queries = await this._generateQueries(query, selectedAgents);

        if (updateProgress) {
          const queryInfo = Object.entries(queries)
            .map(([agent, q]) => `${agent}: "${q}"`)
            .join(', ');
          await updateProgress(`🎯 생성된 쿼리: ${queryInfo}`, 40);
        }

        // 3. 에이전트 병렬 실행
        if (updateProgress) {
          await updateProgress('🚀 에이전트들을 병렬로 실행합니다', 50);
        }

        const agentResults = await this._executeAgents(selectedAgents, queries, {
          sessionId,
          username,
          updateProgress
        });
        
        // 에이전트 결과 저장 (나중에 metadata로 전달하기 위해)
        this.agentResults = agentResults;

        // 에이전트 결과 요약
        if (updateProgress) {
          const summary = Object.entries(agentResults)
            .filter(([_, result]) => result.success)
            .map(([agent, result]) => {
              const count = result.results?.length || 0;
              return `${agent}: ${count}개 결과`;
            })
            .join(', ');
          await updateProgress(`📊 검색 완료: ${summary}`, 70);
        }

        // 4. 결과 취합
        if (updateProgress) {
          await updateProgress('🔄 여러 에이전트의 검색 결과를 취합합니다', 80);
        }

        const aggregatedResult = await this._aggregateResults(query, agentResults);

        // 5. 품질 검증
        if (updateProgress) {
          await updateProgress('🔍 결과의 품질을 검증합니다', 85);
        }

        const qualityEvaluation = await this._validateQuality(query, aggregatedResult, agentResults);

        // 품질 검증 결과 로깅
        const qualityPercent = Math.round(qualityEvaluation.score * 100);
        logger.info(`[SupervisorOrchestrator] 품질 검증 결과: ${qualityPercent}%`);

        // 6. CRAG 적용 (활성화된 경우)
        let finalResult = aggregatedResult;
        let finalQuality = qualityEvaluation.score;

        if (this.enableCRAG && qualityEvaluation.score < this.qualityThreshold) {
          const cragResult = await this._applyCRAG(query, aggregatedResult, qualityEvaluation, agentResults);
          if (cragResult.improved) {
            finalResult = cragResult.result;
            finalQuality = cragResult.quality;
          }
        }

        // Self-Reflection 기반 결과 개선 (품질 임계값 미만인 경우)
        if (finalQuality < this.qualityThreshold && attempts <= this.maxRetries) {
          logger.info(`[SupervisorOrchestrator] Self-Reflection 기반 품질 개선 시작 (현재 품질: ${(finalQuality * 100).toFixed(1)}%)`);
          const retrievedDocuments = this._extractRetrievedDocuments(agentResults);
          const improvedResult = await this._improveResultWithReflection(query, finalResult, qualityEvaluation, retrievedDocuments);

          if (improvedResult.improved) {
            logger.info(`[SupervisorOrchestrator] Self-Reflection 개선 완료 (개선 품질: ${(improvedResult.newQuality * 100).toFixed(1)}%)`);
            finalResult = improvedResult.newResult;
            finalQuality = improvedResult.newQuality;
          } else {
            logger.warn(`[SupervisorOrchestrator] Self-Reflection 품질 개선 실패: ${improvedResult.reasoning}`);
          }
        }

        // 최고 결과 저장
        if (finalQuality > bestQuality) {
          bestResult = finalResult;
          bestQuality = finalQuality;
        }

        // 품질 임계값 통과 시 종료
        if (finalQuality >= this.qualityThreshold) {
          break;
        }

      } catch (error) {
        logger.error(`[SupervisorOrchestrator] 시도 ${attempts} 실패: ${error.message}`);
        if (attempts > this.maxRetries) {
          throw error;
        }
      }
    }

    const success = bestQuality >= this.qualityThreshold;
    logger.info(`[SupervisorOrchestrator] 요청 처리 완료: 성공=${success}, 최종 품질=${(bestQuality * 100).toFixed(1)}%`);

    // 최종 결과에 대한 진행 상태 업데이트
    if (updateProgress) {
      const qualityPercent = Math.round(bestQuality * 100);
      const status = success ? '성공' : '품질 미달';
      await updateProgress(`📊 최종 품질 평가 완료: ${qualityPercent}% (${status})`, 95);
    }

    return {
      success,
      content: bestResult || '죄송합니다, 요청하신 내용에 대한 답변을 생성하지 못했습니다.',
      metadata: {
        selectedAgents: finalSelectedAgents || [],
        qualityScore: bestQuality,
        attempts,
        enableCRAG: this.enableCRAG,
        agentResults: this.agentResults,  // 에이전트 실행 결과 포함
      },
    };
  }

  /**
   * 작업 분석
   *
   * @private
   * @param {string} query - 사용자 질문
   * @returns {Promise<Object>} 분석 결과
   */
  async _analyzeTask(query) {
    const prompt = this.templates.render('taskAnalysis', { query });

    const response = await this.modelManager.chatWithJSON(
      this.modelManager.selectModel('taskAnalysis'),
      [{ role: 'user', content: prompt }],
      {
        agents: 'array',
        reasoning: 'string',
        complexity: 'string',
      }
    );

    // 응답 구조 안전하게 확인
    const result = response?.content || response || {};
    const agents = result.agents || result.selectedAgents || [];

    logger.debug(`[SupervisorOrchestrator] 작업 분석 결과:`, { agents, reasoning: result.reasoning });

    return {
      agents: agents,
      reasoning: result.reasoning || '',
      complexity: result.complexity || 'moderate'
    };
  }

  /**
   * 쿼리 생성
   *
   * @private
   * @param {string} query - 원본 질문
   * @param {Array} selectedAgents - 선정된 에이전트 목록
   * @returns {Promise<Object>} 에이전트별 쿼리
   */
  async _generateQueries(query, selectedAgents) {
    const prompt = this.templates.render('queryGeneration', {
      originalQuery: query,
      targetAgents: selectedAgents,
    });

    const response = await this.modelManager.chatWithJSON(
      this.modelManager.selectModel('queryGeneration'),
      [{ role: 'user', content: prompt }],
      {
        queries: 'object',
      }
    );

    // 응답 구조 안전하게 확인
    const queries = response?.content?.queries || response?.queries || {};
    logger.debug(`[SupervisorOrchestrator] 생성된 쿼리:`, queries);

    return queries;
  }

  /**
   * 에이전트 병렬 실행
   *
   * @private
   * @param {Array} selectedAgents - 선정된 에이전트 목록
   * @param {Object} queries - 에이전트별 쿼리
   * @param {Object} context - 실행 컨텍스트
   * @returns {Promise<Object>} 에이전트 실행 결과
   */
  async _executeAgents(selectedAgents, queries, context) {
    // selectedAgents가 배열인지 확인
    if (!Array.isArray(selectedAgents)) {
      logger.error(`[SupervisorOrchestrator] selectedAgents가 배열이 아님:`, typeof selectedAgents, selectedAgents);
      return {};
    }

    const promises = selectedAgents.map(async (agentName) => {
      const agent = this.agents.get(agentName);
      if (!agent) {
        return { agent: agentName, error: 'Agent not found' };
      }

      try {
        // 에이전트별 쿼리와 컨텍스트를 합쳐서 전달
        const agentContext = {
          query: queries[agentName],
          username: context.username,
          sessionId: context.sessionId,
          options: context.options || {}
        };
        const result = await agent.execute(agentContext);
        return { agent: agentName, success: true, ...result };
      } catch (error) {
        logger.error(`[SupervisorOrchestrator] 에이전트 ${agentName} 실행 실패: ${error.message}`);
        return { agent: agentName, success: false, error: error.message };
      }
    });

    const results = await Promise.all(promises);
    const agentResults = {};
    results.forEach(result => {
      agentResults[result.agent] = result;
    });

    return agentResults;
  }

  /**
   * 결과 취합
   *
   * @private
   * @param {string} query - 원본 질문
   * @param {Object} agentResults - 에이전트 실행 결과
   * @returns {Promise<string>} 취합된 결과
   */
  async _aggregateResults(query, agentResults) {
    const prompt = this.templates.render('resultAggregation', {
      query,
      results: agentResults,
    });

    const response = await this.modelManager.chat(
      this.modelManager.selectModel('resultAggregation'),
      [{ role: 'user', content: prompt }]
    );

    return response.content;
  }

  _getDocumentKey(doc) {
    const content = doc.content || doc.text || '';
    const source = doc.metadata?.filename || doc.metadata?.source || '';
    return `${source}:${content.substring(0, 100)}`;
  }

  /**
   * CRAG 통계 조회
   *
   * @returns {Object} CRAG 관련 통계
   */
  getCRAGStats() {
    return {
      enabled: this.enableCRAG,
      maxRetries: this.maxCRAGRetries,
      graderThresholds: {
        correct: this.documentGrader.correctThreshold,
        incorrect: this.documentGrader.incorrectThreshold
      }
    };
  }
}

export default SupervisorOrchestrator;
