/**
 * CRAG (Corrective RAG) 시스템 테스트
 *
 * Document Grader, Query Rewriter, CRAG 통합 테스트
 *
 * 실행: node services/agent-system/test-crag-system.js
 */

import DocumentGrader, { GRADE, ACTION } from './graders/document-grader.js';
import QueryRewriter, { STRATEGY } from './rewriters/query-rewriter.js';
import { initializeSupervisor, getSupervisor } from './index.js';

// 테스트 결과 저장
const testResults = {
  passed: 0,
  failed: 0,
  tests: []
};

/**
 * 테스트 케이스 실행
 */
async function runTest(name, testFn) {
  const startTime = Date.now();
  try {
    await testFn();
    const duration = Date.now() - startTime;
    testResults.passed++;
    testResults.tests.push({ name, status: 'PASS', duration });
    console.log(`✅ ${name} (${duration}ms)`);
  } catch (error) {
    const duration = Date.now() - startTime;
    testResults.failed++;
    testResults.tests.push({ name, status: 'FAIL', duration, error: error.message });
    console.error(`❌ ${name} (${duration}ms): ${error.message}`);
  }
}

/**
 * 단언 함수
 */
function assert(condition, message) {
  if (!condition) {
    throw new Error(message || 'Assertion failed');
  }
}

/**
 * Document Grader 테스트
 */
async function testDocumentGrader() {
  console.log('\n📋 Document Grader 테스트\n');

  const grader = new DocumentGrader({
    correctThreshold: 0.7,
    incorrectThreshold: 0.3
  });

  // 테스트 1: 관련성 높은 문서 평가
  await runTest('관련성 높은 문서 평가', async () => {
    const query = '파이썬 웹 프레임워크';
    const documents = [
      {
        content: 'Django는 파이썬의 대표적인 웹 프레임워크입니다. MTV 패턴을 사용하며 관리자 페이지를 자동 생성합니다.',
        metadata: { filename: 'django-guide.pdf' },
        scores: { keyword_precision: 0.8, semantic_relevance: 0.9, content_quality: 0.85 }
      },
      {
        content: 'Flask는 경량 파이썬 웹 프레임워크로, 마이크로 프레임워크라고도 불립니다.',
        metadata: { filename: 'flask-intro.pdf' },
        scores: { keyword_precision: 0.75, semantic_relevance: 0.85, content_quality: 0.8 }
      }
    ];

    const result = await grader.gradeDocuments(query, documents);

    assert(result.gradedDocuments.length === 2, '2개 문서가 평가되어야 함');
    assert(result.summary.totalDocuments === 2, '총 문서 수가 2여야 함');
    console.log(`   평균 점수: ${result.summary.avgScore}`);
    console.log(`   액션: ${result.action.type}`);
  });

  // 테스트 2: 빈 문서 배열 처리
  await runTest('빈 문서 배열 처리', async () => {
    const result = await grader.gradeDocuments('테스트 쿼리', []);

    assert(result.action.type === ACTION.FALLBACK, '빈 결과는 FALLBACK이어야 함');
    assert(result.gradedDocuments.length === 0, '평가된 문서가 없어야 함');
  });

  // 테스트 3: 관련성 낮은 문서 평가
  await runTest('관련성 낮은 문서 평가', async () => {
    const query = '머신러닝 알고리즘';
    const documents = [
      {
        content: '오늘 날씨가 좋습니다. 맑은 하늘이 펼쳐져 있습니다.',
        metadata: { filename: 'weather.txt' },
        scores: { keyword_precision: 0.1, semantic_relevance: 0.1, content_quality: 0.3 }
      }
    ];

    const result = await grader.gradeDocuments(query, documents);

    console.log(`   평균 점수: ${result.summary.avgScore}`);
    console.log(`   액션: ${result.action.type}`);
  });
}

/**
 * Query Rewriter 테스트
 */
async function testQueryRewriter() {
  console.log('\n✏️ Query Rewriter 테스트\n');

  const rewriter = new QueryRewriter();

  // 테스트 1: 기본 쿼리 재작성
  await runTest('기본 쿼리 재작성', async () => {
    const originalQuery = 'AI 기술 동향';
    const result = await rewriter.rewriteQuery(originalQuery, {
      failureReason: '검색 결과 관련성 낮음'
    });

    assert(result.rewrittenQueries, '재작성된 쿼리가 있어야 함');
    assert(result.rewrittenQueries.rag, 'RAG 쿼리가 있어야 함');
    assert(result.rewrittenQueries.web, 'Web 쿼리가 있어야 함');

    console.log(`   전략: ${result.strategy}`);
    console.log(`   RAG 쿼리: ${result.rewrittenQueries.rag}`);
    console.log(`   Web 쿼리: ${result.rewrittenQueries.web}`);
  });

  // 테스트 2: 복잡한 쿼리 분해
  await runTest('복잡한 쿼리 재작성', async () => {
    const originalQuery = '우리 회사의 AI 전략과 경쟁사 동향을 비교하고 향후 발전 방향 제시';
    const result = await rewriter.rewriteQuery(originalQuery, {});

    assert(result.rewrittenQueries, '재작성된 쿼리가 있어야 함');
    console.log(`   전략: ${result.strategy}`);
    console.log(`   처리 시간: ${result.processingTime}ms`);
  });

  // 테스트 3: 쿼리 품질 평가
  await runTest('쿼리 품질 평가', async () => {
    const goodQuery = '파이썬 웹 프레임워크 Django Flask 비교';
    const badQuery = '의 을 를 에서';

    const goodResult = rewriter.evaluateQueryQuality(goodQuery);
    const badResult = rewriter.evaluateQueryQuality(badQuery);

    assert(goodResult.score > badResult.score, '좋은 쿼리의 점수가 더 높아야 함');
    console.log(`   좋은 쿼리 점수: ${goodResult.score}`);
    console.log(`   나쁜 쿼리 점수: ${badResult.score}`);
  });
}

/**
 * CRAG 통합 테스트
 */
async function testCRAGIntegration() {
  console.log('\n🔄 CRAG 통합 테스트\n');

  // Supervisor 초기화
  const supervisor = initializeSupervisor();

  // 테스트 1: CRAG 기본 설정 확인 (기본 비활성화)
  await runTest('CRAG 기본 설정 확인 (비활성화)', async () => {
    const stats = supervisor.getCRAGStats();

    assert(stats.enabled === false, 'CRAG가 기본적으로 비활성화되어야 함');
    assert(stats.graderThresholds.correct === 0.7, 'correct 임계값이 0.7이어야 함');
    assert(stats.graderThresholds.incorrect === 0.3, 'incorrect 임계값이 0.3이어야 함');

    console.log(`   CRAG 기본 상태: ${stats.enabled ? '활성화' : '비활성화'}`);
    console.log(`   임계값: CORRECT=${stats.graderThresholds.correct}, INCORRECT=${stats.graderThresholds.incorrect}`);
  });

  // 테스트 1.5: 딥리서치 모드 (CRAG 활성화/비활성화 동적 전환)
  await runTest('CRAG 동적 활성화/비활성화', async () => {
    // 활성화
    supervisor.updateConfig({ enableCRAG: true });
    let stats = supervisor.getCRAGStats();
    assert(stats.enabled === true, 'CRAG가 활성화되어야 함');
    console.log(`   활성화 후: ${stats.enabled}`);

    // 비활성화
    supervisor.updateConfig({ enableCRAG: false });
    stats = supervisor.getCRAGStats();
    assert(stats.enabled === false, 'CRAG가 비활성화되어야 함');
    console.log(`   비활성화 후: ${stats.enabled}`);
  });

  // 테스트 2: 전체 파이프라인 테스트 (RAG 서버 필요)
  await runTest('전체 CRAG 파이프라인 (서버 연결 확인)', async () => {
    try {
      // 딥리서치 모드 시뮬레이션: CRAG 활성화
      supervisor.updateConfig({ enableCRAG: true });

      const result = await supervisor.process({
        query: '파이썬 웹 개발',
        sessionId: 'test-crag-session',
        username: 'test-user',
        options: {}
      });

      // 테스트 후 CRAG 비활성화
      supervisor.updateConfig({ enableCRAG: false });

      console.log(`   성공 여부: ${result.success}`);
      console.log(`   품질 점수: ${result.metadata?.qualityScore || 'N/A'}`);
      console.log(`   시도 횟수: ${result.metadata?.attempts || 'N/A'}`);

      if (result.metadata?.agentResults?.rag?.cragApplied) {
        console.log(`   CRAG 적용됨: ${result.metadata.agentResults.rag.cragApplied}`);
      }
    } catch (error) {
      // 테스트 후 CRAG 비활성화
      supervisor.updateConfig({ enableCRAG: false });

      // RAG 서버가 없어도 테스트는 통과 (연결 오류만 확인)
      if (error.message.includes('ECONNREFUSED') || error.message.includes('timeout')) {
        console.log('   ⚠️ RAG/Ollama 서버 연결 필요 (테스트 스킵)');
        return;
      }
      throw error;
    }
  });
}

/**
 * 성능 테스트
 */
async function testPerformance() {
  console.log('\n⚡ 성능 테스트\n');

  const grader = new DocumentGrader();
  const rewriter = new QueryRewriter();

  // 테스트 1: Document Grader 성능
  await runTest('Document Grader 성능 (10개 문서)', async () => {
    const documents = Array(10).fill(null).map((_, i) => ({
      content: `테스트 문서 ${i + 1}의 내용입니다. 이 문서는 AI 기술에 대해 설명합니다.`,
      metadata: { filename: `doc${i + 1}.pdf` },
      scores: { keyword_precision: Math.random(), semantic_relevance: Math.random(), content_quality: Math.random() }
    }));

    const startTime = Date.now();
    const result = await grader.gradeDocuments('AI 기술', documents);
    const duration = Date.now() - startTime;

    assert(duration < 30000, '10개 문서 평가는 30초 이내여야 함');
    console.log(`   처리 시간: ${duration}ms`);
    console.log(`   문서당 시간: ${Math.round(duration / 10)}ms`);
  });

  // 테스트 2: Query Rewriter 성능
  await runTest('Query Rewriter 성능', async () => {
    const queries = [
      '파이썬 프로그래밍',
      '머신러닝 알고리즘',
      '클라우드 컴퓨팅 서비스'
    ];

    const startTime = Date.now();
    for (const query of queries) {
      await rewriter.rewriteQuery(query, {});
    }
    const duration = Date.now() - startTime;

    assert(duration < 60000, '3개 쿼리 재작성은 60초 이내여야 함');
    console.log(`   총 처리 시간: ${duration}ms`);
    console.log(`   쿼리당 시간: ${Math.round(duration / 3)}ms`);
  });
}

/**
 * 테스트 결과 출력
 */
function printResults() {
  console.log('\n' + '='.repeat(60));
  console.log('📊 테스트 결과 요약');
  console.log('='.repeat(60));
  console.log(`✅ 통과: ${testResults.passed}`);
  console.log(`❌ 실패: ${testResults.failed}`);
  console.log(`📝 총 테스트: ${testResults.passed + testResults.failed}`);

  if (testResults.failed > 0) {
    console.log('\n실패한 테스트:');
    testResults.tests
      .filter(t => t.status === 'FAIL')
      .forEach(t => console.log(`  - ${t.name}: ${t.error}`));
  }

  const totalTime = testResults.tests.reduce((sum, t) => sum + t.duration, 0);
  console.log(`\n⏱️ 총 소요 시간: ${totalTime}ms`);
  console.log('='.repeat(60));
}

/**
 * 메인 실행
 */
async function main() {
  console.log('🚀 CRAG 시스템 테스트 시작\n');
  console.log('='.repeat(60));

  try {
    await testDocumentGrader();
    await testQueryRewriter();
    await testCRAGIntegration();
    await testPerformance();
  } catch (error) {
    console.error('\n❌ 테스트 중 예외 발생:', error);
  }

  printResults();

  // 종료 코드 설정
  process.exit(testResults.failed > 0 ? 1 : 0);
}

main();
