/**
 * Supervisor Multi-Agent 시스템 통합 테스트
 *
 * 전체 시스템의 동작을 검증:
 * 1. SupervisorOrchestrator
 * 2. RAGAgent
 * 3. WebAgent
 * 4. 엔드투엔드 흐름
 */

import SupervisorOrchestrator from './supervisor/supervisor-orchestrator.js';
import RAGAgent from './agents/rag-agent.js';
import WebAgent from './agents/web-agent.js';

async function testRAGAgent() {
  console.log('\n=== RAG Agent 테스트 시작 ===\n');

  try {
    console.log('1. RAG Agent 초기화...');
    const ragAgent = new RAGAgent({
      ragServerUrl: process.env.RAG_SERVER_URL || 'http://localhost:5600',
    });
    console.log('✅ RAG Agent 초기화 성공\n');

    // 상태 확인
    console.log('2. RAG 서버 상태 확인...');
    const health = await ragAgent.checkHealth();
    if (health.success) {
      console.log(`✅ RAG 서버 상태: ${health.status}\n`);
    } else {
      console.log(`⚠️  RAG 서버 접근 불가: ${health.error}\n`);
    }

    // 검색 테스트
    console.log('3. RAG 검색 테스트...');
    const searchResult = await ragAgent.execute({
      query: '인공지능 머신러닝',
      username: 'admin',
      options: { top_k: 3 },
    });

    if (searchResult.success) {
      console.log(`✅ 검색 성공: ${searchResult.results.length}개 결과`);
      if (searchResult.results.length > 0) {
        console.log(`   첫 번째 결과: ${searchResult.results[0].content.slice(0, 100)}...\n`);
      }
    } else {
      console.log(`⚠️  검색 실패: ${searchResult.error}\n`);
    }

    console.log('=== RAG Agent 테스트 완료 ===');
    return true;
  } catch (error) {
    console.error('❌ RAG Agent 테스트 실패:', error.message);
    return false;
  }
}

async function testWebAgent() {
  console.log('\n=== Web Agent 테스트 시작 ===\n');

  try {
    console.log('1. Web Agent 초기화...');
    const webAgent = new WebAgent({
      webSearchUrl: process.env.WEB_SEARCH_URL || 'http://localhost:5610',
    });
    console.log('✅ Web Agent 초기화 성공\n');

    // 상태 확인
    console.log('2. Web Search 서버 상태 확인...');
    const health = await webAgent.checkHealth();
    if (health.success) {
      console.log(`✅ Web Search 서버 상태: ${health.status}\n`);
    } else {
      console.log(`⚠️  Web Search 서버 접근 불가: ${health.error}\n`);
    }

    // 검색 테스트
    console.log('3. Web 검색 테스트...');
    const searchResult = await webAgent.execute({
      query: '2025 AI trends',
      options: { num_results: 3 },
    });

    if (searchResult.success) {
      console.log(`✅ 검색 성공: ${searchResult.results.length}개 결과`);
      if (searchResult.results.length > 0) {
        console.log(`   첫 번째 결과: ${searchResult.results[0].title}\n`);
      }
    } else {
      console.log(`⚠️  검색 실패: ${searchResult.error}\n`);
    }

    console.log('=== Web Agent 테스트 완료 ===');
    return true;
  } catch (error) {
    console.error('❌ Web Agent 테스트 실패:', error.message);
    return false;
  }
}

async function testSupervisorOrchestrator() {
  console.log('\n=== Supervisor Orchestrator 테스트 시작 ===\n');

  try {
    console.log('1. Supervisor 초기화...');

    // 에이전트 생성
    const ragAgent = new RAGAgent();
    const webAgent = new WebAgent();

    // Supervisor 초기화
    const supervisor = new SupervisorOrchestrator({
      ollamaUrl: process.env.OLLAMA_PROXY_SERVER || 'https://api.hamonize.com/ollama',
      maxRetries: 1,
      qualityThreshold: 0.6,
    });

    // 에이전트 등록
    supervisor.registerAgent('rag', ragAgent);
    supervisor.registerAgent('web', webAgent);

    console.log('✅ Supervisor 초기화 및 에이전트 등록 완료\n');

    // 간단한 요청 테스트
    console.log('2. 단순 질문 처리 테스트...');
    const simpleResult = await supervisor.process({
      query: '파이썬이란 무엇인가요?',
      sessionId: 'test-session-1',
      username: 'admin',
      options: {},
    });

    if (simpleResult.success) {
      console.log('✅ 단순 질문 처리 성공');
      console.log(`   선정된 에이전트: ${simpleResult.metadata.selectedAgents.join(', ')}`);
      console.log(`   품질 점수: ${simpleResult.metadata.qualityScore.toFixed(2)}`);
      console.log(`   시도 횟수: ${simpleResult.metadata.attempts}`);
      console.log(`   답변 미리보기: ${simpleResult.content.slice(0, 100)}...\n`);
    } else {
      console.log('⚠️  단순 질문 처리 실패 (품질 미달)');
      console.log(`   품질 점수: ${simpleResult.metadata.qualityScore.toFixed(2)}\n`);
    }

    // 복합 요청 테스트
    console.log('3. 복합 질문 처리 테스트...');
    const complexResult = await supervisor.process({
      query: '우리 회사의 AI 전략과 최신 AI 트렌드를 비교해서 분석해줘',
      sessionId: 'test-session-2',
      username: 'admin',
      options: {},
    });

    if (complexResult.success) {
      console.log('✅ 복합 질문 처리 성공');
      console.log(`   선정된 에이전트: ${complexResult.metadata.selectedAgents.join(', ')}`);
      console.log(`   품질 점수: ${complexResult.metadata.qualityScore.toFixed(2)}`);
      console.log(`   시도 횟수: ${complexResult.metadata.attempts}`);
      console.log(`   답변 미리보기: ${complexResult.content.slice(0, 100)}...\n`);
    } else {
      console.log('⚠️  복합 질문 처리 실패 (품질 미달)');
      console.log(`   품질 점수: ${complexResult.metadata.qualityScore.toFixed(2)}\n`);
    }

    console.log('=== Supervisor Orchestrator 테스트 완료 ===');
    return true;
  } catch (error) {
    console.error('❌ Supervisor Orchestrator 테스트 실패:', error.message);
    console.error(error.stack);
    return false;
  }
}

async function runAllTests() {
  console.log('╔═══════════════════════════════════════════╗');
  console.log('║  AIRUN Supervisor Multi-Agent Test Suite ║');
  console.log('╚═══════════════════════════════════════════╝\n');

  const results = {
    ragAgent: false,
    webAgent: false,
    supervisor: false,
  };

  // RAG Agent 테스트
  results.ragAgent = await testRAGAgent();

  // Web Agent 테스트
  results.webAgent = await testWebAgent();

  // Supervisor Orchestrator 테스트
  results.supervisor = await testSupervisorOrchestrator();

  // 결과 요약
  console.log('\n╔═══════════════════════════════════════════╗');
  console.log('║           테스트 결과 요약                ║');
  console.log('╚═══════════════════════════════════════════╝\n');
  console.log(`RAG Agent:        ${results.ragAgent ? '✅ 통과' : '❌ 실패'}`);
  console.log(`Web Agent:        ${results.webAgent ? '✅ 통과' : '❌ 실패'}`);
  console.log(`Supervisor:       ${results.supervisor ? '✅ 통과' : '❌ 실패'}`);

  const allPassed = Object.values(results).every((r) => r === true);
  console.log(`\n전체 결과: ${allPassed ? '✅ 모든 테스트 통과' : '⚠️  일부 테스트 실패/경고'}`);

  console.log('\n참고: RAG/Web Search 서버가 실행 중이지 않으면 일부 테스트가 실패할 수 있습니다.');

  process.exit(allPassed ? 0 : 1);
}

// 스크립트 직접 실행 시
if (import.meta.url === `file://${process.argv[1]}`) {
  runAllTests().catch((error) => {
    console.error('치명적 오류:', error);
    process.exit(1);
  });
}

export { testRAGAgent, testWebAgent, testSupervisorOrchestrator };
