import crypto from 'crypto';
import dbManager from '../database/index.js';

class ApiKeyService {
    constructor() {}

    // API 키 생성
    async createApiKey(userId, options = {}) {
        const {
            name = '기본 API 키',
            expiresAt: expiresAtCamel = null,
            permissions = null,
            allowedIps = null,
            rateLimit = 1000
        } = options;
        const expiresAtInput = typeof options.expires_at !== 'undefined' ? options.expires_at : expiresAtCamel;
        
        // API 키 생성 (16바이트 랜덤 문자열)
        const randomKey = crypto.randomBytes(16).toString('hex');
        const keyValue = `airun_${userId}_${randomKey}`;

        let normalizedExpiresAt = null;
        if (expiresAtInput !== null && typeof expiresAtInput !== 'undefined') {
            const expiresDate = expiresAtInput instanceof Date ? expiresAtInput : new Date(expiresAtInput);
            if (!Number.isNaN(expiresDate.getTime())) {
                normalizedExpiresAt = expiresDate;
            }
        }

        const result = await dbManager.query(`
            INSERT INTO api_keys 
            (user_id, key_value, name, expires_at, permissions, allowed_ips, rate_limit)
            VALUES ($1, $2, $3, $4, $5, $6, $7)
            RETURNING *
        `, [userId, keyValue, name, normalizedExpiresAt, permissions, allowedIps, rateLimit]);

        return result.rows[0];
    }

    // API 키 검증
    async verifyApiKey(keyValue, ip = null) {
        const result = await dbManager.query(`
            SELECT ak.*, u.role as user_role, u.status as user_status, u.username
            FROM api_keys ak
            JOIN users u ON u.id = ak.user_id
            WHERE ak.key_value = $1 
            AND ak.status = 'active'
            AND (ak.expires_at IS NULL OR ak.expires_at > NOW())
        `, [keyValue]);

        if (result.rows.length === 0) {
            throw new Error('유효하지 않은 API 키입니다.');
        }

        const apiKey = result.rows[0];

        // 사용자 상태가 active가 아닌 경우 접근 차단
        if (apiKey.user_status !== 'active') {
            throw new Error('비활성화된 계정입니다. 관리자에게 문의하세요.');
        }

        // IP 제한 확인
        if (ip && apiKey.allowed_ips && apiKey.allowed_ips.length > 0) {
            if (!apiKey.allowed_ips.includes(ip)) {
                throw new Error('허용되지 않은 IP 주소입니다.');
            }
        }

        // 마지막 사용 시간 업데이트
        await dbManager.query(`
            UPDATE api_keys 
            SET last_used = CURRENT_TIMESTAMP 
            WHERE id = $1
        `, [apiKey.id]);

        return {
            userId: apiKey.user_id,
            permissions: apiKey.permissions,
            rateLimit: apiKey.rate_limit,
            userRole: apiKey.user_role
        };
    }

    // 사용자의 API 키 목록 조회
    async listApiKeys(userId) {
        try {
            const result = await dbManager.query(
                `SELECT 
                    ak.id,
                    ak.user_id,
                    u.username AS user_name,
                    ak.name as api_key_name,
                    ak.key_value,
                    ak.status,
                    ak.last_used,
                    ak.created_at,
                    ak.expires_at,
                    ak.permissions,
                    ak.allowed_ips,
                    ak.rate_limit
                FROM api_keys ak
                JOIN users u ON u.id = ak.user_id
                WHERE ak.user_id = $1
                ORDER BY ak.created_at DESC`
                , [userId]
            );
            return result.rows;
        } catch (error) {
            console.error('API 키 목록 조회 중 오류:', error);
            throw new Error('API 키 목록을 조회할 수 없습니다.');
        }
    }

    // API 키 상태 변경 (활성화/비활성화)
    async updateApiKeyStatus(userId, keyId, status) {
        if (!['active', 'inactive'].includes(status)) {
            throw new Error('유효하지 않은 상태값입니다.');
        }

        const result = await dbManager.query(`
            UPDATE api_keys 
            SET status = $1 
            WHERE id = $2 AND user_id = $3 
            RETURNING *
        `, [status, keyId, userId]);

        if (result.rows.length === 0) {
            throw new Error('API 키를 찾을 수 없습니다.');
        }

        return result.rows[0];
    }

    // API 키 삭제
    async deleteApiKey(userId, keyId) {
        const result = await dbManager.query(`
            DELETE FROM api_keys 
            WHERE id = $1 AND user_id = $2 
            RETURNING *
        `, [keyId, userId]);

        if (result.rows.length === 0) {
            throw new Error('API 키를 찾을 수 없습니다.');
        }

        return result.rows[0];
    }

    // API 키 업데이트
    async updateApiKey(userId, keyId, updates) {
        const allowedUpdates = ['name', 'permissions', 'allowed_ips', 'rate_limit', 'expires_at'];
        const updateFields = [];
        const values = [keyId, userId];
        let paramCount = 3;

        Object.keys(updates).forEach(key => {
            if (allowedUpdates.includes(key)) {
                updateFields.push(`${key} = $${paramCount}`);
                values.push(updates[key]);
                paramCount++;
            }
        });

        if (updateFields.length === 0) {
            throw new Error('업데이트할 필드가 없습니다.');
        }

        const query = `
            UPDATE api_keys 
            SET ${updateFields.join(', ')} 
            WHERE id = $1 AND user_id = $2 
            RETURNING *
        `;

        const result = await dbManager.query(query, values);

        if (result.rows.length === 0) {
            throw new Error('API 키를 찾을 수 없습니다.');
        }

        return result.rows[0];
    }
}

export default ApiKeyService; 