// schema.js - Minimal migration for current database structure
// This file creates tables that match the current database exactly

import bcrypt from 'bcrypt';
import crypto from 'crypto';
import fs from 'fs';
import path from 'path';

const SCHEMA_VERSION = '4.0.3';

/* ---------------------------------- Utils ---------------------------------- */

const compareVersions = (a, b) => {
  const pa = String(a).split('.').map(Number);
  const pb = String(b).split('.').map(Number);
  const len = Math.max(pa.length, pb.length);
  for (let i = 0; i < len; i++) {
    const va = pa[i] || 0;
    const vb = pb[i] || 0;
    if (va > vb) return 1;
    if (va < vb) return -1;
  }
  return 0;
};

const withTxn = async (db, fn) => {
  await db.query('BEGIN');
  try {
    const res = await fn();
    await db.query('COMMIT');
    return res;
  } catch (e) {
    await db.query('ROLLBACK');
    throw e;
  }
};

const log = (...args) => console.log('[SCHEMA]', ...args);
const warn = (...args) => console.warn('[SCHEMA][WARN]', ...args);
const err = (...args) => console.error('[SCHEMA][ERROR]', ...args);

/* ------------------------------ Version table ------------------------------ */

const createMigrationTable = async (db) => {
  await db.query(`
    CREATE TABLE IF NOT EXISTS schema_migrations (
      version VARCHAR(50) PRIMARY KEY,
      applied_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    );
  `);
};

const getCurrentVersion = async (db) => {
  try {
    await createMigrationTable(db);
    const r = await db.query(
      'SELECT version FROM schema_migrations ORDER BY applied_at DESC, version DESC LIMIT 1'
    );
    return r.rows[0]?.version || '0';
  } catch (e) {
    err('마이그레이션 버전 확인 실패:', e);
    return '0';
  }
};

const recordMigration = async (db, version) => {
  await db.query(
    'INSERT INTO schema_migrations (version) VALUES ($1) ON CONFLICT (version) DO NOTHING',
    [version]
  );
};

/* --------------------------------- Helpers --------------------------------- */

const tableExists = async (db, table) => {
  const { rows } = await db.query(`
    SELECT EXISTS (
      SELECT FROM information_schema.tables
      WHERE table_schema='public' AND table_name=$1
    ) AS exists
  `, [table]);
  return !!rows[0]?.exists;
};

const ensureExtension = async (db, ext) => {
  await db.query(`CREATE EXTENSION IF NOT EXISTS ${ext};`);
};

const runSchemaStatements = async (db, statements) => {
  for (const stmt of statements) {
    await db.query(stmt);
  }
};

/* ------------------------------ Schema DDL ------------------------------- */

const schemaStatements = [
  // Core tables
  `CREATE TABLE IF NOT EXISTS users (
    id SERIAL PRIMARY KEY,
    username VARCHAR(50) UNIQUE NOT NULL,
    email VARCHAR(255) UNIQUE,
    password_hash VARCHAR(255) NOT NULL,
    name VARCHAR(100),
    role VARCHAR(50) DEFAULT 'user',
    language VARCHAR(10) DEFAULT 'ko',
    status VARCHAR(20) DEFAULT 'active',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    last_login TIMESTAMP,
    system_prompt TEXT
  );`,
  `CREATE TABLE IF NOT EXISTS permissions (
    id SERIAL PRIMARY KEY,
    name VARCHAR(100) UNIQUE NOT NULL,
    description TEXT
  );`,
  `CREATE TABLE IF NOT EXISTS role_permissions (
    role VARCHAR(50),
    permission_id INTEGER REFERENCES permissions(id) ON DELETE CASCADE,
    PRIMARY KEY (role, permission_id)
  );`,
  `CREATE TABLE IF NOT EXISTS api_keys (
    id SERIAL PRIMARY KEY,
    user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
    key_value VARCHAR(64) UNIQUE NOT NULL,
    name VARCHAR(100),
    status VARCHAR(20) DEFAULT 'active',
    permissions JSONB DEFAULT '{}',
    allowed_ips TEXT[],
    rate_limit INTEGER DEFAULT 1000,
    last_used TIMESTAMP,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    expires_at TIMESTAMP
  );`,
  `CREATE TABLE IF NOT EXISTS activity_logs (
    id SERIAL PRIMARY KEY,
    user_id INTEGER REFERENCES users(id) ON DELETE SET NULL,
    action VARCHAR(255),
    details JSONB DEFAULT '{}',
    ip_address VARCHAR(45),
    user_agent TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  );`,
  `CREATE TABLE IF NOT EXISTS user_ip_logs (
    id SERIAL PRIMARY KEY,
    user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
    ip_address VARCHAR(45) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  );`,
  `CREATE TABLE IF NOT EXISTS sessions (
    id VARCHAR(64) PRIMARY KEY,
    user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
    username VARCHAR(100),
    data JSONB NOT NULL DEFAULT '{}',
    provider VARCHAR(50),
    model VARCHAR(100),
    title VARCHAR(255),
    message_count INTEGER DEFAULT 0,
    last_message TEXT,
    status VARCHAR(20) DEFAULT 'active',
    type VARCHAR(20) DEFAULT 'chat',
    created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
    expires_at TIMESTAMP WITH TIME ZONE
  );`,
  `CREATE TABLE IF NOT EXISTS projects (
    id SERIAL PRIMARY KEY,
    name VARCHAR(255) NOT NULL,
    description TEXT,
    user_id VARCHAR(100) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE(user_id, name)
  );`,
  `CREATE TABLE IF NOT EXISTS document_embeddings (
    id SERIAL PRIMARY KEY,
    is_embedding boolean NOT NULL DEFAULT FALSE,
    doc_id VARCHAR(500) NOT NULL,
    filename VARCHAR(500) NOT NULL,
    chunk_index INTEGER NOT NULL,
    chunk_text TEXT NOT NULL,
    embedding vector(1024),
    image_embedding vector(512),
    user_id VARCHAR(255),
    source TEXT,
    modification_date TIMESTAMP,
    metadata JSONB DEFAULT '{}',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    CONSTRAINT unique_doc_chunk UNIQUE (doc_id, chunk_index)
  );`,
  `CREATE TABLE IF NOT EXISTS chat_documents (
    id SERIAL PRIMARY KEY,
    filename VARCHAR(255) NOT NULL,
    filepath VARCHAR(500) NOT NULL,
    filesize BIGINT,
    mimetype VARCHAR(100),
    user_id VARCHAR(100) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    upload_status VARCHAR(20) DEFAULT 'uploaded' CHECK (upload_status IN ('uploading','uploaded','failed')),
    embedding_status VARCHAR(20) DEFAULT 'pending' CHECK (embedding_status IN ('pending','processing','completed','failed')),
    project_name VARCHAR(100),
    project_id INTEGER REFERENCES projects(id) ON DELETE SET NULL,
    tags TEXT[],
    metadata JSONB DEFAULT '{}',
    error_message TEXT,
    processed_at TIMESTAMP,
    UNIQUE(user_id, filename)
  );`,
  `CREATE TABLE IF NOT EXISTS nlpai_lab_kure_v1_document_embeddings (
    id SERIAL PRIMARY KEY,
    doc_id VARCHAR(500) NOT NULL,
    filename VARCHAR(500) NOT NULL,
    chunk_index INTEGER NOT NULL,
    chunk_text TEXT NOT NULL,
    embedding vector(1024),
    image_embedding vector(512),
    user_id VARCHAR(255),
    source TEXT,
    modification_date TIMESTAMP,
    metadata JSONB DEFAULT '{}',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    is_embedding boolean NOT NULL DEFAULT FALSE
  );`,
  // Report tables
  `CREATE TABLE IF NOT EXISTS report_jobs (
    id SERIAL PRIMARY KEY,
    job_id VARCHAR(255) UNIQUE NOT NULL,
    user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
    username VARCHAR(100),
    title VARCHAR(500),
    executive_summary TEXT,
    template VARCHAR(100),
    output_format VARCHAR(10) DEFAULT 'pdf',
    status VARCHAR(20) DEFAULT 'queued',
    progress INTEGER DEFAULT 0,
    output_path VARCHAR(1000),
    cache_key VARCHAR(255),
    error_message TEXT,
    metadata JSONB DEFAULT '{}',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    completed_at TIMESTAMP
  );`,
  `CREATE TABLE IF NOT EXISTS report_section_cache (
    id SERIAL PRIMARY KEY,
    job_id VARCHAR(255) NOT NULL,
    section_name VARCHAR(255) NOT NULL,
    subsection_name VARCHAR(255),
    content TEXT,
    chart_data JSONB,
    table_data JSONB,
    cache_key VARCHAR(255),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    expires_at TIMESTAMP,
    FOREIGN KEY (job_id) REFERENCES report_jobs(job_id) ON DELETE CASCADE
  );`,
  `CREATE TABLE IF NOT EXISTS user_report_settings (
    id SERIAL PRIMARY KEY,
    user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
    setting_key VARCHAR(255) NOT NULL,
    setting_value TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE(user_id, setting_key)
  );`,
  // Scheduler tables
  `CREATE TABLE IF NOT EXISTS saved_tasks (
    id SERIAL PRIMARY KEY,
    task_id VARCHAR(255) UNIQUE NOT NULL,
    user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
    name VARCHAR(255) NOT NULL,
    description TEXT,
    prompt TEXT NOT NULL,
    python_code TEXT,
    agent_options JSONB DEFAULT '{}',
    execution_count INTEGER DEFAULT 0,
    success_count INTEGER DEFAULT 0,
    metadata JSONB DEFAULT '{}',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    last_used TIMESTAMP
  );`,
  `CREATE TABLE IF NOT EXISTS scheduled_tasks (
    id SERIAL PRIMARY KEY,
    schedule_id VARCHAR(255) UNIQUE NOT NULL,
    user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
    task_id VARCHAR(255) NOT NULL,
    name VARCHAR(255) NOT NULL,
    schedule_type VARCHAR(20) NOT NULL CHECK (schedule_type IN ('once','recurring')),
    scheduled_at TIMESTAMP NOT NULL,
    recurring_pattern JSONB,
    status VARCHAR(20) DEFAULT 'active' CHECK (status IN ('active','inactive','completed','failed','running')),
    next_execution TIMESTAMP,
    last_execution TIMESTAMP,
    execution_count INTEGER DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (task_id) REFERENCES saved_tasks(task_id) ON DELETE CASCADE
  );`,
  `CREATE TABLE IF NOT EXISTS task_execution_logs (
    id SERIAL PRIMARY KEY,
    execution_id VARCHAR(255) UNIQUE NOT NULL,
    task_id VARCHAR(255) NOT NULL,
    schedule_id VARCHAR(255),
    user_id INTEGER REFERENCES users(id) ON DELETE SET NULL,
    execution_type VARCHAR(20) NOT NULL CHECK (execution_type IN ('manual','scheduled')),
    status VARCHAR(20) NOT NULL CHECK (status IN ('pending','running','completed','failed','cancelled')),
    python_code TEXT,
    stdout TEXT,
    stderr TEXT,
    exit_code INTEGER,
    error_message TEXT,
    agent_job_id VARCHAR(255),
    execution_time_ms INTEGER,
    metadata JSONB DEFAULT '{}',
    started_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    completed_at TIMESTAMP,
    FOREIGN KEY (task_id) REFERENCES saved_tasks(task_id) ON DELETE CASCADE,
    FOREIGN KEY (schedule_id) REFERENCES scheduled_tasks(schedule_id) ON DELETE SET NULL
  );`,
  // Web portal tables
  `CREATE TABLE IF NOT EXISTS support_tickets (
    id SERIAL PRIMARY KEY,
    title VARCHAR(255) NOT NULL,
    question TEXT NOT NULL,
    status VARCHAR(20) NOT NULL DEFAULT 'open',
    author_name VARCHAR(255) NOT NULL,
    author_role VARCHAR(20) NOT NULL,
    attachment_image BYTEA,
    attachment_image_mimetype VARCHAR(100),
    created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP
  );`,
  `CREATE TABLE IF NOT EXISTS support_ticket_replies (
    id SERIAL PRIMARY KEY,
    ticket_id INTEGER NOT NULL REFERENCES support_tickets(id) ON DELETE CASCADE,
    content TEXT NOT NULL,
    author_name VARCHAR(255) NOT NULL,
    author_role VARCHAR(20) NOT NULL,
    images BYTEA[] DEFAULT '{}',
    image_mimetypes VARCHAR(100)[] DEFAULT '{}',
    created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP
  );`,
  `CREATE TABLE IF NOT EXISTS support_documents (
    id SERIAL PRIMARY KEY,
    filename VARCHAR(255) NOT NULL,
    filepath TEXT NOT NULL,
    filesize BIGINT,
    mimetype VARCHAR(100),
    user_id VARCHAR(255),
    created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP
  );`,
  `CREATE TABLE IF NOT EXISTS user_memories (
    id SERIAL PRIMARY KEY,
    user_id VARCHAR(255) NOT NULL,
    title VARCHAR(255) NOT NULL,
    content TEXT NOT NULL,
    category VARCHAR(100) DEFAULT 'general',
    importance_level INTEGER DEFAULT 1 CHECK (importance_level BETWEEN 1 AND 5),
    tags TEXT[],
    is_active BOOLEAN DEFAULT true,
    access_count INTEGER DEFAULT 0,
    created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
    last_accessed TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP
  );`,
  // FlowAI tables
  `CREATE TABLE IF NOT EXISTS flowai_workflows (
    id VARCHAR(255) PRIMARY KEY,
    name VARCHAR(255) NOT NULL,
    description TEXT,
    nodes TEXT NOT NULL DEFAULT '[]',
    connections TEXT NOT NULL DEFAULT '[]',
    variables TEXT DEFAULT '{}',
    user_id VARCHAR(255) NOT NULL,
    is_public BOOLEAN DEFAULT false,
    tags TEXT DEFAULT '[]',
    created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP
  );`,
  `CREATE TABLE IF NOT EXISTS flowai_executions (
    id SERIAL PRIMARY KEY,
    workflow_id VARCHAR(255) NOT NULL REFERENCES flowai_workflows(id) ON DELETE CASCADE,
    user_id VARCHAR(255) NOT NULL,
    status VARCHAR(20) NOT NULL DEFAULT 'pending',
    results TEXT DEFAULT '[]',
    total_duration INTEGER,
    error_message TEXT,
    final_output TEXT,
    start_time TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
    end_time TIMESTAMP WITH TIME ZONE
  );`,
  `CREATE TABLE IF NOT EXISTS flowai_template_categories (
    id VARCHAR(255) PRIMARY KEY,
    name VARCHAR(255) NOT NULL,
    description TEXT,
    icon VARCHAR(50) DEFAULT '📁',
    color VARCHAR(7) DEFAULT '#666666',
    order_index INTEGER DEFAULT 999,
    is_active BOOLEAN DEFAULT true,
    created_at TIMESTAMP DEFAULT now(),
    updated_at TIMESTAMP DEFAULT now()
  );`,
  `CREATE TABLE IF NOT EXISTS flowai_templates (
    id VARCHAR(255) PRIMARY KEY,
    name VARCHAR(255) NOT NULL,
    description TEXT,
    category_id VARCHAR(255) REFERENCES flowai_template_categories(id) ON DELETE SET NULL,
    tags TEXT DEFAULT '[]',
    icon VARCHAR(50) DEFAULT '⚡',
    difficulty VARCHAR(50) DEFAULT 'beginner',
    estimated_time INTEGER DEFAULT 5,
    nodes TEXT DEFAULT '[]',
    connections TEXT DEFAULT '[]',
    variables TEXT DEFAULT '[]',
    preview_image TEXT,
    preview_description TEXT,
    author_name VARCHAR(255),
    author_organization VARCHAR(255),
    version VARCHAR(50) DEFAULT '1.0.0',
    is_public BOOLEAN DEFAULT false,
    usage_count INTEGER DEFAULT 0,
    rating NUMERIC(3,2) DEFAULT 0.00,
    user_id INTEGER,
    created_at TIMESTAMP DEFAULT now(),
    updated_at TIMESTAMP DEFAULT now()
  );`,
  `CREATE TABLE IF NOT EXISTS flowai_template_reviews (
    id SERIAL PRIMARY KEY,
    template_id VARCHAR(255) NOT NULL REFERENCES flowai_templates(id) ON DELETE CASCADE,
    user_id INTEGER NOT NULL,
    rating INTEGER CHECK (rating >= 1 AND rating <= 5),
    comment TEXT,
    created_at TIMESTAMP DEFAULT now(),
    updated_at TIMESTAMP DEFAULT now()
  );`,
  `CREATE TABLE IF NOT EXISTS flowai_template_usage (
    id SERIAL PRIMARY KEY,
    template_id VARCHAR(255) NOT NULL REFERENCES flowai_templates(id) ON DELETE CASCADE,
    user_id INTEGER NOT NULL,
    used_at TIMESTAMP DEFAULT now()
  );`,
  // QA Dataset table
  `CREATE TABLE IF NOT EXISTS qa_dataset (
    id SERIAL PRIMARY KEY,
    user_question TEXT NOT NULL,
    ai_answer TEXT NOT NULL,
    user_id VARCHAR(100) NOT NULL,
    rating INTEGER NOT NULL CHECK (rating >= 1 AND rating <= 5),
    feedback_type VARCHAR(50) DEFAULT 'thumbs',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  );`,
  // Knowledge Graph tables
  `CREATE TABLE IF NOT EXISTS graph_entities (
    id SERIAL PRIMARY KEY,
    name VARCHAR(255) NOT NULL,
    type VARCHAR(100) NOT NULL,
    description TEXT,
    properties JSONB DEFAULT '{}',
    source_documents TEXT[],
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE(name, type)
  );`,
  `CREATE TABLE IF NOT EXISTS graph_relationships (
    id SERIAL PRIMARY KEY,
    source_entity_id INTEGER REFERENCES graph_entities(id) ON DELETE CASCADE,
    target_entity_id INTEGER REFERENCES graph_entities(id) ON DELETE CASCADE,
    relationship_type VARCHAR(100) NOT NULL,
    description TEXT,
    weight FLOAT DEFAULT 1.0,
    properties JSONB DEFAULT '{}',
    source_documents TEXT[],
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  );`,
  `CREATE TABLE IF NOT EXISTS graph_communities (
    id SERIAL PRIMARY KEY,
    name VARCHAR(255) NOT NULL,
    description TEXT,
    entities INTEGER[],
    level INTEGER DEFAULT 0,
    properties JSONB DEFAULT '{}',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  );`,
  // Business Suite tables
  `CREATE TABLE IF NOT EXISTS business_announcements (
    id VARCHAR(255) PRIMARY KEY,
    source VARCHAR(50) NOT NULL,
    external_id VARCHAR(255) NOT NULL,
    title TEXT NOT NULL,
    description TEXT,
    category VARCHAR(100),
    subcategory VARCHAR(100),
    budget BIGINT,
    support_amount BIGINT,
    support_ratio FLOAT,
    announcement_date TIMESTAMP,
    application_start TIMESTAMP,
    application_end TIMESTAMP,
    project_start TIMESTAMP,
    project_end TIMESTAMP,
    eligibility JSONB DEFAULT '{}',
    requirements JSONB DEFAULT '{}',
    evaluation JSONB DEFAULT '{}',
    contact JSONB DEFAULT '{}',
    documents JSONB DEFAULT '{}',
    status VARCHAR(20) DEFAULT 'active',
    view_count INTEGER DEFAULT 0,
    raw_data JSONB DEFAULT '{}',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  );`,
  `CREATE TABLE IF NOT EXISTS companies (
    id SERIAL PRIMARY KEY,
    business_number VARCHAR(20) UNIQUE NOT NULL,
    company_name VARCHAR(200) NOT NULL,
    company_size VARCHAR(20),
    industry_code VARCHAR(10),
    industry_name VARCHAR(100),
    address TEXT,
    region VARCHAR(50),
    employee_count INTEGER,
    annual_revenue BIGINT,
    establishment_date TIMESTAMP,
    technologies JSONB DEFAULT '{}',
    certifications JSONB DEFAULT '{}',
    capabilities JSONB DEFAULT '{}',
    business_history JSONB DEFAULT '{}',
    financial_info JSONB DEFAULT '{}',
    contact_person VARCHAR(50),
    contact_phone VARCHAR(20),
    contact_email VARCHAR(100),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  );`,
  `CREATE TABLE IF NOT EXISTS company_announcement_matches (
    id SERIAL PRIMARY KEY,
    company_id INTEGER REFERENCES companies(id) ON DELETE CASCADE,
    announcement_id VARCHAR(255) REFERENCES business_announcements(id) ON DELETE CASCADE,
    total_score FLOAT NOT NULL,
    category_score FLOAT,
    size_score FLOAT,
    location_score FLOAT,
    technology_score FLOAT,
    experience_score FLOAT,
    match_details JSONB DEFAULT '{}',
    recommendation_status VARCHAR(20) DEFAULT 'pending',
    recommendation_reason TEXT,
    matched_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  );`,
  `CREATE TABLE IF NOT EXISTS business_proposals (
    id SERIAL PRIMARY KEY,
    company_id INTEGER REFERENCES companies(id) ON DELETE CASCADE,
    announcement_id VARCHAR(255) REFERENCES business_announcements(id) ON DELETE CASCADE,
    proposal_title VARCHAR(500) NOT NULL,
    proposal_summary TEXT,
    sections JSONB DEFAULT '{}',
    generation_method VARCHAR(50),
    template_used VARCHAR(100),
    ai_model_used VARCHAR(100),
    status VARCHAR(20) DEFAULT 'draft',
    version INTEGER DEFAULT 1,
    file_path VARCHAR(500),
    file_format VARCHAR(10),
    file_size INTEGER,
    quality_score FLOAT,
    completeness_score FLOAT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  );`,
  `CREATE TABLE IF NOT EXISTS data_collection_logs (
    id SERIAL PRIMARY KEY,
    source VARCHAR(50) NOT NULL,
    collection_type VARCHAR(20) NOT NULL,
    total_items INTEGER DEFAULT 0,
    new_items INTEGER DEFAULT 0,
    updated_items INTEGER DEFAULT 0,
    failed_items INTEGER DEFAULT 0,
    start_time TIMESTAMP NOT NULL,
    end_time TIMESTAMP,
    duration_seconds INTEGER,
    status VARCHAR(20) NOT NULL,
    error_message TEXT,
    details JSONB DEFAULT '{}',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  );`,
  `CREATE TABLE IF NOT EXISTS system_configurations (
    id SERIAL PRIMARY KEY,
    config_key VARCHAR(100) NOT NULL UNIQUE,
    config_value TEXT,
    config_type VARCHAR(20) DEFAULT 'string',
    description TEXT,
    category VARCHAR(50),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  );`,
  // Context Memory tables
  `CREATE TABLE IF NOT EXISTS context_memories (
    id SERIAL PRIMARY KEY,
    user_id VARCHAR(255) NOT NULL,
    session_id VARCHAR(255),
    memory_type VARCHAR(50) NOT NULL CHECK (memory_type IN ('entity','fact','preference','pattern','skill','project','relationship')),
    content TEXT NOT NULL,
    summary TEXT,
    entities JSONB DEFAULT '[]',
    keywords TEXT[],
    importance_score FLOAT DEFAULT 0.5 CHECK (importance_score >= 0 AND importance_score <= 1),
    confidence_score FLOAT DEFAULT 0.8 CHECK (confidence_score >= 0 AND confidence_score <= 1),
    valid_from TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    valid_until TIMESTAMP,
    source_context TEXT,
    reference_count INTEGER DEFAULT 0,
    last_referenced TIMESTAMP,
    embedding vector(1024),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    CONSTRAINT unique_user_memory_content UNIQUE (user_id, memory_type, content)
  );`,
  `CREATE TABLE IF NOT EXISTS user_preferences (
    id SERIAL PRIMARY KEY,
    user_id VARCHAR(255) NOT NULL,
    category VARCHAR(100) NOT NULL,
    preference_key VARCHAR(255) NOT NULL,
    preference_value TEXT NOT NULL,
    strength FLOAT DEFAULT 0.5 CHECK (strength >= 0 AND strength <= 1),
    confidence FLOAT DEFAULT 0.8 CHECK (confidence >= 0 AND confidence <= 1),
    learned_from VARCHAR(50) DEFAULT 'conversation',
    evidence_count INTEGER DEFAULT 1,
    first_observed TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    last_confirmed TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    notes TEXT,
    tags TEXT[],
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE(user_id, category, preference_key)
  );`,
  `CREATE TABLE IF NOT EXISTS context_relationships (
    id SERIAL PRIMARY KEY,
    user_id VARCHAR(255) NOT NULL,
    source_memory_id INTEGER REFERENCES context_memories(id) ON DELETE CASCADE,
    target_memory_id INTEGER REFERENCES context_memories(id) ON DELETE CASCADE,
    relationship_type VARCHAR(50) NOT NULL,
    strength FLOAT DEFAULT 0.5 CHECK (strength >= 0 AND strength <= 1),
    description TEXT,
    evidence TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  );`,
  `CREATE TABLE IF NOT EXISTS context_session_summaries (
    id SERIAL PRIMARY KEY,
    session_id VARCHAR(255) UNIQUE NOT NULL,
    user_id VARCHAR(255) NOT NULL,
    title VARCHAR(255),
    summary TEXT,
    key_topics TEXT[],
    mentioned_entities TEXT[],
    message_count INTEGER DEFAULT 0,
    session_type VARCHAR(50),
    provider VARCHAR(50),
    model VARCHAR(100),
    importance_score FLOAT DEFAULT 0.5,
    engagement_score FLOAT DEFAULT 0.5,
    session_start TIMESTAMP,
    session_end TIMESTAMP,
    duration_minutes INTEGER,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  );`,
  // Webhooks tables
  `CREATE TABLE IF NOT EXISTS webhooks (
    id SERIAL PRIMARY KEY,
    user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
    url VARCHAR(500) NOT NULL,
    events TEXT[] NOT NULL,
    secret VARCHAR(255),
    is_active BOOLEAN DEFAULT true,
    retry_count INTEGER DEFAULT 3,
    timeout_seconds INTEGER DEFAULT 30,
    headers JSONB DEFAULT '{}',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  );`,
  `CREATE TABLE IF NOT EXISTS webhook_logs (
    id SERIAL PRIMARY KEY,
    webhook_id INTEGER REFERENCES webhooks(id) ON DELETE CASCADE,
    event_type VARCHAR(100) NOT NULL,
    payload JSONB,
    response_status INTEGER,
    response_body TEXT,
    response_headers JSONB,
    attempts INTEGER DEFAULT 1,
    delivered_at TIMESTAMP,
    error_message TEXT,
    execution_time_ms INTEGER,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  );`,
  // Health Batch System tables
  `CREATE TABLE IF NOT EXISTS health_batch_jobs (
    id VARCHAR(255) PRIMARY KEY,
    user_id VARCHAR(255) NOT NULL,
    filename VARCHAR(255) NOT NULL,
    status VARCHAR(20) DEFAULT 'pending' CHECK (status IN ('pending','processing','completed','failed')),
    total_records INTEGER DEFAULT 0,
    processed_records INTEGER DEFAULT 0,
    successful_records INTEGER DEFAULT 0,
    failed_records INTEGER DEFAULT 0,
    progress FLOAT DEFAULT 0,
    current_patient VARCHAR(255),
    error_message TEXT,
    metadata JSONB DEFAULT '{}',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    completed_at TIMESTAMP
  );`,
  `CREATE TABLE IF NOT EXISTS health_batch_patients (
    id SERIAL PRIMARY KEY,
    job_id VARCHAR(255) NOT NULL REFERENCES health_batch_jobs(id) ON DELETE CASCADE,
    patient_name VARCHAR(255) NOT NULL,
    row_number INTEGER NOT NULL,
    raw_data JSONB DEFAULT '{}',
    blood_pressure_score INTEGER DEFAULT 3,
    cholesterol_score INTEGER DEFAULT 3,
    hba1c_received INTEGER DEFAULT 3,
    blood_sugar_score INTEGER DEFAULT 3,
    obesity_score INTEGER DEFAULT 3,
    urine_protein_score INTEGER DEFAULT 1,
    past_history_treatment TEXT,
    current_history_treatment TEXT,
    final_judgment INTEGER DEFAULT 3,
    analysis_status VARCHAR(20) DEFAULT 'pending' CHECK (analysis_status IN ('pending','processing','completed','failed')),
    analysis_error TEXT,
    analyzed_at TIMESTAMP,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  );`,
  `CREATE TABLE IF NOT EXISTS patient_health_data (
    id SERIAL PRIMARY KEY,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  );`,
  // LLM Fine-tuning tables
  `CREATE TABLE IF NOT EXISTS finetuning_datasets (
    id VARCHAR(255) PRIMARY KEY,
    name VARCHAR(255) NOT NULL,
    description TEXT,
    source VARCHAR(50) NOT NULL,
    file_path TEXT,
    total_samples INTEGER DEFAULT 0,
    status VARCHAR(20) DEFAULT 'pending' CHECK (status IN ('pending','processing','completed','failed')),
    metadata JSONB DEFAULT '{}',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  );`,
  `CREATE TABLE IF NOT EXISTS finetuned_models (
    id VARCHAR(255) PRIMARY KEY,
    base_model VARCHAR(255) NOT NULL,
    dataset_id VARCHAR(255) REFERENCES finetuning_datasets(id) ON DELETE SET NULL,
    status VARCHAR(50) NOT NULL CHECK (status IN ('training','completed','failed','cancelled')),
    output_path TEXT,
    training_config JSONB DEFAULT '{}',
    metrics JSONB DEFAULT '{}',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  );`,
  `CREATE TABLE IF NOT EXISTS finetuning_jobs (
    id SERIAL PRIMARY KEY,
    model_id VARCHAR(255) REFERENCES finetuned_models(id) ON DELETE CASCADE,
    status VARCHAR(20) DEFAULT 'queued' CHECK (status IN ('queued','running','completed','failed','cancelled')),
    progress INTEGER DEFAULT 0,
    logs TEXT,
    error_message TEXT,
    started_at TIMESTAMP,
    completed_at TIMESTAMP,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  );`,
  // ESG Analysis tables
  `CREATE TABLE IF NOT EXISTS esg_batch_jobs (
    id VARCHAR(36) PRIMARY KEY,
    user_id VARCHAR(50) NOT NULL,
    filename VARCHAR(255) NOT NULL,
    status VARCHAR(20) DEFAULT 'pending' CHECK (status IN ('pending','processing','completed','failed')),
    total_records INTEGER DEFAULT 0,
    processed_records INTEGER DEFAULT 0,
    successful_records INTEGER DEFAULT 0,
    failed_records INTEGER DEFAULT 0,
    progress INTEGER DEFAULT 0,
    error_message TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    completed_at TIMESTAMP
  );`,
  `CREATE TABLE IF NOT EXISTS esg_batch_documents (
    id SERIAL PRIMARY KEY,
    job_id VARCHAR(36) REFERENCES esg_batch_jobs(id) ON DELETE CASCADE,
    document_name VARCHAR(255),
    row_number INTEGER,
    raw_content TEXT,
    industry_major_code VARCHAR(10),
    industry_major_name VARCHAR(255),
    industry_sub_code VARCHAR(10),
    industry_sub_name VARCHAR(255),
    industry_confidence DECIMAL(5,2),
    estimated_revenue DECIMAL(15,2),
    revenue_currency VARCHAR(10) DEFAULT 'KRW',
    carbon_intensity DECIMAL(10,4),
    total_carbon_emission FLOAT,
    emission_unit VARCHAR(20) DEFAULT 'tCO2e',
    analysis_status VARCHAR(20) DEFAULT 'pending' CHECK (analysis_status IN ('pending','processing','completed','failed')),
    analysis_error TEXT,
    analyzed_at TIMESTAMP,
    report_content TEXT,
    report_format VARCHAR(20) DEFAULT 'html',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  );`,
  `CREATE TABLE IF NOT EXISTS esg_industry_classifications (
    id SERIAL PRIMARY KEY,
    code VARCHAR(10) NOT NULL UNIQUE,
    major_category VARCHAR(200) NOT NULL,
    sub_categories JSONB DEFAULT '[]',
    carbon_intensity FLOAT NOT NULL,
    description TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  );`,
  // RPA Interface tables
  `CREATE TABLE IF NOT EXISTS rpa_jobs (
    id SERIAL PRIMARY KEY,
    request_id VARCHAR(255) UNIQUE NOT NULL,
    status VARCHAR(20) DEFAULT 'QUEUED' CHECK (status IN ('QUEUED','IN_PROGRESS','SUCCEEDED','FAILED')),
    application_type VARCHAR(100) NOT NULL,
    applicant_name VARCHAR(255) NOT NULL,
    applicant_info JSONB DEFAULT '{}',
    applicant_email VARCHAR(255),
    applicant_phone VARCHAR(50),
    applicant_address TEXT,
    application_data JSONB DEFAULT '{}',
    additional_notes TEXT,
    payload_hash VARCHAR(255),
    schema_version VARCHAR(20) DEFAULT '1.0',
    correlation_id VARCHAR(255),
    user_id INTEGER REFERENCES users(id) ON DELETE SET NULL,
    username VARCHAR(100),
    dici_key VARCHAR(255),
    error_message TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    completed_at TIMESTAMP
  );`,
  `CREATE TABLE IF NOT EXISTS rpa_attachments (
    id SERIAL PRIMARY KEY,
    rpa_job_id INTEGER NOT NULL REFERENCES rpa_jobs(id) ON DELETE CASCADE,
    file_name VARCHAR(255) NOT NULL,
    original_name VARCHAR(255) NOT NULL,
    file_path TEXT NOT NULL,
    file_size BIGINT NOT NULL,
    file_type VARCHAR(100),
    file_hash VARCHAR(255),
    description TEXT,
    uploaded_by INTEGER REFERENCES users(id) ON DELETE SET NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  );`,
  // AI Form Builder tables
  `CREATE TABLE IF NOT EXISTS ai_forms (
    id VARCHAR(255) PRIMARY KEY,
    user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
    username VARCHAR(100),
    title VARCHAR(500) NOT NULL,
    description TEXT,
    document_id VARCHAR(255),
    document_filename VARCHAR(255),
    document_url TEXT,
    analyzed_content TEXT,
    form_schema JSONB NOT NULL DEFAULT '[]',
    share_link VARCHAR(255) UNIQUE,
    view_link VARCHAR(255) UNIQUE,
    status VARCHAR(20) DEFAULT 'active' CHECK (status IN ('draft','active','archived')),
    response_count INTEGER DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  );`,
  `CREATE TABLE IF NOT EXISTS ai_form_responses (
    id SERIAL PRIMARY KEY,
    form_id VARCHAR(255) REFERENCES ai_forms(id) ON DELETE CASCADE,
    response_data JSONB NOT NULL,
    submitted_by VARCHAR(255),
    submitted_ip VARCHAR(45),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  );`,
  `CREATE TABLE IF NOT EXISTS ai_form_response_attachments (
    id SERIAL PRIMARY KEY,
    response_id INTEGER NOT NULL REFERENCES ai_form_responses(id) ON DELETE CASCADE,
    filename VARCHAR(255) NOT NULL,
    original_filename VARCHAR(255) NOT NULL,
    file_path VARCHAR(500) NOT NULL,
    file_size INTEGER NOT NULL,
    mime_type VARCHAR(100),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  );`,
  // Quotation System tables
  `CREATE TABLE IF NOT EXISTS company_profiles (
    id SERIAL PRIMARY KEY,
    user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
    company_name VARCHAR(255) NOT NULL,
    business_number VARCHAR(20),
    ceo_name VARCHAR(100),
    address TEXT,
    phone VARCHAR(50),
    fax VARCHAR(50),
    email VARCHAR(255),
    website VARCHAR(255),
    logo_path VARCHAR(500),
    seal_path VARCHAR(500),
    bank_name VARCHAR(100),
    bank_account VARCHAR(100),
    bank_holder VARCHAR(100),
    default_validity_days INTEGER DEFAULT 30,
    default_payment_terms TEXT,
    default_notes TEXT,
    metadata JSONB DEFAULT '{}',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  );`,
  `CREATE TABLE IF NOT EXISTS price_catalog (
    id SERIAL PRIMARY KEY,
    user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
    category VARCHAR(100),
    item_code VARCHAR(50),
    item_name VARCHAR(255) NOT NULL,
    description TEXT,
    unit VARCHAR(50) DEFAULT '식',
    unit_price DECIMAL(15,2) NOT NULL DEFAULT 0,
    currency VARCHAR(10) DEFAULT 'KRW',
    min_quantity INTEGER DEFAULT 1,
    discount_rate DECIMAL(5,2) DEFAULT 0,
    is_active BOOLEAN DEFAULT true,
    tags TEXT[],
    metadata JSONB DEFAULT '{}',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  );`,
  `CREATE TABLE IF NOT EXISTS quotation_templates (
    id SERIAL PRIMARY KEY,
    user_id INTEGER REFERENCES users(id) ON DELETE CASCADE,
    template_name VARCHAR(255) NOT NULL,
    description TEXT,
    template_type VARCHAR(50) DEFAULT 'standard',
    header_html TEXT,
    body_html TEXT,
    footer_html TEXT,
    css_styles TEXT,
    default_items JSONB DEFAULT '[]',
    default_terms TEXT,
    default_notes TEXT,
    is_default BOOLEAN DEFAULT false,
    is_active BOOLEAN DEFAULT true,
    metadata JSONB DEFAULT '{}',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  );`,
  `CREATE TABLE IF NOT EXISTS quotations (
    id SERIAL PRIMARY KEY,
    user_id VARCHAR(50) NOT NULL,
    company_profile_id INTEGER REFERENCES company_profiles(id) ON DELETE SET NULL,
    template_id INTEGER REFERENCES quotation_templates(id) ON DELETE SET NULL,
    quotation_number VARCHAR(100) UNIQUE NOT NULL,
    quotation_date DATE DEFAULT CURRENT_DATE,
    validity_date DATE,
    client_company VARCHAR(255) NOT NULL,
    client_name VARCHAR(100),
    project_name VARCHAR(100),
    client_contact VARCHAR(100),
    client_phone VARCHAR(50),
    client_email VARCHAR(255),
    client_address TEXT,
    title VARCHAR(500),
    description TEXT,
    subtotal DECIMAL(15,2) DEFAULT 0,
    discount_amount DECIMAL(15,2) DEFAULT 0,
    supply_amount DECIMAL(15,2) DEFAULT 0,
    vat_amount DECIMAL(15,2) DEFAULT 0,
    validity_period INTEGER DEFAULT 30,
    tax_rate DECIMAL(5,2) DEFAULT 10,
    tax_amount DECIMAL(15,2) DEFAULT 0,
    total_amount DECIMAL(15,2) DEFAULT 0,
    currency VARCHAR(10) DEFAULT 'KRW',
    payment_terms TEXT,
    delivery_terms TEXT,
    notes TEXT,
    status VARCHAR(20) DEFAULT 'draft' CHECK (status IN ('draft','sent','accepted','rejected','expired','cancelled')),
    generated_by_ai BOOLEAN DEFAULT false,
    ai_model_used VARCHAR(100),
    ai_prompt TEXT,
    source_rag_documents TEXT[],
    file_path VARCHAR(500),
    pdf_path VARCHAR(500),
    metadata JSONB DEFAULT '{}',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    sent_at TIMESTAMP,
    accepted_at TIMESTAMP
  );`,
  `CREATE TABLE IF NOT EXISTS quotation_items (
    id SERIAL PRIMARY KEY,
    quotation_id INTEGER NOT NULL REFERENCES quotations(id) ON DELETE CASCADE,
    item_order INTEGER DEFAULT 0,
    category VARCHAR(100),
    item_code VARCHAR(50),
    item_name VARCHAR(255) NOT NULL,
    description TEXT,
    specification TEXT,
    unit VARCHAR(50) DEFAULT '식',
    quantity DECIMAL(10,2) DEFAULT 1,
    unit_price DECIMAL(15,2) DEFAULT 0,
    discount_rate DECIMAL(5,2) DEFAULT 0,
    discount_amount DECIMAL(15,2) DEFAULT 0,
    amount DECIMAL(15,2) DEFAULT 0,
    remarks TEXT,
    notes TEXT,
    is_optional BOOLEAN DEFAULT false,
    metadata JSONB DEFAULT '{}',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  );`,
  `CREATE TABLE IF NOT EXISTS quotation_versions (
    id SERIAL PRIMARY KEY,
    quotation_id INTEGER NOT NULL REFERENCES quotations(id) ON DELETE CASCADE,
    version_number INTEGER NOT NULL,
    snapshot JSONB NOT NULL,
    change_reason TEXT,
    changed_by INTEGER REFERENCES users(id) ON DELETE SET NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  );`,
  // Migration table (included for completeness)
  `CREATE TABLE IF NOT EXISTS schema_migrations (
    version VARCHAR(50) PRIMARY KEY,
    applied_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
  );`,
];

/* --------------------------- Seed default data ------------------------------ */

const seedDefaults = async (db) => {


  const homeDir = process.env.HOME || process.env.USERPROFILE || '';
  const datasetDir = homeDir ? path.join(homeDir, '.airun', 'datasets') : null;


  // Permissions
  await db.query(`
    INSERT INTO permissions (name, description) VALUES
      ('read:report','보고서 읽기 권한'),
      ('write:report','보고서 작성 권한'),
      ('delete:report','보고서 삭제 권한'),
      ('share:report','보고서 공유 권한'),
      ('manage:users','사용자 관리 권한'),
      ('manage:api_keys','API 키 관리 권한'),
      ('admin','관리자 권한')
    ON CONFLICT (name) DO NOTHING;
  `);

  await db.query(`
    INSERT INTO role_permissions (role, permission_id)
    SELECT 'admin', id FROM permissions
    ON CONFLICT DO NOTHING;
  `);

  await db.query(`
    INSERT INTO role_permissions (role, permission_id)
    SELECT 'manager', id FROM permissions
    WHERE name IN ('read:report','write:report','share:report','manage:api_keys')
    ON CONFLICT DO NOTHING;
  `);

  await db.query(`
    INSERT INTO role_permissions (role, permission_id)
    SELECT 'user', id FROM permissions
    WHERE name IN ('read:report','write:report')
    ON CONFLICT DO NOTHING;
  `);

  // Default users
  const { rows } = await db.query('SELECT COUNT(*) AS c FROM users');
  if (Number(rows[0].c) === 0) {
    const adminPassword = await bcrypt.hash('admin1234', 10);
    const demoPassword = await bcrypt.hash('demo', 10);

    const adminIns = await db.query(
      `INSERT INTO users (username, email, password_hash, name, role, status)
       VALUES ('admin','admin@airun.local',$1,'하모나이즈 관리자','admin','active') RETURNING id;`,
      [adminPassword]
    );

    const demoIns = await db.query(
      `INSERT INTO users (username, email, password_hash, name, role, status)
       VALUES ('demo','demo@airun.local',$1,'데모 사용자','user','active') RETURNING id;`,
      [demoPassword]
    );

    // Create API keys for both admin and demo users
    if (adminIns.rows.length) {
      const adminId = adminIns.rows[0].id;
      const randomKey = crypto.randomBytes(16).toString('hex');
      const keyValue = `airun_${adminId}_${randomKey}`;
      await db.query(
        `INSERT INTO api_keys (user_id, name, key_value, permissions, rate_limit, status)
         VALUES ($1,$2,$3,$4,$5,$6)`,
        [
          adminId,
          '기본 API 키',
          keyValue,
          {
            reports: { read: true, write: true, delete: true, share: true },
            users: { read: true, write: true, manage: true },
          },
          1000,
          'active',
        ]
      );
    }

    if (demoIns.rows.length) {
      const demoId = demoIns.rows[0].id;
      const randomKey = crypto.randomBytes(16).toString('hex');
      const keyValue = `airun_${demoId}_${randomKey}`;
      await db.query(
        `INSERT INTO api_keys (user_id, name, key_value, permissions, rate_limit, status)
         VALUES ($1,$2,$3,$4,$5,$6)`,
        [
          demoId,
          '데모 API 키',
          keyValue,
          {
            reports: { read: true, write: true, delete: true, share: true },
          },
          500,
          'active',
        ]
      );
    }
    log('기본 사용자 생성 완료 (admin/demo)');
  }


  // 기본 제공 파인튜닝 데이터셋 등록 (외부 FK 제약 충돌 방지)
  const builtinDatasets = [
    {
      id: 'domain_benchmark',
      name: '도메인 평가 데이터셋',
      description: '내장 도메인 평가 데이터셋',
      filename: 'domain_benchmark.json',
      metadata: { type: 'domain_eval', builtin: true }
    },
    {
      id: 'universal_benchmark',
      name: '범용 평가 데이터셋',
      description: '내장 범용 평가 데이터셋',
      filename: 'universal_benchmark.json',
      metadata: { type: 'universal_eval', builtin: true }
    }
  ];

  for (const dataset of builtinDatasets) {
    const filePath = datasetDir ? path.join(datasetDir, dataset.filename) : null;
    let totalSamples = null;

    if (filePath && fs.existsSync(filePath)) {
      try {
        const fileContent = fs.readFileSync(filePath, 'utf-8');
        const parsed = JSON.parse(fileContent);
        const examples = Array.isArray(parsed?.examples) ? parsed.examples : [];
        totalSamples = examples.length;
      } catch (error) {
        warn(`기본 데이터셋(${dataset.id}) 파일 분석 실패:`, error.message);
      }
    }

    await db.query(
      `
        INSERT INTO finetuning_datasets (id, name, description, source, file_path, total_samples, status, metadata)
        VALUES ($1, $2, $3, $4, $5, $6, 'completed', $7)
        ON CONFLICT (id) DO UPDATE SET
          name = EXCLUDED.name,
          description = EXCLUDED.description,
          source = EXCLUDED.source,
          file_path = EXCLUDED.file_path,
          total_samples = COALESCE(EXCLUDED.total_samples, finetuning_datasets.total_samples),
          status = CASE WHEN finetuning_datasets.status = 'pending' THEN EXCLUDED.status ELSE finetuning_datasets.status END,
          metadata = finetuning_datasets.metadata || EXCLUDED.metadata,
          updated_at = CURRENT_TIMESTAMP;
      `,
      [
        dataset.id,
        dataset.name,
        dataset.description,
        'system',
        filePath,
        totalSamples,
        dataset.metadata
      ]
    );
  }
  // system configurations
  await db.query(`
    INSERT INTO system_configurations (config_key, config_value, config_type, description, category) VALUES
      ('data_collection_schedule','{"interval_hours": 6, "enabled": true}','json','데이터 수집 스케줄 설정','collection'),
      ('matching_threshold','70.0','number','기업-공고 매칭 최소 임계점','matching'),
      ('proposal_template_path','templates/business_proposal.docx','string','제안서 템플릿 파일 경로','generation'),
      ('ai_model_config','{"provider":"openai","model":"gpt-4","temperature":0.7}','json','AI 모델 설정','generation'),
      ('notification_settings','{"email_enabled": false, "webhook_enabled": false}','json','알림 설정','notification')
    ON CONFLICT (config_key) DO NOTHING;
  `);
};

/* --------------------------- Migration Runner ------------------------------ */

export const createTables = async (db) => {
  try {
    const currentVersion = await getCurrentVersion(db);
    log(`현재 데이터베이스 버전: ${currentVersion}`);

    if (compareVersions(currentVersion, SCHEMA_VERSION) >= 0) {
      log(`데이터베이스가 이미 최신 버전(${SCHEMA_VERSION})입니다.`);
      return;
    }

    await withTxn(db, async () => {
      log(`데이터베이스 마이그레이션 시작 (→ v${SCHEMA_VERSION})`);

      // Enable vector extension
      await ensureExtension(db, 'vector');

      // Create/align all tables
      await runSchemaStatements(db, schemaStatements);

      // Seed default data if needed
      await seedDefaults(db);

      await recordMigration(db, SCHEMA_VERSION);
      log(`데이터베이스 마이그레이션 완료 (v${SCHEMA_VERSION})`);
    });
  } catch (e) {
    err('데이터베이스 마이그레이션 중 오류:', e);
    throw e;
  }
};
