import fs from 'fs';
import path from 'path';
import { fileURLToPath } from 'url';
import { dirname } from 'path';
import archiver from 'archiver';
import { isElectron } from '../../commons.js';
import chalk from 'chalk';
import os from 'os';
import axios from 'axios';

// electron 패키지는 필요할 때만 동적으로 import
let electronPkg = null;
const getElectronPkg = async () => {
    if (electronPkg) return electronPkg;
    
    try {
        if (isElectron()) {
            electronPkg = await import('electron');
            return electronPkg;
        }
    } catch (error) {
        console.warn('Electron 패키지를 로드할 수 없습니다:', error.message);
    }
    return null;
};

const __filename = fileURLToPath(import.meta.url);
const __dirname = dirname(__filename);

/**
 * 파일을 저장합니다.
 */
export async function saveFile(body) {
    try {
        if (!body?.content) {
            throw new Error('Content is required');
        }
        
        // 기본 저장 경로 설정
        const defaultPath = 'code_export/main.py';
        const fullPath = path.join(__dirname, '../../', body.filePath || defaultPath);
        
        // 디렉토리가 없으면 생성
        const dirPath = path.dirname(fullPath);
        if (!fs.existsSync(dirPath)) {
            await fs.promises.mkdir(dirPath, { recursive: true });
        }
        
        // 파일에 내용 저장
        await fs.promises.writeFile(fullPath, body.content, 'utf8');
        return { success: true };
    } catch (error) {
        console.error('Error saving file:', error);
        return { success: false, error: error.message };
    }
}

/**
 * ZIP 파일을 생성합니다.
 */
export async function createCodeZip(body) {
    if (!isElectron()) {
        return { success: false, error: 'This function requires Electron environment' };
    }

    try {
        // electron 패키지를 동적으로 import
        const electronPkg = await getElectronPkg();
        if (!electronPkg) {
            return { success: false, error: 'Electron package not available' };
        }

        const { dialog, app, BrowserWindow } = electronPkg;
        
        const sourcePath = path.join(__dirname, '../../', body.sourceDir);
        const downloadsPath = app.getPath('downloads');
        const targetPath = path.join(downloadsPath, body.targetZip);

        // 다운로드 폴더에 저장할 것인지 사용자에게 확인
        const window = BrowserWindow.getAllWindows()[0];
        if (!window) {
            throw new Error('No active window found');
        }

        const result = await dialog.showSaveDialog(window, {
            title: body.translations?.dialog?.saveCodeTitle || 'Save Code',
            defaultPath: targetPath,
            filters: [{ name: 'ZIP Files', extensions: ['zip'] }]
        });

        if (result.canceled) {
            return { success: false, canceled: true };
        }

        const finalPath = result.filePath;

        // ZIP 파일 생성
        const output = fs.createWriteStream(finalPath);
        const archive = archiver('zip', { zlib: { level: 9 } });

        await new Promise((resolve, reject) => {
            output.on('close', resolve);
            archive.on('error', reject);
            archive.pipe(output);
            
            // 소스 디렉토리의 모든 파일 추가
            archive.directory(sourcePath, false);
            archive.finalize();
        });

        // 임시 파일 삭제
        const tempFilePath = path.join(sourcePath, 'main.py');
        if (fs.existsSync(tempFilePath)) {
            await fs.promises.unlink(tempFilePath);
        }

        return { success: true, path: finalPath };
    } catch (error) {
        console.error('Error creating ZIP:', error);
        return { success: false, error: error.message };
    }
} 

export async function clearCache(options = {}) {
    try {
        // console.log(chalk.cyan('🧹 캐시 정리를 시작합니다...'));
        
        const homedir = os.homedir();
        const cacheBasePath = path.join(homedir, '.airun', 'cache');
        const ragCachePath = path.join(homedir, '.airun', 'rag_cache.json');
        const taskCachePath = path.join(homedir, '.airun', 'tasks.json');
        const webSearchCachePath = path.join(homedir, '.airun', 'cache', 'web_search');
        
        let clearedItems = 0;
        let errors = [];
        
        // 삭제할 항목 결정
        const itemsToClear = [];
        
        if (options.webOnly) {
            itemsToClear.push({
                type: 'web',
                paths: [
                    path.join(cacheBasePath, 'web_search')
                ]
            });
        } else if (options.ragOnly) {
            itemsToClear.push({
                type: 'rag',
                paths: [ragCachePath]
            });
        } else if (options.reportOnly) {
            itemsToClear.push({
                type: 'report',
                paths: [path.join(cacheBasePath, 'report')]
            });
        } else if (options.taskOnly) {
            itemsToClear.push({
                type: 'task',
                paths: [taskCachePath]
            });
        } else {
            // 모든 캐시 삭제
            itemsToClear.push(
                {
                    type: 'web',
                    paths: [
                        path.join(cacheBasePath, 'web_search')
                    ]
                },
                {
                    type: 'rag',
                    paths: [ragCachePath]
                },
                {
                    type: 'report',
                    paths: [path.join(cacheBasePath, 'report')]
                },
                {
                    type: 'task',
                    paths: [taskCachePath]
                }
            );

            // 웹검색 서버 캐시 제거 로직을 여기에 추가
            try {
                const response = await axios.get('http://localhost:5610/health', { timeout: 2000 })
                    .catch(() => null);

                if (response?.data?.status === 'healthy') {
                    await axios.post('http://localhost:5610/cache/clear')
                        .catch(error => {
                            console.log(chalk.yellow(`⚠️ 웹검색 서버 Redis 캐시 제거 실패: ${error.message}`));
                        });
                    // console.log(chalk.green('✅ RAG 서버 Redis 캐시를 제거했습니다'));
                }
            } catch (error) {
                console.log(chalk.yellow(`⚠️ 웹검색 서버 캐시 제거 중 오류: ${error.message}`));
            }            

            // RAG 서버 캐시 제거 로직을 여기에 추가
            try {
                const response = await axios.get('http://localhost:5600/health', { timeout: 2000 })
                    .catch(() => null);

                if (response?.data?.status === 'healthy') {
                    await axios.post('http://localhost:5600/clear-cache')
                        .catch(error => {
                            console.log(chalk.yellow(`⚠️ RAG 서버 Redis 캐시 제거 실패: ${error.message}`));
                        });
                    // console.log(chalk.green('✅ RAG 서버 Redis 캐시를 제거했습니다'));
                }
            } catch (error) {
                console.log(chalk.yellow(`⚠️ RAG 서버 캐시 제거 중 오류: ${error.message}`));
            }
        }
        
        // 각 항목 처리
        for (const item of itemsToClear) {
            for (const targetPath of item.paths) {
                try {
                    if (fs.existsSync(targetPath)) {
                        if (fs.lstatSync(targetPath).isDirectory()) {
                            await fs.promises.rm(targetPath, { recursive: true, force: true });
                            // 디렉토리인 경우에만 재생성
                            await fs.promises.mkdir(targetPath, { recursive: true });
                            // console.log(chalk.green(`✅ ${item.type} 캐시 디렉토리를 초기화했습니다: ${targetPath}`));
                        } else {
                            // 파일인 경우 삭제 후 빈 파일 생성
                            await fs.promises.unlink(targetPath);
                            if (item.type === 'rag') {
                                await fs.promises.writeFile(targetPath, '{}', 'utf8');
                                // console.log(chalk.green(`✅ ${item.type} 캐시 파일을 초기화했습니다: ${targetPath}`));
                            } else if (item.type === 'task') {
                                await fs.promises.writeFile(targetPath, '[]', 'utf8');
                                // console.log(chalk.green(`✅ ${item.type} 캐시 파일을 초기화했습니다: ${targetPath}`));
                            }
                        }
                        clearedItems++;
                    } else if (item.type !== 'task') { // tasks.json이 없는 경우는 에러가 아님
                        console.log(chalk.yellow(`⚠️ ${item.type} 캐시가 존재하지 않습니다: ${targetPath}`));
                    }
                } catch (error) {
                    errors.push({ type: item.type, path: targetPath, error: error.message });
                    console.error(chalk.red(`❌ ${item.type} 캐시 정리 중 오류 발생: ${error.message}`));
                }
            }
        }
        
        const result = {
            success: true,
            data: {
                clearedItems,
                errors,
                timestamp: new Date().toISOString()
            }
        };
        
        // 결과 보고
        // console.log('\n' + chalk.cyan('캐시 정리 결과:'));
        console.log(chalk.green(`🧹 모든 캐시가 초기화 되었습니다.`));
        // console.log(chalk.green(`🧹 캐시 초기화 완료: ${clearedItems}개`));
        if (errors.length > 0) {
            console.log(chalk.yellow(`⚠️ 오류 발생: ${errors.length}건`));
            errors.forEach(err => {
                console.log(chalk.red(`  - ${err.type} 캐시 (${err.path}): ${err.error}`));
            });
        }
        
        // if (clearedItems > 0) {
        //     console.log(chalk.green('\n🎉 캐시 정리가 완료되었습니다!'));
        // } else {
        //     console.log(chalk.yellow('\n⚠️ 정리할 캐시가 없습니다.'));
        // }
        
        return result;
        
    } catch (error) {
        console.error(chalk.red('\n❌ 캐시 정리 중 치명적인 오류가 발생했습니다:'));
        console.error(chalk.red(error.message));
        return {
            success: false,
            error: {
                message: error.message,
                stack: error.stack,
                timestamp: new Date().toISOString()
            }
        };
    }
}