import chalk from "chalk";
import pg from 'pg';
import { debugLog, logger } from '../../utils/logger.js';
import { getVarVal } from '../../configuration.js';

const { Pool } = pg;

class DatabaseManager {
  constructor() {
    this.pool = null;
    this.isConnected = false;
  }

  async initialize() {
    if (this.pool) {
      return this.pool;
    }

    // 중앙화된 설정에서 DB 연결 정보 가져오기
    const dbHost = await getVarVal('DB_HOST') || 'localhost';
    const dbPort = parseInt(await getVarVal('DB_PORT') || '5433');
    const dbUser = await getVarVal('DB_USER') || 'ivs';
    const dbPassword = await getVarVal('DB_PASSWORD') || "Exitem08**";
    const dbName = await getVarVal('DB_NAME') || 'airun';

    // logger.debug(chalk.cyan("데이터베이스 연결 설정 (중앙화된 설정):"));
    // logger.debug(chalk.cyan(`- 호스트: ${dbHost}`));
    // logger.debug(chalk.cyan(`- 포트: ${dbPort}`));
    // logger.debug(chalk.cyan(`- 데이터베이스: ${dbName}`));
    // logger.debug(chalk.cyan(`- 사용자: ${dbUser}`));

    // 1. postgres 기본 DB로 연결해서 존재 여부 체크
    const adminPool = new Pool({
      user: dbUser,
      host: dbHost,
      database: "postgres",
      password: dbPassword,
      port: dbPort,
    });

    try {
      const client = await adminPool.connect();

      const dbExistsResult = await client.query(
        `SELECT 1 FROM pg_database WHERE datname = $1`,
        [dbName]
      );

      if (dbExistsResult.rowCount === 0) {
        logger.debug(
          chalk.yellow(
            `'${dbName}' 데이터베이스가 존재하지 않습니다. 생성합니다.`
          )
        );
        await client.query(`CREATE DATABASE ${dbName}`);
        // logger.debug(chalk.green(`'${dbName}' 데이터베이스 생성 완료.`));
      } else {
        // logger.debug(chalk.green(`'${dbName}' 데이터베이스가 이미 존재합니다.`));
      }

      client.release();
    } catch (error) {
      logger.error(
        chalk.red("데이터베이스 존재 여부 확인 또는 생성 실패:"),
        error
      );
      throw error;
    } finally {
      await adminPool.end();
    }

    // 2. airun 데이터베이스에 Pool 연결 (중앙화된 설정 사용)
    const maxConnections = parseInt(await getVarVal('DB_POOL_MAX_SIZE') || "20");
    const minConnections = parseInt(await getVarVal('DB_POOL_MIN_SIZE') || "2");

    this.pool = new Pool({
      user: dbUser,
      host: dbHost,
      database: dbName,
      password: dbPassword,
      port: dbPort,
      max: maxConnections,            // 환경변수 기반 최대 연결 수
      min: minConnections,            // 환경변수 기반 최소 연결 수
      idleTimeoutMillis: 300000,      // 5분으로 연장 (유휴 연결 관리)
      connectionTimeoutMillis: 10000, // 10초 연결 타임아웃
      acquireTimeoutMillis: 60000,    // 60초 연결 획득 타임아웃
      application_name: "airun-api",
      keepAlive: true,
      statement_timeout: 300000,      // 5분 쿼리 실행 타임아웃 (RAG 초기화 고려)
      query_timeout: 300000,          // 5분 쿼리 대기 타임아웃
      idle_in_transaction_session_timeout: 600000,  // 10분 트랜잭션 내 유휴 타임아웃
    });

    // 연결 이벤트 리스너
    this.pool.on("connect", (client) => {
      if (!this.isConnected) {
        logger.debug(chalk.green("PostgreSQL 연결 풀 생성됨"));
        this.isConnected = true;
      }
      // 연결별 타임아웃 및 설정
      client.query('SET statement_timeout = 30000');
      client.query('SET idle_in_transaction_session_timeout = 60000');
      client.query('SET lock_timeout = 10000');
    });

    this.pool.on("error", (err, client) => {
      logger.error(chalk.red("PostgreSQL 연결 오류:"), err);
      this.isConnected = false;
      
      // 심각한 연결 오류 시에만 풀 재초기화
      if (err.code === 'ECONNRESET' || err.code === 'ENOTFOUND' || err.code === 'ECONNREFUSED') {
        logger.warn(chalk.yellow("연결 풀을 재초기화합니다..."));
        setTimeout(async () => {
          try {
            await this.pool.end();
            this.pool = null;
            await this.initialize();
          } catch (reinitError) {
            logger.error(chalk.red("연결 풀 재초기화 실패:"), reinitError);
          }
        }, 5000);
      }
    });

    this.pool.on("acquire", () => {
      // logger.debug("클라이언트 연결 획득됨");
    });

    this.pool.on("release", () => {
      // logger.debug("클라이언트 연결 해제됨");
    });

    // 초기 연결 테스트
    try {
      const client = await this.pool.connect();
      const result = await client.query("SELECT NOW()");
      // logger.debug(chalk.green("PostgreSQL 연결 테스트 성공:"), result.rows[0]);
      client.release();
      return this.pool;
    } catch (error) {
      logger.error(chalk.red("PostgreSQL 초기 연결 실패:"));
      logger.error(chalk.red("오류 메시지:"), error.message);
      if (error.code) {
        logger.error(chalk.red("오류 코드:"), error.code);
      }
      throw error;
    }
  }

  async checkTableExists(tableName) {
    try {
      const result = await this.query(
        `
                SELECT EXISTS (
                    SELECT FROM information_schema.tables 
                    WHERE table_schema = 'public' 
                    AND table_name = $1
                );
            `,
        [tableName]
      );
      return result.rows[0].exists;
    } catch (error) {
      logger.error(chalk.red("테이블 존재 여부 확인 실패:"), error);
      return false;
    }
  }

  async getPool() {
    if (!this.pool) {
      await this.initialize();
    }
    return this.pool;
  }

  async query(text, params) {
    const pool = await this.getPool();
    try {
      const result = await pool.query(text, params);
      return result;
    } catch (error) {
      // 오류 정보를 더 자세히 로깅
      const errorInfo = {
        message: error.message,
        code: error.code,
        severity: error.severity,
        detail: error.detail,
        hint: error.hint,
        position: error.position,
        internalPosition: error.internalPosition,
        internalQuery: error.internalQuery,
        where: error.where,
        schema: error.schema,
        table: error.table,
        column: error.column,
        dataType: error.dataType,
        constraint: error.constraint,
        file: error.file,
        line: error.line,
        routine: error.routine
      };
      
      logger.error(chalk.red("쿼리 실행 오류:"), errorInfo);
      
      // 57014 (query_canceled) 오류는 더 부드럽게 처리
      if (error.code === '57014') {
        logger.warn('쿼리가 취소되었습니다. 연결 상태를 확인하세요.');
      }
      
      throw error;
    }
  }

  async getClient() {
    const pool = await this.getPool();
    let client;
    
    try {
      client = await pool.connect();
    } catch (error) {
      logger.error(chalk.red("클라이언트 연결 획득 실패:"), error);
      throw error;
    }

    const originalRelease = client.release.bind(client);
    let released = false;

    // Release 함수를 안전하게 래핑
    client.release = (err) => {
      if (!released) {
        released = true;
        return originalRelease(err);
      }
      logger.warn(chalk.yellow("클라이언트가 이미 해제되었습니다."));
    };

    // 타임아웃 설정 (30초)
    const timeoutId = setTimeout(() => {
      if (!released) {
        logger.warn(chalk.yellow("클라이언트 연결이 30초 동안 해제되지 않았습니다. 강제 해제합니다."));
        client.release();
      }
    }, 30000);

    // 정상 해제 시 타임아웃 취소
    const originalReleaseWrapper = client.release;
    client.release = (err) => {
      clearTimeout(timeoutId);
      return originalReleaseWrapper(err);
    };

    return client;
  }

  async end() {
    if (this.pool) {
      await this.pool.end();
      this.pool = null;
      this.isConnected = false;
      logger.debug(chalk.yellow("PostgreSQL 연결 풀 종료됨"));
    }
  }

  // 스키마 초기화를 위한 새로운 메서드
  async initializeSchema() {
    try {
      logger.debug(chalk.cyan("데이터베이스 스키마 확인 중..."));

      // 필요한 모든 테이블 목록
      const requiredTables = [
        "users",
        "permissions",
        "role_permissions",
        "api_keys",
        "activity_logs",
        "schema_migrations",
      ];

      // 각 테이블의 존재 여부 확인
      const tableChecks = await Promise.all(
        requiredTables.map(async (table) => {
          const exists = await this.checkTableExists(table);
          return { table, exists };
        })
      );

      const missingTables = tableChecks.filter((check) => !check.exists);

      if (missingTables.length > 0) {
        logger.debug(
          chalk.yellow(
            `누락된 테이블 발견: ${missingTables
              .map((t) => t.table)
              .join(", ")}`
          )
        );
        logger.debug(chalk.yellow("스키마를 생성합니다..."));

        const { createTables } = await import("../auth/schema.js");
        await createTables(this);

        logger.debug(chalk.green("스키마 생성이 완료되었습니다."));

        // 테이블 생성 확인
        const result = await this.query(
          `
                    SELECT table_name 
                    FROM information_schema.tables 
                    WHERE table_schema = 'public' 
                    AND table_type = 'BASE TABLE'
                    AND table_name = ANY($1)
                `,
          [requiredTables]
        );

        logger.debug(
          chalk.green(
            `생성된 테이블: ${result.rows.map((r) => r.table_name).join(", ")}`
          )
        );
      } else {
        // 기본 데이터 확인
        const userCount = await this.query("SELECT COUNT(*) FROM users");
        const permissionCount = await this.query(
          "SELECT COUNT(*) FROM permissions"
        );

        logger.debug(chalk.green("모든 필수 테이블이 존재합니다:"));
        // logger.debug(chalk.green(`- 테이블 목록: ${requiredTables.join(', ')}`));
        logger.debug(
          chalk.green(`- 등록된 사용자 수: ${userCount.rows[0].count}`)
        );
        logger.debug(
          chalk.green(`- 등록된 권한 수: ${permissionCount.rows[0].count}`)
        );
      }
    } catch (error) {
      logger.error(chalk.red("스키마 초기화 중 오류:"), error);
      throw error;
    }
  }

  // 세션 저장/업데이트 (에러 핸들링 개선)
  async saveSession(session) {
    if (!session || typeof session !== 'object') {
        throw new Error('유효하지 않은 세션 객체입니다.');
    }

    // 세션 객체 유효성 검사
    if (session.validate) {
        session.validate();
    }

    const client = await this.getClient();
    try {
        const query = `
            INSERT INTO sessions (
                id, user_id, data, provider, model, 
                title, message_count, last_message, status, type,
                created_at, updated_at
            ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12)
            ON CONFLICT (id) DO UPDATE SET
                data = $3,
                provider = $4,
                model = $5,
                title = $6,
                message_count = $7,
                last_message = $8,
                status = $9,
                type = $10,
                updated_at = $12
            RETURNING *
        `;

        // 세션 객체를 JSON으로 변환
        const sessionJSON = session.toJSON ? session.toJSON() : session;
        
        // 히스토리에서 메시지 수와 마지막 메시지 추출
        const history = sessionJSON.history || [];
        logger.debug('[saveSession] 💾 저장 직전 history 분석:', {
            sessionId: sessionJSON.id,
            historyLength: history.length,
            historyType: typeof history,
            isArray: Array.isArray(history),
            historyPreview: history.slice(0, 3).map(h => ({
                role: h?.role,
                contentPreview: h?.content?.substring(0, 30) + '...'
            }))
        });
        const messageCount = history.length;
        const lastMessage = messageCount > 0 ? history[messageCount - 1].content : null;
        
        // 세션 데이터를 JSONB 형식으로 구성
        const sessionData = {
            parameters: sessionJSON.parameters || {},
            history: history,
            requestType: sessionJSON.requestType,
            userId: sessionJSON.userId,
            username: sessionJSON.username,
            sessionType: sessionJSON.sessionType || 'chat',
            hasTools: sessionJSON.hasTools || false,
            created_at: sessionJSON.created_at || new Date().toISOString(),
            updated_at: new Date().toISOString(),
            lastActivity: sessionJSON.lastActivity || new Date().toISOString()
        };

        // userId 처리
        let userId = sessionJSON.userId || (session.getUserId && session.getUserId());
        if (typeof userId !== 'number' && typeof userId !== 'string') {
            userId = 0; // 기본값으로 처리
        }
        let numericUserId = typeof userId === 'number' ? userId : parseInt(userId, 10);
        if (isNaN(numericUserId)) {
            numericUserId = 0; // 변환 불가 시 기본값
        }

        // 세션 제목 생성
        const title = sessionJSON.title || this.generateSessionTitle(history);

        const values = [
            sessionJSON.id,
            numericUserId,
            sessionData,
            sessionJSON.provider || 'openai',
            sessionJSON.model,
            title,
            messageCount,
            lastMessage,
            sessionJSON.status || 'active',
            sessionJSON.type || sessionData.sessionType,
            sessionJSON.created_at || new Date().toISOString(),
            new Date().toISOString()
        ];

        const result = await client.query(query, values);
        return result.rows[0];
    } catch (error) {
        logger.error('세션 저장 중 오류:', error);
        throw error;
    } finally {
        client.release();
    }
  }

  // 세션 제목 생성 헬퍼 함수
  generateSessionTitle(history) {
    if (!history || history.length === 0) {
        return '새 대화';
    }

    // 사용자의 첫 메시지를 찾아 제목으로 사용
    const firstUserMessage = history.find(msg => msg.role === 'user');
    if (firstUserMessage) {
        // 메시지가 너무 길면 잘라내기
        const title = firstUserMessage.content.substring(0, 50);
        return title.length < firstUserMessage.content.length ? `${title}...` : title;
    }

    return '새 대화';
  }

  // 세션 로드 (에러 핸들링 개선)
  async loadSession(sessionId) {
    const startTime = Date.now();
    const client = await this.getClient();
    try {
      // logger.debug('🔍 DB에서 세션 로드 시도:', { sessionId });
      
      const result = await client.query(
        'SELECT * FROM sessions WHERE id = $1',
        [sessionId]
      );
      
      const session = result.rows[0];
      const duration = Date.now() - startTime;
      
      // 성능 로깅 (10ms 이상일 때만)
      // if (duration > 10) {
      //   logger.debug(`[DB-PERFORMANCE] 세션 로드 시간: ${duration}ms (sessionId: ${sessionId})`);
      // }
      
      // if (session) {
      //   logger.debug('✅ DB에서 세션 로드됨:', {
      //     id: session.id,
      //     provider: session.provider,
      //     model: session.model,
      //     type: session.type,
      //     userId: session.user_id
      //   });
      // } else {
      //   logger.debug('❌ DB에서 세션을 찾을 수 없음:', { sessionId });
      // }
      
      return session;
    } catch (error) {
      logger.error('세션 로드 중 오류:', error);
      throw error;
    } finally {
      client.release();
    }
  }

  // 세션 삭제 (에러 핸들링 개선)
  async deleteSession(sessionId) {
    const client = await this.getClient();
    try {
      // logger.debug('[DEBUG] DB에서 세션 삭제 시도:', sessionId);
      const result = await client.query(
        'DELETE FROM sessions WHERE id = $1',
        [sessionId]
      );
      // logger.debug('[DEBUG] 삭제된 row :', result.rowCount);
      return result.rowCount;
    } catch (error) {
      logger.error('[ERROR] 세션 삭제 중 오류:', error);
      throw error;
    } finally {
      client.release();
    }
  }

  // 모든 세션 삭제 (에러 핸들링 개선)
  async deleteAllSessions() {
    const client = await this.getClient();
    try {
      const result = await client.query('DELETE FROM sessions');
      return result.rowCount;
    } catch (error) {
      logger.error('[ERROR] 전체 세션 삭제 중 오류:', error);
      throw error;
    } finally {
      client.release();
    }
  }

  // 세션 제목 업데이트 (에러 핸들링 개선)
  async updateSessionTitle(sessionId, title) {
    const client = await this.getClient();
    try {
      const result = await client.query(
        'UPDATE sessions SET title = $1, updated_at = CURRENT_TIMESTAMP WHERE id = $2 RETURNING *',
        [title, sessionId]
      );
      return result.rows[0];
    } catch (error) {
      logger.error('세션 제목 업데이트 중 오류:', error);
      throw error;
    } finally {
      client.release();
    }
  }

  // 사용자의 세션 목록 조회 (에러 핸들링 개선)
  async listSessions(userId, type = null, limit = 50, offset = 0) {
    const client = await this.getClient();
    try {
      let query = `
        SELECT s.*, u.username 
        FROM sessions s 
        LEFT JOIN users u ON s.user_id = u.id 
        WHERE s.user_id = $1
      `;
      const params = [userId];
      
      if (type) {
        query += ' AND s.type = $2';
        params.push(type);
      }
      
      query += ' ORDER BY s.updated_at DESC';
      
      if (limit) {
        query += ' LIMIT $' + (params.length + 1);
        params.push(limit);
      }
      
      if (offset) {
        query += ' OFFSET $' + (params.length + 1);
        params.push(offset);
      }
      
      const result = await client.query(query, params);
      return result.rows;
    } catch (error) {
      logger.error('세션 목록 조회 중 오류:', error);
      throw error;
    } finally {
      client.release();
    }
  }

  // 세션 상태 업데이트 (에러 핸들링 개선)
  async updateSessionStatus(sessionId, status) {
    const client = await this.getClient();
    try {
      const result = await client.query(
        'UPDATE sessions SET status = $1, updated_at = CURRENT_TIMESTAMP WHERE id = $2 RETURNING *',
        [status, sessionId]
      );
      return result.rows[0];
    } catch (error) {
      logger.error('세션 상태 업데이트 중 오류:', error);
      throw error;
    } finally {
      client.release();
    }
  }

  // 오래된 세션 정리 (에러 핸들링 및 타임아웃 개선)
  async cleanupExpiredSessions(daysOld = 30) {
    const client = await this.getClient();
    let totalDeleted = 0;
    
    try {
      // 더 긴 타임아웃 설정 (2분) 및 idle_in_transaction_session_timeout 비활성화
      await client.query('SET statement_timeout = 120000');
      await client.query('SET idle_in_transaction_session_timeout = 0');
      
      // 배치 크기로 점진적 삭제 (PostgreSQL 성능 최적화)
      const batchSize = 1000;
      let deletedCount = 0;
      
      do {
        const result = await client.query(
          `DELETE FROM sessions 
          WHERE id IN (
            SELECT id FROM sessions 
            WHERE updated_at < NOW() - INTERVAL '1 day' * $1
            ORDER BY updated_at ASC
            LIMIT $2
          )
          RETURNING id`,
          [daysOld, batchSize]
        );
        
        deletedCount = result.rows.length;
        totalDeleted += deletedCount;
        
        // 배치 간 잠시 대기 (DB 부하 분산)
        if (deletedCount > 0) {
          await new Promise(resolve => setTimeout(resolve, 100));
        }
        
      } while (deletedCount === batchSize);
      
      if (totalDeleted > 0) {
        logger.debug(chalk.yellow(`${totalDeleted}개의 만료된 세션을 정리했습니다.`));
      }
      
      return Array.from({length: totalDeleted}, (_, i) => ({id: i}));
      
    } catch (error) {
      // 57014 (query_canceled) 또는 read timeout 오류는 로그만 남기고 예외를 던지지 않음
      if (error.code === '57014' || error.message?.includes('timeout')) {
        logger.warn(`세션 정리 중 타임아웃 (${totalDeleted}개 처리됨). 다음 정리 주기에 계속합니다.`);
        return Array.from({length: totalDeleted}, (_, i) => ({id: i}));
      }
      logger.error('만료된 세션 정리 중 오류:', error);
      throw error;
    } finally {
      // 타임아웃 설정 복원
      try {
        await client.query('SET statement_timeout = DEFAULT');
        await client.query('SET idle_in_transaction_session_timeout = DEFAULT');
      } catch (resetError) {
        logger.warn('타임아웃 설정 복원 실패:', resetError.message);
      }
      client.release();
    }
  }
}

const dbManager = new DatabaseManager();

export default dbManager;
