/**
 * 견적서 AI 에이전트
 *
 * 사용자의 자연어 요청을 분석하여 견적서를 자동으로 생성합니다.
 * RAG를 통해 과거 견적서와 단가 정보를 검색하고,
 * AI 모델을 활용하여 적절한 품목과 금액을 결정합니다.
 *
 * 사용 모델:
 * - hamonize:latest - 추론 및 분석
 * - airun-chat - JSON 응답 생성
 * - chandra (vLLM) - 문서 이해 및 분석
 */

import { debugLog, logger } from '../../utils/logger.js';
import toolRegistry from '../../tools/tools-registry.js';
import toolLoader from '../../tools/tool-loader.js';

// 도구 로드 확인 (지연 로딩)
const ensureToolsLoaded = async () => {
    await toolLoader.loadAllTools();
};

// 데이터베이스 연결
const getDBPool = async () => {
    const { default: dbManager } = await import('../database/index.js');
    return dbManager;
};

// Ollama API 호출
const callOllamaAPI = async (model, prompt, options = {}) => {
    const ollamaUrl = process.env.OLLAMA_PROXY_SERVER || 'https://api.hamonize.com/ollama';

    try {
        const response = await fetch(`${ollamaUrl}/api/generate`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                model: model || 'hamonize:latest',
                prompt,
                stream: false,
                options: {
                    temperature: options.temperature || 0.3,
                    num_predict: options.maxTokens || 4096,
                    ...options
                }
            })
        });

        if (!response.ok) {
            throw new Error(`Ollama API error: ${response.status}`);
        }

        const data = await response.json();
        return data.response;
    } catch (error) {
        logger.error('[QuotationAgent] Ollama API 호출 실패:', error);
        throw error;
    }
};

// JSON 파싱 헬퍼
const parseJSON = (text) => {
    try {
        // JSON 블록 추출 시도
        const jsonMatch = text.match(/```json\s*([\s\S]*?)\s*```/);
        if (jsonMatch) {
            return JSON.parse(jsonMatch[1]);
        }

        // 직접 JSON 파싱 시도
        const cleanText = text.trim();
        if (cleanText.startsWith('{') || cleanText.startsWith('[')) {
            return JSON.parse(cleanText);
        }

        // JSON 객체/배열 찾기
        const match = cleanText.match(/(\{[\s\S]*\}|\[[\s\S]*\])/);
        if (match) {
            return JSON.parse(match[1]);
        }

        return null;
    } catch (e) {
        logger.warn('[QuotationAgent] JSON 파싱 실패:', e.message);
        return null;
    }
};

/**
 * 견적서 에이전트 클래스
 */
class QuotationAgent {
    constructor() {
        this.model = 'hamonize:latest';
        this.jsonModel = 'airun-chat';
    }

    /**
     * 사용자 요청 분석
     * 자연어 요청에서 견적서 생성에 필요한 정보를 추출합니다.
     */
    async analyzeRequest(userPrompt, context = {}) {
        logger.info('[QuotationAgent] 요청 분석 시작:', userPrompt.substring(0, 100));

        const analysisPrompt = `당신은 견적서 작성을 도와주는 AI 에이전트입니다.
다음 사용자 요청을 분석하여 견적서 생성에 필요한 정보를 JSON 형식으로 추출하세요.

사용자 요청:
"${userPrompt}"

다음 JSON 형식으로 응답하세요:
\`\`\`json
{
  "title": "견적서 제목 (예: AI Agent 개발 견적서)",
  "clientCompany": "고객사명",
  "clientContact": "담당자명 (있으면)",
  "projectDescription": "프로젝트 설명",
  "targetTotal": 목표 총액 (숫자, 부가세 포함 금액, 없으면 null),
  "includeTax": true/false (부가세 포함 여부),
  "suggestedItems": [
    {
      "itemName": "품목명",
      "description": "품목 설명",
      "estimatedRatio": 비중 (0-1 사이 숫자)
    }
  ],
  "specialRequirements": "특별 요구사항 (있으면)"
}
\`\`\`

JSON만 응답하세요.`;

        try {
            const response = await callOllamaAPI(this.jsonModel, analysisPrompt, { temperature: 0.1 });
            const analysis = parseJSON(response);

            if (!analysis) {
                throw new Error('요청 분석 결과를 파싱할 수 없습니다.');
            }

            logger.info('[QuotationAgent] 요청 분석 완료:', {
                clientCompany: analysis.clientCompany,
                targetTotal: analysis.targetTotal,
                itemCount: analysis.suggestedItems?.length
            });

            return analysis;
        } catch (error) {
            logger.error('[QuotationAgent] 요청 분석 실패:', error);
            throw error;
        }
    }

    /**
     * 관련 정보 수집
     * RAG와 데이터베이스에서 관련 정보를 수집합니다.
     */
    async gatherInformation(analysis, context = {}) {
        logger.info('[QuotationAgent] 정보 수집 시작');

        const results = {
            priceCatalog: [],
            pastQuotations: [],
            companyInfo: null,
            ragDocuments: []
        };

        const toolContext = {
            username: context.username || 'admin',
            updateProgress: context.updateProgress
        };

        try {
            // 1. 단가 카탈로그 검색
            if (context.updateProgress) {
                await context.updateProgress('📊 단가 정보 검색 중...', 20);
            }

            const priceTool = toolRegistry.getTool('price_catalog_search');
            if (priceTool) {
                const priceResult = await priceTool.handler(
                    { query: analysis.projectDescription || analysis.title },
                    toolContext
                );
                if (priceResult.success) {
                    results.priceCatalog = priceResult.data.items || [];
                }
            }

            // 2. 과거 견적서 검색
            if (context.updateProgress) {
                await context.updateProgress('📋 과거 견적서 검색 중...', 40);
            }

            const pastTool = toolRegistry.getTool('past_quotation_search');
            if (pastTool) {
                const pastResult = await pastTool.handler(
                    {
                        query: analysis.projectDescription || analysis.title,
                        clientCompany: analysis.clientCompany
                    },
                    toolContext
                );
                if (pastResult.success) {
                    results.pastQuotations = pastResult.data.quotations || [];
                    results.ragDocuments = pastResult.data.ragResults || [];
                }
            }

            // 3. 회사 정보 조회
            if (context.updateProgress) {
                await context.updateProgress('🏢 회사 정보 조회 중...', 60);
            }

            const companyTool = toolRegistry.getTool('company_info_lookup');
            if (companyTool) {
                const companyResult = await companyTool.handler({}, toolContext);
                if (companyResult.success) {
                    results.companyInfo = companyResult.data;
                }
            }

            logger.info('[QuotationAgent] 정보 수집 완료:', {
                priceItems: results.priceCatalog.length,
                pastQuotations: results.pastQuotations.length,
                hasCompanyInfo: !!results.companyInfo
            });

            return results;
        } catch (error) {
            logger.error('[QuotationAgent] 정보 수집 실패:', error);
            return results;
        }
    }

    /**
     * 견적 품목 결정
     * 수집된 정보를 바탕으로 AI가 견적 품목과 금액을 결정합니다.
     */
    async determineItems(analysis, gatheredInfo, context = {}) {
        logger.info('[QuotationAgent] 품목 결정 시작');

        if (context.updateProgress) {
            await context.updateProgress('🤖 AI가 견적 품목 구성 중...', 70);
        }

        // 참조할 정보 구성
        const referenceInfo = {
            priceCatalog: gatheredInfo.priceCatalog.slice(0, 10),
            pastQuotations: gatheredInfo.pastQuotations.slice(0, 5).map(q => ({
                title: q.title,
                totalAmount: q.totalAmount,
                items: (q.items || []).slice(0, 5)
            })),
            ragSnippets: gatheredInfo.ragDocuments.slice(0, 3).map(d => d.content?.substring(0, 200))
        };

        const itemsPrompt = `당신은 전문 견적서 작성 AI입니다.
다음 정보를 참고하여 견적서 품목을 구성하세요.

## 프로젝트 정보
- 제목: ${analysis.title}
- 고객사: ${analysis.clientCompany}
- 설명: ${analysis.projectDescription || ''}
- 목표 총액: ${analysis.targetTotal ? `${analysis.targetTotal.toLocaleString()}원 (부가세 포함)` : '지정 없음'}

## 참고 단가 정보
${JSON.stringify(referenceInfo.priceCatalog, null, 2)}

## 과거 유사 견적서
${JSON.stringify(referenceInfo.pastQuotations, null, 2)}

## AI 제안 품목 (분석 결과)
${JSON.stringify(analysis.suggestedItems, null, 2)}

## 요청사항
1. 프로젝트에 적합한 품목들을 구성하세요
2. 각 품목의 수량과 단가를 현실적으로 책정하세요
3. 참고 단가가 있으면 활용하세요
4. 목표 총액이 있으면 그에 맞게 조정하세요 (부가세 10% 포함)
5. 품목은 3-8개 정도로 구성하세요

다음 JSON 형식으로 응답하세요:
\`\`\`json
{
  "items": [
    {
      "itemName": "품목명",
      "description": "품목 설명",
      "unit": "식/개/시간/MM 등",
      "quantity": 수량 (숫자),
      "unitPrice": 단가 (숫자)
    }
  ],
  "reasoning": "품목 구성 이유 간략 설명"
}
\`\`\`

JSON만 응답하세요.`;

        try {
            const response = await callOllamaAPI(this.jsonModel, itemsPrompt, { temperature: 0.2 });
            const itemsResult = parseJSON(response);

            if (!itemsResult || !itemsResult.items || itemsResult.items.length === 0) {
                // 기본 품목 생성
                return this.generateDefaultItems(analysis);
            }

            logger.info('[QuotationAgent] 품목 결정 완료:', {
                itemCount: itemsResult.items.length,
                reasoning: itemsResult.reasoning?.substring(0, 100)
            });

            return itemsResult;
        } catch (error) {
            logger.error('[QuotationAgent] 품목 결정 실패:', error);
            return this.generateDefaultItems(analysis);
        }
    }

    /**
     * 기본 품목 생성 (AI 실패 시 폴백)
     */
    generateDefaultItems(analysis) {
        const targetTotal = analysis.targetTotal || 5000000;
        const netTotal = Math.round(targetTotal / 1.1); // 부가세 제외

        const defaultItems = [
            { itemName: '요구사항 분석 및 설계', description: '시스템 요구사항 분석 및 설계 문서 작성', unit: '식', ratio: 0.15 },
            { itemName: '개발', description: '시스템 개발 및 구현', unit: '식', ratio: 0.50 },
            { itemName: '테스트 및 검증', description: '시스템 테스트 및 품질 검증', unit: '식', ratio: 0.15 },
            { itemName: '문서화 및 교육', description: '사용자 문서 작성 및 교육', unit: '식', ratio: 0.10 },
            { itemName: '유지보수 지원', description: '초기 운영 지원 및 유지보수', unit: '식', ratio: 0.10 }
        ];

        if (analysis.suggestedItems && analysis.suggestedItems.length > 0) {
            // AI가 제안한 품목 사용
            const totalRatio = analysis.suggestedItems.reduce((sum, item) =>
                sum + (item.estimatedRatio || 1 / analysis.suggestedItems.length), 0);

            return {
                items: analysis.suggestedItems.map(item => ({
                    itemName: item.itemName,
                    description: item.description || '',
                    unit: '식',
                    quantity: 1,
                    unitPrice: Math.round(netTotal * (item.estimatedRatio || 1 / analysis.suggestedItems.length) / totalRatio)
                })),
                reasoning: '사용자 요청에서 추출한 품목으로 구성'
            };
        }

        return {
            items: defaultItems.map(item => ({
                itemName: item.itemName,
                description: item.description,
                unit: item.unit,
                quantity: 1,
                unitPrice: Math.round(netTotal * item.ratio)
            })),
            reasoning: '표준 소프트웨어 개발 프로젝트 품목 구성'
        };
    }

    /**
     * 견적서 검증
     * RAG에서 견적서 작성 원칙을 검색하고 생성된 견적서를 검증합니다.
     */
    async verifyQuotation(quotationData, context = {}) {
        logger.info('[QuotationAgent] 견적서 검증 시작');

        try {
            // RAG에서 견적서 작성 원칙 검색
            const docSearchTool = toolRegistry.getTool('doc_search');
            let principlesText = '';

            if (docSearchTool) {
                const ragResult = await docSearchTool.handler({
                    query: '견적서 작성 원칙 검증 기준 단가 적정성',
                    ragSearchScope: 'all'
                }, { username: context.username || 'admin' });

                // doc_search 도구의 응답 구조: { success, data: { results: { documents: [[...]], metadatas: [[...]] }, count }, message }
                if (ragResult.success && ragResult.data?.results?.documents?.[0]?.length > 0) {
                    principlesText = ragResult.data.results.documents[0].join('\n\n');
                    logger.info('[QuotationAgent] RAG에서 견적서 원칙 문서 검색됨:', ragResult.data.count, '개');
                } else {
                    logger.warn('[QuotationAgent] RAG에서 견적서 원칙 문서를 찾지 못함');
                }
            } else {
                logger.warn('[QuotationAgent] doc_search 도구를 찾을 수 없음');
            }

            // 검증 프롬프트 구성
            const quotationSummary = {
                title: quotationData.title,
                clientCompany: quotationData.clientCompany,
                supplyAmount: quotationData.subtotal,
                vatAmount: quotationData.taxAmount,
                totalAmount: quotationData.totalAmount,
                itemCount: quotationData.items?.length || 0,
                items: quotationData.items?.map(item => ({
                    name: item.itemName,
                    unit: item.unit,
                    quantity: item.quantity,
                    unitPrice: item.unitPrice,
                    amount: item.amount
                }))
            };

            // 1단계: hamonize 모델로 추론/분석 수행
            const analysisPrompt = `당신은 견적서 품질 검증 전문가입니다.
다음 견적서가 작성 원칙에 맞게 작성되었는지 상세히 분석해주세요.

## 견적서 작성 원칙
${principlesText || `
- 최소 3개 이상 품목 권장
- 시간당 단가: 50,000원 ~ 300,000원 범위
- 일당 단가: 400,000원 ~ 2,400,000원 범위
- 공급가액의 10%가 부가세로 계산되어야 함
- 총액 = 공급가액 + 부가세
- 프로젝트 유형에 맞는 품목 구성
`}

## 검증 대상 견적서
${JSON.stringify(quotationSummary, null, 2)}

## 검증해야 할 항목
1. 품목 구성 적절성: 품목 수가 3개 이상인지, 프로젝트에 필요한 항목들이 포함되었는지
2. 단가 적정성: 각 품목의 단위(시간, 일, 개, 식)에 맞는 적정 단가인지 확인
   - 시간당 단가: 50,000원 ~ 300,000원
   - 일당 단가: 400,000원 ~ 2,400,000원
   - 일괄(식) 단가: 프로젝트 규모에 적합한지
3. 금액 계산 정확성: 품목금액=단가×수량, 부가세=공급가액×10%
4. 프로젝트 연관성: 견적서 제목과 품목 구성이 일치하는지
5. 전체 품질: 전반적인 견적서 완성도

각 항목별로 통과/실패 여부와 그 이유를 상세히 설명해주세요.
마지막에 전체 등급(A/B/C/D)과 점수(0-100)를 제시해주세요.`;

            logger.info('[QuotationAgent] 1단계: hamonize 모델로 추론 분석 시작');
            const analysisResult = await callOllamaAPI(
                'hamonize:latest',
                analysisPrompt,
                { temperature: 0.3 }
            );
            logger.debug('[QuotationAgent] hamonize 분석 결과:', analysisResult?.substring?.(0, 500) || analysisResult);

            // 2단계: airun-chat 모델로 JSON 형식 출력 생성
            const jsonPrompt = `다음 견적서 검증 분석 결과를 JSON 형식으로 변환해주세요.

## 분석 결과
${analysisResult}

## 필요한 JSON 형식
반드시 아래 형식의 순수 JSON만 출력하세요. 다른 설명이나 마크다운 없이 JSON만 출력합니다.

{
    "verified": true 또는 false (등급이 A 또는 B이면 true, C 또는 D이면 false),
    "grade": "A" 또는 "B" 또는 "C" 또는 "D",
    "score": 0부터 100 사이의 숫자,
    "checks": {
        "itemComposition": { "pass": true/false, "message": "분석 결과 요약" },
        "priceAppropriateness": { "pass": true/false, "message": "분석 결과 요약" },
        "calculationAccuracy": { "pass": true/false, "message": "분석 결과 요약" },
        "projectRelevance": { "pass": true/false, "message": "분석 결과 요약" },
        "overallQuality": { "pass": true/false, "message": "분석 결과 요약" }
    },
    "summary": "전체 검증 결과 한 문장 요약",
    "recommendations": ["개선이 필요한 경우 권고사항 목록"]
}`;

            logger.info('[QuotationAgent] 2단계: airun-chat 모델로 JSON 출력 생성');
            let verificationResult = await callOllamaAPI(
                'airun-chat:latest',
                jsonPrompt,
                { temperature: 0.1 }
            );

            // airun-chat JSON 실패 시 재시도 (이슈 #218 참고: airun-chat은 JSON 품질 100%)
            if (!verificationResult || typeof verificationResult !== 'string' || !verificationResult.includes('{')) {
                logger.warn('[QuotationAgent] airun-chat 첫 번째 시도 실패, 재시도');
                verificationResult = await callOllamaAPI(
                    'airun-chat:latest',
                    jsonPrompt + '\n\n반드시 JSON 형식으로만 응답하세요.',
                    { temperature: 0.05 }
                );
            }

            // 결과 파싱
            let verification;
            try {
                logger.debug('[QuotationAgent] AI 검증 응답 타입:', typeof verificationResult);

                let resultToParse = verificationResult;

                // 문자열인 경우 JSON 추출 시도
                if (typeof resultToParse === 'string') {
                    logger.debug('[QuotationAgent] AI 검증 응답 (raw):', resultToParse.substring(0, 500));

                    // 마크다운 코드 블록에서 JSON 추출
                    const jsonMatch = resultToParse.match(/```(?:json)?\s*([\s\S]*?)```/);
                    if (jsonMatch) {
                        resultToParse = jsonMatch[1].trim();
                        logger.debug('[QuotationAgent] 코드 블록에서 JSON 추출됨');
                    }

                    // JSON 객체 패턴 추출 (중괄호로 시작하는 부분)
                    const jsonObjMatch = resultToParse.match(/\{[\s\S]*\}/);
                    if (jsonObjMatch) {
                        resultToParse = jsonObjMatch[0];
                    }

                    verification = JSON.parse(resultToParse);
                } else if (typeof resultToParse === 'object' && resultToParse !== null) {
                    verification = resultToParse;
                } else {
                    throw new Error('유효하지 않은 응답 형식');
                }

                // 필수 필드 검증 및 기본값 설정
                verification = {
                    verified: verification.verified ?? true,
                    grade: verification.grade || 'B',
                    score: verification.score ?? 75,
                    checks: verification.checks || {},
                    summary: verification.summary || '검증 완료',
                    recommendations: verification.recommendations || []
                };

            } catch (e) {
                logger.warn('[QuotationAgent] 검증 결과 파싱 실패:', e.message);
                logger.debug('[QuotationAgent] 파싱 실패한 원본:', typeof verificationResult === 'string' ? verificationResult.substring(0, 300) : verificationResult);
                verification = {
                    verified: true,
                    grade: 'B',
                    score: 75,
                    checks: {},
                    summary: '기본 검증 완료',
                    recommendations: []
                };
            }

            logger.info('[QuotationAgent] 검증 완료:', {
                verified: verification.verified,
                grade: verification.grade,
                score: verification.score
            });

            return {
                success: true,
                verification: verification
            };
        } catch (error) {
            logger.error('[QuotationAgent] 검증 실패:', error);
            return {
                success: false,
                error: error.message,
                verification: {
                    verified: true,
                    grade: 'N/A',
                    score: 0,
                    summary: '검증 프로세스 오류로 인해 검증을 건너뛰었습니다.',
                    recommendations: []
                }
            };
        }
    }

    /**
     * 견적서 생성 실행
     */
    async generateQuotation(userPrompt, context = {}) {
        logger.info('[QuotationAgent] 견적서 생성 프로세스 시작');

        try {
            // 도구 로드 확인
            await ensureToolsLoaded();
            // 1. 요청 분석
            if (context.updateProgress) {
                await context.updateProgress('🔍 요청 분석 중...', 10);
            }
            const analysis = await this.analyzeRequest(userPrompt, context);

            // 2. 정보 수집
            const gatheredInfo = await this.gatherInformation(analysis, context);

            // 3. 품목 결정
            const itemsResult = await this.determineItems(analysis, gatheredInfo, context);

            // 4. 견적서 생성
            if (context.updateProgress) {
                await context.updateProgress('📝 견적서 생성 중...', 85);
            }

            const generatorTool = toolRegistry.getTool('quotation_generator');
            if (!generatorTool) {
                throw new Error('견적서 생성 도구를 찾을 수 없습니다.');
            }

            const quotationResult = await generatorTool.handler({
                title: analysis.title,
                clientCompany: analysis.clientCompany,
                clientContact: analysis.clientContact || '',
                items: itemsResult.items,
                targetTotal: analysis.targetTotal,
                taxRate: 10,
                notes: analysis.specialRequirements || '',
                userPrompt: userPrompt || ''  // 사용자 요청 프롬프트 저장
            }, {
                username: context.username || 'admin',
                updateProgress: context.updateProgress
            });

            if (!quotationResult.success) {
                throw new Error(quotationResult.error || '견적서 생성 실패');
            }

            // 5. 견적서 검증 및 자동 재작성
            const MAX_RETRY_ATTEMPTS = 3;
            let currentQuotationResult = quotationResult;
            let currentItemsResult = itemsResult;
            let verificationResult = null;
            let attemptCount = 1;

            while (attemptCount <= MAX_RETRY_ATTEMPTS) {
                if (context.updateProgress) {
                    const progressMsg = attemptCount === 1
                        ? '🔍 견적서 검증 중...'
                        : `🔄 검증 기반 재작성 중... (${attemptCount}/${MAX_RETRY_ATTEMPTS})`;
                    await context.updateProgress(progressMsg, 85 + (attemptCount * 3));
                }

                try {
                    verificationResult = await this.verifyQuotation(currentQuotationResult.data, context);
                    logger.info(`[QuotationAgent] 검증 결과 (시도 ${attemptCount}):`, {
                        grade: verificationResult?.verification?.grade,
                        score: verificationResult?.verification?.score,
                        verified: verificationResult?.verification?.verified
                    });
                } catch (verifyError) {
                    logger.warn('[QuotationAgent] 검증 중 오류 발생 (계속 진행):', verifyError.message);
                    verificationResult = {
                        success: false,
                        error: verifyError.message,
                        verification: {
                            verified: false,
                            grade: 'N/A',
                            score: 0,
                            summary: '검증을 수행할 수 없습니다.',
                            recommendations: ['검증 서비스를 확인해 주세요.']
                        }
                    };
                    break; // 검증 오류 시 재시도 중단
                }

                // 검증 통과 (등급 A 또는 B) 시 반복 종료
                const grade = verificationResult?.verification?.grade;
                if (grade === 'A' || grade === 'B') {
                    logger.info(`[QuotationAgent] 검증 통과 (등급: ${grade}), 시도 횟수: ${attemptCount}`);
                    break;
                }

                // 최대 시도 횟수 도달 시 종료
                if (attemptCount >= MAX_RETRY_ATTEMPTS) {
                    logger.warn(`[QuotationAgent] 최대 시도 횟수(${MAX_RETRY_ATTEMPTS}) 도달, 현재 결과 사용`);
                    break;
                }

                // 등급 C 또는 D: 권고사항 기반 재작성
                logger.info(`[QuotationAgent] 등급 ${grade}, 권고사항 기반 재작성 시작`);

                const recommendations = verificationResult?.verification?.recommendations || [];
                const failedChecks = Object.entries(verificationResult?.verification?.checks || {})
                    .filter(([_, check]) => !check?.pass)
                    .map(([key, check]) => `${key}: ${check?.message || ''}`)
                    .join(', ');

                // 권고사항을 반영한 품목 재결정
                const improvementPrompt = `
이전 견적서가 검증에서 등급 ${grade}를 받았습니다.

문제점:
${failedChecks}

개선 권고사항:
${recommendations.map((r, i) => `${i + 1}. ${r}`).join('\n')}

위 내용을 반영하여 품목과 단가를 수정해주세요. 특히 단가가 기준 범위를 벗어난 경우 적정 범위 내로 조정해주세요.
`;

                const improvedAnalysis = {
                    ...analysis,
                    specialRequirements: (analysis.specialRequirements || '') + '\n' + improvementPrompt
                };

                // 품목 재결정
                currentItemsResult = await this.determineItems(improvedAnalysis, gatheredInfo, context);

                // 견적서 재생성
                currentQuotationResult = await generatorTool.handler({
                    title: analysis.title,
                    clientCompany: analysis.clientCompany,
                    clientContact: analysis.clientContact || '',
                    items: currentItemsResult.items,
                    targetTotal: analysis.targetTotal,
                    taxRate: 10,
                    notes: analysis.specialRequirements || '',
                    userPrompt: userPrompt || ''  // 사용자 요청 프롬프트 저장
                }, {
                    username: context.username || 'admin',
                    updateProgress: context.updateProgress
                });

                if (!currentQuotationResult.success) {
                    logger.error('[QuotationAgent] 재작성 중 오류:', currentQuotationResult.error);
                    break;
                }

                attemptCount++;
            }

            // 검증 결과에 시도 횟수 추가
            if (verificationResult?.verification) {
                verificationResult.verification.attemptCount = attemptCount;
                verificationResult.verification.maxAttempts = MAX_RETRY_ATTEMPTS;
            }

            // 검증 결과를 DB metadata에 저장
            const quotationId = currentQuotationResult.data?.quotationId;
            if (quotationId && verificationResult?.verification) {
                try {
                    const db = await getDBPool();
                    await db.query(
                        `UPDATE quotations SET metadata = jsonb_set(
                            COALESCE(metadata, '{}'::jsonb),
                            '{aiVerification}',
                            $1::jsonb
                        ) WHERE id = $2`,
                        [JSON.stringify(verificationResult.verification), quotationId]
                    );
                    logger.info(`[QuotationAgent] AI 검증 결과 저장 완료 (quotationId: ${quotationId})`);
                } catch (saveError) {
                    logger.warn('[QuotationAgent] AI 검증 결과 저장 실패:', saveError.message);
                }
            }

            // 6. 최종 결과 구성
            if (context.updateProgress) {
                const finalGrade = verificationResult?.verification?.grade || 'N/A';
                const statusEmoji = (finalGrade === 'A' || finalGrade === 'B') ? '✅' : '⚠️';
                await context.updateProgress(`${statusEmoji} 견적서 생성 완료 (등급: ${finalGrade})`, 100);
            }

            return {
                success: true,
                data: {
                    ...currentQuotationResult.data,
                    analysis: {
                        originalRequest: userPrompt,
                        extractedInfo: analysis,
                        itemsReasoning: currentItemsResult.reasoning
                    },
                    references: {
                        usedPriceCatalog: gatheredInfo.priceCatalog.length > 0,
                        usedPastQuotations: gatheredInfo.pastQuotations.length > 0,
                        usedRagDocuments: gatheredInfo.ragDocuments.length > 0
                    },
                    companyInfo: gatheredInfo.companyInfo,
                    verification: verificationResult?.verification || null
                },
                message: quotationResult.message
            };
        } catch (error) {
            logger.error('[QuotationAgent] 견적서 생성 실패:', error);
            return {
                success: false,
                error: error.message,
                message: `견적서 생성 실패: ${error.message}`
            };
        }
    }

    /**
     * 견적서 수정 요청 처리
     */
    async modifyQuotation(quotationId, modificationRequest, context = {}) {
        logger.info('[QuotationAgent] 견적서 수정 요청:', quotationId);

        try {
            const db = await getDBPool();

            // 기존 견적서 조회
            const quotationResult = await db.query(
                `SELECT q.*,
                    (SELECT json_agg(qi.*) FROM quotation_items qi WHERE qi.quotation_id = q.id) as items
                 FROM quotations q WHERE q.id = $1`,
                [quotationId]
            );

            if (quotationResult.rows.length === 0) {
                throw new Error('견적서를 찾을 수 없습니다.');
            }

            const currentQuotation = quotationResult.rows[0];

            // 수정 요청 분석
            const modifyPrompt = `현재 견적서 정보:
${JSON.stringify(currentQuotation, null, 2)}

수정 요청:
"${modificationRequest}"

수정 사항을 JSON으로 응답하세요:
\`\`\`json
{
  "modifications": {
    "title": "수정된 제목 (변경 시)",
    "items": [...] // 수정된 품목 배열 (변경 시)
  },
  "explanation": "수정 내용 설명"
}
\`\`\``;

            const response = await callOllamaAPI(this.jsonModel, modifyPrompt, { temperature: 0.2 });
            const modifications = parseJSON(response);

            if (!modifications) {
                throw new Error('수정 요청을 분석할 수 없습니다.');
            }

            // 수정 적용 로직은 API에서 처리
            return {
                success: true,
                data: {
                    quotationId,
                    currentQuotation,
                    proposedModifications: modifications
                },
                message: '수정 사항이 분석되었습니다.'
            };
        } catch (error) {
            logger.error('[QuotationAgent] 견적서 수정 실패:', error);
            return {
                success: false,
                error: error.message,
                message: `견적서 수정 실패: ${error.message}`
            };
        }
    }
}

// 싱글톤 인스턴스
const quotationAgent = new QuotationAgent();

export { QuotationAgent, quotationAgent };
export default quotationAgent;
