#!/usr/bin/env python3
"""
AIRUN Report Generation Server - Full Implementation
완전한 보고서 생성 서버 (report_generator.py 통합)
RAG 서비스와 동일한 패턴으로 구현 + 데이터베이스 기반 작업 관리
"""

import os
import sys
import asyncio
import uvicorn
import logging
import time
import json
import uuid
import hashlib
import shutil
import warnings
import importlib.util
import glob
from datetime import datetime
from pathlib import Path
from typing import Optional, Dict, Any, List, Union
from contextlib import asynccontextmanager
from dataclasses import dataclass
from concurrent.futures import ThreadPoolExecutor
from threading import Lock
import traceback

# 데이터베이스 연결
import psycopg2
from psycopg2.extras import RealDictCursor
from psycopg2.pool import ThreadedConnectionPool
import psycopg2.sql as sql

# FastAPI 관련
from fastapi import FastAPI, HTTPException, BackgroundTasks
from fastapi.responses import JSONResponse, StreamingResponse
from fastapi.middleware.cors import CORSMiddleware
from pydantic import BaseModel, Field
import queue
import threading

# 모든 경고 무시
warnings.filterwarnings('ignore', category=DeprecationWarning)
warnings.filterwarnings('ignore', category=UserWarning)
warnings.filterwarnings('ignore', category=FutureWarning)

# 프로젝트 루트 디렉토리를 Python 경로에 추가
current_dir = Path(__file__).parent.absolute()
project_root = current_dir.parent.parent
sys.path.insert(0, str(project_root))

# 가상환경 utils 패키지 경로 추가 (컴파일된 utils.so 파일 우선 로드를 위해)
venv_site_packages = os.path.expanduser("~/.airun_venv/lib/python3.12/site-packages")
if os.path.exists(venv_site_packages):
    # 컴파일된 utils.so 파일이 있는지 확인
    compiled_utils_so = os.path.join(venv_site_packages, "utils.cpython-312-x86_64-linux-gnu.so")
    if os.path.exists(compiled_utils_so):
        print(f"[DEBUG] 컴파일된 utils.so 파일 발견: {compiled_utils_so}")
        # 가상환경 경로를 추가하되, utils 패키지 디렉터리는 제외
        sys.path.insert(0, venv_site_packages)
        print(f"[DEBUG] 가상환경 사이트 패키지 경로 추가: {venv_site_packages}")
    else:
        print(f"[DEBUG] 컴파일된 utils.so 파일 없음, 가상환경 경로 추가")
        sys.path.insert(0, venv_site_packages)

# .env 파일 로드
def load_env_file():
    """프로젝트 루트의 .env 파일을 로드"""
    env_file = project_root / ".env"
    if env_file.exists():
        with open(env_file, 'r', encoding='utf-8') as f:
            for line in f:
                line = line.strip()
                if line and not line.startswith('#') and '=' in line:
                    key, value = line.split('=', 1)
                    os.environ[key.strip()] = value.strip()
        print(f"[DEBUG] .env 파일 로드됨: {env_file}")
    else:
        print(f"[DEBUG] .env 파일 없음: {env_file}")

# .env 파일 로드 실행
load_env_file()

# 기본 설정 및 경로 정의
BASE_DIR = os.path.expanduser("~/.airun")
LOG_DIR = os.path.join(BASE_DIR, "logs")
LOG_FILE = os.path.join(LOG_DIR, "airun-report-server.log")
CONFIG_FILE = os.path.join(BASE_DIR, "airun.conf")

# 서드파티 라이브러리
try:
    import pandas as pd
    import numpy as np
    import matplotlib.pyplot as plt
    import matplotlib
    matplotlib.use('Agg')
    import requests
    from openai import OpenAI
except ImportError as e:
    logging.warning(f"일부 라이브러리 임포트 실패: {e}")

# 프로젝트 모듈들 - RAG 서비스와 동일한 패턴으로 임포트
try:
    # utils 모듈 임포트 - 컴파일된 .so 파일 우선 로드
    compiled_utils_so = os.path.join(venv_site_packages, "utils.cpython-312-x86_64-linux-gnu.so")
    utils_py_file = os.path.join(project_root, "utils.py")
    
    if os.path.exists(compiled_utils_so):
        # 컴파일된 .so 파일 직접 로드
        print(f"[DEBUG] 컴파일된 utils.so 파일 사용: {compiled_utils_so}")
        spec = importlib.util.spec_from_file_location("utils", compiled_utils_so)
        utils = importlib.util.module_from_spec(spec)
        sys.modules["utils"] = utils
        spec.loader.exec_module(utils)
        from utils import load_config, create_matplotlib
        print(f"[DEBUG] 컴파일된 utils.so 로드 완료 - AIProvider 클래스 존재: {hasattr(utils, 'AIProvider')}")
    elif os.path.exists(utils_py_file):
        # 프로젝트 루트의 .py 파일 사용
        print(f"[DEBUG] 프로젝트 루트의 utils.py 사용: {utils_py_file}")
        spec = importlib.util.spec_from_file_location("utils", utils_py_file)
        utils = importlib.util.module_from_spec(spec)
        sys.modules["utils"] = utils
        spec.loader.exec_module(utils)
        from utils import load_config, create_matplotlib
        print(f"[DEBUG] utils.py 로드 완료 - AIProvider 클래스 존재: {hasattr(utils, 'AIProvider')}")
    else:
        # 일반 import 시도 (폴백)
        print(f"[DEBUG] 일반 import 시도 (폴백)")
        import utils
        from utils import load_config, create_matplotlib
        print(f"[DEBUG] 일반 import 완료 - AIProvider 클래스 존재: {hasattr(utils, 'AIProvider')}")
    
    # report_generator 모듈 동적 임포트 (RAG 서비스와 동일한 패턴)
    script_dir = os.path.dirname(os.path.abspath(__file__))
    py_file = os.path.join(script_dir, "report_generator.py")
    
    if os.path.exists(py_file):
        # .py 파일이 있으면 우선 사용 (개발 모드)
        module_path = py_file
        print(f"[DEBUG] Using Python source file: report_generator.py")
    else:
        # 컴파일된 파일 찾기
        if sys.platform.startswith('win'):
            module_pattern = os.path.join(script_dir, "report_generator.cp*-win_amd64.pyd")
        elif sys.platform.startswith('darwin'):
            module_pattern = os.path.join(script_dir, "report_generator.cpython-*-darwin.so")
        else:  # Linux
            module_pattern = os.path.join(script_dir, "report_generator.cpython-312-x86_64-linux-gnu.so")
        
        module_files = glob.glob(module_pattern)
        if module_files:
            module_path = module_files[0]
            print(f"[DEBUG] Using compiled module: {os.path.basename(module_path)}")
        else:
            # 정확한 파일명으로 시도
            exact_file = os.path.join(script_dir, "report_generator.cpython-312-x86_64-linux-gnu.so")
            if os.path.exists(exact_file):
                module_path = exact_file
                print(f"[DEBUG] Using exact compiled module: {os.path.basename(exact_file)}")
            else:
                raise ImportError(f"report_generator 모듈을 찾을 수 없습니다. 시도한 경로: {py_file}, {module_pattern}, {exact_file}")
    
    # 모듈 로드
    spec = importlib.util.spec_from_file_location("report_generator", module_path)
    if spec is None:
        raise ImportError(f"Failed to create module spec for {module_path}")
    
    report_generator_module = importlib.util.module_from_spec(spec)
    sys.modules["report_generator"] = report_generator_module
    spec.loader.exec_module(report_generator_module)
    
    # 필요한 클래스들 임포트
    from report_generator import (
        BizPlanContentGenerator,
        SectionConfig,
        BizPlanDocument,
        RAGServiceClient,
        WebSearchServiceClient
    )
    
    print(f"[DEBUG] Successfully imported report_generator module")
    
except ImportError as e:
    print(f"WARNING: report_generator 모듈 임포트 실패: {e}")
    # 기본 구현으로 폴백
    report_generator_module = None
    BizPlanContentGenerator = None
    SectionConfig = None
    RAGServiceClient = None
    WebSearchServiceClient = None

# 로깅 설정 (logger 정의를 앞으로 이동)
log_dir = os.path.expanduser('~/.airun/logs')
os.makedirs(log_dir, exist_ok=True)
log_file = os.path.join(log_dir, 'airun-report.log')

# 환경변수에서 로그 레벨 가져오기
def get_log_level_from_env():
    log_level_str = os.getenv('LOG_LEVEL', 'INFO').upper()
    level_mapping = {
        'DEBUG': logging.DEBUG,
        'INFO': logging.INFO,
        'WARNING': logging.WARNING,
        'ERROR': logging.ERROR,
        'CRITICAL': logging.CRITICAL
    }
    return level_mapping.get(log_level_str, logging.INFO)

log_level = get_log_level_from_env()

logging.basicConfig(
    level=log_level,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s',
    handlers=[
        logging.FileHandler(log_file),
        logging.StreamHandler()
    ]
)
logger = logging.getLogger('airun-report')
logger.info(f"로그 레벨 설정: {logging.getLevelName(log_level)}")

# 전역 변수 (메모리 기반 제거, DB 기반으로 변경)
# jobs_storage = {}  # 제거
# jobs_lock = Lock()  # 제거
# active_jobs = {}  # 제거 - DB로 관리

# 활성 작업 관리 (메모리 기반에서 DB 기반으로 변경)
server_stats = {
    "start_time": datetime.now(),
    "total_requests": 0,
    "completed_jobs": 0,
    "failed_jobs": 0,
    "active_jobs": 0
}

# SSE 스트리밍 관리 (메모리 기반 유지 - 실시간 스트리밍은 메모리에서 처리)
streaming_queues = {}  # job_id -> queue.Queue
streaming_locks = {}   # job_id -> threading.Lock

# 전역 서비스 인스턴스
report_service = None
db_manager = None
clarification_service = None  # 🔍 Clarification 서비스 추가

# 모듈 전역 변수
generate_bizplan_module = None

# 상수 정의 - 환경변수 사용
RAG_SERVICE_URL = f"http://127.0.0.1:{os.getenv('RAG_SERVER_PORT', '5600')}"
WEB_SEARCH_URL = f"http://127.0.0.1:{os.getenv('WEB_SEARCH_PORT', '5610')}"

# AIRUN 시스템 경로 설정
project_root = Path(__file__).parent.parent.parent
sys.path.insert(0, str(project_root))

# 기본 템플릿 설정 (폴백용)
DEFAULT_TEMPLATE_CONFIG = {
    "simple": {
        "title": "간단한 보고서",
        "sections": {
            "executive_summary": {
                "title": "요약",
                "prompt": "다음 내용을 바탕으로 간단한 요약을 작성하세요: {prompt}",
                "section_type": "summary"
            },
            "main_content": {
                "title": "주요 내용", 
                "prompt": "다음 주제에 대해 상세히 설명하세요: {prompt}",
                "section_type": "content"
            },
            "conclusion": {
                "title": "결론",
                "prompt": "다음 내용을 바탕으로 결론을 작성하세요: {prompt}",
                "section_type": "conclusion"
            }
        }
    }
}

# =============================================================================
# 데이터베이스 관리자 클래스 추가
# =============================================================================

class DatabaseManager:
    """데이터베이스 관리자 클래스 - 보고서 작업 관리용"""
    
    def __init__(self):
        self.pool = None
        self.is_connected = False
        self.logger = logger
        
    def initialize(self):
        """데이터베이스 연결 초기화"""
        if self.pool:
            return True
            
        try:
            # 환경변수에서 데이터베이스 설정 가져오기 (중앙집중화된 DB_ 환경변수 사용)
            db_config = {
                'host': os.getenv('DB_HOST', os.getenv('POSTGRES_HOST', 'localhost')),
                'port': int(os.getenv('DB_PORT', os.getenv('POSTGRES_PORT', '5433'))),
                'database': os.getenv('DB_NAME', os.getenv('POSTGRES_DB', 'airun')),
                'user': os.getenv('DB_USER', os.getenv('POSTGRES_USER', 'ivs')),
                'password': os.getenv('DB_PASSWORD', os.getenv('POSTGRES_PASSWORD', "Exitem08**")),
                'minconn': 1,
                'maxconn': 10
            }
            
            self.logger.info(f"🔗 데이터베이스 연결 초기화 중... (호스트: {db_config['host']}:{db_config['port']})")
            
            # 연결 풀 생성
            self.pool = ThreadedConnectionPool(**db_config)
            self.is_connected = True
            
            # 연결 테스트
            conn = self.get_connection()
            try:
                with conn.cursor() as cur:
                    cur.execute("SELECT 1")
                    result = cur.fetchone()
                    if result:
                        self.logger.info("✅ 데이터베이스 연결 성공")
                    else:
                        raise Exception("테스트 쿼리 실행 실패")
            finally:
                self.return_connection(conn)
                
            return True
            
        except Exception as e:
            self.logger.error(f"❌ 데이터베이스 연결 실패: {str(e)}")
            self.is_connected = False
            return False
    
    def get_connection(self):
        """연결 풀에서 연결 가져오기"""
        if not self.pool:
            raise Exception("데이터베이스 연결 풀이 초기화되지 않았습니다")
        return self.pool.getconn()
    
    def return_connection(self, conn):
        """연결 풀에 연결 반환"""
        if self.pool and conn:
            self.pool.putconn(conn)
    
    def cleanup(self):
        """연결 풀 정리"""
        if self.pool:
            self.pool.closeall()
            self.pool = None
            self.is_connected = False
            self.logger.info("🧹 데이터베이스 연결 풀 정리 완료")
    
    async def execute_query(self, query: str, params: tuple = None, fetch: bool = False):
        """쿼리 실행 (비동기)"""
        if not self.is_connected:
            raise Exception("데이터베이스에 연결되지 않았습니다")
            
        conn = None
        try:
            conn = self.get_connection()
            
            # 트랜잭션 내에서 실행
            with conn.cursor(cursor_factory=RealDictCursor) as cur:
                cur.execute(query, params)
                
                if fetch:
                    if 'SELECT' in query.upper():
                        result = cur.fetchall()
                    else:
                        result = cur.fetchone()
                    conn.commit()
                    return result
                else:
                    conn.commit()
                    return cur.rowcount
                    
        except Exception as e:
            if conn:
                conn.rollback()
            self.logger.error(f"❌ 쿼리 실행 실패: {str(e)}")
            raise
        finally:
            if conn:
                self.return_connection(conn)
    
    # 🔄 보고서 작업 관리 메서드들
    async def create_report_job(self, job_data: dict) -> dict:
        """보고서 작업 생성"""
        query = """
            INSERT INTO report_jobs (
                job_id, username, title, executive_summary, template, 
                output_format, status, progress, metadata
            ) VALUES (
                %s, %s, %s, %s, %s, %s, %s, %s, %s
            ) RETURNING *
        """
        
        params = (
            job_data['job_id'],
            job_data.get('username', ''),
            job_data.get('title', ''),
            job_data.get('executive_summary', ''),
            job_data.get('template', 'simple'),
            job_data.get('output_format', 'pdf'),
            job_data.get('status', 'queued'),
            job_data.get('progress', 0),
            json.dumps(job_data.get('metadata', {}))
        )
        
        results = await self.execute_query(query, params, fetch=True)
        if results:
            if isinstance(results, list) and len(results) > 0:
                return dict(results[0])  # 첫 번째 레코드를 dict로 변환
            elif not isinstance(results, list):
                return dict(results)  # 단일 레코드를 dict로 변환
        return None
    
    async def update_report_job(self, job_id: str, updates: dict) -> dict:
        """보고서 작업 상태 업데이트"""
        # 업데이트할 필드 구성
        set_clauses = []
        params = []
        
        for key, value in updates.items():
            if key == 'metadata':
                set_clauses.append(f"{key} = %s")
                params.append(json.dumps(value))
            elif key not in ['updated_at', 'completed_at']:  # 이미 있는 것은 제외
                set_clauses.append(f"{key} = %s")
                params.append(value)
        
        # updated_at 항상 추가 (중복 방지)
        if 'updated_at' not in updates:
            set_clauses.append("updated_at = CURRENT_TIMESTAMP")
        
        # 작업 완료 시 completed_at 추가 (중복 방지)
        if updates.get('status') == 'completed' and 'completed_at' not in updates:
            set_clauses.append("completed_at = CURRENT_TIMESTAMP")
        
        query = f"""
            UPDATE report_jobs 
            SET {', '.join(set_clauses)}
            WHERE job_id = %s
            RETURNING *
        """
        params.append(job_id)
        
        results = await self.execute_query(query, tuple(params), fetch=True)
        if results:
            if isinstance(results, list) and len(results) > 0:
                return dict(results[0])  # 첫 번째 레코드를 dict로 변환
            elif not isinstance(results, list):
                return dict(results)  # 단일 레코드를 dict로 변환
        return None
    
    async def get_report_job(self, job_id: str) -> dict:
        """보고서 작업 조회"""
        query = """
            SELECT * FROM report_jobs WHERE job_id = %s
        """
        
        results = await self.execute_query(query, (job_id,), fetch=True)
        if results:
            if isinstance(results, list) and len(results) > 0:
                return dict(results[0])  # 첫 번째 레코드를 dict로 변환
            elif not isinstance(results, list):
                return dict(results)  # 단일 레코드를 dict로 변환
        return None
    
    async def list_report_jobs(self, username: str = None, status: str = None, limit: int = 50) -> List[dict]:
        """보고서 작업 목록 조회"""
        query = """
            SELECT * FROM report_jobs WHERE 1=1
        """
        params = []
        
        if username:
            query += " AND username = %s"
            params.append(username)
        
        if status:
            query += " AND status = %s"
            params.append(status)
        
        query += " ORDER BY created_at DESC LIMIT %s"
        params.append(limit)
        
        results = await self.execute_query(query, tuple(params), fetch=True)
        return [dict(row) for row in results] if results else []
    
    async def delete_report_job(self, job_id: str) -> bool:
        """보고서 작업 삭제"""
        query = "DELETE FROM report_jobs WHERE job_id = %s"
        row_count = await self.execute_query(query, (job_id,))
        return row_count > 0
    
    async def get_job_stats(self) -> dict:
        """작업 통계 조회"""
        query = """
            SELECT 
                status,
                COUNT(*) as count,
                COUNT(DISTINCT username) as unique_users
            FROM report_jobs 
            WHERE username IS NOT NULL AND username != ''
            GROUP BY status
        """
        
        results = await self.execute_query(query, fetch=True)
        stats = {}
        
        if results:
            for row in results:
                stats[row['status']] = {
                    'count': row['count'],
                    'unique_users': row['unique_users']
                }
        
        return stats

# =============================================================================
# 데이터 모델들
# =============================================================================

@dataclass
class SectionConfigFallback:
    """섹션 설정 정보 (폴백용)"""
    title: str
    prompt: str
    requires_chart: bool = False
    requires_flow: bool = False
    requires_table: bool = False
    requires_rag: bool = False
    requires_web: bool = False
    requires_hide: bool = False
    requires_competitor_analysis: bool = False
    requires_diagram: bool = False
    section_type: str = "default"
    search_keywords: List[str] = None
    response_format: str = "text"

    @classmethod
    def from_template(cls, title: str, template_data: Union[str, Dict]) -> 'SectionConfigFallback':
        """템플릿 데이터로부터 섹션 설정 생성"""
        if isinstance(template_data, str):
            return cls(title=title, prompt=template_data)
        else:
            return cls(
                title=title,
                prompt=template_data.get("prompt", ""),
                requires_chart=template_data.get("requires_chart", False),
                requires_flow=template_data.get("requires_flow", False),
                requires_table=template_data.get("requires_table", False),
                requires_rag=template_data.get("requires_rag", False),
                requires_web=template_data.get("requires_web", False),
                requires_hide=template_data.get("requires_hide", False),
                requires_competitor_analysis=template_data.get("requires_competitor_analysis", False),
                requires_diagram=template_data.get("requires_diagram", False),
                section_type=template_data.get("section_type", "default"),
                search_keywords=template_data.get("search_keywords", []),
                response_format=template_data.get("response_format", "text")
            )

    def __post_init__(self):
        if self.search_keywords is None:
            self.search_keywords = []

# 🔍 Clarification Card 관련 데이터 모델들 추가
class ClarificationQuestion(BaseModel):
    """명확화 질문"""
    id: str = Field(..., description="질문 ID")
    type: str = Field(..., description="질문 유형 (text, select, multiselect, number, date)")
    question: str = Field(..., description="질문 내용")
    required: bool = Field(default=True, description="필수 여부")
    options: Optional[List[str]] = Field(default=None, description="선택지 (select/multiselect용)")
    placeholder: Optional[str] = Field(default=None, description="입력 힌트")
    validation: Optional[Dict] = Field(default=None, description="유효성 검사 규칙")

class ClarificationResponse(BaseModel):
    """명확화 응답"""
    question_id: str = Field(..., description="질문 ID")
    answer: Union[str, List[str], int, float] = Field(..., description="답변")

class ClarificationRequest(BaseModel):
    """명확화 요청"""
    prompt: str = Field(..., description="초기 사용자 입력")
    template: Optional[str] = Field(default="auto", description="템플릿 (auto면 자동 선택)")
    format: str = Field(default="pdf", description="출력 형식")
    username: Optional[str] = Field(default=None, description="사용자명")

class ClarificationResult(BaseModel):
    """명확화 결과"""
    success: bool
    analysis: Dict = Field(..., description="입력 분석 결과")
    questions: List[ClarificationQuestion] = Field(..., description="명확화 질문들")
    suggested_template: str = Field(..., description="추천 템플릿")
    estimated_sections: List[str] = Field(..., description="예상 섹션 목록")
    todo_list: List[str] = Field(..., description="생성될 작업 목록")
    confidence_score: float = Field(..., description="분석 신뢰도 (0-1)")

class EnhancedReportRequest(BaseModel):
    """향상된 보고서 생성 요청 (Clarification 결과 포함)"""
    original_prompt: str = Field(..., description="원본 프롬프트")
    clarification_responses: List[ClarificationResponse] = Field(..., description="명확화 응답들")
    confirmed_template: str = Field(..., description="확정된 템플릿")
    format: str = Field(default="pdf", description="출력 형식")
    email: Optional[str] = Field(default=None, description="완료 알림 이메일")
    username: Optional[str] = Field(default=None, description="사용자명")
    use_cache: Optional[bool] = Field(default=True, description="캐시 사용 여부")
    project_hash: Optional[str] = Field(default=None, description="프로젝트 해시")
    job_id: Optional[str] = Field(default=None, description="작업 ID")

class ReportGenerationRequest(BaseModel):
    """보고서 생성 요청"""
    prompt: str = Field(..., description="보고서 생성 프롬프트")
    format: str = Field(default="pdf", description="출력 형식 (pdf, hwp, docx, pptx)")
    template: Optional[str] = Field(default="simple", description="템플릿 이름")
    email: Optional[str] = Field(default=None, description="완료 알림 이메일")
    username: Optional[str] = Field(default=None, description="사용자명")
    use_cache: Optional[bool] = Field(default=True, description="캐시 사용 여부")
    project_hash: Optional[str] = Field(default=None, description="프로젝트 해시")
    job_id: Optional[str] = Field(default=None, description="작업 ID")
    
    # 섹션별 수정 기능을 위한 새 필드들
    modification_mode: Optional[str] = Field(default="full", description="수정 모드 ('full': 전체 재생성, 'partial': 섹션별 수정)")
    existing_document_path: Optional[str] = Field(default=None, description="수정할 기존 문서 파일 경로")
    existing_document_content: Optional[str] = Field(default=None, description="수정할 기존 문서 내용 (텍스트)")
    sections_to_modify: Optional[List[str]] = Field(default=None, description="수정할 섹션 목록 (예: ['섹션1/하위섹션1', '섹션2/하위섹션1'])")
    preserve_formatting: Optional[bool] = Field(default=True, description="기존 문서 서식 유지 여부")

    # 🔍 Clarification 결과 필드들 추가
    clarification_responses: Optional[List[ClarificationResponse]] = Field(default=None, description="명확화 응답들")
    enhanced_prompt: Optional[str] = Field(default=None, description="명확화 결과로 향상된 프롬프트")
    
    # 🤖 AI 모델 설정 필드들 추가
    provider: Optional[str] = Field(default=None, description="AI 프로바이더 (openai, anthropic, ollama 등)")
    model: Optional[str] = Field(default=None, description="AI 모델명")

class ReportGenerationResponse(BaseModel):
    """보고서 생성 응답"""
    success: bool
    job_id: str
    message: str
    status: str = "queued"
    estimated_time: Optional[int] = None

class JobStatusResponse(BaseModel):
    """작업 상태 응답"""
    success: bool
    job_id: str
    status: str  # queued, processing, completed, failed
    progress: Optional[int] = None
    file_path: Optional[str] = None
    error: Optional[str] = None
    created_at: Optional[str] = None
    updated_at: Optional[str] = None

# =============================================================================
# 주간 보고서 변환 요청/응답 모델
# =============================================================================

class CompletedTaskItem(BaseModel):
    """완료된 업무 항목"""
    taskName: str = Field(..., description="업무명")
    result: str = Field(..., description="결과")
    timeSpent: Optional[str] = Field(default=None, description="소요 시간")

class OngoingTaskItem(BaseModel):
    """진행 중인 업무 항목"""
    taskName: str = Field(..., description="업무명")
    progress: str = Field(..., description="진행 상황")
    nextPlan: str = Field(..., description="다음 계획")

class IssueItem(BaseModel):
    """이슈 항목"""
    category: str = Field(..., description="분류")
    content: str = Field(..., description="내용")
    solution: str = Field(..., description="해결 방안")
    status: str = Field(..., description="상태")

class NextPeriodPlanItem(BaseModel):
    """다음 기간 계획 항목"""
    taskName: str = Field(..., description="업무명")
    priority: str = Field(..., description="우선순위")
    targetResult: str = Field(..., description="목표 결과")

class WeeklyReportData(BaseModel):
    """주간 보고서 데이터"""
    summary: str = Field(..., description="업무 요약")
    completedTasks: List[CompletedTaskItem] = Field(default=[], description="완료된 업무 목록")
    ongoingTasks: List[OngoingTaskItem] = Field(default=[], description="진행 중인 업무 목록")
    issues: List[IssueItem] = Field(default=[], description="이슈 목록")
    nextPeriodPlans: List[NextPeriodPlanItem] = Field(default=[], description="다음 기간 계획")

class WeeklyReportMetadata(BaseModel):
    """주간 보고서 메타데이터"""
    reportType: str = Field(default="weekly", description="보고서 유형 (weekly/monthly)")
    startDate: str = Field(..., description="시작 날짜")
    endDate: str = Field(..., description="종료 날짜")
    department: Optional[str] = Field(default=None, description="부서명")
    author: Optional[str] = Field(default=None, description="작성자")
    documentCount: Optional[int] = Field(default=0, description="참조 문서 수")
    generatedAt: Optional[str] = Field(default=None, description="생성 시간")

class WeeklyReportConvertRequest(BaseModel):
    """주간 보고서 변환 요청"""
    report: WeeklyReportData = Field(..., description="보고서 데이터")
    metadata: WeeklyReportMetadata = Field(..., description="보고서 메타데이터")
    format: str = Field(default="pdf", description="출력 형식 (pdf, docx, hwpx)")

# =============================================================================
# 보고서 생성 서비스 (싱글톤 패턴)
# =============================================================================

# 🔍 Clarification 서비스 클래스 추가
class ClarificationService:
    """사용자 요청 분석 및 명확화 질문 생성 서비스"""
    
    def __init__(self, logger=None):
        self.logger = logger or logging.getLogger("clarification")
        self.ai_provider = None
        self._initialize_ai_provider()
    
    def _initialize_ai_provider(self):
        """AI 프로바이더 초기화 - utils.py의 AIProvider 사용"""
        try:
            # 1. 이미 로드된 utils 모듈 확인
            if 'utils' in sys.modules:
                utils_module = sys.modules['utils']
                if hasattr(utils_module, 'AIProvider'):
                    self.ai_provider = utils_module.AIProvider()
                    self.logger.info("✅ Clarification용 AI 프로바이더 초기화 완료 (기존 모듈)")
                    self._log_available_providers()
                    return
            
            # 2. utils.py 경로 확인 및 동적 로드
            current_dir = os.path.dirname(os.path.abspath(__file__))
            utils_path = os.path.join(current_dir, '..', '..', 'utils.py')
            utils_path = os.path.normpath(utils_path)
            
            self.logger.info(f"🔍 utils.py 경로 확인: {utils_path}")
            
            if os.path.exists(utils_path):
                # utils 모듈 동적 로드
                import importlib.util
                spec = importlib.util.spec_from_file_location("utils", utils_path)
                utils_module = importlib.util.module_from_spec(spec)
                spec.loader.exec_module(utils_module)
                
                # AIProvider 클래스 인스턴스 생성
                if hasattr(utils_module, 'AIProvider'):
                    self.ai_provider = utils_module.AIProvider()
                    self.logger.info("✅ Clarification용 AI 프로바이더 초기화 완료 (동적 로드)")
                    self._log_available_providers()
                    return
                else:
                    raise ImportError("AIProvider 클래스를 찾을 수 없습니다")
            
            # 3. 직접 import 시도
            try:
                from utils import AIProvider
                self.ai_provider = AIProvider()
                self.logger.info("✅ Clarification용 AI 프로바이더 초기화 완료 (직접 import)")
                self._log_available_providers()
                return
            except ImportError as import_error:
                self.logger.warning(f"직접 import 실패: {import_error}")
                
            raise FileNotFoundError(f"utils.py 파일을 찾을 수 없습니다: {utils_path}")
                
        except Exception as e:
            self.logger.error(f"❌ utils.AIProvider 초기화 실패: {str(e)}")
            self.logger.info("🔄 OpenAI 직접 초기화로 폴백 시도...")
            
            # OpenAI 직접 초기화 폴백
            try:
                import openai
                from openai import OpenAI
                
                # 환경 변수에서 API 키 확인
                api_key = os.getenv('OPENAI_API_KEY')
                if api_key:
                    self.ai_provider = OpenAI(api_key=api_key)
                    self.logger.info("✅ OpenAI 직접 초기화 완료")
                else:
                    self.logger.error("❌ OPENAI_API_KEY 환경 변수가 설정되지 않았습니다")
                    self.ai_provider = None
            except ImportError:
                self.logger.error("❌ OpenAI 라이브러리도 사용할 수 없습니다")
                self.ai_provider = None
    
    def _log_available_providers(self):
        """사용 가능한 AI 프로바이더 로그"""
        try:
            if not self.ai_provider:
                return
                
            available_providers = []
            
            # utils.AIProvider의 경우
            if hasattr(self.ai_provider, 'openai') and self.ai_provider.openai:
                available_providers.append("OpenAI")
            if hasattr(self.ai_provider, 'anthropic') and self.ai_provider.anthropic:
                available_providers.append("Anthropic")
            if hasattr(self.ai_provider, 'ollama') and self.ai_provider.ollama:
                available_providers.append("Ollama")
            if hasattr(self.ai_provider, 'google') and self.ai_provider.google:
                available_providers.append("Google")
            
            if available_providers:
                self.logger.info(f"🔗 사용 가능한 AI 프로바이더: {', '.join(available_providers)}")
            else:
                self.logger.warning("⚠️ 사용 가능한 AI 프로바이더가 없습니다")
        except Exception as e:
            self.logger.debug(f"프로바이더 정보 확인 중 오류: {e}")
    
    async def analyze_user_request(self, request: ClarificationRequest) -> ClarificationResult:
        """사용자 요청을 분석하여 명확화 질문들을 생성"""
        try:
            self.logger.info(f"🔍 사용자 요청 분석 시작: {request.prompt[:100]}...")
            
            # 1. 입력 분석을 위한 AI 프롬프트 구성
            analysis_prompt = f"""
다음 사용자 요청을 분석하여 문서 생성에 필요한 정보가 충분한지 평가하고, 부족한 정보를 식별해주세요.

사용자 요청:
{request.prompt}

요청된 출력 형식: {request.format}
지정된 템플릿: {request.template}

다음 JSON 형식으로 분석 결과를 제공해주세요:

{{
    "document_type": "식별된 문서 유형 (예: 사업계획서, 제안서, 보고서 등)",
    "purpose": "문서의 목적",
    "target_audience": "대상 독자",
    "scope": "문서 범위 (상세도 수준)",
    "missing_info": [
        {{
            "category": "부족한 정보 카테고리",
            "description": "구체적으로 부족한 내용",
            "importance": "high/medium/low",
            "question_type": "text/select/multiselect/number/date"
        }}
    ],
    "suggested_template": "추천 템플릿명",
    "confidence_score": 0.85,
    "estimated_sections": ["예상 섹션1", "예상 섹션2", "..."],
    "complexity_level": "simple/medium/complex"
}}

분석 시 고려사항:
1. 사업계획서라면: 사업 개요, 시장 분석, 재무 계획, 팀 구성 등이 필요
2. 제안서라면: 문제 정의, 해결 방안, 예산, 일정 등이 필요
3. 보고서라면: 목적, 범위, 데이터 출처, 결론 등이 필요
4. 부족한 정보는 실제로 문서 생성에 중요한 것들만 포함
5. 질문은 사용자가 쉽게 답할 수 있는 형태로 구성
"""

            # 2. AI를 통한 분석 수행
            if not self.ai_provider:
                raise Exception("AI 프로바이더가 초기화되지 않았습니다")
            
            analysis_result = await self._call_ai_api(
                analysis_prompt, 
                require_json=True,
                system_message="당신은 문서 분석 전문가입니다. 사용자의 요청을 정확히 분석하여 부족한 정보를 식별하고 적절한 질문을 제안해주세요."
            )
            
            self.logger.info(f"✅ AI 분석 완료: {analysis_result}")
            
            # 3. 분석 결과를 바탕으로 명확화 질문들 생성
            questions = await self._generate_clarification_questions(analysis_result, request.prompt)
            
            # 4. Todo 리스트 생성
            todo_list = self._generate_todo_list(analysis_result)
            
            # 5. 결과 구성
            result = ClarificationResult(
                success=True,
                analysis=analysis_result,
                questions=questions,
                suggested_template=analysis_result.get('suggested_template', 'simple'),
                estimated_sections=analysis_result.get('estimated_sections', []),
                todo_list=todo_list,
                confidence_score=analysis_result.get('confidence_score', 0.7)
            )
            
            self.logger.info(f"🎯 명확화 분석 완료: {len(questions)}개 질문 생성")
            return result
            
        except Exception as e:
            self.logger.error(f"❌ 사용자 요청 분석 실패: {str(e)}")
            # 폴백: 기본 질문들 반환
            return await self._create_fallback_clarification(request)
    
    async def _generate_clarification_questions(self, analysis: Dict, original_prompt: str) -> List[ClarificationQuestion]:
        """분석 결과를 바탕으로 구체적인 명확화 질문들 생성"""
        questions = []
        
        missing_info = analysis.get('missing_info', [])
        document_type = analysis.get('document_type', '문서')
        
        for idx, info in enumerate(missing_info):
            question_id = f"q_{idx + 1}"
            category = info.get('category', '')
            description = info.get('description', '')
            importance = info.get('importance', 'medium')
            question_type = info.get('question_type', 'text')
            
            # 카테고리별 구체적인 질문 생성
            question_text = await self._generate_specific_question(
                category, description, document_type, original_prompt
            )
            
            # 질문 타입별 옵션 설정
            options = None
            placeholder = None
            validation = None
            
            if question_type == 'select':
                options = await self._generate_question_options(category, description)
            elif question_type == 'multiselect':
                options = await self._generate_question_options(category, description, multiple=True)
            elif question_type == 'number':
                validation = {"min": 0, "type": "number"}
                placeholder = "숫자를 입력하세요"
            elif question_type == 'date':
                validation = {"type": "date"}
                placeholder = "YYYY-MM-DD"
            else:  # text
                placeholder = f"{description}에 대해 자세히 설명해주세요"
            
            question = ClarificationQuestion(
                id=question_id,
                type=question_type,
                question=question_text,
                required=importance == 'high',
                options=options,
                placeholder=placeholder,
                validation=validation
            )
            
            questions.append(question)
        
        # 기본 질문들 추가 (항상 포함)
        if not any(q.id == 'target_audience' for q in questions):
            questions.append(ClarificationQuestion(
                id="target_audience",
                type="select",
                question=f"이 {document_type}의 주요 대상 독자는 누구인가요?",
                required=False,
                options=["경영진", "투자자", "고객", "내부 팀", "일반 대중", "기타"],
                placeholder="대상 독자를 선택하세요"
            ))
        
        if not any(q.id == 'detail_level' for q in questions):
            questions.append(ClarificationQuestion(
                id="detail_level",
                type="select",
                question="문서의 상세도 수준을 어느 정도로 하시겠습니까?",
                required=False,
                options=["간략하게 (5-10페이지)", "보통 (10-20페이지)", "상세하게 (20페이지 이상)"],
                placeholder="상세도를 선택하세요"
            ))
        
        return questions
    
    async def _generate_specific_question(self, category: str, description: str, document_type: str, original_prompt: str) -> str:
        """카테고리와 설명을 바탕으로 구체적인 질문 생성"""
        question_prompt = f"""
다음 정보를 바탕으로 사용자에게 물어볼 구체적이고 명확한 질문을 한국어로 생성해주세요.

문서 유형: {document_type}
부족한 정보 카테고리: {category}
구체적 설명: {description}
원본 요청: {original_prompt}

질문 생성 가이드라인:
1. 사용자가 쉽게 이해하고 답할 수 있어야 함
2. 구체적이고 명확해야 함
3. 문서 생성에 실제로 도움이 되는 정보를 얻을 수 있어야 함
4. 한 문장으로 간결하게 작성
5. 존댓말 사용

예시:
- "사업의 주요 목표 고객층은 어떻게 되나요?"
- "예상 프로젝트 기간은 얼마나 되나요?"
- "주요 경쟁사나 경쟁 제품이 있다면 알려주세요"

질문만 반환해주세요:
"""
        
        try:
            if self.ai_provider:
                question = await self._call_ai_api(question_prompt, require_json=False)
                return question.strip().strip('"').strip("'")
            else:
                # 폴백: 기본 질문 템플릿
                return f"{category}에 대해 더 자세한 정보를 알려주세요"
        except Exception as e:
            self.logger.warning(f"질문 생성 실패: {str(e)}")
            return f"{category}에 대해 더 자세한 정보를 알려주세요"
    
    async def _generate_question_options(self, category: str, description: str, multiple: bool = False) -> List[str]:
        """선택형 질문의 옵션들 생성"""
        # 카테고리별 기본 옵션들
        default_options = {
            "예산": ["100만원 미만", "100만원-500만원", "500만원-1천만원", "1천만원-5천만원", "5천만원 이상"],
            "기간": ["1개월 미만", "1-3개월", "3-6개월", "6개월-1년", "1년 이상"],
            "규모": ["개인/소규모", "중소기업", "대기업", "글로벌 기업"],
            "산업": ["IT/소프트웨어", "제조업", "서비스업", "금융", "교육", "의료", "기타"],
            "지역": ["서울", "수도권", "전국", "해외", "글로벌"],
            "단계": ["아이디어 단계", "계획 단계", "개발 단계", "출시 단계", "성장 단계"]
        }
        
        # 카테고리 키워드 매칭
        for key, options in default_options.items():
            if key in category or key in description:
                return options
        
        # 기본 옵션
        return ["예", "아니오", "잘 모르겠음"]
    
    def _generate_todo_list(self, analysis: Dict) -> List[str]:
        """분석 결과를 바탕으로 Todo 리스트 생성"""
        todo_list = []
        
        document_type = analysis.get('document_type', '문서')
        estimated_sections = analysis.get('estimated_sections', [])
        complexity = analysis.get('complexity_level', 'medium')
        
        # 기본 작업들
        todo_list.append(f"📋 {document_type} 구조 설계")
        
        # 섹션별 작업들
        for section in estimated_sections:
            todo_list.append(f"✍️ {section} 작성")
        
        # 복잡도에 따른 추가 작업들
        if complexity in ['medium', 'complex']:
            todo_list.append("📊 차트 및 그래프 생성")
            todo_list.append("🔍 관련 정보 조사 및 분석")
        
        if complexity == 'complex':
            todo_list.append("📈 상세 데이터 분석")
            todo_list.append("🎨 고급 시각화 요소 추가")
            todo_list.append("📝 참고문헌 및 부록 작성")
        
        # 마무리 작업들
        todo_list.append("🎯 최종 검토 및 품질 확인")
        todo_list.append(f"📄 {analysis.get('format', 'PDF').upper()} 파일 생성")
        
        return todo_list
    
    async def _create_fallback_clarification(self, request: ClarificationRequest) -> ClarificationResult:
        """AI 분석 실패 시 기본 명확화 결과 생성"""
        self.logger.info("🔄 폴백 모드: 기본 명확화 질문 생성")
        
        # 기본 질문들
        questions = [
            ClarificationQuestion(
                id="purpose",
                type="text",
                question="이 문서의 주요 목적이나 목표를 알려주세요",
                required=True,
                placeholder="예: 투자 유치를 위한 사업계획서 작성"
            ),
            ClarificationQuestion(
                id="target_audience",
                type="select",
                question="주요 대상 독자는 누구인가요?",
                required=False,
                options=["경영진", "투자자", "고객", "내부 팀", "일반 대중", "기타"]
            ),
            ClarificationQuestion(
                id="timeline",
                type="select",
                question="문서 작성 완료까지 희망하는 시간은?",
                required=False,
                options=["즉시", "1시간 이내", "하루 이내", "일주일 이내"]
            ),
            ClarificationQuestion(
                id="detail_level",
                type="select",
                question="문서의 상세도 수준을 선택해주세요",
                required=False,
                options=["간략하게 (5-10페이지)", "보통 (10-20페이지)", "상세하게 (20페이지 이상)"]
            )
        ]
        
        return ClarificationResult(
            success=True,
            analysis={
                "document_type": "문서",
                "purpose": "명시되지 않음",
                "target_audience": "명시되지 않음",
                "scope": "보통",
                "complexity_level": "medium"
            },
            questions=questions,
            suggested_template="simple",
            estimated_sections=["개요", "주요 내용", "결론"],
            todo_list=[
                "📋 문서 구조 설계",
                "✍️ 주요 내용 작성",
                "📊 필요시 차트 생성",
                "🎯 최종 검토",
                f"📄 {request.format.upper()} 파일 생성"
            ],
            confidence_score=0.5
        )
    
    async def _call_ai_api(self, prompt: str, require_json: bool = False, system_message: str = "") -> Union[str, Dict]:
        """AI API 호출 - utils.py의 AIProvider 사용"""
        try:
            if not self.ai_provider:
                raise Exception("AI 프로바이더가 사용할 수 없습니다")
            
            # JSON 응답이 필요한 경우 프롬프트 수정
            if require_json:
                prompt = f"""Please provide your response in JSON format.

{prompt}

중요: 응답은 반드시 유효한 JSON 형식이어야 합니다. 다른 텍스트 없이 JSON만 반환하세요."""

            # 메시지 구성
            messages = []
            if system_message:
                messages.append({"role": "system", "content": system_message})
            else:
                messages.append({"role": "system", "content": "당신은 도움이 되는 어시스턴트입니다."})
            
            if require_json and messages[0]["role"] == "system":
                messages[0]["content"] += "\n\n중요: 응답은 반드시 유효한 JSON 형식이어야 합니다."
            
            messages.append({"role": "user", "content": prompt})
            
            # utils.AIProvider 사용
            if hasattr(self.ai_provider, '_call_provider'):
                # utils.AIProvider 인스턴스인 경우
                max_tokens = 4000 if require_json else None
                format_type = "json" if require_json else None
                
                result = self.ai_provider._call_provider(
                    messages=messages,
                    max_tokens=max_tokens,
                    format=format_type
                )
                
                # JSON 응답 파싱
                if require_json and isinstance(result, str):
                    try:
                        return json.loads(result)
                    except json.JSONDecodeError as e:
                        self.logger.error(f"JSON 파싱 실패: {str(e)}")
                        self.logger.error(f"응답 내용: {result}")
                        # JSON 파싱 실패 시 텍스트로 반환
                        return result
                
                return result
                
            # OpenAI 직접 호출 폴백 (utils.AIProvider가 아닌 경우)
            elif hasattr(self.ai_provider, 'chat'):
                # OpenAI 클라이언트인 경우
                params = {
                    "model": "gpt-4o-mini",
                    "messages": messages,
                    "temperature": 0.3 if require_json else 0.7
                }
                
                if require_json:
                    try:
                        params["response_format"] = {"type": "json_object"}
                    except Exception:
                        pass
                
                response = await asyncio.get_event_loop().run_in_executor(
                    None,
                    lambda: self.ai_provider.chat.completions.create(**params)
                )
                result = response.choices[0].message.content
                
                # JSON 응답 파싱
                if require_json:
                    try:
                        return json.loads(result)
                    except json.JSONDecodeError as e:
                        self.logger.error(f"JSON 파싱 실패: {str(e)}")
                        self.logger.error(f"응답 내용: {result}")
                        return result
                
                return result
            else:
                raise Exception("지원되지 않는 AI 프로바이더 타입입니다")
                
        except Exception as e:
            self.logger.error(f"AI API 호출 실패: {str(e)}")
            # 에러 발생 시 기본 응답 반환
            if require_json:
                return {
                    "error": "AI API 호출 실패",
                    "message": str(e),
                    "fallback": True
                }
            else:
                return f"AI API 호출 실패: {str(e)}"

class ReportGenerationService:
    """통합 보고서 생성 서비스 (싱글톤) - RAG 서비스와 동일한 패턴"""
    _instance = None
    _lock = Lock()
    _initialized = False
    
    def __new__(cls):
        if cls._instance is None:
            with cls._lock:
                if cls._instance is None:
                    cls._instance = super(ReportGenerationService, cls).__new__(cls)
        return cls._instance
    
    def __init__(self):
        if not self._initialized:
            self.logger = logger
            self.thread_pool = None
            self.openai_available = False
            self.rag_available = False
            self.web_search_available = False
            self.bizplan_generator_available = False
            self._initialized = False
    
    def initialize(self):
        """서비스 초기화"""
        if self._initialized:
            return True
            
        try:
            self.logger.info("보고서 생성 서비스 초기화 시작...")
            
            # 1. 스레드 풀 초기화
            self._initialize_thread_pool()
            
            # 2. 외부 서비스 연결 확인
            self._check_external_services()
            
            # 3. 비즈니스 플랜 생성기 확인
            self._check_bizplan_generator()
            
            self._initialized = True
            self.logger.info("보고서 생성 서비스 초기화 완료")
            
            return True
            
        except Exception as e:
            self.logger.error(f"서비스 초기화 실패: {str(e)}")
            return False
    
    def _initialize_thread_pool(self):
        """스레드 풀 초기화"""
        try:
            import multiprocessing
            max_workers = min(4, multiprocessing.cpu_count())
            self.thread_pool = ThreadPoolExecutor(
                max_workers=max_workers,
                thread_name_prefix="ReportGen"
            )
            self.logger.info(f"스레드 풀 초기화 완료 (워커 수: {max_workers})")
        except Exception as e:
            self.logger.error(f"스레드 풀 초기화 실패: {str(e)}")
            raise
    
    def _check_external_services(self):
        """외부 서비스 및 AIRUN 프로바이더 설정 확인"""
        # RAG 서비스 확인
        try:
            response = requests.get(f"{RAG_SERVICE_URL}/health", timeout=5)
            if response.status_code == 200:
                self.rag_available = True
                self.logger.info("RAG 서비스 연결 확인")
            else:
                self.logger.warning("RAG 서비스 연결 실패")
        except Exception as e:
            self.logger.warning(f"RAG 서비스 연결 확인 실패: {str(e)}")

        # AIRUN 프로바이더 설정 확인
        try:
            # utils 모듈에서 AIProvider 클래스 가져오기 (동적 로드된 모듈 사용)
            if 'utils' in sys.modules:
                utils_module = sys.modules['utils']
                if hasattr(utils_module, 'AIProvider'):
                    AIProvider = getattr(utils_module, 'AIProvider')
                    self.ai_provider = AIProvider()
                    self.openai_available = True
                    self.logger.info("AIRUN AI 프로바이더 설정 확인")
                else:
                    raise AttributeError("AIProvider 클래스를 찾을 수 없습니다")
            else:
                # 폴백: 일반 import 시도
                from utils import AIProvider
                self.ai_provider = AIProvider()
                self.openai_available = True
                self.logger.info("AIRUN AI 프로바이더 설정 확인 (폴백 방식)")
        except Exception as e:
            self.logger.warning(f"AIRUN AI 프로바이더 설정 확인 실패: {str(e)}")
            self.openai_available = False

        # 웹 검색 서비스 확인
        try:
            response = requests.get(f"{WEB_SEARCH_URL}/health", timeout=5)
            if response.status_code == 200:
                self.web_search_available = True
                self.logger.info("웹 검색 서비스 연결 확인")
            else:
                self.logger.warning("웹 검색 서비스 연결 실패")
        except Exception as e:
            self.logger.warning(f"웹 검색 서비스 연결 확인 실패: {str(e)}")

    def _check_bizplan_generator(self):
        """비즈니스 플랜 생성기 확인 및 사전 초기화"""
        try:
            # generate_bizplan 모듈 동적 로딩
            global generate_bizplan_module
            if 'generate_bizplan' in sys.modules:
                generate_bizplan_module = sys.modules['generate_bizplan']
            else:
                # 현재 디렉토리에서 generate_bizplan 모듈 로드
                current_dir = os.path.dirname(os.path.abspath(__file__))
                generate_bizplan_path = os.path.join(current_dir, 'generate_bizplan.py')
                
                if os.path.exists(generate_bizplan_path):
                    import importlib.util
                    spec = importlib.util.spec_from_file_location("generate_bizplan", generate_bizplan_path)
                    generate_bizplan_module = importlib.util.module_from_spec(spec)
                    spec.loader.exec_module(generate_bizplan_module)
                    sys.modules['generate_bizplan'] = generate_bizplan_module
                else:
                    # 일반 import 시도
                    import generate_bizplan as generate_bizplan_module
                    
            # 필요한 함수가 있는지 확인
            if hasattr(generate_bizplan_module, 'generate_business_plan'):
                self.bizplan_generator_available = True
                
                # 🚀 사전 초기화: 첫 번째 요청 시 지연 방지
                self.logger.info("🔄 보고서 생성기 사전 초기화 중...")
                try:
                    # BizPlanContentGenerator 인스턴스 미리 생성하여 초기화 완료
                    if BizPlanContentGenerator:
                        self._warmup_generator = BizPlanContentGenerator(output_format='pdf', logger=self.logger)
                        
                        # 🚀 추가 최적화: 기본 템플릿 미리 로드
                        try:
                            # 기본 템플릿 로드 (simple 템플릿)
                            self._warmup_generator.load_prompt_templates(template='simple')
                            self.logger.info("✅ 기본 템플릿 사전 로드 완료")
                            
                            # 🚀 추가 최적화: 기본 캐시 디렉토리 미리 생성
                            base_cache_dir = Path.home() / ".airun" / "cache" / "report"
                            base_cache_dir.mkdir(parents=True, exist_ok=True)
                            self.logger.debug(f"기본 캐시 디렉토리 생성: {base_cache_dir}")
                            
                        except Exception as template_error:
                            self.logger.warning(f"⚠️ 기본 템플릿 로드 실패: {str(template_error)}")
                        
                        self.logger.info("✅ 보고서 생성기 사전 초기화 완료")
                    else:
                        self.logger.warning("⚠️ BizPlanContentGenerator 클래스를 찾을 수 없음")
                except Exception as warmup_error:
                    self.logger.warning(f"⚠️ 보고서 생성기 사전 초기화 실패: {str(warmup_error)}")
                    # 사전 초기화 실패해도 서비스는 계속 사용 가능
                
                return
            else:
                self.bizplan_generator_available = False
                self.logger.warning("❌ generate_business_plan 함수가 없어 비즈니스 플랜 생성기 사용 불가")
                
        except Exception as e:
            self.bizplan_generator_available = False
            self.logger.warning(f"❌ 비즈니스 플랜 생성기 확인 실패: {str(e)}")

    def cleanup(self):
        """서비스 정리"""
        try:
            if self.thread_pool:
                self.thread_pool.shutdown(wait=True)
                self.logger.info("스레드 풀 정리 완료")
        except Exception as e:
            self.logger.error(f"서비스 정리 중 오류: {str(e)}")

    async def generate_report(self, request: ReportGenerationRequest, progress_callback=None) -> Dict:
        """보고서 생성 - generate_bizplan 모듈 사용 (사전 초기화된 인스턴스 활용)"""
        try:
            # 사용자 정보 로깅
            user_info = f"(사용자: {request.username})" if request.username else "(사용자: 미지정)"
            
            # 진행률 콜백
            if progress_callback:
                await progress_callback("보고서 생성 시작", 0)

            # generate_bizplan 모듈이 사용 가능한지 확인
            if not self.bizplan_generator_available or not generate_bizplan_module:
                raise Exception("generate_bizplan 모듈을 사용할 수 없습니다. 서비스 초기화를 확인해주세요.")

            self.logger.info(f"generate_bizplan 모듈 사용 {user_info}")
            
            if progress_callback:
                await progress_callback("보고서 생성기 실행 중...", 20)

            # 🚀 성능 개선: 사용자별 출력 디렉토리 미리 생성 (username 기반)
            if request.username:
                user_output_dir = project_root / "output" / request.username
                user_output_dir.mkdir(parents=True, exist_ok=True)
                self.logger.debug(f"사용자별 출력 디렉토리 확인: {user_output_dir}")
                
                # 🚀 성능 개선: 사용자별 캐시 디렉토리 미리 생성
                user_cache_dir = Path.home() / ".airun" / "cache" / "report" / request.username
                user_cache_dir.mkdir(parents=True, exist_ok=True)
                self.logger.debug(f"사용자별 캐시 디렉토리 확인: {user_cache_dir}")
            
            # 요청 데이터 구성 (generate_bizplan.py 형식에 맞춤)
            input_data = {
                'executive_summary': request.prompt,
                'format': request.format,
                'template': request.template,
                'email': request.email,
                'user_id': request.username,  # username을 user_id로 전달 (파일시스템 호환성)
                'username': request.username,
                'use_cache': request.use_cache,
                'project_hash': request.project_hash,
                'job_id': request.job_id,
                
                # AI 모델 설정 추가
                'provider': request.provider,
                'model': request.model,
                
                # 부분 수정 모드 관련 파라미터 추가
                'modification_mode': request.modification_mode,
                'existing_document_path': request.existing_document_path,
                'existing_document_content': request.existing_document_content,
                'sections_to_modify': request.sections_to_modify,
                'preserve_formatting': request.preserve_formatting
            }

            # 🚀 성능 개선: 사전 초기화된 인스턴스가 있으면 재사용하여 초기화 시간 단축
            if hasattr(self, '_warmup_generator') and self._warmup_generator:
                self.logger.info(f"✅ 사전 초기화된 보고서 생성기 사용 - 빠른 시작 {user_info}")
                # 사용자 ID 설정
                if request.username:
                    self._warmup_generator.set_user_id(request.username)
                if progress_callback:
                    await progress_callback("사전 초기화된 생성기 사용 중...", 30)
            else:
                self.logger.info(f"⚠️ 사전 초기화된 보고서 생성기 없음 - 일반 초기화 진행 {user_info}")
                if progress_callback:
                    await progress_callback("보고서 생성기 초기화 중...", 25)

            # generate_bizplan 모듈 사용 (비동기 함수를 직접 호출)
            result = await generate_bizplan_module.generate_business_plan(
                json.dumps(input_data),
                progress_callback=progress_callback  # progress_callback 전달
            )
            
            if progress_callback:
                await progress_callback("보고서 생성 완료", 100)
            
            self.logger.info(f"보고서 생성 완료 {user_info}")
            return result
            
        except Exception as e:
            self.logger.error(f"보고서 생성 실패 {user_info}: {str(e)}")
            raise

# =============================================================================
# 전역 변수 및 서비스 관리
# =============================================================================

# 전역 변수 (DB 기반으로 변경)
server_stats = {
    "start_time": datetime.now(),
    "total_requests": 0,
    "active_jobs": 0,
    "completed_jobs": 0,
    "failed_jobs": 0
}

# 보고서 생성 서비스 (서버 시작 시 한 번만 초기화)
report_service = None
db_manager = None
clarification_service = None  # 🔍 Clarification 서비스 추가

# =============================================================================
# 서비스 초기화 및 관리
# =============================================================================

async def initialize_services():
    """서비스 초기화 - RAG 서비스와 동일한 패턴"""
    global report_service, db_manager, clarification_service
    
    try:
        logger.info("🚀 AIRUN Report Server 초기화 시작...")
        
        # 데이터베이스 연결 초기화
        db_manager = DatabaseManager()
        db_success = db_manager.initialize()
        
        if not db_success:
            logger.error("❌ 데이터베이스 연결 실패")
            return False
        
        # 기본 디렉토리 생성
        output_dir = project_root / "output"
        output_dir.mkdir(parents=True, exist_ok=True)
        
        log_dir = project_root / "logs"
        log_dir.mkdir(parents=True, exist_ok=True)
        
        cache_dir = project_root / "cache"
        cache_dir.mkdir(parents=True, exist_ok=True)
        
        # 보고서 생성 서비스 초기화 (싱글톤 패턴)
        if report_service is None:
            report_service = ReportGenerationService()
        
        success = report_service.initialize()
        
        if not success:
            logger.error("❌ 보고서 생성 서비스 초기화 실패")
            return False
        
        # 🔍 Clarification 서비스 초기화 추가
        if clarification_service is None:
            clarification_service = ClarificationService(logger=logger)
            logger.info("✅ Clarification 서비스 초기화 완료")
        
        # 서버 통계 초기화 (DB에서 기존 통계 불러오기)
        try:
            db_stats = await db_manager.get_job_stats()
            server_stats["completed_jobs"] = sum(stats.get('count', 0) for status, stats in db_stats.items() if status == 'completed')
            server_stats["failed_jobs"] = sum(stats.get('count', 0) for status, stats in db_stats.items() if status == 'failed')
            server_stats["active_jobs"] = sum(stats.get('count', 0) for status, stats in db_stats.items() if status in ['queued', 'processing'])
            logger.info(f"📊 DB에서 기존 통계 로드: 완료 {server_stats['completed_jobs']}, 실패 {server_stats['failed_jobs']}, 활성 {server_stats['active_jobs']}")
        except Exception as e:
            logger.warning(f"⚠️ DB 통계 로드 실패: {str(e)}")
        
        server_stats["start_time"] = datetime.now()
        
        logger.info(f"✅ 초기화 완료!")
        logger.info(f"   - 데이터베이스: 연결됨")
        logger.info(f"   - 출력 디렉토리: {output_dir}")
        logger.info(f"   - 로그 디렉토리: {log_dir}")
        logger.info(f"   - 캐시 디렉토리: {cache_dir}")
        logger.info(f"   - 보고서 생성 서비스: 초기화됨")
        logger.info(f"   - Clarification 서비스: 초기화됨")  # 🔍 추가
        
        # 기능 상태 로깅
        features = {
            "database": db_manager.is_connected,
            "rag_service": getattr(report_service, 'rag_available', False),
            "web_search": getattr(report_service, 'web_search_available', False),
            "bizplan_generator": getattr(report_service, 'bizplan_generator_available', False),
            "thread_pool": getattr(report_service, 'thread_pool', None) is not None,
            "warmup_generator": hasattr(report_service, '_warmup_generator') and report_service._warmup_generator is not None,
            "clarification": clarification_service is not None  # 🔍 추가
        }
        
        logger.info("🔧 사용 가능한 기능:")
        for feature, available in features.items():
            status = "✅" if available else "❌"
            feature_name = {
                "database": "데이터베이스",
                "rag_service": "RAG 서비스",
                "web_search": "웹 검색",
                "bizplan_generator": "보고서 생성기",
                "thread_pool": "스레드 풀",
                "warmup_generator": "사전 초기화",
                "clarification": "명확화 서비스"  # 🔍 추가
            }.get(feature, feature)
            logger.info(f"   - {feature_name}: {status}")
        
        # 🚀 성능 개선 효과 예상 시간 로깅
        if features["warmup_generator"]:
            logger.info("⚡ 부스터 모드 활성화")
        else:
            logger.info("⚠️ 부스터 모드 비활성화")
        
        return True
        
    except Exception as e:
        logger.error(f"❌ 초기화 실패: {str(e)}")
        logger.error(traceback.format_exc())
        return False

async def cleanup_services():
    """서비스 정리 - RAG 서비스와 동일한 패턴"""
    global report_service, db_manager
    
    logger.info("🧹 AIRUN Report Server 정리 중...")
    
    try:
        # 활성 작업들 정리 (DB에서 가져오기)
        if db_manager and db_manager.is_connected:
            try:
                active_jobs = await db_manager.list_report_jobs(status='queued') + await db_manager.list_report_jobs(status='processing')
                for job in active_jobs:
                    await db_manager.update_report_job(job['job_id'], {"status": "cancelled"})
                    logger.info(f"⏸️ 작업 취소 (DB): {job['job_id']}")
            except Exception as e:
                logger.warning(f"⚠️ 활성 작업 정리 실패: {str(e)}")
        
        # 데이터베이스 연결 정리
        if db_manager:
            db_manager.cleanup()

        # 보고서 생성 서비스 정리
        if report_service:
            report_service.cleanup()
        
        logger.info("✅ 서비스 정리 완료")
        
    except Exception as e:
        logger.error(f"서비스 정리 중 오류: {e}")

@asynccontextmanager
async def lifespan(app: FastAPI):
    """FastAPI 라이프사이클 관리 - RAG 서비스와 동일한 패턴"""
    # 시작
    try:
        success = await initialize_services()
        if not success:
            logger.warning("⚠️ 서비스 초기화 실패 - 제한된 기능으로 서버를 계속 실행합니다.")
        
        yield
        
    except Exception as e:
        logger.error(f"서버 실행 중 오류: {e}")
        raise
    finally:
        # 종료
        await cleanup_services()

# FastAPI 앱 생성
app = FastAPI(
    title="AIRUN Report Generation Server",
    description="완전한 보고서 생성 서비스 (report_generator.py 통합)",
    version="2.0.0",
    lifespan=lifespan
)

# CORS 미들웨어 추가
app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

# =============================================================================
# 유틸리티 함수들
# =============================================================================

def generate_job_id() -> str:
    """작업 ID 생성"""
    return str(uuid.uuid4())[:8]

async def update_job_status(job_id: str, status: str, progress: int = None, 
                          file_path: str = None, error: str = None):
    """작업 상태 업데이트"""
    global db_manager
    if not db_manager:
        logger.warning("DB 매니저가 초기화되지 않아 작업 상태를 업데이트할 수 없습니다.")
        return

    try:
        updates = {}
        if status:
            updates['status'] = status
        if progress is not None:
            updates['progress'] = progress
        if file_path:
            updates['output_path'] = file_path
        if error:
            updates['error_message'] = error
        if status == 'completed' or status == 'failed' or status == 'cancelled':
            updates['completed_at'] = datetime.now().isoformat()
        # updated_at은 update_report_job 메서드에서 자동으로 추가됨

        if updates:
            await db_manager.update_report_job(job_id, updates)
            logger.info(f"✅ 작업 상태 업데이트: {job_id}, 상태: {status}, 진행률: {progress}")
            
            # 🔄 SSE 스트리밍으로 상태 업데이트 전송
            await send_streaming_update(job_id, {
                "type": "status",
                "status": status,
                "progress": progress,
                "file_path": file_path,
                "error": error,
                "timestamp": datetime.now().isoformat()
            })
            
            # 📊 서버 통계 업데이트
            global server_stats
            if status == "completed":
                server_stats["completed_jobs"] += 1
                server_stats["active_jobs"] = max(0, server_stats["active_jobs"] - 1)
            elif status == "failed":
                server_stats["failed_jobs"] += 1
                server_stats["active_jobs"] = max(0, server_stats["active_jobs"] - 1)
            elif status == "cancelled":
                server_stats["active_jobs"] = max(0, server_stats["active_jobs"] - 1)
        else:
            logger.info(f"❌ 작업 상태 업데이트 시 변경된 필드가 없습니다: {job_id}")

    except Exception as e:
        logger.error(f"❌ 작업 상태 업데이트 중 오류: {job_id} - {e}")
        logger.error(traceback.format_exc())

def create_streaming_queue(job_id: str):
    """job_id에 대한 스트리밍 큐 생성"""
    streaming_queues[job_id] = queue.Queue()
    streaming_locks[job_id] = threading.Lock()
    logger.info(f"🔄 스트리밍 큐 생성: {job_id}")

def cleanup_streaming_queue(job_id: str):
    """job_id에 대한 스트리밍 큐 정리"""
    if job_id in streaming_queues:
        # 종료 신호 전송
        try:
            streaming_queues[job_id].put({"type": "close"}, timeout=1)
        except:
            pass
        del streaming_queues[job_id]
    if job_id in streaming_locks:
        del streaming_locks[job_id]
    logger.info(f"🧹 스트리밍 큐 정리: {job_id}")

async def send_streaming_update(job_id: str, data: dict):
    """SSE 스트리밍으로 업데이트 전송"""
    if job_id not in streaming_queues:
        return
    
    try:
        with streaming_locks[job_id]:
            streaming_queues[job_id].put(data, timeout=1)
        
        # 섹션 데이터 전송 시 상세 로깅
        if data.get('type') == 'section':
            section_name = data.get('section_name', 'unknown')
            content_length = len(data.get('content', '')) if data.get('content') else 0
            logger.info(f"📝 섹션 데이터 큐에 추가: {job_id} - {section_name} (내용: {content_length}자)")
            if content_length > 0:
                content_preview = data.get('content', '')[:100] + ('...' if content_length > 100 else '')
                logger.debug(f"📝 섹션 내용 미리보기: {content_preview}")
        else:
            logger.debug(f"📡 스트리밍 데이터 전송: {job_id} - {data.get('type', 'unknown')}")
    except queue.Full:
        logger.warning(f"⚠️ 스트리밍 큐 풀: {job_id}")
    except Exception as e:
        logger.error(f"❌ 스트리밍 전송 오류: {job_id} - {e}")

def generate_sse_stream(job_id: str):
    """SSE 스트림 생성기"""
    try:
        if job_id not in streaming_queues:
            yield f"data: {json.dumps({'type': 'error', 'message': 'Job not found'})}\n\n"
            return
        
        logger.info(f"📡 SSE 스트림 시작: {job_id}")
        
        # 초기 연결 확인 메시지
        yield f"data: {json.dumps({'type': 'connected', 'job_id': job_id, 'timestamp': datetime.now().isoformat()})}\n\n"
        
        while True:
            try:
                # 큐에서 데이터 가져오기 (타임아웃 2초로 단축)
                data = streaming_queues[job_id].get(timeout=2)
                
                if data.get("type") == "close":
                    logger.info(f"📡 SSE 스트림 종료: {job_id}")
                    break
                
                # 섹션 데이터 전송 시 상세 로깅
                if data.get('type') == 'section':
                    section_name = data.get('section_name', 'unknown')
                    content_length = len(data.get('content', '')) if data.get('content') else 0
                    logger.info(f"📤 클라이언트로 섹션 데이터 전송: {job_id} - {section_name} (내용: {content_length}자)")
                
                # SSE 형식으로 데이터 전송
                yield f"data: {json.dumps(data)}\n\n"
                
                # 작업 완료시 스트림 종료
                if data.get("type") == "status" and data.get("status") in ["completed", "failed", "cancelled"]:
                    logger.info(f"📡 SSE 스트림 작업 완료로 종료: {job_id}")
                    break
                    
            except queue.Empty:
                # 주기적인 heartbeat
                yield f"data: {json.dumps({'type': 'heartbeat', 'timestamp': datetime.now().isoformat()})}\n\n"
            except Exception as e:
                error_message = str(e) if str(e).strip() else "알 수 없는 오류가 발생했습니다"
                logger.error(f"❌ SSE 스트림 오류: {job_id} - {error_message}")
                
                # Job ID가 에러 메시지로 전달되는 경우 스트림 종료 (무한루프 방지)
                if error_message.replace("'", "").replace('"', '').strip().isdigit():
                    logger.warning(f"⚠️ Job ID가 에러로 잘못 전달됨: {error_message} - 스트림 종료")
                    yield f"data: {json.dumps({'type': 'heartbeat', 'timestamp': datetime.now().isoformat()})}\n\n"
                    break  # 무한루프 방지를 위해 스트림 종료
                
                # 실제 오류인 경우에만 에러 메시지 전송
                try:
                    yield f"data: {json.dumps({'type': 'error', 'message': error_message, 'timestamp': datetime.now().isoformat()})}\n\n"
                except Exception as json_error:
                    logger.error(f"❌ SSE JSON 직렬화 오류: {job_id} - {json_error}")
                    yield f"data: {json.dumps({'type': 'error', 'message': 'JSON 직렬화 오류'})}\n\n"
                    break
                
    finally:
        # 정리
        cleanup_streaming_queue(job_id)

# =============================================================================
# 백그라운드 작업 함수들
# =============================================================================

async def generate_report_background(request: ReportGenerationRequest, job_id: str):
    """백그라운드에서 실제 보고서 생성"""
    global db_manager
    try:
        # 스트리밍 큐 생성
        create_streaming_queue(job_id)
        
        # 작업 상태를 'processing'으로 업데이트
        await update_job_status(job_id, "processing", 0)
        
        # 사용자 정보 로깅 (username 기반)
        user_info = f"(사용자: {request.username})" if request.username else "(사용자: 미지정)"
        logger.info(f"🚀 보고서 생성 시작 - Job ID: {job_id} {user_info}")
        
        # 초기 시작 메시지 전송 (약간의 지연 후)
        await asyncio.sleep(0.5)  # SSE 연결이 안정화될 시간 제공
        await send_streaming_update(job_id, {
            "type": "section",
            "section_name": "initialization",
            "title": "문서 생성 시작",
            "content": f"📄 {request.prompt[:100]}{'...' if len(request.prompt) > 100 else ''}\n\n문서 생성을 시작합니다...",
            "status": "processing",
            "progress": 0,
            "timestamp": datetime.now().isoformat()
        })
        logger.info(f"📝 초기 메시지 전송 완료: {job_id}")
        
        # 진행률 콜백 함수 - 스트리밍 지원
        async def progress_callback(message: str, progress: int, section_name: str = None, section_data: dict = None):
            logger.info(f"📊 [{job_id}] {user_info} {message}")
            if progress >= 0:
                await update_job_status(job_id, "processing", progress)
            
            # 섹션 데이터가 있으면 구조화된 데이터를 스트리밍으로 전송
            if section_name and section_data:
                # 섹션 데이터를 스트리밍으로 전송
                streaming_data = {
                    "type": "section",
                    "section_name": section_name,
                    "title": message,
                    "content": section_data.get('content', ''),
                    "status": "completed",
                    "progress": progress,
                    "timestamp": datetime.now().isoformat()
                }
                
                # 추가 데이터 포함 (차트, 테이블, 다이어그램 등)
                if section_data.get('has_chart'):
                    streaming_data["chart_data"] = section_data.get('chart_data')
                if section_data.get('has_table'):
                    streaming_data["table_data"] = section_data.get('table_data')
                if section_data.get('has_diagram'):
                    streaming_data["diagram_data"] = section_data.get('diagram_data')
                if section_data.get('has_flowchart'):
                    streaming_data["flowchart_data"] = section_data.get('flowchart_data')
                if section_data.get('has_gantt'):
                    streaming_data["gantt_data"] = section_data.get('gantt_data')
                if section_data.get('has_sequence'):
                    streaming_data["sequence_data"] = section_data.get('sequence_data')
                if section_data.get('visualization_paths'):
                    streaming_data["visualization_paths"] = section_data.get('visualization_paths')
                
                await send_streaming_update(job_id, streaming_data)
            else:
                # 일반 진행률 업데이트
                await send_streaming_update(job_id, {
                    "type": "progress",
                    "message": message,
                    "progress": progress,
                    "timestamp": datetime.now().isoformat()
                })
        
        # request에 job_id 설정
        request.job_id = job_id
        
        # 실제 보고서 생성 (report_service 사용)
        result = await report_service.generate_report(request, progress_callback)
        
        # 완료 처리 - 결과 구조에 따라 파일 경로 추출
        file_path = None
        if result.get('success') and result.get('result'):
            file_path = result['result'].get('file_path')
            logger.info(f"📊 [DEBUG] 파일 경로 추출 (success.result) {user_info}: {file_path}")
        elif result.get('file_path'):
            file_path = result['file_path']
            logger.info(f"📊 [DEBUG] 파일 경로 추출 (direct file_path) {user_info}: {file_path}")
        elif result.get('output_path'):
            file_path = result['output_path']
            logger.info(f"📊 [DEBUG] 파일 경로 추출 (output_path) {user_info}: {file_path}")
        else:
            logger.warning(f"📊 [DEBUG] 파일 경로를 찾을 수 없음 {user_info}: {result}")
        
        # 파일이 실제로 존재하고 완전히 생성되었는지 확인
        if file_path and os.path.exists(file_path):
            # 파일 크기가 0이 아닌지 확인 (빈 파일이 아닌지)
            file_size = os.path.getsize(file_path)
            if file_size > 0:
                logger.info(f"📊 [DEBUG] 파일 검증 완료 {user_info}: {file_path} (크기: {file_size:,} bytes)")
                # 파일이 완전히 생성되었을 때만 작업 상태를 'completed'로 업데이트
                await update_job_status(job_id, "completed", 100, file_path)
            else:
                logger.error(f"📊 [ERROR] 파일이 비어있음 {user_info}: {file_path}")
                await update_job_status(job_id, "failed", 100, None, "생성된 파일이 비어있습니다.")
                return
        else:
            logger.error(f"📊 [ERROR] 파일이 존재하지 않음 {user_info}: {file_path}")
            await update_job_status(job_id, "failed", 100, None, "파일 생성에 실패했습니다.")
            return
        
        # 완료 메시지 스트리밍
        await send_streaming_update(job_id, {
            "type": "section",
            "section_name": "completion",
            "title": "문서 생성 완료",
            "content": f"✅ 문서 생성이 성공적으로 완료되었습니다!\n\n📁 파일: {file_path}\n🎉 모든 섹션이 작성되었습니다.",
            "status": "completed",
            "progress": 100,
            "file_path": file_path,
            "timestamp": datetime.now().isoformat()
        })
        
        logger.info(f"✅ 보고서 생성 완료 - Job ID: {job_id} {user_info}")
        logger.info(f"   파일: {file_path}")
        
        # 완료 메시지 전송 후 충분한 지연을 주어 클라이언트가 데이터를 수신할 시간 확보
        logger.info(f"📤 클라이언트 데이터 수신 대기 중: {job_id} (10초)")
        await asyncio.sleep(10)
        logger.info(f"📤 스트리밍 큐 정리 시작: {job_id}")
        cleanup_streaming_queue(job_id)
        
        # 이메일 발송 로직 - 간단하고 안전한 방식
        if request.email and file_path:
            try:
                # 파일이 실제로 존재하는지 확인
                if not os.path.exists(file_path):
                    logger.warning(f"⚠️ 파일을 찾을 수 없어 이메일을 발송하지 않습니다: {file_path}")
                else:
                    # 파일 크기 확인
                    file_size = os.path.getsize(file_path)
                    file_format = request.format.upper()
                    
                    # 이메일 제목 및 본문 구성
                    subject = f"AI.RUN {file_format} 문서 생성 완료"
                    body = f"""안녕하세요,

요청하신 {file_format} 문서가 성공적으로 생성되었습니다.

📋 작업 정보:
- Job ID: {job_id}
- 프롬프트: {request.prompt[:100]}{'...' if len(request.prompt) > 100 else ''}
- 출력 형식: {file_format}
- 템플릿: {request.template}
- 파일 크기: {file_size:,} bytes
- 사용자: {request.username or '미지정'}

첨부 파일을 확인해 주세요.

감사합니다.
AI.RUN 시스템"""

                    # 직접 이메일 발송 시도
                    try:
                        from utils import send_email
                        attachments = [file_path]
                        result = send_email(request.email, subject, body, attachments)
                        
                        if result:
                            logger.info(f"📧 이메일 발송 성공: {request.email}")
                        else:
                            logger.warning(f"📧 이메일 발송 실패: {request.email}")
                            
                    except Exception as email_send_error:
                        logger.error(f"📧 이메일 발송 오류: {request.email} - {str(email_send_error)}")
                        
            except Exception as email_process_error:
                logger.error(f"📧 이메일 처리 중 전체 오류: {str(email_process_error)}")
                
    except Exception as e:
        # 에러 처리
        error_msg = str(e)
        logger.error(f"❌ 보고서 생성 실패 - Job ID: {job_id}: {error_msg}")
        logger.error(traceback.format_exc())
        
        await update_job_status(job_id, "failed", None, None, error_msg)
        
        # 에러 메시지 스트리밍
        await send_streaming_update(job_id, {
            "type": "section",
            "section_name": "error",
            "title": "문서 생성 실패",
            "content": f"❌ 문서 생성 중 오류가 발생했습니다:\n\n{error_msg}\n\n다시 시도해 주세요.",
            "status": "failed",
            "progress": 0,
            "error": error_msg,
            "timestamp": datetime.now().isoformat()
        })
        
        # 스트리밍 정리
        await asyncio.sleep(1)
        cleanup_streaming_queue(job_id)

# =============================================================================
# API 엔드포인트들
# =============================================================================

# 🔍 Clarification 관련 API 엔드포인트들 추가
@app.post("/analyze", response_model=ClarificationResult)
async def analyze_user_request(request: ClarificationRequest):
    """사용자 요청 분석 및 명확화 질문 생성"""
    global clarification_service
    try:
        if not clarification_service:
            raise HTTPException(status_code=500, detail="Clarification 서비스가 초기화되지 않았습니다.")
        
        logger.info(f"🔍 사용자 요청 분석 시작: {request.username} - {request.prompt[:50]}...")
        
        result = await clarification_service.analyze_user_request(request)
        
        logger.info(f"✅ 분석 완료: {len(result.questions)}개 질문, 신뢰도: {result.confidence_score}")
        
        return result
        
    except Exception as e:
        logger.error(f"❌ 사용자 요청 분석 실패: {str(e)}")
        logger.error(traceback.format_exc())
        raise HTTPException(status_code=500, detail=f"요청 분석 실패: {str(e)}")

@app.post("/generate-enhanced", response_model=ReportGenerationResponse)
async def generate_enhanced_report(request: EnhancedReportRequest, background_tasks: BackgroundTasks):
    """명확화 결과를 포함한 향상된 보고서 생성"""
    global db_manager
    try:
        server_stats["total_requests"] += 1
        server_stats["active_jobs"] += 1
        
        logger.info(f"📝 향상된 보고서 생성 요청: {request.username} - {request.original_prompt[:50]}...")
        
        # 명확화 응답들을 프롬프트에 통합
        enhanced_prompt = await _integrate_clarification_responses(
            request.original_prompt, 
            request.clarification_responses
        )
        
        logger.info(f"🔧 프롬프트 향상 완료: {len(enhanced_prompt)} 자")
        
        # 기존 ReportGenerationRequest로 변환
        report_request = ReportGenerationRequest(
            prompt=enhanced_prompt,
            format=request.format,
            template=request.confirmed_template,
            email=request.email,
            username=request.username,
            use_cache=request.use_cache,
            project_hash=request.project_hash,
            job_id=request.job_id,
            clarification_responses=request.clarification_responses,
            enhanced_prompt=enhanced_prompt
        )
        
        # Job ID 생성 및 중복 확인
        job_id = report_request.job_id if report_request.job_id else generate_job_id()
        
        existing_job = await db_manager.get_report_job(job_id)
        if existing_job:
            import time
            job_id = f"{job_id}_{int(time.time())}"
        
        # 작업 정보 저장
        job_data = {
            'job_id': job_id,
            'username': request.username,
            'title': f"[향상됨] {request.original_prompt[:50]}...",
            'executive_summary': enhanced_prompt,
            'template': request.confirmed_template,
            'output_format': request.format,
            'status': 'queued',
            'progress': 0,
            'metadata': {
                'original_prompt': request.original_prompt,
                'enhanced_prompt': enhanced_prompt,
                'clarification_responses': [resp.dict() for resp in request.clarification_responses],
                'format': request.format,
                'template': request.confirmed_template,
                'email': request.email,
                'username': request.username,
                'use_cache': request.use_cache,
                'project_hash': request.project_hash,
                'is_enhanced': True  # 향상된 요청임을 표시
            }
        }
        
        await db_manager.create_report_job(job_data)
        logger.info(f"📝 향상된 보고서 생성 요청 접수 - Job ID: {job_id}")
        
        # 백그라운드 작업 추가
        background_tasks.add_task(generate_report_background, report_request, job_id)
        
        return ReportGenerationResponse(
            success=True,
            job_id=job_id,
            message="명확화된 정보로 향상된 보고서 생성 작업이 시작되었습니다.",
            status="queued",
            estimated_time=90  # 향상된 보고서는 더 오래 걸릴 수 있음
        )
        
    except Exception as e:
        logger.error(f"❌ 향상된 보고서 생성 요청 처리 중 오류: {e}")
        logger.error(traceback.format_exc())
        raise HTTPException(status_code=500, detail=f"향상된 보고서 생성 요청 처리 실패: {e}")

async def _integrate_clarification_responses(original_prompt: str, responses: List[ClarificationResponse]) -> str:
    """명확화 응답들을 원본 프롬프트에 통합하여 향상된 프롬프트 생성"""
    try:
        # 응답들을 구조화된 정보로 변환
        clarifications = {}
        for response in responses:
            clarifications[response.question_id] = response.answer
        
        # 향상된 프롬프트 구성
        enhanced_sections = [
            f"원본 요청\n{original_prompt}",
            "\n## 추가 수집된 정보"
        ]
        
        # 응답별로 정보 추가
        important_info = []
        additional_info = []
        
        for response in responses:
            answer = response.answer
            if isinstance(answer, list):
                answer = ", ".join(answer)
            
            # 질문 ID를 기반으로 카테고리 분류
            if response.question_id in ['purpose', 'target_audience', 'detail_level']:
                important_info.append(f"- {response.question_id}: {answer}")
            else:
                additional_info.append(f"- {response.question_id}: {answer}")
        
        if important_info:
            enhanced_sections.append("### 핵심 정보")
            enhanced_sections.extend(important_info)
        
        if additional_info:
            enhanced_sections.append("\n### 추가 세부사항")
            enhanced_sections.extend(additional_info)
        
        # 생성 지침 추가
        enhanced_sections.append("""
## 문서 생성 지침
위의 원본 요청과 추가 명확화된 정보를 모두 반영하여 완성도 높은 문서를 생성해주세요. 
명확화된 정보들은 문서의 구체성과 정확성을 높이는데 활용해주세요.""")
        
        enhanced_prompt = "\n".join(enhanced_sections)
        
        logger.info(f"🔧 프롬프트 통합 완료: {len(responses)}개 응답 통합, 최종 길이: {len(enhanced_prompt)}자")
        
        return enhanced_prompt
        
    except Exception as e:
        logger.error(f"❌ 프롬프트 통합 실패: {str(e)}")
        # 실패 시 원본 프롬프트 반환
        return original_prompt

@app.get("/health")
async def health_check():
    """서버 상태 확인 - 기본 상태 정보 제공"""
    try:
        uptime = datetime.now() - server_stats["start_time"]
        
        # 기본 기능 상태
        features = {
            "rag_service": False,
            "web_search": False,
            "thread_pool": False
        }
        
        # 서비스 상태 확인
        if report_service:
            try:
                features["rag_service"] = getattr(report_service, 'rag_available', False)
                features["web_search"] = getattr(report_service, 'web_search_available', False)
                features["ai_provider"] = getattr(report_service, 'openai_available', False)
                features["bizplan_generator"] = getattr(report_service, 'bizplan_generator_available', False)
                features["thread_pool"] = getattr(report_service, 'thread_pool', None) is not None
            except Exception as e:
                logger.warning(f"서비스 상태 확인 중 오류: {e}")
        
        # 전체 상태 결정
        status = "healthy" if features["thread_pool"] else "degraded"
        
        # DB에서 활성 작업 수 가져오기
        active_jobs_count = 0
        total_jobs_count = 0
        unique_users_count = 0
        
        try:
            if db_manager and db_manager.is_connected:
                active_jobs_list = await db_manager.list_report_jobs(status='queued') + await db_manager.list_report_jobs(status='processing')
                active_jobs_count = len(active_jobs_list)
                
                # 전체 작업 수 계산
                job_stats = await db_manager.get_job_stats()
                total_jobs_count = sum(stats.get('count', 0) for stats in job_stats.values())
                
                # 고유 사용자 수 계산
                unique_users_count = len(set(job.get('username') for job in active_jobs_list if job.get('username')))
        except Exception as e:
            logger.warning(f"DB 상태 조회 실패: {e}")
        
        return {
            "status": status,
            "service": "airun-report-server",
            "version": "2.0.0",
            "timestamp": datetime.now().isoformat(),
            "uptime": str(uptime).split('.')[0],
            "active_jobs": active_jobs_count,
            "total_jobs": total_jobs_count,
            "features": features,
            "performance": {
                "total_requests": server_stats["total_requests"],
                "completed_jobs": server_stats["completed_jobs"],
                "failed_jobs": server_stats["failed_jobs"],
                "success_rate": round(server_stats["completed_jobs"] / max(server_stats["total_requests"], 1) * 100, 2),
                "unique_users": unique_users_count
            }
        }
    except Exception as e:
        logger.error(f"Health check 오류: {e}")
        return {
            "status": "error",
            "service": "airun-report-server",
            "version": "2.0.0",
            "timestamp": datetime.now().isoformat(),
            "error": str(e)
        }

@app.get("/stats")
async def get_server_stats():
    """서버 통계 조회"""
    uptime = datetime.now() - server_stats["start_time"]
    
    # 사용자별 통계 계산
    user_stats = {}
    active_jobs_count = 0
    total_jobs_count = 0
    
    try:
        if db_manager and db_manager.is_connected:
            conn = db_manager.get_connection()
            try:
                with conn.cursor(cursor_factory=RealDictCursor) as cur:
                    # 사용자별 통계 조회
                    cur.execute("SELECT username, COUNT(*) as total, SUM(CASE WHEN status IN ('queued', 'processing') THEN 1 ELSE 0 END) as active, SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed, SUM(CASE WHEN status = 'failed' THEN 1 ELSE 0 END) as failed FROM report_jobs WHERE username IS NOT NULL AND username != '' GROUP BY username")
                    results = cur.fetchall()
                    for row in results:
                        user_stats[row['username']] = {
                            "total": row['total'],
                            "active": row['active'],
                            "completed": row['completed'],
                            "failed": row['failed'],
                            "username": row['username']
                        }
                    
                    # 활성 작업 수 계산
                    cur.execute("SELECT COUNT(*) as count FROM report_jobs WHERE status IN ('queued', 'processing')")
                    active_jobs_count = cur.fetchone()['count']
                    
                    # 전체 작업 수 계산
                    cur.execute("SELECT COUNT(*) as count FROM report_jobs")
                    total_jobs_count = cur.fetchone()['count']
                    
            finally:
                db_manager.return_connection(conn)
    except Exception as e:
        logger.warning(f"DB 통계 조회 실패: {e}")
    
    return {
        "server_uptime": str(uptime).split('.')[0],
        "total_requests": server_stats["total_requests"],
        "active_jobs": active_jobs_count,
        "completed_jobs": server_stats["completed_jobs"],
        "failed_jobs": server_stats["failed_jobs"],
        "total_jobs": total_jobs_count,
        "user_stats": user_stats,
        "unique_users": len(user_stats)
    }

@app.post("/generate", response_model=ReportGenerationResponse)
async def generate_report(request: ReportGenerationRequest, background_tasks: BackgroundTasks):
    """보고서 생성 요청"""
    global db_manager
    try:
        server_stats["total_requests"] += 1
        server_stats["active_jobs"] += 1
        
        # 부분 수정 모드 유효성 검증
        if request.modification_mode == 'partial':
            if not request.existing_document_path and not request.existing_document_content:
                raise HTTPException(
                    status_code=400, 
                    detail="부분 수정 모드에는 기존 문서 파일 경로(existing_document_path) 또는 내용(existing_document_content)이 필요합니다."
                )
            
            if request.existing_document_path and not os.path.exists(request.existing_document_path):
                raise HTTPException(
                    status_code=400, 
                    detail=f"지정된 기존 문서 파일을 찾을 수 없습니다: {request.existing_document_path}"
                )
            
            if request.sections_to_modify and not isinstance(request.sections_to_modify, list):
                raise HTTPException(
                    status_code=400, 
                    detail="sections_to_modify는 문자열 배열이어야 합니다."
                )
            
            logger.info(f"📝 부분 수정 모드 요청 - 수정 대상: {request.sections_to_modify}")
        
        # Job ID 디버깅 로그 추가
        logger.info(f"📋 [DEBUG] 요청에서 받은 Job ID: {request.job_id}")
        logger.info(f"📋 [DEBUG] 요청 전체 내용: {request}")
        
        # Job ID 사용 (큐에서 전달받은 것을 우선 사용, 없으면 새로 생성)
        job_id = request.job_id if request.job_id else generate_job_id()
        logger.info(f"📋 [DEBUG] 최종 사용할 Job ID: {job_id}")
        
        # 기존 작업 존재 여부 확인 (중복 방지)
        existing_job = await db_manager.get_report_job(job_id)
        if existing_job:
            logger.warning(f"⚠️ Job ID 중복 발견: {job_id} - 기존 작업이 존재함")
            # 중복되는 경우 새로 생성 (안전장치)
            import time
            job_id = f"{job_id}_{int(time.time())}"
            logger.info(f"✅ 중복 회피를 위해 수정된 Job ID: {job_id}")
        
        # 작업 정보 저장 (username 기반으로 단순화)
        job_data = {
            'job_id': job_id,
            'username': request.username,
            'title': f"{request.prompt[:50]}...",
            'executive_summary': request.prompt,
            'template': request.template,
            'output_format': request.format,
            'status': 'queued',
            'progress': 0,
            'metadata': {
                'prompt': request.prompt,
                'format': request.format,
                'template': request.template,
                'email': request.email,
                'username': request.username,
                'use_cache': request.use_cache,
                'project_hash': request.project_hash,
                'modification_mode': request.modification_mode,
                'existing_document_path': request.existing_document_path,
                'sections_to_modify': request.sections_to_modify,
                'preserve_formatting': request.preserve_formatting,
                'provider': request.provider,
                'model': request.model
            }
        }
        
        await db_manager.create_report_job(job_data)
        logger.info(f"📝 보고서 생성 요청 접수 - Job ID: {job_id} (사용자: {request.username})")
        logger.info(f"   프롬프트: {request.prompt[:50]}...")
        logger.info(f"   형식: {request.format}")
        logger.info(f"   템플릿: {request.template}")
        logger.info(f"   예상 시간: {60}초") # 기본 60초 (실제 AI 처리 시간)
        
        # 백그라운드 작업 추가
        background_tasks.add_task(generate_report_background, request, job_id)
        
        return ReportGenerationResponse(
            success=True,
            job_id=job_id,
            message="보고서 생성 작업이 시작되었습니다.",
            status="queued",
            estimated_time=60
        )
        
    except Exception as e:
        logger.error(f"보고서 생성 요청 처리 중 오류: {e}")
        logger.error(traceback.format_exc())
        raise HTTPException(status_code=500, detail=f"보고서 생성 요청 처리 실패: {e}")

@app.get("/status/{job_id}", response_model=JobStatusResponse)
async def get_job_status(job_id: str):
    """작업 상태 조회"""
    global db_manager
    try:
        job_info = await db_manager.get_report_job(job_id)
        if not job_info:
            raise HTTPException(status_code=404, detail="작업을 찾을 수 없습니다.")
        
        return JobStatusResponse(
            success=True,
            job_id=job_info['job_id'],
            status=job_info['status'],
            progress=job_info['progress'],
            file_path=job_info.get('output_path'),
            error=job_info.get('error_message'),
            created_at=job_info['created_at'].isoformat() if job_info['created_at'] else None,
            updated_at=job_info['updated_at'].isoformat() if job_info['updated_at'] else None
        )
    except Exception as e:
        logger.error(f"작업 상태 조회 중 오류: {job_id} - {e}")
        raise HTTPException(status_code=500, detail=f"작업 상태 조회 실패: {e}")

@app.get("/jobs")
async def list_jobs(username: Optional[str] = None):
    """작업 목록 조회 (사용자별 필터링 지원)"""
    global db_manager
    try:
        jobs_data = await db_manager.list_report_jobs(username=username)
        
        jobs = []
        for job in jobs_data:
            jobs.append({
                "job_id": job['job_id'],
                "status": job['status'],
                "progress": job['progress'],
                "username": job['username'],
                "created_at": job['created_at'],
                "updated_at": job['updated_at'],
                "title": job['title'],
                "template": job['template'],
                "output_format": job['output_format'],
                "request": {
                    "prompt": job['executive_summary'],
                    "format": job['output_format'],
                    "template": job['template'],
                    "email": job.get('email'),
                    "username": job['username'],
                    "use_cache": job.get('use_cache', True),
                    "project_hash": job.get('project_hash')
                }
            })
        
        return {
            "success": True,
            "jobs": jobs,
            "total": len(jobs)
        }
    except Exception as e:
        logger.error(f"작업 목록 조회 중 오류: {e}")
        raise HTTPException(status_code=500, detail=f"작업 목록 조회 실패: {e}")

@app.delete("/job/{job_id}")
async def cancel_job(job_id: str):
    """작업 취소"""
    global db_manager
    try:
        if not db_manager:
            raise HTTPException(status_code=500, detail="데이터베이스 매니저가 초기화되지 않아 작업을 취소할 수 없습니다.")

        job_info = await db_manager.get_report_job(job_id)
        if not job_info:
            raise HTTPException(status_code=404, detail="작업을 찾을 수 없습니다.")
        
        if job_info['status'] in ["completed", "failed"]:
            raise HTTPException(status_code=400, detail="이미 완료된 작업은 취소할 수 없습니다.")
        
        await update_job_status(job_id, "cancelled")
        
        return {
            "success": True,
            "message": f"작업 {job_id}이 취소되었습니다."
        }
    except Exception as e:
        logger.error(f"작업 취소 중 오류: {job_id} - {e}")
        raise HTTPException(status_code=500, detail=f"작업 취소 실패: {e}")

@app.get("/stream/{job_id}")
async def stream_job_progress(job_id: str):
    """작업 진행 상황 스트리밍"""
    global db_manager
    
    # 작업 존재 확인
    job_info = await db_manager.get_report_job(job_id)
    if not job_info:
        raise HTTPException(status_code=404, detail="작업을 찾을 수 없습니다.")
    
    # 스트리밍 큐가 없으면 생성
    if job_id not in streaming_queues:
        create_streaming_queue(job_id)
    
    # SSE 스트림 반환
    return StreamingResponse(
        generate_sse_stream(job_id),
        media_type="text/event-stream",
        headers={
            "Cache-Control": "no-cache",
            "Connection": "keep-alive",
            "Access-Control-Allow-Origin": "*",
            "Access-Control-Allow-Headers": "Cache-Control"
        }
    )

# =============================================================================
# 주간 보고서 변환 엔드포인트
# =============================================================================

@app.post("/weekly-report/convert")
async def convert_weekly_report(request: WeeklyReportConvertRequest):
    """주간/월간 보고서를 PDF/DOCX/HWP 형식으로 변환 (공문서 스타일)"""
    import tempfile
    from fastapi.responses import FileResponse

    try:
        logger.info(f"📝 주간 보고서 변환 요청 - 형식: {request.format}")

        report = request.report
        metadata = request.metadata
        output_format = request.format.lower()

        # 지원하는 형식 확인
        if output_format not in ['pdf', 'docx', 'hwpx']:
            raise HTTPException(status_code=400, detail=f"지원하지 않는 형식입니다: {output_format}")

        # utils 모듈에서 문서 클래스 가져오기
        if 'utils' not in sys.modules:
            raise HTTPException(status_code=500, detail="utils 모듈이 로드되지 않았습니다")

        utils_module = sys.modules['utils']

        # 문서 객체 생성
        if output_format == 'pdf':
            doc = utils_module.PDFDocument()
        elif output_format == 'docx':
            doc = utils_module.DOCXDocument()
        elif output_format == 'hwpx':
            doc = utils_module.HWPDocument()

        # 헬퍼 함수: 테이블 추가 (HWP만 style/header_style 지원)
        def add_styled_table(data, header, style=2, header_style=7):
            if output_format == 'hwpx':
                doc.add_table(data=data, header=header, style=style, header_style=header_style)
            else:
                # PDF/DOCX는 style/header_style 미지원
                doc.add_table(data=data, header=header)

        # 헬퍼 함수: 간격 추가 (PDF만 지원, DOCX/HWP는 빈 단락으로 대체)
        def add_spacing(size=10):
            if output_format == 'pdf':
                doc.add_spacing(size)
            else:
                # DOCX/HWP는 add_spacing 미지원, 빈 단락으로 대체
                if size >= 10:
                    doc.add_paragraph("")

        # 보고서 제목 설정
        report_type_label = "주간" if metadata.reportType == "weekly" else "월간"
        title = f"{report_type_label} 업무보고서"
        next_period_label = "차주" if metadata.reportType == "weekly" else "차월"

        # ============================================
        # 공문서 스타일 설정 (지원되는 문서 유형에서만 적용)
        # ============================================

        # 머릿말 설정 (회사명/부서) - PDF/DOCX만 지원
        header_text = metadata.department if metadata.department else "AI.RUN"
        try:
            if output_format == 'pdf':
                doc.set_header(header_text, align="left")
            elif output_format == 'docx':
                doc.set_header(header_text)
            # HWP는 set_header 미지원
        except (AttributeError, TypeError) as e:
            logger.debug(f"머릿말 설정 미지원: {e}")

        # 꼬릿말 설정 (저작권/날짜) - PDF/DOCX만 지원
        footer_text = f"© 2025 AI.RUN | 작성일: {metadata.generatedAt[:10] if metadata.generatedAt else datetime.now().strftime('%Y-%m-%d')}"
        try:
            if output_format == 'pdf':
                doc.set_footer(footer_text, align="center")
            elif output_format == 'docx':
                doc.set_footer(footer_text)
            # HWP는 set_footer 미지원
        except (AttributeError, TypeError) as e:
            logger.debug(f"꼬릿말 설정 미지원: {e}")

        # 로고 추가 (있는 경우) - 우측 상단에 작게 표시
        logo_path = os.path.join(os.path.dirname(__file__), '..', '..', 'workspaces', 'web', 'public', 'images', 'logos', 'logo.png')
        
        if os.path.exists(logo_path):
            try:
                # 로고 크기를 작게 조정 (너비 30mm로 고정)
                if output_format == 'pdf':
                    # PDF의 경우 우측 정렬 로고 추가
                    doc._add_right_aligned_logo(logo_path)
                elif output_format == 'docx':
                    # DOCX의 경우 작은 크기로 추가
                    doc.add_image(logo_path, width=30*mm, height=None)
                elif output_format == 'hwpx':
                    # HWP의 경우 작은 크기로 추가
                    doc.add_image(logo_path, width=30*mm, height=None)
                    # HWP는 이미지 추가 후 간격 추가
                    add_spacing(5) if hasattr(doc, 'add_spacing') else doc.add_paragraph("")
            except Exception as e:
                logger.warning(f"로고 추가 실패: {e}")

        # 문서 제목 (큰 제목)
        doc.add_heading(title, level=1)
        add_spacing(5)

        # ============================================
        # 보고서 기본 정보 (표 형식)
        # ============================================
        info_header = ["항목", "내용"]
        info_data = [
            ["보고 기간", f"{metadata.startDate} ~ {metadata.endDate}"],
        ]
        if metadata.department:
            info_data.append(["부서", metadata.department])
        if metadata.author:
            info_data.append(["작성자", metadata.author])
        if metadata.generatedAt:
            info_data.append(["작성일", metadata.generatedAt[:10]])
        if metadata.documentCount and metadata.documentCount > 0:
            info_data.append(["참조 문서 수", f"{metadata.documentCount}건"])

        add_styled_table(data=info_data, header=info_header)
        add_spacing(15)

        # ============================================
        # 1. 업무 요약
        # ============================================
        doc.add_heading("1. 업무 요약", level=2)
        add_spacing(5)
        if report.summary:
            doc.add_paragraph(report.summary)
        else:
            doc.add_paragraph("(요약 없음)")
        add_spacing(10)

        # ============================================
        # 2. 완료된 업무 (표 형식)
        # ============================================
        doc.add_heading("2. 완료된 업무", level=2)
        add_spacing(5)
        if report.completedTasks and len(report.completedTasks) > 0:
            completed_header = ["No.", "업무명", "결과/성과", "소요시간"]
            completed_data = []
            for idx, task in enumerate(report.completedTasks, 1):
                completed_data.append([
                    str(idx),
                    task.taskName or "-",
                    task.result or "-",
                    task.timeSpent or "-"
                ])
            add_styled_table(data=completed_data, header=completed_header)
        else:
            doc.add_paragraph("(완료된 업무 없음)")
        add_spacing(10)

        # ============================================
        # 3. 진행 중인 업무 (표 형식)
        # ============================================
        doc.add_heading("3. 진행 중인 업무", level=2)
        add_spacing(5)
        if report.ongoingTasks and len(report.ongoingTasks) > 0:
            ongoing_header = ["No.", "업무명", "진행률", f"{next_period_label} 계획"]
            ongoing_data = []
            for idx, task in enumerate(report.ongoingTasks, 1):
                ongoing_data.append([
                    str(idx),
                    task.taskName or "-",
                    task.progress or "-",
                    task.nextPlan or "-"
                ])
            add_styled_table(data=ongoing_data, header=ongoing_header)
        else:
            doc.add_paragraph("(진행 중인 업무 없음)")
        add_spacing(10)

        # ============================================
        # 4. 이슈 및 해결 방안 (표 형식)
        # ============================================
        doc.add_heading("4. 이슈 및 해결 방안", level=2)
        add_spacing(5)
        if report.issues and len(report.issues) > 0:
            issues_header = ["No.", "분류", "이슈 내용", "해결 방안", "상태"]
            issues_data = []
            for idx, issue in enumerate(report.issues, 1):
                issues_data.append([
                    str(idx),
                    issue.category or "-",
                    issue.content or "-",
                    issue.solution or "-",
                    issue.status or "-"
                ])
            add_styled_table(data=issues_data, header=issues_header)
        else:
            doc.add_paragraph("(이슈 없음)")
        add_spacing(10)

        # ============================================
        # 5. 다음 기간 계획 (표 형식)
        # ============================================
        doc.add_heading(f"5. {next_period_label} 계획", level=2)
        add_spacing(5)
        if report.nextPeriodPlans and len(report.nextPeriodPlans) > 0:
            plans_header = ["No.", "계획 업무", "우선순위", "목표 성과"]
            plans_data = []
            for idx, plan in enumerate(report.nextPeriodPlans, 1):
                plans_data.append([
                    str(idx),
                    plan.taskName or "-",
                    plan.priority or "-",
                    plan.targetResult or "-"
                ])
            add_styled_table(data=plans_data, header=plans_header)
        else:
            doc.add_paragraph("(계획 없음)")
        add_spacing(15)

        # ============================================
        # 6. 업무 현황 차트 (시각화)
        # ============================================
        try:
            from utils import create_matplotlib

            # 차트용 임시 디렉토리
            chart_temp_dir = os.path.join(os.path.expanduser('~'), '.airun', 'temp', 'charts')
            os.makedirs(chart_temp_dir, exist_ok=True)
            chart_timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')

            charts_added = False

            # 1. 업무 상태 파이 차트 (완료 vs 진행중)
            completed_count = len(report.completedTasks) if report.completedTasks else 0
            ongoing_count = len(report.ongoingTasks) if report.ongoingTasks else 0

            if completed_count > 0 or ongoing_count > 0:
                try:
                    plt.switch_backend('Agg')
                    plt.close('all')

                    fig, ax, font_prop = create_matplotlib(figsize=(8, 5))

                    # 파이 차트 데이터
                    sizes = [completed_count, ongoing_count]
                    labels = [f'완료 ({completed_count}건)', f'진행중 ({ongoing_count}건)']
                    colors = ['#4CAF50', '#2196F3']  # 녹색(완료), 파랑(진행중)
                    explode = (0.05, 0)

                    wedges, texts, autotexts = ax.pie(
                        sizes,
                        labels=labels,
                        colors=colors,
                        autopct='%1.1f%%',
                        startangle=90,
                        explode=explode,
                        textprops={'fontproperties': font_prop}
                    )
                    plt.setp(autotexts, size=10, weight="bold")
                    plt.setp(texts, size=11)
                    ax.set_title(f'{report_type_label} 업무 현황', fontproperties=font_prop, fontsize=14, pad=20)

                    chart_path_status = os.path.join(chart_temp_dir, f'status_chart_{chart_timestamp}.png')
                    plt.tight_layout()
                    fig.savefig(chart_path_status, dpi=150, bbox_inches='tight', facecolor='white')
                    plt.close(fig)

                    if os.path.exists(chart_path_status):
                        if not charts_added:
                            doc.add_heading("6. 업무 현황 분석", level=2)
                            add_spacing(5)
                            charts_added = True
                        doc.add_image(chart_path_status)
                        add_spacing(10)
                        logger.info(f"📊 업무 상태 파이 차트 생성 완료: {chart_path_status}")
                except Exception as chart_err:
                    logger.warning(f"업무 상태 차트 생성 실패: {chart_err}")

            # 2. 진행중인 업무 진행률 막대 차트
            if report.ongoingTasks and len(report.ongoingTasks) > 0:
                try:
                    plt.switch_backend('Agg')
                    plt.close('all')

                    fig, ax, font_prop = create_matplotlib(figsize=(10, 6))

                    # 진행률 데이터 추출
                    task_names = []
                    progress_values = []
                    for task in report.ongoingTasks[:10]:  # 최대 10개
                        name = task.taskName or "미정"
                        # 업무명이 너무 길면 줄임
                        if len(name) > 20:
                            name = name[:17] + "..."
                        task_names.append(name)

                        # 진행률 파싱 (예: "50%" -> 50)
                        progress_str = task.progress or "0%"
                        try:
                            progress = int(progress_str.replace('%', '').strip())
                        except ValueError:
                            progress = 0
                        progress_values.append(min(max(progress, 0), 100))  # 0-100 범위로 제한

                    if task_names and progress_values:
                        # 가로 막대 차트
                        y_pos = range(len(task_names))
                        colors = ['#4CAF50' if p >= 80 else '#FFC107' if p >= 50 else '#FF5722' for p in progress_values]

                        bars = ax.barh(y_pos, progress_values, color=colors, height=0.6)
                        ax.set_yticks(y_pos)
                        ax.set_yticklabels(task_names, fontproperties=font_prop, fontsize=10)
                        ax.set_xlabel('진행률 (%)', fontproperties=font_prop, fontsize=11)
                        ax.set_title('진행 중인 업무 현황', fontproperties=font_prop, fontsize=14, pad=15)
                        ax.set_xlim(0, 100)
                        ax.grid(True, axis='x', linestyle='--', alpha=0.5)

                        # 진행률 값 표시
                        for bar, val in zip(bars, progress_values):
                            ax.text(val + 2, bar.get_y() + bar.get_height()/2,
                                   f'{val}%', va='center', fontproperties=font_prop, fontsize=9)

                        chart_path_progress = os.path.join(chart_temp_dir, f'progress_chart_{chart_timestamp}.png')
                        plt.tight_layout()
                        fig.savefig(chart_path_progress, dpi=150, bbox_inches='tight', facecolor='white')
                        plt.close(fig)

                        if os.path.exists(chart_path_progress):
                            if not charts_added:
                                doc.add_heading("6. 업무 현황 분석", level=2)
                                add_spacing(5)
                                charts_added = True
                            doc.add_image(chart_path_progress)
                            add_spacing(10)
                            logger.info(f"📊 진행률 막대 차트 생성 완료: {chart_path_progress}")
                except Exception as chart_err:
                    logger.warning(f"진행률 차트 생성 실패: {chart_err}")

            # 3. 우선순위 분포 차트 (다음 기간 계획 기준)
            if report.nextPeriodPlans and len(report.nextPeriodPlans) > 0:
                try:
                    plt.switch_backend('Agg')
                    plt.close('all')

                    fig, ax, font_prop = create_matplotlib(figsize=(7, 5))

                    # 우선순위별 개수 집계
                    priority_counts = {'높음': 0, '중간': 0, '낮음': 0, '기타': 0}
                    for plan in report.nextPeriodPlans:
                        priority = (plan.priority or '').strip()
                        if priority in ['높음', '상', 'high', 'High', 'HIGH']:
                            priority_counts['높음'] += 1
                        elif priority in ['중간', '중', 'medium', 'Medium', 'MEDIUM', '보통']:
                            priority_counts['중간'] += 1
                        elif priority in ['낮음', '하', 'low', 'Low', 'LOW']:
                            priority_counts['낮음'] += 1
                        else:
                            priority_counts['기타'] += 1

                    # 0인 항목 제외
                    filtered_priorities = {k: v for k, v in priority_counts.items() if v > 0}

                    if filtered_priorities:
                        labels = list(filtered_priorities.keys())
                        sizes = list(filtered_priorities.values())
                        color_map = {'높음': '#F44336', '중간': '#FFC107', '낮음': '#4CAF50', '기타': '#9E9E9E'}
                        colors = [color_map.get(l, '#9E9E9E') for l in labels]

                        wedges, texts, autotexts = ax.pie(
                            sizes,
                            labels=[f'{l} ({s}건)' for l, s in zip(labels, sizes)],
                            colors=colors,
                            autopct='%1.1f%%',
                            startangle=90,
                            textprops={'fontproperties': font_prop}
                        )
                        plt.setp(autotexts, size=10, weight="bold")
                        plt.setp(texts, size=11)
                        ax.set_title(f'{next_period_label} 계획 우선순위 분포', fontproperties=font_prop, fontsize=14, pad=20)

                        chart_path_priority = os.path.join(chart_temp_dir, f'priority_chart_{chart_timestamp}.png')
                        plt.tight_layout()
                        fig.savefig(chart_path_priority, dpi=150, bbox_inches='tight', facecolor='white')
                        plt.close(fig)

                        if os.path.exists(chart_path_priority):
                            if not charts_added:
                                doc.add_heading("6. 업무 현황 분석", level=2)
                                add_spacing(5)
                                charts_added = True
                            doc.add_image(chart_path_priority)
                            add_spacing(10)
                            logger.info(f"📊 우선순위 분포 차트 생성 완료: {chart_path_priority}")
                except Exception as chart_err:
                    logger.warning(f"우선순위 차트 생성 실패: {chart_err}")

            if charts_added:
                add_spacing(15)

        except Exception as chart_section_err:
            logger.warning(f"차트 섹션 생성 중 오류 (문서 생성은 계속): {chart_section_err}")

        # ============================================
        # 서명란 (공문서 스타일)
        # ============================================
        doc.add_paragraph("")
        doc.add_paragraph("─" * 50)
        doc.add_paragraph("")
        signature_header = ["작성자", "검토자", "승인자"]
        signature_data = [
            [metadata.author or "(서명)", "(서명)", "(서명)"],
            ["", "", ""],
        ]
        add_styled_table(data=signature_data, header=signature_header)
        add_spacing(10)

        # 문서 끝 표시
        doc.add_paragraph("")
        doc.add_paragraph("- 끝 -")

        # 임시 파일로 저장
        ext_map = {'pdf': '.pdf', 'docx': '.docx', 'hwpx': '.hwpx'}
        ext = ext_map.get(output_format, '.pdf')

        # 임시 디렉토리에 파일 생성
        temp_dir = os.path.join(os.path.expanduser('~'), '.airun', 'temp')
        os.makedirs(temp_dir, exist_ok=True)

        timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
        filename = f"weekly_report_{timestamp}{ext}"
        output_path = os.path.join(temp_dir, filename)

        # 문서 저장
        doc.save(output_path)

        if not os.path.exists(output_path):
            raise HTTPException(status_code=500, detail="문서 생성에 실패했습니다")

        file_size = os.path.getsize(output_path)
        logger.info(f"✅ 주간 보고서 변환 완료: {output_path} ({file_size:,} bytes)")

        # MIME 타입 설정
        mime_types = {
            'pdf': 'application/pdf',
            'docx': 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'hwpx': 'application/hwp+zip'
        }
        mime_type = mime_types.get(output_format, 'application/octet-stream')

        # 파일 응답 반환
        return FileResponse(
            path=output_path,
            filename=filename,
            media_type=mime_type,
            headers={
                "Content-Disposition": f"attachment; filename*=UTF-8''{filename}"
            }
        )

    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"❌ 주간 보고서 변환 실패: {str(e)}")
        logger.error(traceback.format_exc())
        raise HTTPException(status_code=500, detail=f"보고서 변환 실패: {str(e)}")

# =============================================================================
# 메인 실행 함수
# =============================================================================

def main():
    """메인 실행 함수 - RAG 서비스와 동일한 패턴"""
    try:
        # 프로세스 이름 설정
        try:
            from setproctitle import setproctitle
            setproctitle("airun-report-server")
        except ImportError:
            pass  # setproctitle이 없어도 계속 진행
        
        # 환경 변수에서 설정 읽기
        host = os.getenv("REPORT_SERVER_HOST", "127.0.0.1")  # 로컬호스트로 변경 (보안)
        port = int(os.getenv("REPORT_SERVER_PORT", "5620"))
        workers = int(os.getenv("REPORT_SERVER_WORKERS", "1"))  # 단일 워커로 유지
        
        logger.info(f"🚀 AIRUN Report Server v2.0 시작...")
        logger.info(f"   - Host: {host}")
        logger.info(f"   - Port: {port}")
        logger.info(f"   - Workers: {workers}")
        logger.info(f"   - Health Check: http://{host}:{port}/health")
        logger.info(f"   - API Documentation: http://{host}:{port}/docs")
        logger.info(f"   - Features: 최적화된 보고서 생성 엔진")
        logger.info(f"   - Pattern: RAG 서비스와 동일한 아키텍처")
        
        # uvicorn 서버 실행
        uvicorn.run(
            app,  # 모듈 경로 대신 앱 객체 직접 사용
            host=host,
            port=port,
            workers=workers,
            log_level="info",
            reload=False,
            access_log=True
        )
        
    except Exception as e:
        logger.error(f"❌ 서버 시작 실패: {str(e)}")
        logger.error(traceback.format_exc())
        sys.exit(1)
    finally:
        logger.info("🛑 AIRUN Report Server 종료")

if __name__ == "__main__":
    main() 