import LanguageDetector from './LanguageDetector.js';
import DependencyAnalyzer from './DependencyAnalyzer.js';
import { aiChat } from '../../../aiFeatures.js';

class TaskAnalyzer {
    constructor() {
        this.languageDetector = new LanguageDetector();
        this.dependencyAnalyzer = new DependencyAnalyzer();
    }

    async analyzeTask(request) {
        const language = await this.languageDetector.detect(request);
        const prompt = this.getAnalysisPrompt(language, request);
        
        try {
            // 직접 aiChat 호출 (포트 없이 독립적 실행)
            const fullPrompt = prompt + " Task: " + request + "\n\n중요: 응답은 반드시 다음과 같은 JSON 형식이어야 합니다:\n" + 
            `{
                "originalRequest": "사용자의 원본 요청",
                "paths": {
                    "inputs": ["입력 경로 1", "입력 경로 2", ...],
                    "output": "출력 경로"
                },
                "steps": [
                    {
                        "id": "0",
                        "number": 1,
                        "description": "작업 설명",
                        "dependencies": []
                    },
                    ...
                ]
            }`;

            const analysis = await aiChat([{role: 'user', content: fullPrompt}], {}, undefined, [], null, false);
            if (!analysis || typeof analysis !== 'string') {
                throw new Error('aiChat에서 유효한 응답을 받지 못했습니다');
            }
            
            return this.processAnalysisResponse(analysis, request);
        } catch (error) {
            console.error('aiChat 호출 실패, 간단한 분석으로 대체:', error.message);
            return this.createSimpleTaskAnalysis(request);
        }
    }
    
    
    processAnalysisResponse(analysis, originalRequest) {
        // JSON 파싱 시도
        let analysisData;
        try {
            // JSON 문자열 찾기
            const jsonMatch = analysis.match(/({[\s\S]*})/);
            if (jsonMatch) {
                analysisData = JSON.parse(jsonMatch[1]);
            } else {
                throw new Error('JSON 형식의 응답을 찾을 수 없습니다');
            }
        } catch (error) {
            console.warn('JSON 파싱 실패, 텍스트 응답을 구조화된 형식으로 변환 시도:', error);
            
            // 텍스트 응답을 파싱하여 구조화된 형식으로 변환
            analysisData = this.parseTextAnalysis(analysis, originalRequest);
        }
        
        // 의존성 분석 및 단계 정보 추가
        const steps = this.dependencyAnalyzer.parseTaskBreakdown(analysisData.steps);
        
        // 각 단계에 분석 결과와 작업 설명을 context에 추가
        steps.forEach(step => {
            step.context = {
                ...step.context,
                originalRequest: analysisData.originalRequest,
                paths: analysisData.paths,
                taskSteps: analysisData.steps.map(s => ({
                    ...s,
                    status: 'pending',
                    result: null,
                    startTime: null,
                    endTime: null,
                    error: null
                }))
            };
        });
        
        return steps;
    }
    
    createSimpleTaskAnalysis(request) {
        // API 서버 호출 실패시 사용하는 간단한 작업 분석
        const analysisData = {
            originalRequest: request,
            paths: {
                inputs: [process.cwd()],
                output: process.cwd() + '/output'
            },
            steps: [
                {
                    id: "0",
                    number: 1,
                    description: request,
                    dependencies: []
                }
            ]
        };
        
        // 의존성 분석 및 단계 정보 추가
        const steps = this.dependencyAnalyzer.parseTaskBreakdown(analysisData.steps);
        
        // 각 단계에 분석 결과와 작업 설명을 context에 추가
        steps.forEach(step => {
            step.context = {
                ...step.context,
                originalRequest: analysisData.originalRequest,
                paths: analysisData.paths,
                taskSteps: analysisData.steps.map(s => ({
                    ...s,
                    status: 'pending',
                    result: null,
                    startTime: null,
                    endTime: null,
                    error: null
                }))
            };
        });
        
        return steps;
    }

    parseTextAnalysis(analysis, originalRequest) {
        const lines = analysis.split('\n');
        const paths = {
            inputs: [],
            output: ''
        };
        const steps = [];
        
        let isCollectingSteps = false;
        
        for (let i = 0; i < lines.length; i++) {
            const line = lines[i].trim();
            
            // 입력 경로 파싱
            if (line.match(/입력\s*경로:|Input\s*Paths?:/i)) {
                const pathLine = line.replace(/^[^:]+:\s*/, '');
                paths.inputs = pathLine
                    .split(/[,，]/)
                    .map(p => p.trim())
                    .filter(p => p && !p.includes('[') && !p.includes(']'));
                continue;
            }
            
            // 출력 경로 파싱
            if (line.match(/출력\s*경로:|Output\s*Path?:/i)) {
                const pathLine = line.replace(/^[^:]+:\s*/, '').trim();
                if (!pathLine.includes('[') && !pathLine.includes(']')) {
                    paths.output = pathLine;
                }
                continue;
            }
            
            // 작업 단계 파싱
            if (line.match(/^작업\s*단계:|Task\s*Steps:/i)) {
                isCollectingSteps = true;
                continue;
            }
            
            if (isCollectingSteps && line.match(/^\d+\./)) {
                const stepNumber = steps.length;
                const description = line.replace(/^\d+\.\s*/, '').trim();
                steps.push({
                    id: String(stepNumber),
                    number: stepNumber + 1,
                    description: description,
                    dependencies: []
                });
            }
            
            // 의존성 섹션에서 중단
            if (line.match(/^의존성:|Dependencies:/i)) {
                isCollectingSteps = false;
            }
        }
        
        return {
            originalRequest,
            paths,
            steps
        };
    }

    getAnalysisPrompt(language, request) {
        if (language === 'ko') {
            return `당신은 작업을 분석하여 실행 가능한 단계로 나누는 AI 분석가입니다.
다음 작업을 분석하여 수행해야 할 주요 단계들을 나열해주세요.

작업: ${request}

다음 형식으로 응답해주세요:

경로 정보:
- 입력 경로: [작업에서 사용되는 모든 입력 경로를 전체 경로로 나열]
- 출력 경로: [작업의 결과물이 저장될 전체 경로]

작업 단계:
1. [구체적인 기능 단위 작업]
2. [구체적인 기능 단위 작업]
...

의존성:
- 작업 간의 의존성을 명시

주의사항:
1. 가장 중요한 원칙: 작업을 최소한의 단계로 나누세요
   - 모든 연관된 작업은 반드시 하나의 단계로 통합하세요
   - 데이터 수집, 처리, 출력은 절대로 분리하지 마세요
   - 중간 결과를 저장하거나 전달하는 단계를 만들지 마세요
   - 작업이 단순하다면 하나의 단계로 충분합니다
   
2. 각 단계는 완전히 독립적인 의미 단위여야 합니다
   - 단순히 이전 단계의 결과를 사용하는 것은 독립적인 단계가 아닙니다
   - 데이터 처리나 변환 없이 단순히 출력만 하는 단계는 만들지 마세요
   
3. 기술적인 세부사항은 제외하세요
   - 라이브러리 임포트, 변수 설정 등은 제외
   - 파일 열기/닫기, 예외 처리 등의 상세 구현은 제외

4. 작업 분할 시 다음을 고려하세요:
   - 작업이 단순하다면 무조건 하나의 단계로 처리하세요
   - 여러 단계로 나눌 때는 매우 신중하게 판단하세요
   - 확실하지 않다면 하나의 단계로 통합하세요

5. 경로 정보는 반드시 전체 경로로 명시하세요:
   - 입력 경로는 작업에서 읽어들일 모든 파일이나 디렉토리의 전체 경로
   - 출력 경로는 결과물이 저장될 파일이나 디렉토리의 전체 경로
   - 상대 경로는 사용하지 마세요
   
예시:
[매우 잘못된 예]
경로 정보:
- 입력 경로: ./input
- 출력 경로: ./output/result.txt
(이유: 상대 경로 사용)

[올바른 예]
경로 정보:
- 입력 경로: /home/user/documents/input
- 출력 경로: /home/user/documents/output/result.txt
(이유: 전체 경로 사용)

[매우 잘못된 예]
1. 파일 목록 가져오기
2. js 파일 찾기
3. 결과 출력하기
(이유: 하나의 작업을 불필요하게 세 단계로 나눔)

[잘못된 예]
1. 파일 목록 가져오기
2. js 파일만 필터링하여 출력하기
(이유: 데이터 수집과 처리를 분리함)

[올바른 예]
1. 현재 디렉토리에서 js 파일만 찾아서 출력하기
(이유: 데이터 수집, 필터링, 출력을 하나의 단계로 통합)`;
        }
        
        return `You are an AI analyst who breaks down tasks into executable steps.
Please analyze the following task and list the main steps that need to be performed.

Task: ${request}

Please respond in the following format:

Path Information:
- Input Paths: [List all input paths used in the task with full paths]
- Output Path: [Full path where the task results will be saved]

Task Steps:
1. [Specific functional unit task]
2. [Specific functional unit task]
...

Dependencies:
- Specify dependencies between tasks

Important Notes:
1. Most important principle: Break down tasks into minimum steps
   - Always integrate all related operations into one step
   - Never separate data collection, processing, and output
   - Don't create steps just to store or pass intermediate results
   - If the task is simple, one step is enough
   
2. Each step must be a completely independent unit of meaning
   - Simply using results from previous step is not an independent step
   - Don't create steps that only output data without processing
   
3. Exclude technical details
   - Exclude library imports, variable setup, etc.
   - Exclude implementation details like file open/close, error handling

4. When breaking down tasks, consider:
   - If the task is simple, always handle it in one step
   - Be very careful when deciding to split into multiple steps
   - When in doubt, integrate into one step

5. Always specify paths as full paths:
   - Input paths should be full paths for all files/directories to be read
   - Output path should be full path where results will be saved
   - Never use relative paths
   
Example:
[Very Incorrect]
Path Information:
- Input Paths: ./input
- Output Path: ./output/result.txt
(Reason: Uses relative paths)

[Correct]
Path Information:
- Input Paths: /home/user/documents/input
- Output Path: /home/user/documents/output/result.txt
(Reason: Uses full paths)

[Very Incorrect]
1. Get file list
2. Find js files
3. Print results
(Reason: Unnecessarily splits one task into three steps)

[Incorrect]
1. Get file list
2. Filter and print js files
(Reason: Separates data collection from processing)

[Correct]
1. Find and print js files from current directory
(Reason: Integrates data collection, filtering, and output into one step)`;
    }
}

export default TaskAnalyzer; 