#!/usr/bin/env python3
"""
AIRUN 웹검색 서비스 - 진짜 최적화 버전 (Monkey Patch)
- 포트: 5610
- 기능: 구글, 네이버, 다음 통합 웹검색
- 특징: utils.py 함수들을 monkey patch로 최적화
"""

import asyncio
import json
import logging
import os
import sys
import time
import traceback
import warnings
import threading
import signal
from contextlib import asynccontextmanager
from datetime import datetime
from pathlib import Path
from typing import Dict, List, Optional, Any
from concurrent.futures import ThreadPoolExecutor
from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as EC
from selenium.webdriver.common.by import By
from urllib.parse import urljoin
import aiohttp

# 외부 라이브러리 경고 억제
import warnings
warnings.filterwarnings("ignore", category=UserWarning)

# trafilatura 경고 억제 (구글 검색 시 발생)
import logging as trafilatura_logging
trafilatura_logging.getLogger('trafilatura.core').setLevel(logging.ERROR)

# urllib3 연결 경고 억제 (Selenium WebDriver 연결 시 발생)
trafilatura_logging.getLogger('urllib3.connectionpool').setLevel(logging.ERROR)

# Selenium 관련 경고 억제
trafilatura_logging.getLogger('selenium').setLevel(logging.ERROR)

# 프로젝트 루트 경로 설정
project_root = Path(__file__).parent.parent.parent
sys.path.insert(0, str(project_root))

# utils.py를 직접 임포트
import utils

# 필요한 패키지들 임포트
try:
    import uvicorn
    from fastapi import FastAPI, HTTPException, Query
    from fastapi.middleware.cors import CORSMiddleware
    from pydantic import BaseModel, Field
    import requests
    import urllib.parse
    import re
    from bs4 import BeautifulSoup
except ImportError as e:
    print(f"필요한 패키지가 설치되지 않았습니다: {e}")
    sys.exit(1)

# 로깅 설정 - 개별 로그 파일에 기록
log_dir = Path.home() / ".airun" / "logs"
log_dir.mkdir(parents=True, exist_ok=True)
log_file = log_dir / "airun-websearch.log"

# 로거 설정
logger = logging.getLogger("airun-websearch-server")
logger.setLevel(logging.INFO)

# 파일 핸들러 생성
file_handler = logging.FileHandler(log_file, encoding='utf-8')
file_handler.setLevel(logging.INFO)

# 콘솔 핸들러 생성 (선택적)
console_handler = logging.StreamHandler()
console_handler.setLevel(logging.WARNING)  # 콘솔에는 WARNING 이상만 출력

# 포맷터 설정
formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
file_handler.setFormatter(formatter)
console_handler.setFormatter(formatter)

# 핸들러 추가
logger.addHandler(file_handler)
logger.addHandler(console_handler)

# 다른 로거들의 중복 출력 방지
logger.propagate = False

# 루트 로거의 핸들러들 제거 (중복 방지)
root_logger = logging.getLogger()
for handler in root_logger.handlers[:]:
    root_logger.removeHandler(handler)

# 모든 경고 메시지 무시
warnings.filterwarnings('ignore')

# 전역 변수들
driver_pool: Optional['UltraFastDriverPool'] = None
websearch_cache: Optional['WebSearchCache'] = None
original_get_selenium_driver = None

# 서버 통계
server_stats = {
    "start_time": None,
    "total_requests": 0,
    "successful_searches": 0,
    "failed_searches": 0,
    "cache_hits": 0,
    "cache_misses": 0,
    "avg_response_time": 0.0,
    "driver_reuse_count": 0
}

# =============================================================================
# 초고속 드라이버 풀 (Monkey Patch 방식)
# =============================================================================

class UltraFastDriverPool:
    """초고속 드라이버 풀 (utils.py monkey patch)"""
    
    def __init__(self, pool_size: int = 3):
        self.pool_size = pool_size
        self.drivers = []
        self.available_drivers = []
        self.lock = threading.Lock()
        self.initialized = False
        
    def initialize(self) -> bool:
        """드라이버 풀 초기화 및 monkey patch 적용"""
        try:
            logger.info(f"🚀 초고속 드라이버 풀 초기화 시작 (크기: {self.pool_size})")
            
            # 원본 함수 백업
            global original_get_selenium_driver
            original_get_selenium_driver = utils.get_selenium_driver
            
            # 미리 여러 개의 드라이버 생성
            for i in range(self.pool_size):
                try:
                    logger.info(f"🔧 드라이버 {i+1}/{self.pool_size} 생성 중...")
                    driver = original_get_selenium_driver()
                    
                    if driver:
                        self.drivers.append(driver)
                        self.available_drivers.append(driver)
                        logger.info(f"✅ 드라이버 {i+1} 생성 완료")
                    else:
                        logger.warning(f"⚠️ 드라이버 {i+1} 생성 실패")
                        
                except Exception as e:
                    logger.error(f"❌ 드라이버 {i+1} 생성 중 오류: {e}")
            
            self.initialized = len(self.drivers) > 0
            
            if self.initialized:
                # Monkey patch 적용!
                utils.get_selenium_driver = self._get_driver_from_pool
                logger.info(f"✅ Monkey Patch 적용 완료: utils.get_selenium_driver -> driver_pool")
                logger.info(f"✅ 초고속 드라이버 풀 초기화 완료: {len(self.drivers)}개 드라이버 준비")
            else:
                logger.error("❌ 드라이버 풀 초기화 실패")
            
            return self.initialized
            
        except Exception as e:
            logger.error(f"❌ 드라이버 풀 초기화 실패: {e}")
            logger.error(f"상세 오류:\n{traceback.format_exc()}")
            return False
    
    def _get_driver_from_pool(self):
        """개선된 Monkey patch 함수: 풀에서 드라이버 가져오기"""
        # 즉시 사용 가능한 드라이버 확인
        with self.lock:
            if self.available_drivers:
                driver = self.available_drivers.pop(0)
                server_stats["driver_reuse_count"] += 1
                logger.debug(f"🚗 드라이버 풀에서 할당 (재사용 #{server_stats['driver_reuse_count']}, 남은: {len(self.available_drivers)})")
                return driver
        
        # 풀이 비어있으면 대기 없이 바로 새 드라이버 생성
        logger.info("⚡ 드라이버 풀이 비어있음 - 새 드라이버 즉시 생성")
        try:
            new_driver = original_get_selenium_driver()
            server_stats["driver_create_count"] = server_stats.get("driver_create_count", 0) + 1
            return new_driver
        except Exception as e:
            logger.error(f"새 드라이버 생성 실패: {e}")
            return None
    
    def return_driver(self, driver):
        """개선된 드라이버 풀 반환"""
        if not driver:
            return
            
        try:
            # 풀에서 생성된 드라이버인지 확인
            if driver in self.drivers:
                with self.lock:
                    if driver not in self.available_drivers:
                        # 드라이버 상태 확인 후 반환
                        try:
                            driver.current_url  # 드라이버가 살아있는지 확인
                            self.available_drivers.append(driver)
                            logger.debug(f"🔄 드라이버 풀에 반환 (사용 가능: {len(self.available_drivers)})")
                        except:
                            # 드라이버가 죽어있으면 풀에서 제거
                            if driver in self.drivers:
                                self.drivers.remove(driver)
                            logger.warning("💀 죽은 드라이버 발견 - 풀에서 제거")
            else:
                # 풀에서 생성되지 않은 드라이버는 바로 종료
                try:
                    driver.quit()
                    logger.debug("🗑️ 임시 드라이버 종료")
                except:
                    pass
        except Exception as e:
            logger.error(f"드라이버 반환 중 오류: {e}")
            # 오류 발생 시 드라이버 강제 종료
            try:
                driver.quit()
            except:
                pass
    
    def cleanup(self):
        """드라이버 풀 정리 및 monkey patch 복원"""
        logger.info("🧹 드라이버 풀 정리 시작")
        
        # Monkey patch 복원
        if original_get_selenium_driver:
            utils.get_selenium_driver = original_get_selenium_driver
            logger.info("✅ Monkey Patch 복원 완료")
        
        # 드라이버들 정리
        with self.lock:
            for driver in self.drivers:
                try:
                    if hasattr(driver, 'quit'):
                        driver.quit()
                except Exception as e:
                    logger.error(f"드라이버 정리 중 오류: {e}")
            
            self.drivers.clear()
            self.available_drivers.clear()
            self.initialized = False
        logger.info("✅ 드라이버 풀 정리 완료")

# =============================================================================
# 초고속 웹검색 엔진 (utils.py 직접 사용)
# =============================================================================

class WebSearch:
    """utils.py 함수들을 직접 사용하는 웹검색 엔진"""
    
    def __init__(self, driver_pool: UltraFastDriverPool):
        self.driver_pool = driver_pool
    
    def search_duckduckgo(self, query: str, max_results: int = None) -> List[Dict[str, Any]]:
        """DuckDuckGo 검색을 수행하고 결과를 반환합니다."""
        try:
            results = []
            max_results = max_results or 10
            
            # DuckDuckGo 검색 URL
            headers = {
                'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36'
            }
            
            # DuckDuckGo HTML 검색
            search_url = f"https://html.duckduckgo.com/html/?q={urllib.parse.quote_plus(query)}"
            logger.debug(f"DuckDuckGo 검색 URL: {search_url}")
            
            response = requests.get(search_url, headers=headers, timeout=10)
            response.raise_for_status()
            
            soup = BeautifulSoup(response.text, 'html.parser')
            
            # DuckDuckGo HTML 결과 파싱
            search_results = soup.find_all('div', class_='result')
            logger.debug(f"DuckDuckGo 결과 블록 수: {len(search_results)}")
            
            for result in search_results[:max_results]:
                try:
                    # 제목과 URL 추출
                    link_elem = result.find('a', class_='result__a')
                    if not link_elem:
                        continue
                        
                    title = self._clean_text(link_elem.get_text().strip())
                    url = link_elem.get('href', '')
                    
                    # URL 정리
                    if url.startswith('/l/?uddg='):
                        # DuckDuckGo redirect URL 디코딩
                        url = urllib.parse.unquote(url.split('uddg=')[1])
                    
                    # 설명 추출
                    desc_elem = result.find('a', class_='result__snippet')
                    description = ""
                    if desc_elem:
                        description = self._clean_text(desc_elem.get_text().strip())
                    
                    # URL 검증
                    if not self._is_valid_url(url):
                        continue
                    
                    # 제목이나 URL이 비어있는 경우 스킵
                    if not title or not url:
                        continue
                    
                    # 중복 제거
                    if not any(r["url"] == url for r in results):
                        results.append({
                            'title': title,
                            'url': url,
                            'description': description or title,
                            'type': 'web',
                            'engine': 'duckduckgo'
                        })
                        
                except Exception as e:
                    logger.debug(f"DuckDuckGo 결과 처리 중 오류: {str(e)}")
                    continue
            
            logger.info(f"✅ DuckDuckGo 검색 완료: {len(results)}개 결과")
            return results
            
        except Exception as e:
            logger.error(f"❌ DuckDuckGo 검색 중 오류: {str(e)}")
            return []
    
    def _clean_text(self, text):
        """검색용 텍스트 정리"""
        if not text:
            return ""
        # HTML 태그 제거
        text = re.sub(r'<[^>]+>', '', text)
        # 특수문자 및 연속 공백 정리  
        text = re.sub(r'\s+', ' ', text)
        return text.strip()
    
    def _is_valid_url(self, url):
        """URL 유효성 검사"""
        return url and url.startswith(('http://', 'https://'))
        
    async def search(self, query: str, max_results: int = 10, engine: str = "auto") -> List[Dict[str, Any]]:
        """간소화된 웹검색 - SearXNG 우선, 실패 시 DuckDuckGo fallback"""
        start_time = time.time()
        max_results = 10
        try:
            logger.info(f"🔍 웹검색 시작: '{query}' (최대 {max_results}개)")
            
            # 1단계: SearXNG 검색 시도
            logger.info("🎯 1단계: SearXNG 검색 시도")
            searxng_results = await self._search_searxng_optimized(query, max_results, silent=False)
            
            if searxng_results and len(searxng_results) > 0:
                # 결과 품질 검증 (유효한 URL이 있는지 확인)
                valid_results = [r for r in searxng_results if r.get('url') and r.get('title')]
                if valid_results:
                    # SearXNG 검색 성공
                    elapsed = time.time() - start_time
                    logger.info(f"✅ SearXNG 검색 완료: {len(valid_results)}개 결과 ({elapsed:.3f}초)")
                    return valid_results
                else:
                    logger.warning("⚠️ SearXNG 결과가 있지만 유효하지 않음, fallback 진행")
            
            # 2단계: SearXNG 실패 시 DuckDuckGo fallback
            logger.info("� 2단계: SearXNG 실패, DuckDuckGo로 fallback")
            duckduckgo_results = self.search_duckduckgo(query, max_results)
            
            if duckduckgo_results and len(duckduckgo_results) > 0:
                elapsed = time.time() - start_time
                logger.info(f"✅ DuckDuckGo 검색 완료: {len(duckduckgo_results)}개 결과 ({elapsed:.3f}초)")
                return duckduckgo_results
            
            # 모든 검색 실패
            elapsed = time.time() - start_time
            logger.error(f"❌ 모든 검색 엔진 실패 ({elapsed:.3f}초) - 빈 배열 반환")
            logger.error(f"❌ 검색 실패 상세: SearXNG={len(searxng_results) if searxng_results else 0}개, DuckDuckGo={len(duckduckgo_results) if duckduckgo_results else 0}개")
            return []
                
        except Exception as e:
            execution_time = time.time() - start_time
            logger.error(f"❌ 웹검색 실패: {str(e)} ({execution_time:.2f}초)")
            logger.error(f"상세 오류:\n{traceback.format_exc()}")
            return []
    

    

    
    def _extract_searxng_json(self, json_data: dict) -> List[Dict[str, Any]]:
        """
        SearXNG JSON API 응답에서 검색 결과를 추출
        
        JSON 응답 형식:
        {
            "query": "검색어",
            "number_of_results": 0,
            "results": [
                {
                    "title": "제목",
                    "url": "https://...",
                    "content": "설명",
                    "engine": "duckduckgo",
                    "score": 1.0,
                    "category": "general",
                    ...
                },
                ...
            ],
            "answers": [],
            "corrections": [],
            "infoboxes": [],
            "suggestions": [],
            "unresponsive_engines": []
        }
        """
        import logging
        logger = logging.getLogger("airun-websearch-server")
        out: List[Dict[str, Any]] = []
        
        try:
            # results 배열 추출
            results = json_data.get("results", [])
            
            if not results:
                logger.warning("[SearXNG JSON] 검색 결과가 비어있음")
                # 응답하지 않은 엔진 정보 로깅
                unresponsive = json_data.get("unresponsive_engines", [])
                if unresponsive:
                    logger.warning(f"[SearXNG JSON] 응답하지 않은 엔진: {unresponsive}")
                return []
            
            logger.info(f"[SearXNG JSON] {len(results)}개 결과 파싱 시작")
            
            for i, result in enumerate(results):
                try:
                    # 필수 필드 추출
                    title = result.get("title", "").strip()
                    url = result.get("url", "").strip()
                    
                    # 제목과 URL 유효성 검사
                    if not title or not url:
                        logger.debug(f"[SearXNG JSON] 결과 {i+1}: 제목 또는 URL 누락")
                        continue
                    
                    # URL 유효성 검사
                    if not self._is_valid_url(url):
                        logger.debug(f"[SearXNG JSON] 결과 {i+1}: 유효하지 않은 URL - {url}")
                        continue
                    
                    # 설명 추출 (content 필드 사용)
                    description = result.get("content", "").strip()
                    if not description:
                        description = title
                    
                    # 공백 정리
                    title = " ".join(title.split())
                    description = " ".join(description.split())
                    
                    # 추가 메타데이터 (선택적)
                    engine = result.get("engine", "searxng")
                    score = result.get("score", 0.0)
                    category = result.get("category", "general")
                    published_date = result.get("publishedDate")
                    thumbnail = result.get("thumbnail", "")
                    
                    # 결과 데이터 생성
                    result_data = {
                        "title": title,
                        "url": url,
                        "description": description,
                        "engine": engine,
                        "score": score,
                        "category": category,
                    }
                    
                    # 선택적 필드 추가
                    if published_date:
                        result_data["published_date"] = published_date
                    if thumbnail:
                        result_data["thumbnail"] = thumbnail
                    
                    out.append(result_data)
                    
                    # 처음 3개 결과는 상세 로그 출력
                    if i < 3:
                        logger.info(f"[SearXNG JSON] 결과 {i+1}:")
                        logger.info(f"  제목: {title[:50]}{'...' if len(title) > 50 else ''}")
                        logger.info(f"  URL: {url[:60]}{'...' if len(url) > 60 else ''}")
                        logger.info(f"  엔진: {engine}, 점수: {score:.2f}")
                        
                except Exception as e:
                    logger.warning(f"[SearXNG JSON] 결과 {i+1} 파싱 오류: {e}")
                    continue
            
            # 추가 정보 로깅
            query = json_data.get("query", "")
            suggestions = json_data.get("suggestions", [])
            answers = json_data.get("answers", [])
            
            if suggestions:
                logger.info(f"[SearXNG JSON] 제안 검색어: {suggestions[:3]}")
            if answers:
                logger.info(f"[SearXNG JSON] 즉답: {answers[:2]}")
            
            logger.info(f"[SearXNG JSON] 최종 파싱 결과: {len(out)}개 항목 (쿼리: '{query}')")
            return out
            
        except Exception as e:
            logger.error(f"[SearXNG JSON] 파싱 전체 실패: {e}")
            import traceback
            logger.error(f"[SearXNG JSON] 상세 오류:\n{traceback.format_exc()}")
            return []

    def _extract_searxng_heuristic(self, soup, base_url) -> List[Dict[str, Any]]:
        """선택자 기반 파싱 실패 시 구조 기반 휴리스틱 파싱"""
        import logging
        logger = logging.getLogger("airun-websearch-server")
        out = []
        
        try:
            # 모든 h3, h4 태그를 찾아서 링크가 있는지 확인
            headers = soup.find_all(['h3', 'h4'])
            
            for header in headers:
                link = header.find('a', href=True)
                if not link:
                    continue
                    
                title = link.get_text(strip=True)
                url = link['href']
                
                # URL 처리
                if url.startswith("/"):
                    url = urljoin(base_url, url)
                elif not url.startswith(("http://", "https://")):
                    continue # 이상한 URL 무시
                    
                # 설명 찾기 (헤더 다음의 p 태그나 div 텍스트)
                description = ""
                
                # 1. 부모의 형제나 자식에서 찾기
                container = header.parent
                if container:
                    # content 클래스를 가진 요소 찾기
                    content = container.select_one(".content, .result-content, p")
                    if content:
                        description = content.get_text(strip=True)
                    else:
                        # 텍스트만 추출 (제목 제외)
                        full_text = container.get_text(" ", strip=True)
                        description = full_text.replace(title, "").strip()
                
                if not description:
                    description = title
                    
                out.append({
                    "title": title,
                    "url": url,
                    "description": description,
                    "engine": "searxng"
                })
                
            logger.info(f"[SearXNG] 휴리스틱 파싱 결과: {len(out)}개")
            return out
            
        except Exception as e:
            logger.error(f"[SearXNG] 휴리스틱 파싱 실패: {e}")
            return []

    def _extract_searxng_results(self, html: str, base_url: str) -> List[Dict[str, Any]]:
        """
        SearXNG 결과 페이지(HTML)에서 title / url / description을 추출 (실제 HTML 구조 기반)
        """
        import logging
        logger = logging.getLogger("airun-websearch-server")
        
        try:
            soup = BeautifulSoup(html, "html.parser")
            out: List[Dict[str, Any]] = []

            # 실제 SearXNG HTML 구조에 맞는 선택자
            # <article class="result result-default category-general">
            articles = soup.select("article.result")
            
            if not articles:
                logger.warning("[SearXNG] article.result 선택자로 결과를 찾을 수 없음")
                
                # --- 디버깅 정보 추가 ---
                # 1. HTML 구조 확인
                logger.info(f"[SearXNG] HTML 샘플 (처음 1000자): {html[:1000]}")
                
                # 2. 주요 태그 존재 여부 확인
                h3_tags = soup.find_all("h3")
                logger.info(f"[SearXNG] h3 태그 개수: {len(h3_tags)}")
                
                article_tags = soup.find_all("article")
                logger.info(f"[SearXNG] article 태그 개수: {len(article_tags)}")
                if article_tags:
                    logger.info(f"[SearXNG] 첫 article 클래스: {article_tags[0].get('class')}")

                div_results = soup.select("div.result")
                logger.info(f"[SearXNG] div.result 개수: {len(div_results)}")
                
                # 3. 휴리스틱 파싱 시도 (선택자 실패 시)
                if h3_tags:
                    logger.info("[SearXNG] 휴리스틱 파싱 시도 (h3 태그 기반)")
                    return self._extract_searxng_heuristic(soup, base_url)

                # 대안 선택자들 시도 (기존 로직 유지)
                alt_selectors = ["div.result", ".result-default", ".result"]
                for selector in alt_selectors:
                    articles = soup.select(selector)
                    if articles:
                        logger.info(f"[SearXNG] 대안 선택자 '{selector}'로 {len(articles)}개 결과 발견")
                        break
                
                if not articles:
                    return []
                
            logger.info(f"[SearXNG] HTML 파싱: {len(articles)}개 결과 항목 발견")
                
            for i, article in enumerate(articles):
                try:
                    # 1. 제목과 메인 URL 추출 - h3 > a 구조 사용
                    title_link = article.select_one("h3 > a")
                    if not title_link:
                        logger.debug(f"[SearXNG] 결과 {i+1}: h3 > a 링크를 찾을 수 없음")
                        continue

                    title = title_link.get_text(strip=True)
                    if not title:
                        logger.debug(f"[SearXNG] 결과 {i+1}: 제목이 비어있음")
                        continue
                        
                    # 제목 정리
                    title = " ".join(title.split())

                    # 2. URL 추출 - url_header가 우선, 없으면 제목 링크 사용
                    url_header_link = article.select_one("a.url_header")
                    if url_header_link and url_header_link.get("href"):
                        url = url_header_link.get("href").strip()
                    else:
                        url = title_link.get("href", "").strip()
                    
                    if not url:
                        logger.debug(f"[SearXNG] 결과 {i+1}: URL을 찾을 수 없음")
                        continue

                    # URL 처리 - 상대 경로를 절대 경로로 변환
                    if url.startswith("/"):
                        url = urljoin(base_url, url)
                    elif not url.startswith(("http://", "https://")):
                        url = urljoin(base_url, "/" + url)

                    # 3. 설명 추출 - p.content 사용
                    description = ""
                    content_p = article.select_one("p.content")
                    if content_p:
                        description = content_p.get_text(strip=True)
                        description = " ".join(description.split())  # 공백 정리
                    
                    # 설명이 없으면 제목을 설명으로 사용
                    if not description:
                        description = title

                    # 4. 유효성 검사
                    if not self._is_valid_url(url):
                        logger.debug(f"[SearXNG] 결과 {i+1}: 유효하지 않은 URL - {url}")
                        continue

                    # 5. 결과 데이터 생성
                    result_data = {
                        "title": title,
                        "url": url,
                        "description": description,
                        "engine": "searxng"
                    }
                    out.append(result_data)
                    
                    # 처음 3개 결과는 상세 로그 출력
                    if i < 3:
                        logger.info(f"[SearXNG] 파싱된 결과 {i+1}:")
                        logger.info(f"  제목: {title[:50]}{'...' if len(title) > 50 else ''}")
                        logger.info(f"  URL: {url[:60]}{'...' if len(url) > 60 else ''}")
                        logger.info(f"  설명: {description[:60]}{'...' if len(description) > 60 else ''}")
                        
                except Exception as e:
                    logger.warning(f"[SearXNG] 결과 {i+1} 파싱 오류: {e}")
                    continue

            logger.info(f"[SearXNG] 최종 파싱 결과: {len(out)}개 항목 추출 완료")
            return out
            
        except Exception as e:
            logger.error(f"[SearXNG] HTML 파싱 전체 실패: {e}")
            logger.error(f"[SearXNG] HTML 길이: {len(html)}자")
            return []


    async def _search_searxng_optimized(
        self, query: str, max_results: int, silent: bool = False
    ) -> List[Dict[str, Any]]:
        """
        SearXNG 공식 API 기반 검색 (JSON 응답 우선, HTML fallback)
        - format=json 파라미터로 JSON 응답 요청
        - JSON 파싱 실패 시 HTML 파싱으로 fallback
        """
        import logging
        logger = logging.getLogger("airun-websearch-server")
        if not silent:
            logger.info(f"🚀 SearXNG API 검색 시작: '{query}'")

        base_url = getattr(self, "searxng_base", "http://localhost:5650")
        
        # SearXNG API 표준 헤더 (JSON 응답 요청)
        headers = {
            "User-Agent": "Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36",
            "Accept": "application/json, text/javascript, */*; q=0.01",
            "Accept-Language": "ko-KR,ko;q=0.9,en-US;q=0.8,en;q=0.7",
            "Accept-Encoding": "gzip, deflate",
            "Referer": f"{base_url}/",
        }

        results: List[Dict[str, Any]] = []
        timeout = aiohttp.ClientTimeout(total=12)

        logger.info(f"[SearXNG] base_url={base_url} query='{query}' 최대 {max_results}건")

        async with aiohttp.ClientSession(timeout=timeout, headers=headers) as session:
            # SearXNG 공식 API 파라미터 (JSON 응답 요청)
            params = {
                "q": query,                    # 필수: 검색 쿼리
                "language": "ko",              # 한국어 우선
                "pageno": "1",                 # 첫 번째 페이지
                "time_range": "",              # 시간 범위 (빈값 = 전체)
                "format": "json",              # JSON 응답 요청
                "engines": "google,duckduckgo,bing,naver",  # 사용 엔진
                "categories": "general,news",       # 일반 웹 검색
            }
            
            try:
                search_url = urljoin(base_url, "/search")
                logger.info(f"[SearXNG] GET 요청: {search_url}")
                logger.info(f"[SearXNG] 파라미터: {params}")
                
                async with session.get(search_url, params=params, allow_redirects=True) as resp:
                    logger.info(f"[SearXNG] 응답 상태: {resp.status}")
                    
                    if resp.status != 200:
                        logger.error(f"[SearXNG] API 호출 실패 - HTTP {resp.status}")
                        return []
                    
                    # Content-Type 확인
                    content_type = resp.headers.get("Content-Type", "")
                    logger.info(f"[SearXNG] Content-Type: {content_type}")
                    
                    # 응답 텍스트 읽기
                    text = await resp.text(errors="ignore")
                    logger.info(f"[SearXNG] 응답 길이: {len(text)}자")
                    
                    # 응답 유효성 검사
                    if not text or len(text.strip()) == 0:
                        logger.error("[SearXNG] 빈 응답")
                        return []
                    
                    # JSON 파싱 시도
                    if "application/json" in content_type or text.strip().startswith("{"):
                        try:
                            json_data = json.loads(text)
                            logger.info("[SearXNG] JSON 응답 파싱 시도")
                            
                            # JSON 파싱 메서드 사용
                            results = self._extract_searxng_json(json_data)
                            
                            if results:
                                logger.info(f"[SearXNG] JSON 파싱 성공: {len(results)}건")
                            else:
                                logger.warning("[SearXNG] JSON 파싱 결과 없음, HTML fallback 시도")
                                # HTML fallback (format=json이 무시된 경우)
                                results = self._extract_searxng_results(text, base_url)
                                
                        except json.JSONDecodeError as e:
                            logger.warning(f"[SearXNG] JSON 파싱 실패: {e}, HTML fallback 시도")
                            # HTML 파싱으로 fallback
                            results = self._extract_searxng_results(text, base_url)
                    else:
                        # HTML 응답인 경우 기존 파싱 사용
                        logger.info("[SearXNG] HTML 응답 - HTML 파싱 사용")
                        results = self._extract_searxng_results(text, base_url)
                    
                    logger.info(f"[SearXNG] 파싱 결과: {len(results)}건")
                    
                    # 상위 결과 미리보기
                    for i, result in enumerate(results[:min(3, len(results))]):
                        title = result.get('title', 'NO_TITLE')[:40]
                        url = result.get('url', 'NO_URL')[:50]
                        logger.info(f"[SearXNG] 결과 {i+1}: {title}... | {url}...")
                        
            except asyncio.TimeoutError:
                logger.error("[SearXNG] 요청 타임아웃")
                return []
            except aiohttp.ClientError as e:
                logger.error(f"[SearXNG] 네트워크 오류: {e}")
                return []
            except Exception as e:
                logger.error(f"[SearXNG] 예외 발생: {e}")
                if not silent:
                    logger.error(f"[SearXNG] 상세 오류:\n{traceback.format_exc()}")
                return []

        # 결과 정리 및 중복 제거
        unique_results = []
        seen_urls = set()
        seen_titles = set()
        
        for result in results:
            title = result.get("title", "").strip()
            url = result.get("url", "").strip()
            
            # 기본 유효성 검사
            if not title or not url:
                continue
                
            # URL 유효성 검사
            if not url.startswith(("http://", "https://")):
                continue
                
            title_key = title.lower()
            
            # 중복 제거 (URL과 제목 기준)
            if url not in seen_urls and title_key not in seen_titles:
                seen_urls.add(url)
                seen_titles.add(title_key)
                unique_results.append(result)
                
                if len(unique_results) >= max_results:
                    break

        final_results = unique_results[:max_results]

        if not silent:
            if final_results:
                logger.info(f"✅ SearXNG 검색 완료: {len(final_results)}개 유효 결과")
                desc_count = len([r for r in final_results if r.get("description")])
                logger.info(f"📄 설명 포함 결과: {desc_count}/{len(final_results)}개")
            else:
                logger.warning("⚠️ SearXNG 검색 결과 없음")

        return final_results

    def _clean_text_for_search(self, text: str) -> str:
        """검색용 텍스트 정리"""
        if not text:
            return ""
        return text.strip().replace('\n', ' ').replace('\r', ' ')
    
    def _is_valid_url(self, url: str) -> bool:
        """URL 유효성 검사"""
        if not url:
            return False
        return url.startswith(('http://', 'https://')) and len(url) > 10
    


# =============================================================================
# API 모델들
# =============================================================================

class WebSearchRequest(BaseModel):
    """웹검색 요청"""
    query: str = Field(..., description="검색 쿼리")
    max_results: int = Field(default=10, description="최대 결과 수", ge=1, le=20)
    engine: str = Field(default="auto", description="검색 엔진 (auto: SearXNG 우선, 실패시 DuckDuckGo fallback)")
    use_cache: bool = Field(default=True, description="캐시 사용 여부")

class WebSearchResponse(BaseModel):
    """웹검색 응답"""
    success: bool
    results: List[Dict[str, Any]]
    query: str
    total_results: int
    execution_time: float
    cached: bool = False
    engine_stats: Optional[Dict[str, int]] = None
    error: Optional[str] = None
    optimization_info: Optional[Dict[str, Any]] = None

class HealthResponse(BaseModel):
    """상태 확인 응답"""
    status: str
    service: str
    version: str
    timestamp: str
    uptime: str
    stats: Dict[str, Any]

# =============================================================================
# 캐시 시스템
# =============================================================================

class WebSearchCache:
    """초고속 메모리 캐시"""
    
    def __init__(self):
        self.cache = {}
        self.max_size = 2000
        self.ttl = 1800  # 30분
        
    def _generate_key(self, query: str, max_results: int, engine: str) -> str:
        """캐시 키 생성"""
        return f"{query.strip().lower()}:{max_results}:{engine}"
    
    def get(self, query: str, max_results: int, engine: str) -> Optional[List[Dict]]:
        """캐시에서 결과 조회"""
        key = self._generate_key(query, max_results, engine)
        item = self.cache.get(key)
        
        if not item:
            return None
            
        # TTL 체크
        if time.time() - item['timestamp'] > self.ttl:
            del self.cache[key]
            return None
            
        server_stats["cache_hits"] += 1
        return item['data']
    
    def set(self, query: str, max_results: int, engine: str, results: List[Dict]):
        """캐시에 결과 저장"""
        if len(self.cache) >= self.max_size:
            # 가장 오래된 항목 제거
            oldest_key = min(self.cache.keys(), key=lambda k: self.cache[k]['timestamp'])
            del self.cache[oldest_key]
            
        key = self._generate_key(query, max_results, engine)
        self.cache[key] = {
            'data': results,
            'timestamp': time.time()
        }
        
        server_stats["cache_misses"] += 1
    
    def clear(self):
        """캐시 전체 삭제"""
        self.cache.clear()

# =============================================================================
# 초고속 검색 실행 함수
# =============================================================================

async def execute_ultra_fast_search(query: str, max_results: int = 10, engine: str = "auto", use_cache: bool = True) -> Dict[str, Any]:
    """초고속 웹검색 실행 (utils.py + monkey patch)"""
    start_time = time.time()
    
    try:
        # 캐시 확인
        cached_results = None
        if use_cache and websearch_cache:
            cached_results = websearch_cache.get(query, max_results, engine)
            if cached_results:
                execution_time = time.time() - start_time
                logger.info(f"캐시 히트: {len(cached_results)}개 결과 ({execution_time*1000:.1f}ms)")
                
                return {
                    "success": True,
                    "results": cached_results,
                    "query": query,
                    "total_results": len(cached_results),
                    "execution_time": execution_time,
                    "cached": True,
                    "engine_stats": None,
                    "error": None,
                    "optimization_info": {
                        "cache_hit": True,
                        "monkey_patch_used": False,
                        "performance": "⚡ ULTRA_FAST"
                    }
                }
        
        # 드라이버 풀이 초기화되지 않은 경우
        if not driver_pool or not driver_pool.initialized:
            raise RuntimeError("드라이버 풀이 초기화되지 않았습니다")
        
        # 웹검색 엔진으로 검색 수행
        web_search_engine = WebSearch(driver_pool)
        results = await web_search_engine.search(query, max_results, engine)
        
        # 결과 확인 로그 추가
        if not results or len(results) == 0:
            logger.error(f"❌ WebSearch.search() 결과가 비어있음: {len(results) if results else 0}개")
        else:
            logger.info(f"✅ WebSearch.search() 결과: {len(results)}개")
        
        # 엔진별 결과 통계 (간소화)
        searxng_count = len([r for r in results if r.get('engine') == 'searxng'])
        duckduckgo_count = len([r for r in results if r.get('engine') == 'duckduckgo'])
        
        engine_stats = {
            "searxng": searxng_count,
            "duckduckgo": duckduckgo_count,
            "total": len(results)
        }
        
        # 캐시에 저장
        if use_cache and websearch_cache and results:
            websearch_cache.set(query, max_results, engine, results)
        
        execution_time = time.time() - start_time
        
        # 평균 응답 시간 업데이트
        if server_stats["total_requests"] > 0:
            server_stats["avg_response_time"] = (
                (server_stats["avg_response_time"] * (server_stats["total_requests"] - 1) + execution_time) /
                server_stats["total_requests"]
            )
        
        return {
            "success": True,
            "results": results,
            "query": query,
            "total_results": len(results),
            "execution_time": execution_time,
            "cached": False,
            "engine_stats": engine_stats,
            "error": None,
            "optimization_info": {
                "cache_hit": False,
                "monkey_patch_used": True,
                "driver_reuse_count": server_stats["driver_reuse_count"],
                "available_drivers": len(driver_pool.available_drivers),
                "performance": "⚡ ULTRA_FAST" if execution_time < 3 else ("🚀 FAST" if execution_time < 6 else "🐢 SLOW")
            }
        }
        
    except Exception as e:
        execution_time = time.time() - start_time
        error_msg = f"초고속 검색 실행 오류: {str(e)}"
        logger.error(error_msg)
        logger.error(f"상세 오류:\n{traceback.format_exc()}")
        
        return {
            "success": False,
            "results": [],
            "query": query,
            "total_results": 0,
            "execution_time": execution_time,
            "cached": False,
            "engine_stats": None,
            "error": error_msg,
            "optimization_info": {
                "cache_hit": False,
                "monkey_patch_used": False,
                "performance": "❌ ERROR"
            }
        }

# =============================================================================
# FastAPI 앱 설정
# =============================================================================

@asynccontextmanager
async def lifespan(app: FastAPI):
    """앱 라이프사이클 관리"""
    global websearch_cache, driver_pool
    
    # 시작
    logger.info("🚀 AIRUN 초고속 웹검색 서비스 시작 (Monkey Patch 최적화)")
    server_stats["start_time"] = datetime.now()
    
    # 캐시 초기화
    websearch_cache = WebSearchCache()
    
    # 초고속 드라이버 풀 초기화 (Monkey Patch 적용!)
    driver_pool = UltraFastDriverPool(pool_size=5)  # 3에서 5로 증가하여 병렬 처리 개선
    if not driver_pool.initialize():
        logger.error("❌ 초고속 드라이버 풀 초기화 실패 - 서비스를 종료합니다")
        sys.exit(1)
    
    logger.info("⚡ AIRUN 초고속 웹검색 서비스 준비 완료 (Monkey Patch 최적화)")
    
    yield
    
    # 종료
    logger.info("🛑 AIRUN 초고속 웹검색 서비스 종료")
    if driver_pool:
        driver_pool.cleanup()

app = FastAPI(
    title="AIRUN 초고속 웹검색 서비스 (Monkey Patch)",
    description="utils.py Monkey Patch로 초고속 웹검색 API",
    version="5.0.0",
    lifespan=lifespan
)

# CORS 설정
app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

# =============================================================================
# API 엔드포인트
# =============================================================================

@app.get("/health", response_model=HealthResponse)
async def health_check():
    """서비스 상태 확인"""
    uptime = "0초"
    if server_stats["start_time"]:
        uptime_delta = datetime.now() - server_stats["start_time"]
        uptime = str(uptime_delta).split('.')[0]
    
    return HealthResponse(
        status="healthy",
        service="airun-websearch-server-monkeypatch",
        version="5.0.0",
        timestamp=datetime.now().isoformat(),
        uptime=uptime,
        stats={
            "total_requests": server_stats["total_requests"],
            "successful_searches": server_stats["successful_searches"],
            "failed_searches": server_stats["failed_searches"],
            "cache_hits": server_stats["cache_hits"],
            "cache_misses": server_stats["cache_misses"],
            "cache_hit_rate": (
                server_stats["cache_hits"] / max(1, server_stats["cache_hits"] + server_stats["cache_misses"])
            ) * 100,
            "avg_response_time": server_stats["avg_response_time"],
            "driver_reuse_count": server_stats["driver_reuse_count"],
            "driver_pool_initialized": driver_pool.initialized if driver_pool else False,
            "available_drivers": len(driver_pool.available_drivers) if driver_pool else 0,
            "total_drivers": len(driver_pool.drivers) if driver_pool else 0,
            "cache_size": len(websearch_cache.cache) if websearch_cache else 0
        }
    )

@app.post("/search", response_model=WebSearchResponse)
async def search_web(request: WebSearchRequest):
    """고속 웹검색 실행 (Monkey Patch 최적화)"""
    try:
        server_stats["total_requests"] += 1
        
        logger.info(f"웹검색 요청: '{request.query}'")
        
        # 초고속 검색 실행
        result = await execute_ultra_fast_search(
            query=request.query,
            max_results=request.max_results,
            engine=request.engine,
            use_cache=request.use_cache
        )
        
        if result["success"]:
            server_stats["successful_searches"] += 1
            perf_icon = result.get("optimization_info", {}).get("performance", "")
            logger.info(f"✅ 검색 완료: {result['total_results']}개 결과 ({result['execution_time']:.3f}초) {perf_icon}")
            if result.get("cached"):
                logger.info("⚡ 캐시된 결과 사용")
        else:
            server_stats["failed_searches"] += 1
            logger.error(f"❌ 검색 실패: {result.get('error', '알 수 없는 오류')}")
        
        return WebSearchResponse(**result)
        
    except Exception as e:
        server_stats["failed_searches"] += 1
        error_msg = f"초고속 웹검색 API 오류: {str(e)}"
        logger.error(error_msg)
        logger.error(f"상세 오류:\n{traceback.format_exc()}")
        
        raise HTTPException(status_code=500, detail=error_msg)

@app.post("/cache/clear")
async def clear_cache():
    """캐시 삭제 - 빠른 응답을 위한 백그라운드 처리"""
    try:
        # 즉시 응답하고 백그라운드에서 처리
        asyncio.create_task(background_cache_clear())
        logger.info("🗑️ 웹검색 캐시 삭제 요청 접수")
        return {"message": "캐시 삭제가 요청되었습니다", "success": True}
    except Exception as e:
        error_msg = f"캐시 삭제 요청 실패: {str(e)}"
        logger.error(error_msg)
        raise HTTPException(status_code=500, detail=error_msg)

async def background_cache_clear():
    """백그라운드에서 실제 캐시 삭제 수행"""
    try:
        if websearch_cache:
            websearch_cache.clear()
        logger.info("🗑️ 웹검색 캐시 삭제 완료 (백그라운드)")
    except Exception as e:
        logger.error(f"백그라운드 캐시 삭제 오류: {e}")

@app.get("/stats")
async def get_stats():
    """서비스 통계"""
    try:
        uptime = "0초"
        if server_stats["start_time"]:
            uptime_delta = datetime.now() - server_stats["start_time"]
            uptime = str(uptime_delta).split('.')[0]
            
        return {
            "service": "airun-websearch-server-monkeypatch",
            "version": "5.0.0",
            "uptime": uptime,
            "stats": server_stats,
            "cache_size": len(websearch_cache.cache) if websearch_cache else 0,
            "driver_pool_status": {
                "initialized": driver_pool.initialized if driver_pool else False,
                "total_drivers": len(driver_pool.drivers) if driver_pool else 0,
                "available_drivers": len(driver_pool.available_drivers) if driver_pool else 0,
                "busy_drivers": len(driver_pool.drivers) - len(driver_pool.available_drivers) if driver_pool else 0,
                "reuse_count": server_stats["driver_reuse_count"]
            },
            "optimization_features": [
                "✅ SearXNG 메인 검색 엔진",
                "✅ DuckDuckGo 자동 fallback",
                "✅ 초고속 메모리 캐시",
                "✅ 간소화된 검색 로직",
                "🎯 단순하고 안정적인 구조",
                "⚡ 빠른 응답 시간"
            ]
        }
    except Exception as e:
        error_msg = f"통계 조회 실패: {str(e)}"
        logger.error(error_msg)
        raise HTTPException(status_code=500, detail=error_msg)

# =============================================================================
# 메인 함수
# =============================================================================

def main():
    """서버 시작"""
    
    # 시그널 핸들러 설정
    def signal_handler(signum, frame):
        logger.info(f"🛑 시그널 {signum} 수신 - 서버 종료 중...")
        if 'driver_pool' in globals() and driver_pool:
            driver_pool.cleanup()
        sys.exit(0)
    
    signal.signal(signal.SIGTERM, signal_handler)
    signal.signal(signal.SIGINT, signal_handler)
    
    try:
        # 환경 설정
        host = os.getenv('WEBSEARCH_HOST', '0.0.0.0')
        port = int(os.getenv('WEBSEARCH_PORT', 5610))
        
        logger.info(f"⚡ 초고속 서버 시작: http://{host}:{port}")
        
        # Uvicorn 서버 실행
        uvicorn.run(
            app,
            host=host,
            port=port,
            log_level="info",
            access_log=True,
            workers=1  # Monkey Patch 관리를 위해 단일 워커 사용
        )
        
    except Exception as e:
        logger.error(f"❌ 서버 시작 실패: {e}")
        sys.exit(1)

if __name__ == "__main__":
    main() 
